package org.lsst.ccs.subsystem.refrig.ui;

import java.io.PrintWriter;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;
import javax.swing.JComponent;
import javax.swing.JFrame;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.BusMessage;
import org.lsst.ccs.bus.CommandReply;
import org.lsst.ccs.bus.MetadataStatus;
import org.lsst.ccs.bus.ModuleInvokerCommand;
import org.lsst.ccs.bus.TrendingStatus;
import org.lsst.ccs.subsystem.refrig.data.RefrigFullState;
import org.lsst.ccs.subsystem.refrig.status.RefrigStateStatus;

/**
 *
 * @author max
 */
public class RefrigGUISubsystem extends Subsystem {

    private static final String refrig_dest = "ccs-refrig";
    private PrintWriter out = new PrintWriter(System.out, true);
    private RefrigAssembly assembly;
    private RefrigTrendingTable table;
    private RefrigControlPanel panel;
    private long lastHeartBeat = 0, lastStateRequest = 0;
    private boolean enabled = false;
    String stateCorrelId;

    public RefrigGUISubsystem() {
        assembly = new RefrigAssembly(this);
        table = assembly.getTrendingTable();
        panel = assembly.getControlPanel();
    }

    public void initGui() {
        setName("RefrigerationGUIModule");
        setListenToStatus(true);
        setStatusBroadcastPeriod(0);
        start();

        Timer t = new Timer();
        t.scheduleAtFixedRate(new TimerTask() {

            @Override
            public void run() {
                long delta = (System.currentTimeMillis() - lastHeartBeat) / 1000;
                if (enabled && delta > 10) {
                    enabled = false;
                    panel.disableSystem();
                }
            }
        }, new Date(), 5000);
    }

    public JComponent getGuiLayout() {
        return assembly;
    }

    public void resetGui() {
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        System.setProperty("java.naming.factory.initial",
                           "com.sun.enterprise.naming.SerialInitContextFactory");
        System.setProperty("java.naming.factory.url.pkgs",
                           "com.sun.enterprise.naming");
        System.setProperty("java.naming.factory.state",
                           "com.sun.corba.ee.impl.presentation.rmi.JNDIStateFactoryImpl");
        System.setProperty("org.omg.CORBA.ORBInitialHost",
                           args.length >= 1 ? args[0] : "localhost");
        System.setProperty("org.omg.CORBA.ORBInitialPort", "3700");

//        Properties props = FrameworkUtils.loadProperties(FrameworkUtils.getDefaultJndiProperties(),RefrigGUISubsystem.class.getClass(),"jndi.properties");

        RefrigGUISubsystem t = new RefrigGUISubsystem();

        JFrame frame = new JFrame("Refrigeration Subsystem");
        frame.setContentPane(t.getGuiLayout());
        frame.pack();
        frame.setVisible(true);
        t.initGui();
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

    public void setPrintWriter(PrintWriter out) {
        this.out = out;
    }

    @Override
    public void onReply(CommandReply s) {
        //FIXME temporary fix to prevent multiple GUIs from interfering with each other.
        //See https://jira.slac.stanford.edu/browse/LSSTCCS-9
        if (s.getCorrelId().equals(stateCorrelId)) {
            RefrigFullState r = (RefrigFullState)s.getReply();
            log.debug("RefrigGui received RefrigFullStateStatus reply");
            table.updateTableModel(r);
            panel.updateControlPanel(r.getRefrigState());
            panel.updateTime();
            enabled = true;
        }
    }

    @Override
    public void onStatus(BusMessage s) {
        if (!s.getOrigin().equals(refrig_dest)) return;

        lastHeartBeat = s.getTimeStamp();

        if (!enabled) {
            long currentTime = System.currentTimeMillis();
            if (currentTime - lastStateRequest > 2000) {
                lastStateRequest = currentTime;
                ModuleInvokerCommand cmd = new ModuleInvokerCommand("getFullState");
                sendCommand(cmd);
                stateCorrelId = cmd.getCorrelId();
            }
            return;
        }

        if (s instanceof TrendingStatus) {
            log.debug("RefrigGui received TrendingStatus message");
            table.updateTableValue((TrendingStatus)s);
            panel.updateTime();
        }
        else if (s instanceof MetadataStatus) {
            log.debug("RefrigGui received MetadataStatus message");
            table.updateTableValue((MetadataStatus)s);
        }
        else if (s instanceof RefrigStateStatus) {
            log.debug("RefrigGui received RefrigStateStatus message");
            table.updateTableValue(((RefrigStateStatus)s).getRefrigState());
            panel.updateControlPanel(((RefrigStateStatus)s).getRefrigState());
        }
    }

    public void submitLimit(int index, double value, boolean isLowLimit) {
        Integer arg1 = index;
        Double arg2 = value;
        String command = isLowLimit ? "setLowLimit" : "setHighLimit";
        sendCommand(new ModuleInvokerCommand(command, arg1, arg2));
    }

    public void setMainPowerStatus(boolean on) {
        sendCommand(new ModuleInvokerCommand("setMainPowerEnable", on ? 1 : 0));
    }

    public void setLoadTripStatus(boolean on) {
        sendCommand(new ModuleInvokerCommand("setLoadTripEnable", on ? 1 : 0));
    }

    public void setTickMillis(int value) {
        sendCommand(new ModuleInvokerCommand("setTickMillis", value));
    }

    public void saveConfiguration() {
        sendCommand(new ModuleInvokerCommand("saveConfiguration"));
    }

    private void sendCommand(ModuleInvokerCommand cmd) {
        cmd.setDestination(refrig_dest);
        updateCurrentSubsystem();
        fac.sendCommand(cmd);
    }

}
