package org.lsst.ccs.subsystem.refrig.data;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import org.lsst.ccs.bus.annotations.SkipEncoding;
import org.lsst.ccs.subsystem.refrig.constants.CompConditions;
import org.lsst.ccs.subsystem.refrig.constants.CompLatches;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;
import org.lsst.ccs.subsystem.refrig.constants.CompSwitches;
import org.lsst.ccs.subsystem.refrig.constants.CompTypes;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.SwCondState;

/**
 *  Contains the state of a cold compressor
 *
 *  @author Owen Saxton
 */
@SkipEncoding
public class CompState implements Serializable {
    
    public static final String KEY = "CompressorState";

    private final int type;
    private final Collection validSwitches, validLatches, validConditions, validSwConditions;
    private String name;    // The compressor name
    private int index;      // The compressor index
    private CompressorState compState = CompressorState.OFFLINE;  // The compressor state
    private final SwitchState[] switchState = new SwitchState[CompSwitches.NUM_SWITCHES];  // Switch state array
    private final LatchState[] latchState = new LatchState[CompLatches.NUM_LATCHES];  // Latched PLC conditions array
    private final ConditionState[] conditionState = new ConditionState[CompConditions.NUM_CONDITIONS];  // PLC conditions array
    private final SwCondState[] swCondState = new SwCondState[CompSwConds.NUM_SW_CONDITIONS];  // Software conditions array
    private double bypassValve;    // Cold system bypass va;ve position
    private double vfdFrequency;   // Cold syaytem VFD frequency

    public CompState(int type) {
        this.type = type;
        validSwitches = type == CompTypes.TYPE_COLD ? ColdCompState.SWITCHES : CryoCompState.SWITCHES;
        validLatches = type == CompTypes.TYPE_COLD ? ColdCompState.LATCHES : CryoCompState.LATCHES;
        validConditions = type == CompTypes.TYPE_COLD ? ColdCompState.CONDITIONS : CryoCompState.CONDITIONS;
        validSwConditions = type == CompTypes.TYPE_COLD ? ColdCompState.SW_CONDITIONS : CryoCompState.SW_CONDITIONS;
        Arrays.fill(switchState, SwitchState.OFFLINE);
        Arrays.fill(latchState, LatchState.OFFLINE);
        Arrays.fill(conditionState, ConditionState.OFF);
        Arrays.fill(swCondState, SwCondState.CLEAR);
    }

    public int getType() {
        return type;
    }

    public Collection<Integer> getValidSwitches() {
        return validSwitches;
    }

    public Collection<Integer> getValidLatches() {
        return validLatches;
    }

    public Collection<Integer> getValidConditions() {
        return validConditions;
    }

    public Collection<Integer> getValidSwConditions() {
        return validSwConditions;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getName() {
        return name;
    }

    public void setIndex(int index) {
        this.index = index;
    }

    public int getIndex() {
        return index;
    }

    public void setCompressorState(CompressorState value) {
        compState = value;
    }

    public CompressorState getCompressorState() {
        return compState;
    }

    public void setSwitchState(int sw, SwitchState state) {
        switchState[sw] = state;
    }

    public SwitchState getSwitchState(int sw) {
        return switchState[sw];
    }

    public void setLatchState(int latch, LatchState state) {
        latchState[latch] = state;
    }

    public LatchState getLatchState(int latch) {
        return latchState[latch];
    }

    public void setConditionState(int cond, ConditionState state) {
        conditionState[cond] = state;
    }

    public ConditionState getConditionState(int cond) {
        return conditionState[cond];
    }

    public void setSwConditionState(int cond, SwCondState state) {
        swCondState[cond] = state;
    }

    public SwCondState getSwConditionState(int cond) {
        return swCondState[cond];
    }

    public void setBypassValve(double posn) {
        bypassValve = posn;
    }

    public double getBypassValve() {
        return bypassValve;
    }

    public void setVfdFrequency(double freq) {
        vfdFrequency = freq;
    }

    public double getVfdFrequency() {
        return vfdFrequency;
    }

    @Override
    public String toString() {
        return super.toString();  // Replace this
    }

    private static final long serialVersionUID = 6286834552639649094L;
}
