package org.lsst.ccs.subsystem.refrig.data;

import java.io.Serializable;

/**
 *  Contains the state of the refrigeration system
 *
 *  @author Owen Saxton
 */
public class RefrigState1 implements Serializable {

    public static final String KEY = "RefrigState";

    public static final int
        MAX_COMPRESSORS          = 8,
        POWER_OFF                = 0,
        POWER_ON                 = 1,
        POWER_TRIPPED            = 2,
        CONTROL_OFF              = 0,
        CONTROL_ON               = 1,
        CONTROL_AUTO             = 2,
        CONTROL_NONE             = 3;

    private static final int
        NUM_CMPR_POWER_BITS      = 2,
        NUM_HEATER_POWER_BITS    = 2,
        HEATER_POWER_BIT_NUM     = 2 * MAX_COMPRESSORS,
        NUM_HEATER_CONTROL_BITS  = 2,
        HEATER_CONTROL_BIT_NUM   = HEATER_POWER_BIT_NUM + NUM_HEATER_POWER_BITS,
        ALARM_DISABLED_BIT_NUM   = HEATER_CONTROL_BIT_NUM + NUM_HEATER_CONTROL_BITS;

    private int    systemState,        // Mask of state bits
                   numCmprs,           // The number of compressors
                   tickMillis;         // Data status update period (ms)
    private double heaterPower,        // Heater power set point
                   loadTemp;           // Heat load temperature set point
    
 
    void setSystemState(int state) {
        systemState = state;
    }

    int getSystemState() {
        return systemState;
    }

    public void setCmprPowerState(int cmpr, int state) {
        setSystemStateBits(NUM_CMPR_POWER_BITS, NUM_CMPR_POWER_BITS * cmpr, state);
    }

    public int getCmprPowerState(int cmpr) {
        return getSystemStateBits(NUM_CMPR_POWER_BITS, NUM_CMPR_POWER_BITS * cmpr);
    }

    public void setHeaterPowerState(int state) {
        setSystemStateBits(NUM_HEATER_POWER_BITS, HEATER_POWER_BIT_NUM, state);
    }

    public int getHeaterPowerState() {
        return getSystemStateBits(NUM_HEATER_POWER_BITS, HEATER_POWER_BIT_NUM);
    }

    public void setHeaterControlState(int state) {
        setSystemStateBits(NUM_HEATER_CONTROL_BITS, HEATER_CONTROL_BIT_NUM, state);
    }

    public int getHeaterControlState() {
        return getSystemStateBits(NUM_HEATER_CONTROL_BITS, HEATER_CONTROL_BIT_NUM);
    }

    public void setAlarmDisabled(boolean value) {
        setSystemStateBit(ALARM_DISABLED_BIT_NUM, value);
    }

    public boolean isAlarmDisabled() {
        return isSystemStateBitSet(ALARM_DISABLED_BIT_NUM);
    }

    public void setNumCmprs(int value) {
        numCmprs = value;
    }

    public int getNumCmprs() {
        return numCmprs;
    }

    public void setTickMillis(int value) {
        tickMillis = value;
    }

    public int getTickMillis() {
        return tickMillis;
    }

    public void setHeaterPower(double value) {
        heaterPower = value;
    }

    public double getHeaterPower() {
        return heaterPower;
    }

    public double getLoadTemp() {
        return loadTemp;
    }

    public void setLoadTemp(double value) {
        loadTemp = value;
    }

    private void setSystemStateBits(int nBits, int bitNum, int value) {
        int mask = (1 << nBits) - 1;
        systemState = (systemState & ~(mask << bitNum)) | ((value & mask) << bitNum);
    }

    private int getSystemStateBits(int nBits, int bitNum) {
        return (systemState >> bitNum) & ((1 << nBits) - 1);
    }

    private void setSystemStateBit(int bitNum, boolean value) {
        if (value) {
            systemState |= (1 << bitNum);
        }
        else {
            systemState &= ~(1 << bitNum);
        }
    }

    private boolean isSystemStateBitSet(int bitNum) {
        return ((systemState >> bitNum) & 1) != 0;
    }

    private static final long serialVersionUID = 6334818327504457789L;
}
