package org.lsst.ccs.subsystem.refrig.data;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.SwCondState;

/**
 *  Contains the state of a cold compressor
 *
 *  @author Owen Saxton
 */
public class CompState implements Serializable {
    
    public static final int
        TYPE_COLD = 0,
        TYPE_CRYO = 1;
    public static final int
        SW_ENABLE          = 0,
        SW_HEATER          = 1,
        SW_LIGHTS          = 2,
        SW_ORIFICE_VALVE   = 3,
        SW_COOLANT_VALVE   = 4,
        SW_BYPASS_VALVE    = 5,
        SW_SURGE_HEATER    = 6,
        NUM_SWITCHES       = 7;
    public static final int
        LATCH_DISCHARGE_TEMP  = 0,
        LATCH_SUCTION_TEMP    = 1,
        LATCH_LIQUID_TEMP     = 2,
        LATCH_DISCHARGE_PRESS = 3,
        LATCH_POWER           = 4,
        LATCH_EXT_PERMIT      = 5,
        LATCH_SMOKE_DETC      = 6,
        LATCH_SENSORS_VALID   = 7,
        LATCH_AFTER_COOLER    = 8,
        LATCH_OIL_LEVEL       = 9,
        NUM_LATCHES           = 10;
    public static final int
        COND_KEYSWITCH_ON     = 0,
        COND_CMP_ENABLED      = 1,
        COND_CMP_WAITING      = 2,
        COND_CMP_POWERED      = 3,
        COND_DISC_TEMP_VALID  = 4,
        COND_SUCT_TEMP_VALID  = 5,
        COND_LIQD_TEMP_VALID  = 6,
        COND_DISC_PRESS_VALID = 7,
        COND_SUCT_PRESS_VALID = 8,
        COND_CURRENT_VALID    = 9,
        COND_VOLTAGE_VALID    = 10,
        COND_LATCHES_CLEAR    = 11,
        COND_POWER_LED        = 12,
        COND_CURR_SENSOR_ERR  = 13,
        COND_CMP_ON_8HRS      = 14,
        COND_OIL_LEVEL_VALID  = 15,
        NUM_CONDITIONS        = 16;
    public static final int
        SWC_DISC_PRESS     = 0,
        SWC_DISC_TEMP      = 1,
        SWC_SUCT_TEMP      = 2,
        SWC_CMPR_POWER     = 3,
        SWC_PHASE_SEP_TEMP = 4,
        SWC_PLATE_TEMP     = 5,
        SWC_OIL_LEVEL      = 6,
        SWC_LIQUID_TEMP    = 7,
        NUM_SW_CONDITIONS  = 8;

    public static final String KEY = "CompressorState";

    private final int type;
    private final Collection validSwitches, validLatches, validConditions, validSwConditions;
    private String name;    // The compressor name
    private int index;      // The compressor index
    private CompressorState compState = CompressorState.OFFLINE;  // The compressor state
    private final SwitchState[] switchState = new SwitchState[NUM_SWITCHES];  // Switch state array
    private final LatchState[] latchState = new LatchState[NUM_LATCHES];  // Latched PLC conditions array
    private final ConditionState[] conditionState = new ConditionState[NUM_CONDITIONS];  // PLC conditions array
    private final SwCondState[] swCondState = new SwCondState[NUM_SW_CONDITIONS];  // Software conditions array

    public CompState(int type) {
        this.type = type;
        validSwitches = type == TYPE_COLD ? ColdCompState.SWITCHES : CryoCompState.SWITCHES;
        validLatches = type == TYPE_COLD ? ColdCompState.LATCHES : CryoCompState.LATCHES;
        validConditions = type == TYPE_COLD ? ColdCompState.CONDITIONS : CryoCompState.CONDITIONS;
        validSwConditions = type == TYPE_COLD ? ColdCompState.SW_CONDITIONS : CryoCompState.SW_CONDITIONS;
        Arrays.fill(switchState, SwitchState.OFFLINE);
        Arrays.fill(latchState, LatchState.OFFLINE);
        Arrays.fill(conditionState, ConditionState.OFF);
        Arrays.fill(swCondState, SwCondState.CLEAR);
    }

    public int getType() {
        return type;
    }

    public Collection<Integer> getValidSwitches() {
        return validSwitches;
    }

    public Collection<Integer> getValidLatches() {
        return validLatches;
    }

    public Collection<Integer> getValidConditions() {
        return validConditions;
    }

    public Collection<Integer> getValidSwConditions() {
        return validSwConditions;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getName() {
        return name;
    }

    public void setIndex(int index) {
        this.index = index;
    }

    public int getIndex() {
        return index;
    }

    public void setCompressorState(CompressorState value) {
        compState = value;
    }

    public CompressorState getCompressorState() {
        return compState;
    }

    public void setSwitchState(int sw, SwitchState state) {
        switchState[sw] = state;
    }

    public SwitchState getSwitchState(int sw) {
        return switchState[sw];
    }

    public void setLatchState(int latch, LatchState state) {
        latchState[latch] = state;
    }

    public LatchState getLatchState(int latch) {
        return latchState[latch];
    }

    public void setConditionState(int cond, ConditionState state) {
        conditionState[cond] = state;
    }

    public ConditionState getConditionState(int cond) {
        return conditionState[cond];
    }

    public void setSwConditionState(int cond, SwCondState state) {
        swCondState[cond] = state;
    }

    public SwCondState getSwConditionState(int cond) {
        return swCondState[cond];
    }

    @Override
    public String toString() {
        return super.toString();  // Replace this
    }

    private static final long serialVersionUID = -5419137656635568916L;
}
