package org.lsst.ccs.subsystem.refrig.data;

import java.io.Serializable;
import java.util.Arrays;
import org.lsst.ccs.subsystem.refrig.constants.AuxHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.TrimHeaterState;

/**
 *  Contains the state of the thermal system
 *
 *  @author Owen Saxton
 */
public class ThermalState implements Serializable {

    public static final String KEY = "ThermalState";

    public static final int
        TRIM_HEATER_COLD     = 0,
        TRIM_HEATER_CRYO     = 1,
        NUM_TRIM_HEATERS     = 2,
        COLD_SECTION_MINUS_Y = 0,
        COLD_SECTION_PLUS_Y  = 1,
        NUM_COLD_SECTIONS    = 2,
        AUX_HEATER_MINUS_Y   = 0,
        AUX_HEATER_PLUS_Y    = 1,
        AUX_HEATER_CENTER    = 2,
        NUM_AUX_HEATERS      = 3;

    private int tickMillis;         // Data status update period (ms)
    private final TrimHeaterState[] trimHeaterCtrlState = new TrimHeaterState[NUM_TRIM_HEATERS];  // Trim heater state
    private final double[] trimHeaterPower = new double[NUM_TRIM_HEATERS];          // Trim heater power set point
    private final double[] plateTemperature = new double[NUM_TRIM_HEATERS];         // Plate temperature set point
    private final boolean[] coldTrimEnabled = new boolean[NUM_COLD_SECTIONS];       // Cold trim heater section enables
    private final AuxHeaterState[] auxHeaterState = new AuxHeaterState[NUM_AUX_HEATERS];  // Aux heater state
    private final double[] auxHeaterPower = new double[NUM_AUX_HEATERS];                  // Aux heater power set point

    public ThermalState() {
        Arrays.fill(auxHeaterState, AuxHeaterState.OFFLINE);
        Arrays.fill(trimHeaterCtrlState, TrimHeaterState.OFFLINE);
        Arrays.fill(coldTrimEnabled, true);
    }

    public void setTickMillis(int value) {
        tickMillis = value;
    }

    public int getTickMillis() {
        return tickMillis;
    }

    public void setTrimHeaterState(int heater, TrimHeaterState state) {
        trimHeaterCtrlState[heater] = state;
    }

    public TrimHeaterState getTrimHeaterState(int heater) {
        return trimHeaterCtrlState[heater];
    }

    public void setTrimHeaterPower(int heater, double value) {
        trimHeaterPower[heater] = value;
    }

    public double getTrimHeaterPower(int heater) {
        return trimHeaterPower[heater];
    }

    public void setPlateTemperature(int heater, double value) {
        plateTemperature[heater] = value;
    }

    public double getPlateTemperature(int heater) {
        return plateTemperature[heater];
    }

    public void enableColdSection(int section, boolean value) {
        coldTrimEnabled[section] = value;
    }

    public boolean isColdSectionEnabled(int section) {
        return coldTrimEnabled[section];
    }

    public void setAuxHeaterState(int heater, AuxHeaterState state) {
        auxHeaterState[heater] = state;
    }

    public AuxHeaterState getAuxHeaterState(int heater) {
        return auxHeaterState[heater];
    }

    public void setAuxHeaterPower(int heater, double value) {
        auxHeaterPower[heater] = value;
    }

    public double getAuxHeaterPower(int heater) {
        return auxHeaterPower[heater];
    }

    private static final long serialVersionUID = -6495386488275056259L;
}
