package org.lsst.ccs.subsystem.refrig.data;

import java.io.Serializable;
import java.util.Arrays;
import org.lsst.ccs.subsystem.refrig.constants.HeaterPowerState;
import org.lsst.ccs.subsystem.refrig.constants.HeaterControlState;

/**
 *  Contains the state of the thermal system
 *
 *  @author Owen Saxton
 */
public class ThermalState implements Serializable {

    public static final String KEY = "ThermalState";

    public static final int
        TRIM_HEATER_COLD     = 0,
        TRIM_HEATER_CRYO     = 1,
        NUM_TRIM_HEATERS     = 2,
        COLD_SECTION_MINUS_Y = 0,
        COLD_SECTION_PLUS_Y  = 1,
        NUM_COLD_SECTIONS    = 2,
        AUX_HEATER_MINUS_Y   = 0,
        AUX_HEATER_PLUS_Y    = 1,
        AUX_HEATER_CENTER    = 2,
        NUM_AUX_HEATERS      = 3;

    private int tickMillis;         // Data status update period (ms)
    private final HeaterPowerState[] trimHeaterState = new HeaterPowerState[NUM_TRIM_HEATERS];  // Trim heater power state
    private final HeaterControlState[] trimHeaterCtrlState = new HeaterControlState[NUM_TRIM_HEATERS];  // Trim heater state
    private final double[] trimHeaterPower = new double[NUM_TRIM_HEATERS];          // Trim heater power set point
    private final double[] plateTemperature = new double[NUM_TRIM_HEATERS];         // Plate temperature set point
    private final boolean[] coldTrimEnabled = new boolean[NUM_COLD_SECTIONS];       // Cold trim heater section enables
    private final HeaterPowerState[] auxHeaterState = new HeaterPowerState[NUM_AUX_HEATERS];  // Aux heater state
    private final double[] auxHeaterPower = new double[NUM_AUX_HEATERS];                  // Aux heater power set point

    public ThermalState() {
        Arrays.fill(auxHeaterState, HeaterPowerState.OFFLINE);
        Arrays.fill(trimHeaterState, HeaterPowerState.OFFLINE);
        Arrays.fill(trimHeaterCtrlState, HeaterControlState.OFFLINE);
        Arrays.fill(coldTrimEnabled, true);
    }

    public void setTickMillis(int value) {
        tickMillis = value;
    }

    public int getTickMillis() {
        return tickMillis;
    }

    public void setTrimHeaterState(int heater, HeaterPowerState state) {
        trimHeaterState[heater] = state;
    }

    public HeaterPowerState getTrimHeaterState(int heater) {
        return trimHeaterState[heater];
    }

    public void setTrimHeaterCtrlState(int heater, HeaterControlState state) {
        trimHeaterCtrlState[heater] = state;
    }

    public HeaterControlState getTrimHeaterCtrlState(int heater) {
        return trimHeaterCtrlState[heater];
    }

    public void setTrimHeaterPower(int heater, double value) {
        trimHeaterPower[heater] = value;
    }

    public double getTrimHeaterPower(int heater) {
        return trimHeaterPower[heater];
    }

    public void setPlateTemperature(int heater, double value) {
        plateTemperature[heater] = value;
    }

    public double getPlateTemperature(int heater) {
        return plateTemperature[heater];
    }

    public void enableColdSection(int section, boolean value) {
        coldTrimEnabled[section] = value;
    }

    public boolean isColdSectionEnabled(int section) {
        return coldTrimEnabled[section];
    }

    public void setAuxHeaterState(int heater, HeaterPowerState state) {
        auxHeaterState[heater] = state;
    }

    public HeaterPowerState getAuxHeaterState(int heater) {
        return auxHeaterState[heater];
    }

    public void setAuxHeaterPower(int heater, double value) {
        auxHeaterPower[heater] = value;
    }

    public double getAuxHeaterPower(int heater) {
        return auxHeaterPower[heater];
    }

    private static final long serialVersionUID = 6388078567612517895L;
}
