package org.lsst.ccs.subsystem.rafts;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.config.BulkValidationException;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.reb.BoardDacs;
import org.lsst.ccs.drivers.reb.PowerAdcs;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.subsystem.rafts.config.BiasDACS;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.REBDevice.AdcData;
import org.lsst.ccs.utilities.ccd.ITLCCDType;

/**
 * Implements control of CCD biases and CS gate.
 *
 * @author Owen Saxton
 */
public class BiasControl extends Control implements ConfigurationBulkChangeHandler {

    /**
     * Constants.
     */
    public static final String GD_P = "gdP", OD_P = "odP", OG_P = "ogP", RD_P = "rdP", CS_GATE_P = "csGateP",
                                 GD = "gd", OD = "od", OG = "og", OG_SH = "ogSh", RD = "rd", CS_GATE = "csGate";

    private static final int GD_CH = 0x01, OD_CH = 0x02, OG_CH = 0x04, OG_SH_CH = 0x08, RD_CH = 0x10, CSGATE_CH = 0x20;

    private static final String[] checkList = { GD, OD, OG, RD };

    private static final int[] OD_ADCS = { REBDevice.ADC_OD_0, REBDevice.ADC_OD_1, REBDevice.ADC_OD_2 },
            GD_ADCS = { REBDevice.ADC_GD_0, REBDevice.ADC_GD_1, REBDevice.ADC_GD_2 },
            RD_ADCS = { REBDevice.ADC_RD_0, REBDevice.ADC_RD_1, REBDevice.ADC_RD_2 },
            OG_ADCS = { REBDevice.ADC_OG_0, REBDevice.ADC_OG_1, REBDevice.ADC_OG_2 };

    // Test parameters
    private static final double DAC_TOLERANCE = 0.33;
    private static final int DEFAULT_LOAD_WAIT = 160, DEFAULT_CLEAR_WAIT = 50;
    private static final int SLOW_ADCS_DELTA = 6, SLOW_ADCS_SAMPLE_DELAY = 34;
    private static final double TEST_VOLTS = 0.5, ZERO_ERROR = 0.2, VALUE_ERROR = 0.2, OD_SHORTS_LIMIT = 0.005,
            CLKH_SHORTS_LIMIT = 0.005, CLKL_SHORTS_LIMIT = 0.005;

    /*
     * Data fields.
     */
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;
    @LookupField(strategy = LookupField.Strategy.ANCESTORS)
    private REBDevice rebDevc;
    @LookupPath
    String biasPath;

    // Supplied configurable fields - bias voltages, N.B. ranges are inclusive
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Guard diode", range = ("13.0..27.0"), units = "volts")
    private volatile double gdP;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Output drain", range = ("19.0..28.0"),units = "volts")
    private volatile double odP;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Output gate", range = ("-4.0..4.0"), units = "volts")
    private volatile double ogP;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Reset drain",range = ("10.0..16.0"), units = "volts")
    private volatile double rdP;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Current source", units = "mA")
    private volatile double csGateP;

    // Configurable bias voltage limits
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Guard diode min", units = "volts")
    private volatile double gdMin;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Guard diode max", units = "volts")
    private volatile double gdMax;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Output drain min", units = "volts")
    private volatile double odMin;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Output drain max", units = "volts")
    private volatile double odMax;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Output gate min", units = "volts")
    private volatile double ogMin;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Output gate max", units = "volts")
    private volatile double ogMax;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Reset drain min", units = "volts")
    private volatile double rdMin;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Reset drain max", units = "volts")
    private volatile double rdMax;

    // Supplied configurable fields - bias DAC settings
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Guard diode", units = "DAC counts")
    private volatile int gd;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Output drain", units = "DAC counts")
    private volatile int od;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Output gate", units = "DAC counts")
    private volatile int og;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Output gate shifted", units = "DAC counts")
    private volatile int ogSh;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Reset drain", units = "DAC counts")
    private volatile int rd;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Current source", units = "DAC counts")
    private volatile int csGate;

    // PowerCCDsOn test parameters
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.00..0.7"), description = "Guard diode tolerance", units = "volts")
    private volatile double gdTol;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.90..1.10"), description = "GD convert factor", units = "none")
    private volatile double gdCal;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.00..0.7"), description = "Output drain tolerance", units = "volts")
    private volatile double odTol;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.90..1.10"), description = "OD convert factor", units = "none")
    private volatile double odCal;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.00..0.7"), description = "Output gate tolerance", units = "volts")
    private volatile double ogTol;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.85..1.15"), description = "OG convert factor", units = "none")
    private volatile double ogCal;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.00..0.7"), description = "Reset drain tolerance", units = "volts")
    private volatile double rdTol;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.90..1.10"), description = "RD convert factor", units = "none")
    private volatile double rdCal;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Max ODI value", units = "Amps")
    private volatile double odIMax;

    // Shorts test parameters (N.B. OD is not shorts tested but only power state)
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Shorts Test Voltage for GD", units = "Volts")
    private volatile double gdTestVolts;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for GD at 0V", units = "Volts")
    private volatile double gdZeroErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for GD at TestVolts", units = "Volts")
    private volatile double gdValueErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Shorts Test Voltage for OG", units = "Volts")
    private volatile double ogTestVolts;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for OG at 0V", units = "Volts")
    private volatile double ogZeroErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for OG at TestVolts", units = "Volts")
    private volatile double ogValueErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Shorts Test Voltage for RD", units = "Volts")
    private volatile double rdTestVolts;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for RD at 0V", units = "Volts")
    private volatile double rdZeroErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for RD at TestVolts", units = "Volts")
    private volatile double rdValueErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for OD at 0V", units = "Volts")
    private volatile double odZeroErr;

    // Set from Groovy file
    private boolean raw = false;

    // Other
    private static final Logger LOG = Logger.getLogger(BiasControl.class.getName());
    private boolean cfgValid;
    private int hwVersion;
    private BoardDacs dac;
    private int changed = 0;
    private final Map<String, Double> lowLimitMap = new HashMap<>();
    private final Map<String, Double> highLimitMap = new HashMap<>();
    private double gdConv, odConv, rdConv, ogConv, ogShiftConv, csConv, csOffset;
    private int adcType;
    private double gdDelta, odDelta, rdDelta, ogDelta;

    /**
     * Constructor.
     */
    public BiasControl() {
        for (String pName : checkList) {
            lowLimitMap.put(pName, 0.0);
            highLimitMap.put(pName, 0.0);
        }
    }

    /**
     * Configures channel description.
     *
     * @param mon  The associated monitor object
     * @param devc The associated device object
     */
    @Override
    protected void configure(Monitor mon, Device devc) {
        super.configure(mon, devc);
        dac = rebDevc.getBoardDacs();
    }

    /**
     * Checks configuration.
     *
     * @return Whether configuration is okay
     */
    public boolean checkConfig() {
        int numStrips;
        hwVersion = getHwVersion(dac);
        if (hwVersion == REB.VERSION_UNSUPP) {
            LOG.log(Level.SEVERE, "{0} configuration incompatible with the firmware", getName());
            return false;
        }

        try {
            numStrips = dac.getNumStrips();
        } catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error getting number of strips for {0}", getName());
            return false;
        }
        if (hwChan < 0 || hwChan >= numStrips) {
            LOG.log(Level.SEVERE, "{0} hwChan:{1} not in range: 0:{2}", new Object[]{biasPath, hwChan, numStrips});
            return false;
        }

        if (!raw) {
            switch (hwVersion) {
                case REB.VERSION_1:
                    odConv = REBDevice.DAC_CONV / (1.0 + 49.9 / 10.0);
                    gdConv = odConv;
                    rdConv = odConv;
                    break;
                case REB.VERSION_2:
                    odConv = REBDevice.DAC_CONV / (1.0 + 62.0 / 9.9); // Zener diode leakage
                    gdConv = REBDevice.DAC_CONV / (1.0 + 62.0 / 10.0);
                    rdConv = REBDevice.DAC_CONV / (1.0 + 40.2 / 10.0);
                    break;
                default:
                    odConv = REBDevice.DAC_CONV / (1.0 + 62.0 / 10.0);
                    gdConv = odConv;
                    rdConv = REBDevice.DAC_CONV / (1.0 + 40.2 / 10.0);
                    break;
            }
            double ogGain = 10.0 / 10.0;
            double fudge = hwVersion == REB.VERSION_3 ? 1.0 : 1.1;
            ogShiftConv = fudge * REBDevice.DAC_CONV / ogGain;
            ogConv = fudge * REBDevice.DAC_CONV / (1.0 + ogGain);
            csConv = REBDevice.DAC_CONV * 2.0; // 2 Kohm -> 1 mA = 2 V
            csOffset = -0.95; // current offset in mA
        } else {
            cfgValid = true; // No checking of values
        }
        adcType = hwVersion == REB.VERSION_2 ? REBDevice.TYPE_BIAS_VOLT : REBDevice.TYPE_CR_VOLT;
        changed = 0;

        return true;
    }

    /**
     * Gets the raw state.
     *
     * @return Whether configuration data are raw ADC values
     */
    public boolean isRaw() {
        return raw;
    }

    /**
     * Gets the hardware version.
     *
     * @param dac The board DAC object
     * @return The hardware version
     */
    public static int getHwVersion(BoardDacs dac) {
        int version = dac.getVersion();
        switch (version) {
            case BoardDacs.VERSION_WREB1:
            case BoardDacs.VERSION_WREB2:
            case BoardDacs.VERSION_GREB1:
                version = REB.VERSION_1;
                break;
            case BoardDacs.VERSION_REB3:
            case BoardDacs.VERSION_REB5:
                version = REB.VERSION_2;
                break;
            case BoardDacs.VERSION_GREB2:
            case BoardDacs.VERSION_WREB4:
                version = REB.VERSION_3;
                break;
            default:
                version = REB.VERSION_UNSUPP;
                break;
        }

        return version;
    }

    /**
     * Gets the hardware version.
     *
     * @return The hardware version
     */
    @Deprecated
    public int getHwVersion() {
        return hwVersion;
    }


    /**
     * Validates configuration parameter change.
     *
     * @param params The map of configuration parameter values
     * @throws IllegalArgumentException
     */
    @Override
    public void validateBulkChange(Map<String, Object> params) throws IllegalArgumentException {
        // System.out.println(getName() + " validateBulkChange called");
        String errMsg = null;
        if (!raw) {
            for (String pName : checkList) {
                double value = (Double) params.get(pName + "P");
                final Double max = (Double) params.get(pName + "Max");
                if (value > max) {
                    errMsg = String.format("%s: %sP (%5.5g) is above high limit (%5.5g)", getName(), pName, value, max);
                    break;
                }
                final Double min = (Double) params.get(pName + "Min");
                if (value < min) {
                    errMsg = String.format("%s: %sP (%5.5g) is below low limit (%5.5g)", getName(), pName, value, min);
                    break;
                }
            }
        }
        if (errMsg != null) {
            LOG.log(Level.SEVERE, "Configuration failure: {0}", errMsg);
            throw new IllegalArgumentException(errMsg);
        } else {
            cfgValid = true;
        }
    }


    /**
     * Augments setting of parameters to trigger CCDsPowerState update
     * Parameters are set by ConfigurationChanger methods
     * If parameters match the pattern the rebDevice updateCCDsPowerState flag is set
     * If parameters match the pattern the matching bits in this.changed are set
     *
     * @param params The map of configuration parameter values
     * @throws IllegalArgumentException
     */
    @Override
    public void setParameterBulk(Map<String, Object> params) throws IllegalArgumentException {
        StringBuilder changedString = new StringBuilder();
        Pattern changedPattern = Pattern.compile("(rd|od|gd|og)(P|Cal|Tol)|csGateP");
        if (!raw) {
            int matchCount = 0;
            for (String param: params.keySet()) {  // loop over changed parameters
                Matcher m = changedPattern.matcher(param);
                if (m.matches()) {
                    if        (param.startsWith("gd")) {
                        changed |= GD_CH;
                    } else if (param.startsWith("od")) {
                        changed |= OD_CH;
                    } else if (param.startsWith("og")) {
                        changed |= OG_CH;
                        changed |= OG_SH_CH;
                    } else if (param.startsWith("rd")) {
                        changed |= RD_CH;
                    } else if (param.equals("csGateP")) {
                        changed |= CSGATE_CH;
                    } else {
                        LOG.log(Level.WARNING, "Programmer error: Unexpected parameter change match: "+param);
                    }
                    changedString.append(" "+param);
                    matchCount++;
                }
            }
            if (matchCount > 0) rebDevc.setUpdateCCDsPowerState();  // trigger call in REBDevice
            LOG.log(Level.INFO, biasPath+": "+matchCount+" parameters changed:"+changedString);
            LOG.log(Level.FINE, String.format("%s: changed = 0x%x", biasPath, changed));
        }
    }


    /**
     * Gets the changed variable value
     *
     * @return The value
     */
    @Command(type = CommandType.QUERY, description = "Get the changed variable value")
    public String getChanged() {
        return String.format("0x%x", changed);
    }

    /**
     * Sets the Guard Diode voltage.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setGdP(double value) {
        gdP = value;
        changed |= GD_CH;
    }

    /**
     * Gets the Guard Diode voltage.
     *
     * @return The value
     */
    @Command(type = CommandType.QUERY, description = "Get the Guard Diode voltage")
    public double getGdP() {
        return gdP;
    }

    /**
     * Sets the Guard Diode voltage maximum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setGdMax(double value) {
        gdMax = value;
        highLimitMap.put(GD, value);
    }

    /**
     * Sets the Guard Diode voltage minimum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setGdMin(double value) {
        gdMin = value;
        lowLimitMap.put(GD, value);
    }

    /**
     * Sets the Guard Diode DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setGd(int value) {
        gd = value;
        changed |= GD_CH;
    }

    /**
     * Gets the Guard Diode DAC value.
     *
     * @return value
     */
    @Command(type = CommandType.QUERY, description = "Get the Guard Diode DAC value")
    public int getGd() {
        return gd;
    }

    /**
     * Sets the Output Drain voltage.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setOdP(double value) {
        odP = value;
        changed |= OD_CH;
    }

    /**
     * Gets the Output Drain voltage.
     *
     * @return The value
     */
    @Command(type = CommandType.QUERY, description = "Get the readout mode Output Drain voltage")
    public double getOdP() {
        return odP;
    }

    /**
     * Sets the Output Drain voltage maximum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setOdMax(double value) {
        odMax = value;
        highLimitMap.put(OD, value);
    }

    /**
     * Sets the Output Drain voltage minimum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setOdMin(double value) {
        odMin = value;
        lowLimitMap.put(OD, value);
    }

    /**
     * Sets the Output Drain DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setOd(int value) {
        od = value;
        changed |= OD_CH;
    }

    /**
     * Gets the Output Drain DAC value.
     *
     * @return value
     */
    @Command(type = CommandType.QUERY, description = "Get the readout mode Output Drain DAC value")
    public int getOd() {
        return od;
    }

    /**
     * Sets the Output Gate voltage.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setOgP(double value) {
        ogP = value;
        changed |= OG_CH;
    }

    /**
     * Gets the Output Gate voltage.
     *
     * @return The value
     */
    @Command(type = CommandType.QUERY, description = "Get the Output Gate voltage")
    public double getOgP() {
        return ogP;
    }

    /**
     * Sets the Output Gate voltage maximum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setOgMax(double value) {
        ogMax = value;
        highLimitMap.put(OG, value);
    }

    /**
     * Sets the Output Gate voltage minimum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setOgMin(double value) {
        ogMin = value;
        lowLimitMap.put(OG, value);
    }

    /**
     * Sets the Output Gate DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setOg(int value) {
        og = value;
        changed |= OG_CH;
    }

    /**
     * Gets the Output Gate DAC value.
     *
     * @return value
     */
    @Command(type = CommandType.QUERY, description = "Get the Output Gate DAC value")
    public int getOg() {
        return og;
    }

    /**
     * Sets the Output Gate shift DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setOgSh(int value) {
        ogSh = value;
        changed |= OG_SH_CH;
    }

    /**
     * Gets the Output Gate shift DAC value.
     *
     * @return value
     */
    @Command(type = CommandType.QUERY, description = "Get the Output Gate shift DAC value")
    public int getOgSh() {
        return ogSh;
    }

    /**
     * Sets the Reset Drain voltage.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRdP(double value) {
        rdP = value;
        changed |= RD_CH;
    }

    /**
     * Gets the Reset Drain voltage.
     *
     * @return The value
     */
    @Command(type = CommandType.QUERY, description = "Get the Reset Drain voltage")
    public double getRdP() {
        return rdP;
    }

    /**
     * Sets the Reset Drain voltage maximum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRdMax(double value) {
        rdMax = value;
        highLimitMap.put(RD, value);
    }

    /**
     * Sets the Reset Drain voltage minimum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRdMin(double value) {
        rdMin = value;
        lowLimitMap.put(RD, value);
    }

    /**
     * Sets the Reset Drain DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRd(int value) {
        rd = value;
        changed |= RD_CH;
    }

    /**
     * Gets the Reset Drain DAC value.
     *
     * @return value
     */
    @Command(type = CommandType.QUERY, description = "Get the Reset Drain DAC value")
    public int getRd() {
        return rd;
    }

    /**
     * Sets the Current Source gate current.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setCsGateP(double value) {
        csGateP = value;
        changed |= CSGATE_CH;
    }

    /**
     * Gets the Current Source gate current.
     *
     * @return The value
     */
    @Command(type = CommandType.QUERY, description = "Get the Current Source gate")
    public double getCsGateP() {
        return csGateP;
    }

    /**
     * Sets the Current Source gate DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setCsGate(int value) {
        csGate = value;
        changed |= CSGATE_CH;
    }

    /**
     * Gets the Current Source gate DAC value.
     *
     * @return value
     */
    @Command(type = CommandType.QUERY, description = "Get the Current Source gate")
    public int getCsGate() {
        return csGate;
    }

    /**
     * Sets all the configuration data.
     * 
     * @param bias The bias DAC values
     * @throws RaftException
     */
    @Deprecated
    void setConfig(BiasDACS bias) throws RaftException {
        String name = getName();
        if (!raw) {
            double[] values = bias.getPValues();
            sce.submitChange(name, GD_P, values[BiasDACS.GD]);
            sce.submitChange(name, OD_P, values[BiasDACS.OD]);
            sce.submitChange(name, OG_P, values[BiasDACS.OG]);
            sce.submitChange(name, RD_P, values[BiasDACS.RD]);
            sce.submitChange(name, CS_GATE_P, values[BiasDACS.CS_GATE]);
        } else {
            int[] values = bias.getValues();
            sce.submitChange(name, GD, values[BiasDACS.GD]);
            sce.submitChange(name, OD, values[BiasDACS.OD]);
            sce.submitChange(name, OG, values[BiasDACS.OG]);
            sce.submitChange(name, OG_SH, values[BiasDACS.OG_SH]);
            sce.submitChange(name, RD, values[BiasDACS.RD]);
            sce.submitChange(name, CS_GATE, values[BiasDACS.CS_GATE]);
        }
        try {
            sce.commitBulkChange();
        } catch (BulkValidationException e) {
            sce.dropSubmittedChangesForComponent(name);
            throw new RaftException("Bulk configuration validation failed");
        }
    }

    /**
     * Gets all the configuration data.
     * 
     * @return The bias DAC values
     */
    BiasDACS getConfig() {
        BiasDACS bias = new BiasDACS();
        if (!raw) {
            // Note, this returns the values array by reference
            // so changing the values here also changes them inside the
            // bias object
            @SuppressWarnings("MismatchedReadAndWriteOfArray")
            double[] values = bias.getPValues();
            values[BiasDACS.GD] = gdP;
            values[BiasDACS.OD] = odP;
            values[BiasDACS.OG] = ogP;
            values[BiasDACS.RD] = rdP;
            values[BiasDACS.CS_GATE] = csGateP;
        } else {
            @SuppressWarnings("MismatchedReadAndWriteOfArray")
            int[] values = bias.getValues();
            values[BiasDACS.GD] = gd;
            values[BiasDACS.OD] = od;
            values[BiasDACS.OG] = og;
            values[BiasDACS.OG_SH] = ogSh;
            values[BiasDACS.RD] = rd;
            values[BiasDACS.CS_GATE] = csGate;
        }
        return bias;
    }

    /**
     * Loads configuration data to the biases DAC.
     *
     * @param wait     The time to wait after each DAC setting (ms)
     * @param check    Whether to check the read-back value
     * @param powerOn  Whether being called in sequenced powerOn or delta's to ON state
     * @param dataList List to receive ADC data
     * @return The number of DAC channels loaded: 6
     * @throws RaftException
     */
    int load(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        if (!cfgValid) {
            throw new RaftException("DAC load error: invalid configuration");
        }
        if (!rebDevc.isSerialNumValid()) {
            throw new RaftException("DAC load error: invalid REB serial number");
        }
        rebDevc.readAllAdcs(dataList); // ~38 ms
        if (isITLManufacturedCCD()) {
            loadOg(wait, check, powerOn, dataList);
            loadGd(wait, check, powerOn, dataList);
            loadRd(wait, check, powerOn, dataList);
            loadOd(wait, check, powerOn, dataList);
        } else {
            loadRd(wait, check, powerOn, dataList);
            loadOd(wait, check, powerOn, dataList);
            loadGd(wait, check, powerOn, dataList);
            loadOg(wait, check, powerOn, dataList);
        }
        loadCsGate(0);
        changed = 0;
        return 6;
    }

    /**
     * Loads changed configuration data to the biases DAC.
     * N.B. This is only to be called in the ON|DELTA power state
     *
     * @param wait     The time to wait after each DAC setting (ms)
     * @param check    Whether to check the read-back value
     * @param dataList List to receive read-back values
     * @return The number of DAC channels loaded
     * @throws RaftException
     */
    int loadChanged(int wait, boolean check, List<AdcData> dataList) throws RaftException {
        if (!cfgValid) {
            throw new RaftException("DAC load error: invalid configuration");
        }
        if (!rebDevc.isSerialNumValid()) {
            throw new RaftException("DAC load error: invalid REB serial number");
        }
        int count = 0;
        rebDevc.readAllAdcs(dataList);
        if (isITLManufacturedCCD()) {
            if ((changed & OG_CH) != 0) {
                loadOg(wait, check, false, dataList);
                count += 2;
            }
            if ((changed & GD_CH) != 0) {
                loadGd(wait, check, false, dataList);
                count++;
            }
            if ((changed & RD_CH) != 0) {
                loadRd(wait, check, false, dataList);
                count++;
            }
            if ((changed & OD_CH) != 0) {
                loadOd(wait, check, false, dataList);
                count++;
            }
        } else {
            if ((changed & RD_CH) != 0) {
                loadRd(wait, check, false, dataList);
                count++;
            }
            if ((changed & OD_CH) != 0) {
                loadOd(wait, check, false, dataList);
                count++;
            }
            if ((changed & GD_CH) != 0) {
                loadGd(wait, check, false, dataList);
                count++;
            }
            if ((changed & OG_CH) != 0) {
                loadOg(wait, check, false, dataList);
                count += 2;
            }
        }
        if ((changed & CSGATE_CH) != 0) {
            loadCsGate(0);
            count++;
        }
        changed = 0;
        return count;
    }

    /**
     * Clears (zeroes) the biases DAC values.
     *
     * @throws RaftException
     */
    void clear() throws RaftException {
        clear(DEFAULT_CLEAR_WAIT);
    }

    /**
     * Clears (zeroes) the biases DAC values.
     *
     * @param wait The time to wait after each DAC setting (ms)
     * @throws RaftException
     */
    void clear(int wait) throws RaftException {
        clearCsGate(0);
        if (isITLManufacturedCCD()) {
            clearOd(wait);
            clearRd(wait);
            clearGd(wait);
            clearOg(wait);  // TODO: this could be 0, review waits
        } else {
            clearOg(wait);
            clearGd(wait);
            clearOd(wait);
            clearRd(wait);
        }
        changed = -1;
    }

    /**
     * Tests for biases shorts.
     *
     * @param dataList A list to receive read ADC values
     * @param tmsgList A list of strings to contain messages
     * @return Whether the test failed
     * @throws RaftException
     *
     *                       Note OD not in test: driver doesn't turn on below ~15V
     */
    public boolean testShorts(List<AdcData> dataList, List<String> tmsgList) throws RaftException {
        boolean failed;
        failed = testShorts(BoardDacs.CHAN_RD, rdConv, RD_ADCS[hwChan], rdTestVolts, rdValueErr, rdZeroErr, dataList,
                tmsgList);
        failed |= testShorts(BoardDacs.CHAN_GD, gdConv, GD_ADCS[hwChan], gdTestVolts, gdValueErr, gdZeroErr, dataList,
                tmsgList);
        failed |= testShorts(BoardDacs.CHAN_OG, ogConv, OG_ADCS[hwChan], ogTestVolts, ogValueErr, ogZeroErr, dataList,
                tmsgList);
        // read one more time but not needed to evaluate test, just to get good trending
        // points
        rebDevc.readAllAdcs(dataList);
        return failed;
    }

    /**
     * Tests for biases shorts. Apply voltages and take readings: 1 before, 2@testV,
     * 2 @0V, then perform tests on accumulated data.
     *
     * @param dac          The DAC channel number
     * @param conv         The conversion factor
     * @param adc          The corresponding ADC number
     * @param adcTestVolts The test voltage for that adc
     * @param adcValueErr  The tolerance for the test voltage
     * @param adcZeroErr   The tolerance for change in other biases
     * @param dataList     A list to receive read slow ADC values
     * @param tmsgList     A list to contain test result messages
     * @return Whether the test failed
     * @throws RaftException
     */
    private boolean testShorts(int dac, double conv, int adc, double adcTestVolts, double adcValueErr,
            double adcZeroErr, List<AdcData> dataList, List<String> tmsgList) throws RaftException {
        boolean failed = false;
        // read base
        double[] origPower = rebDevc.readPowerAdcs(dataList);
        double[] origAdcs = rebDevc.readSlowAdcs(dataList);
        // load the test voltage and record
        setDac(dac, (int) (conv * adcTestVolts));
        loadDac(DEFAULT_LOAD_WAIT, dataList);
        rebDevc.readAllAdcs(dataList);
        // give it a little extra time to settle
        // try {
        //     Thread.sleep(DEFAULT_CLEAR_WAIT);
        // } catch (InterruptedException e) {
        // }
        // read for test at test voltage
        double[] curPower = rebDevc.readPowerAdcs(dataList);
        double[] currAdcs = rebDevc.readSlowAdcs(dataList);
        setDac(dac, 0);
        loadDac(DEFAULT_CLEAR_WAIT, dataList);
        // give it a little extra time to settle
        try {
            Thread.sleep(DEFAULT_CLEAR_WAIT);
        } catch (InterruptedException e) {
        }
        // final read at zero voltage
        rebDevc.readAllAdcs(dataList);
        // Perform voltage checks
        for (int ad = 0; ad < currAdcs.length; ad++) { // cycle through the "SlowAdcs"
            double value = currAdcs[ad] - origAdcs[ad]; // change in voltage from baseline
            if (ad == adc) { // where the test voltage was applied
                if (Math.abs(value - adcTestVolts) > adcValueErr) {
                    tmsgList.add(String.format("%s %s value (%.2f) is out of bounds %.2f (+/-%.2f)", biasPath,
                            REBDevice.getAdcName(ad), value, adcTestVolts, adcValueErr));
                    failed = true;
                }
            } else { // all others should stay at zero
                if (Math.abs(value) > adcZeroErr) { // note this is ad == adc!, reflects effect of "ad" change
                    tmsgList.add(String.format("%s %s value (%.2f) is non-zero while testing %s", biasPath,
                            REBDevice.getAdcName(ad), value, REBDevice.getAdcName(adc)));
                    failed = true;
                }
            }
        }
        // Perform current increase checks
        if (curPower[PowerAdcs.ADC_OD_CURRENT] > origPower[PowerAdcs.ADC_OD_CURRENT] + OD_SHORTS_LIMIT) {
            tmsgList.add(String.format("%s OD current increased excessively from %.1f mA to %.1f mA while testing %s",
                    biasPath, 1000.0 * origPower[PowerAdcs.ADC_OD_CURRENT],
                    1000.0 * curPower[PowerAdcs.ADC_OD_CURRENT], REBDevice.getAdcName(adc)));
            failed = true;
        }
        if (curPower[PowerAdcs.ADC_CLKH_CURRENT] > origPower[PowerAdcs.ADC_CLKH_CURRENT] + CLKH_SHORTS_LIMIT) {
            tmsgList.add(String.format("%s CLKH current increased excessively from %.1f mA to %.1f mA while testing %s",
                    biasPath, 1000.0 * origPower[PowerAdcs.ADC_CLKH_CURRENT],
                    1000.0 * curPower[PowerAdcs.ADC_CLKH_CURRENT], REBDevice.getAdcName(adc)));
            failed = true;
        }
        if (curPower[PowerAdcs.ADC_CLKL_CURRENT] > origPower[PowerAdcs.ADC_CLKL_CURRENT] + CLKL_SHORTS_LIMIT) {
            tmsgList.add(String.format("%s CLKL current increased excessively from %.1f mA to %.1f mA while testing %s",
                    biasPath, 1000.0 * origPower[PowerAdcs.ADC_CLKL_CURRENT],
                    1000.0 * curPower[PowerAdcs.ADC_CLKL_CURRENT], REBDevice.getAdcName(adc)));
            failed = true;
        }
        return failed;
    }

    /**
     * Loads the Guard Diode DAC.
     *
     * setDac(), loadDac() both can throw, which caller must deal with
     * checkAdc(check==true) handles out of bounds condition and re-throws
     */
    private void loadGd(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        if (!raw) {
            setDac(BoardDacs.CHAN_GD, (int) (gdConv * gdCal * gdP));
            loadDac(wait, dataList);
            try {
                checkAdc(GD_ADCS[hwChan], dataList, check, powerOn, gdP, gdTol);
            } catch (RaftException e) {
                if (powerOn) clearGd(0);  // only during powerOn sequence, not deltas
                throw e;
            }
        } else {
            setDac(BoardDacs.CHAN_GD, gd);
            loadDac(wait);
        }
    }

    /**
     * Loads the Output Drain DAC.
    private void loadOd(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        loadOd(wait, check, powerOn, dataList, false);
    }
     */

    /**
     * Loads the Output Drain DAC.
     *
     * setDac(), loadDac() both can throw, which caller must deal with
     * checkAdc(check==true) handles out of bounds condition and re-throws
     */
    private void loadOd(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        if (!raw) {
            if (powerOn) {
                double odStepP = 15.0;
                if (odP > odStepP) { // N.B. 2X odTol at step since not calibrated at 15.0
                    setDac(BoardDacs.CHAN_OD, (int) (odConv * odCal * odStepP));
                    loadDac(2 * wait / 3, dataList);
                    try {
                        checkAdc(OD_ADCS[hwChan], dataList, check, powerOn, odStepP, 2.0 * odTol);
                    } catch (RaftException e) {
                        if (powerOn) clearOd(0);
                        throw e;
                    }
                } else { // no clear needed since nothing loaded
                    throw new RaftException(
                            String.format("odStepP (%.2f) > odP (%.2f), odP setting too low", odStepP, odP));
                }
            }
            setDac(BoardDacs.CHAN_OD, (int) (odConv * odCal * odP));
            loadDac(powerOn ? wait / 3 : wait, dataList);
            try {
                checkAdc(OD_ADCS[hwChan], dataList, check, powerOn, odP, odTol);
            } catch (RaftException e) {
                if (powerOn) clearOd(0);
                throw e;
            }
        } else {
            setDac(BoardDacs.CHAN_OD, od);
            loadDac(wait);
        }
    }

    /**
     * Loads the Output Gate DAC.
     *
     * setDac(), loadDac() both can throw, which caller must deal with
     * checkAdc(check==true) handles out of bounds condition and re-throws
     */
    private void loadOg(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        if (!raw) {
            setDac(BoardDacs.CHAN_OG, ogP > 0.0 ? (int) (ogConv * ogCal * ogP) : 0);
            setDac(BoardDacs.CHAN_OG_SH, ogP < 0.0 ? (int) (-ogShiftConv * ogCal * ogP) : 0);
            loadDac(wait, dataList);
            try {
                checkAdc(OG_ADCS[hwChan], dataList, check, powerOn, ogP, ogTol);
            } catch (RaftException e) {
                if (powerOn) clearOg(0);
                throw e;
            }
        } else {
            setDac(BoardDacs.CHAN_OG, og);
            setDac(BoardDacs.CHAN_OG_SH, ogSh);
            loadDac(wait);
        }
    }

    /**
     * Loads the Reset Drain DAC.
     *
     * setDac(), loadDac() both can throw, which caller must deal with
     * checkAdc(check==true) handles out of bounds condition and re-throws
     */
    private void loadRd(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        if (!raw) {
            setDac(BoardDacs.CHAN_RD, (int) (rdConv * rdCal * rdP));
            loadDac(wait, dataList);
            try {
                checkAdc(RD_ADCS[hwChan], dataList, check, powerOn, rdP, rdTol);
            } catch (RaftException e) {
                if (powerOn) clearRd(0);
                throw e;
            }
        } else {
            setDac(BoardDacs.CHAN_RD, rd);
            loadDac(wait);
        }
    }

    /**
     * Loads the Current Source gate DAC.
     */
    private void loadCsGate(int wait) throws RaftException {
        if (!raw) {
            setDac(BoardDacs.CHAN_CSGATE, (int) (csConv * (csGateP + csOffset)));
        } else {
            setDac(BoardDacs.CHAN_CSGATE, csGate);
        }
        loadDac(wait);
    }

    /**
     * Obtains an ADC value and checks voltage and overcurrent
     *
     * @param adc      index of Bias channel in SlowAdcs being checked
     * @param dataList stores measured values for later publication
     * @param check    record and perform check (true) otherwise just record values
     * @param setPoint voltage set point for channel being checked
     * @param adcTol   Tolerance for comparison
     * @return void
     * @throws RaftExecption
     * 
     */
    private void checkAdc(int adc, List<AdcData> dataList, boolean check, boolean powerOn, double setPoint, double adcTol)
            throws RaftException {
        StringBuilder exString = new StringBuilder();
        boolean failed = false;
        if (!check && dataList == null)
            return; // no action taken
        double[] adcValues = rebDevc.readSlowAdcs(dataList); // these can throw
        double[] curPower = rebDevc.readPowerAdcs(dataList);
        // do the checking with failure text appended to exception string as a warning
        if (Math.abs(adcValues[adc] - setPoint) > adcTol) {
            exString.append(String.format("%s %s out of range: setpt=%.2f, reading: %.2f, tol: %.2f", biasPath,
                    REBDevice.getAdcName(adc), setPoint, adcValues[adc], adcTol));
            failed = true;
        }
        if (powerOn) {
            if (curPower[PowerAdcs.ADC_OD_CURRENT] > odIMax) { // this is RXY/RebN/BiasM/odIMax so can step M=0,1,2
                exString.append(String.format(",  %s %s Overcurrent measured on ODI: %.3f > %.3f (odIMax (A))",
                        biasPath, REBDevice.getAdcName(adc), curPower[PowerAdcs.ADC_OD_CURRENT], odIMax));
                failed = true;
            }
        } else { // compare to configured current limit
            double odIlimitHi = Double.parseDouble(sce.getConfigurationParameterValue(rebDevc.rebPath + "/ODI", "limitHi"));
            int rebType = rebDevc.getRebType();  // TODO this seems dodgy
            if (odIlimitHi < (80 - 30.0 * rebType) || odIlimitHi > 110.0) {
                exString.append(String.format("%s limitHi value: %.2f out of allowed range (%d..110)",
                          rebDevc.rebPath + "/ODI", odIlimitHi, (80 - 30.0 * rebType))); 
                failed = true;
            } else {
                if ((1000.0 * curPower[PowerAdcs.ADC_OD_CURRENT]) > odIlimitHi) {
                    exString.append(String.format("%s %s Overcurrent measured on ODI (mA): %.3f > %.3f (ODI:limitHi)",
                            biasPath, REBDevice.getAdcName(adc), 1000.0 * curPower[PowerAdcs.ADC_OD_CURRENT], odIlimitHi));
                    failed = true;
                }
            }
        }
        if (failed) {
            LOG.log(Level.SEVERE, exString.toString());
            if (check) {
                throw new RaftException(exString.toString());
            }
        }
    }

    /**
     * Checks whether Bias voltage values are ON, OFF, DELTA
     *
     * @param slowAdcs[] array of values measued in REBDevice
     * @param tmsgList   List to contain test messages
     * @return powerState On=1; Off=0; ;DELTA>1, error=-1
     * @throws RaftException
     * 
     */
    public int getBiasDacsPowerState(double[] slowAdcs, List<String> tmsgList) {
        double value;
        double delta;
        double maxDelta = rebDevc.getMaxDelta();
        int offCnt = 0;
        int onCnt = 0;
        int lowCnt = 0;
        int dltaCnt = 0;
        int changes = 0;

        // RD
        value = slowAdcs[RD_ADCS[hwChan]];
        delta = value - rdP;
        if (Math.abs(value) < rdZeroErr) {
            offCnt += 1;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for OFF", biasPath,
                    REBDevice.getAdcName(RD_ADCS[hwChan]), value));
        } else if (Math.abs(delta) < rdTol) {
            onCnt += 1;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for ON", biasPath,
                    REBDevice.getAdcName(RD_ADCS[hwChan]), value));
        } else if (Math.abs(delta) < maxDelta) {
            onCnt += 1;
            dltaCnt += 1;
            changes |= RD_CH;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for DELTA", biasPath,
                    REBDevice.getAdcName(RD_ADCS[hwChan]), value));
        } else {
            tmsgList.add(String.format("%s %s voltage = %.2f NOT in range for ON|OFF", biasPath,
                    REBDevice.getAdcName(RD_ADCS[hwChan]), value));
        }
        // OD
        value = slowAdcs[OD_ADCS[hwChan]];
        delta = value - odP;
        if (Math.abs(value) < odZeroErr) {
            offCnt += 1;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for OFF", biasPath,
                    REBDevice.getAdcName(OD_ADCS[hwChan]), value));
        } else if (Math.abs(delta) < odTol) {
            onCnt += 1;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for ON", biasPath,
                    REBDevice.getAdcName(OD_ADCS[hwChan]), value));
        } else if (Math.abs(delta) < maxDelta) {
            onCnt += 1;
            dltaCnt += 1;
            changes |= OD_CH;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for DELTA", biasPath,
                    REBDevice.getAdcName(OD_ADCS[hwChan]), value));
        } else {
            tmsgList.add(String.format("%s %s voltage = %.2f NOT in range for ON|OFF", biasPath,
                    REBDevice.getAdcName(OD_ADCS[hwChan]), value));
        }
        // GD
        value = slowAdcs[GD_ADCS[hwChan]];
        delta = value - gdP;
        if (Math.abs(value) < gdZeroErr) {
            offCnt += 1;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for OFF", biasPath,
                    REBDevice.getAdcName(GD_ADCS[hwChan]), value));
        } else if (Math.abs(delta) < gdTol) {
            onCnt += 1;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for ON", biasPath,
                    REBDevice.getAdcName(GD_ADCS[hwChan]), value));
        } else if (Math.abs(delta) < maxDelta) {
            onCnt += 1;
            dltaCnt += 1;
            changes |= GD_CH;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for DELTA", biasPath,
                    REBDevice.getAdcName(GD_ADCS[hwChan]), value));
        } else {
            tmsgList.add(String.format("%s %s voltage = %.2f NOT in range for ON|OFF", biasPath,
                    REBDevice.getAdcName(GD_ADCS[hwChan]), value));
        }
        // OG -- special case since setpoint could be near zero
        lowCnt = 0;
        value = slowAdcs[OG_ADCS[hwChan]];
        delta = value - ogP;
        if (Math.abs(value) < ogZeroErr) {
            offCnt += 1;
            lowCnt += 1;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for OFF", biasPath,
                    REBDevice.getAdcName(OG_ADCS[hwChan]), value));
        }
        if (Math.abs(delta) < ogTol) {
            onCnt += 1;
            lowCnt += 1;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for ON", biasPath,
                    REBDevice.getAdcName(OG_ADCS[hwChan]), value));
        } else if (Math.abs(delta) < maxDelta) {
            onCnt += 1;
            dltaCnt += 1;
            lowCnt += 1;
            changes |= OG_CH;
            tmsgList.add(String.format("%s %s voltage = %.2f in range for DELTA", biasPath,
                    REBDevice.getAdcName(OG_ADCS[hwChan]), value));
        }
        if (lowCnt == 0) {
            tmsgList.add(String.format("%s %s voltage = %.2f NOT in range for ON|OFF", biasPath,
                    REBDevice.getAdcName(OG_ADCS[hwChan]), value));
        } else if (lowCnt == 2) {
            tmsgList.add(String.format("%s %s value %.2f is in both OFF and ON allowed range", biasPath,
                    REBDevice.getAdcName(OG_ADCS[hwChan]), value));
        }
        tmsgList.add(String.format("%s: getBiasDacsPowerState() found %d(%d) ON(OFF) of biases", biasPath,
                onCnt, offCnt));

        if (offCnt == 4) {
            changed = -1;  // sets all bits
            return 0;
        } else if (onCnt == 4) {  // add deltas and preserve existing changes
            changed |= changes;
            return 1 + dltaCnt;
        } else {
            changed = -1;
            return -1;
        }
    }

    /**
     * Measure and Set BiasDac Calibration to converge measured and configured
     * voltages
     *
     * @param slowAdcs array of measured voltages for CCD biases
     * @param tmsgList List to contain results returned to caller
     * @param action   true|false to actually submit config changes returns 0 on
     *                 success or a count of errors (voltages with out of range
     *                 offsets)
     */
    public int setBiasDacsCalibration(double[] slowAdcs, List<String> tmsgList, boolean action) {
        int retval = 0;
        double delta;
        double minTol = rebDevc.getMinTol();
        double maxDelta = rebDevc.getMaxDelta();

        // table format strings
        String formatStr0 = "%-8.8s %-8.8s %6.2f %6.2f  %6.3f --> %-6.3f"
                + "  %8.3f --> %-8.3f  %5d --> %5d  %6.2f --> %-6.2f";

        String formatStr1 = "%-8.8s %-8.8s %6.2f %6.2f  %6.3f --> ERROR Delta larger than %5.3f max";

        // RD
        double rdCalNew = rdCal; // nominal is 1.0
        double rdTolNew = rdTol;
        delta = rdP - slowAdcs[RD_ADCS[hwChan]];

        if (Math.abs(delta) < maxDelta) { // Small enough to correct
            rdCalNew = rdCal * Math.abs(rdP / slowAdcs[RD_ADCS[hwChan]]);
            rdTolNew = Math.abs(delta) / 4.0 > minTol ? Math.abs(delta) / 4.0 : minTol;

            tmsgList.add(String.format(formatStr0, biasPath, REBDevice.getAdcName(RD_ADCS[hwChan]), rdP,
                    slowAdcs[RD_ADCS[hwChan]], rdCal, rdCalNew, rdTol, rdTolNew, (int) (rdConv * rdCal * rdP),
                    (int) (rdConv * rdCalNew * rdP), rdConv * rdCal, rdConv * rdCalNew));
        } else { // invalid offset for this voltage, rd*New's are unchanged
            retval += 1;
            tmsgList.add(String.format(formatStr1, biasPath, REBDevice.getAdcName(RD_ADCS[hwChan]), rdP,
                    slowAdcs[RD_ADCS[hwChan]], rdCal, maxDelta));
        }
        // OD
        double odCalNew = odCal;
        double odTolNew = odTol;
        delta = odP - slowAdcs[OD_ADCS[hwChan]];

        if (Math.abs(delta) < maxDelta) { // Small enough to correct
            odCalNew = odCal * Math.abs(odP / slowAdcs[OD_ADCS[hwChan]]);
            odTolNew = Math.abs(delta) / 4.0 > minTol ? Math.abs(delta) / 4.0 : minTol;

            tmsgList.add(String.format(formatStr0, biasPath, REBDevice.getAdcName(OD_ADCS[hwChan]), odP,
                    slowAdcs[OD_ADCS[hwChan]], odCal, odCalNew, odTol, odTolNew, (int) (odConv * odCal * odP),
                    (int) (odConv * odCalNew * odP), odConv * odCal, odConv * odCalNew));
        } else { // invalid offset for this voltage, rd*New's are unchanged
            retval += 1;
            tmsgList.add(String.format(formatStr1, biasPath, REBDevice.getAdcName(OD_ADCS[hwChan]), odP,
                    slowAdcs[OD_ADCS[hwChan]], odCal, maxDelta));
        }
        // GD *** N.B.: using double tolerance to compensate the slow GD rise time
        double gdCalNew = gdCal;
        double gdTolNew = gdTol;
        delta = gdP - slowAdcs[GD_ADCS[hwChan]];

        if (Math.abs(delta) < maxDelta) { // Small enough to correct
            gdCalNew = gdCal * Math.abs(gdP / slowAdcs[GD_ADCS[hwChan]]);
            gdTolNew = Math.abs(delta) / 4.0 > (2.0 * minTol) ? Math.abs(delta) / 4.0 : (2.0 * minTol);
            tmsgList.add(String.format(formatStr0, biasPath, REBDevice.getAdcName(GD_ADCS[hwChan]), gdP,
                        slowAdcs[GD_ADCS[hwChan]], gdCal, gdCalNew, gdTol, gdTolNew, (int) (gdConv * gdCal * gdP),
                        (int) (gdConv * gdCalNew * gdP), gdConv * gdCal, gdConv * gdCalNew));
        } else { // invalid offset for this voltage, rd*New's are unchanged
            retval += 1;
            tmsgList.add(String.format(formatStr1, biasPath, REBDevice.getAdcName(GD_ADCS[hwChan]), gdP,
                    slowAdcs[GD_ADCS[hwChan]], gdCal, maxDelta));
        }
        // OG
        double ogCalNew = ogCal;
        double ogTolNew = ogTol;
        delta = ogP - slowAdcs[OG_ADCS[hwChan]];

        if (Math.signum(ogP) == Math.signum(slowAdcs[OG_ADCS[hwChan]]) && Math.abs(delta) < maxDelta) { // Small enough to correct
            ogCalNew = ogCal * Math.abs(ogP / slowAdcs[OG_ADCS[hwChan]]);
            ogTolNew = Math.abs(delta) / 4.0 > minTol ? Math.abs(delta) / 4.0 : minTol;
            tmsgList.add(String.format(formatStr0, biasPath, REBDevice.getAdcName(OG_ADCS[hwChan]), ogP,
                        slowAdcs[OG_ADCS[hwChan]], ogCal, ogCalNew, ogTol, ogTolNew, (int) (ogConv * ogCal * ogP),
                        (int) (ogConv * ogCalNew * ogP), ogConv * ogCal, ogConv * ogCalNew));
        } else { // invalid offset for this voltage, rd*New's are unchanged
            retval += 1;
            tmsgList.add(String.format(formatStr1, biasPath, REBDevice.getAdcName(OG_ADCS[hwChan]), ogP,
                    slowAdcs[OG_ADCS[hwChan]], ogCal, maxDelta));
        }

        // Submit if all offsets are valid
        if (action && retval == 0) {
            sce.submitChange(biasPath, "rdCal", rdCalNew);
            sce.submitChange(biasPath, "rdTol", rdTolNew);
            sce.submitChange(biasPath, "odCal", odCalNew);
            sce.submitChange(biasPath, "odTol", odTolNew);
            sce.submitChange(biasPath, "gdCal", gdCalNew);
            sce.submitChange(biasPath, "gdTol", gdTolNew);
            sce.submitChange(biasPath, "ogCal", ogCalNew);
            sce.submitChange(biasPath, "ogTol", ogTolNew);
        }
        return retval;
    }


    /**
     * Measure BiasDac offsets to prepare for a jump
     *
     * @param slowAdcs array of measured voltages for CCD biases
     * @param tmsgList List to contain results returned to caller
     *
     * @return true|false on okay to jump (step is less than maxStep)
     */
    public int checkBiasStepLimits(double[] slowAdcs, List<String> tmsgList) {
        int retval = 0;
        boolean isOk;
        double delta;
        double maxStep = rebDevc.getMaxStep();

        String formatStr1 = "%-8.8s %-8.8s %6.2f --> %-6.2f: %s";

        // odP 
        delta = odP - slowAdcs[OD_ADCS[hwChan]];
        retval += (isOk = Math.abs(delta) < maxStep) ? 0 : 1;
        tmsgList.add(String.format(formatStr1, biasPath, REBDevice.getAdcName(OD_ADCS[hwChan]),
                    slowAdcs[OD_ADCS[hwChan]], odP, isOk ? "OK" : "STEPTOOLARGE"));
        // rdP
        delta = rdP - slowAdcs[RD_ADCS[hwChan]];
        retval += (isOk = Math.abs(delta) < maxStep) ? 0 : 1;
        tmsgList.add(String.format(formatStr1, biasPath, REBDevice.getAdcName(RD_ADCS[hwChan]),
                    slowAdcs[RD_ADCS[hwChan]], rdP, isOk ? "OK" : "STEPTOOLARGE"));
        // gdP
        delta = gdP - slowAdcs[GD_ADCS[hwChan]];
        retval += (isOk = Math.abs(delta) < maxStep) ? 0 : 1;
        tmsgList.add(String.format(formatStr1, biasPath, REBDevice.getAdcName(GD_ADCS[hwChan]),
                    slowAdcs[GD_ADCS[hwChan]], gdP, isOk ? "OK" : "STEPTOOLARGE"));
        // ogP
        delta = ogP - slowAdcs[OG_ADCS[hwChan]];
        retval += (isOk = Math.abs(delta) < maxStep) ? 0 : 1;
        tmsgList.add(String.format(formatStr1, biasPath, REBDevice.getAdcName(OG_ADCS[hwChan]),
                    slowAdcs[OG_ADCS[hwChan]], ogP, isOk ? "OK" : "STEPTOOLARGE"));
        return retval;
    }

    /**
     * Checks whether a read value is okay
     */
    private static boolean isValueOkay(double read, double set) {
        return Math.abs(read - set) <= DAC_TOLERANCE;
    }

    /**
     * Clears the Guard Diode DAC.
     */
    private void clearGd(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_GD, 0);
        loadDac(wait);
    }

    /**
     * Clears the Output Drain DAC.
     */
    private void clearOd(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_OD, 0);
        loadDac(wait);
    }

    /**
     * Clears the Output Gate DAC.
     */
    private void clearOg(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_OG, 0);
        setDac(BoardDacs.CHAN_OG_SH, 0);
        loadDac(wait);
    }

    /**
     * Clears the Reset Drain DAC.
     */
    private void clearRd(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_RD, 0);
        loadDac(wait);
    }

    /**
     * Clears the Current Source gate DAC.
     */
    private void clearCsGate(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_CSGATE, 0);
        loadDac(wait);
    }

    /**
     * Sets a DAC value.
     *
     * @param chan  The channel number
     * @param value The value to set
     * @throws RaftException
     */
    private void setDac(int chan, int value) throws RaftException {
        try {
            dac.set(hwChan, chan, Math.max(0, Math.min(value, REBDevice.DAC_LIMIT)));
        } catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }

    /**
     * Loads DAC values and waits
     *
     * @param wait The time to wait after loading the DAC (ms)
     * @throws RaftException
     */
    private void loadDac(int wait) throws RaftException {
        try {
            dac.loadStrip(hwChan);
        } catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
        try {
            Thread.sleep(wait);
        } catch (InterruptedException e) {
        }
    }

    /**
     * Loads DAC values and waits with sampling during wait.
     *
     * @param wait The time to wait after loading the DAC (ms)
     * @throws RaftException
     */
    private void loadDac(int wait, List<AdcData> dataList) throws RaftException {
        int delay = wait;
        try {
            dac.loadStrip(hwChan);
        } catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
        while (delay > (SLOW_ADCS_DELTA + SLOW_ADCS_SAMPLE_DELAY)) {  // sample to get shape
            try {
                Thread.sleep(SLOW_ADCS_SAMPLE_DELAY);
            } catch (InterruptedException e) {
            }
            rebDevc.readSlowAdcs(dataList);
            delay -= (SLOW_ADCS_DELTA + SLOW_ADCS_SAMPLE_DELAY);
        }
        if (delay > 0) {
            try {
                Thread.sleep(delay);
            } catch (InterruptedException e) {
            }
        }
    }

    /**
     * Determines whether CCD is manufactured by ITL
     * 
     * @return <code>true</code> iff ccd manufacturer is ITL
     */
    private boolean isITLManufacturedCCD() {
        return rebDevc.isITLManufacturedCCD();
    }

}
