package org.lsst.ccs.subsystem.rafts;

import java.util.Set;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.config.BulkValidationException;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.drivers.reb.Asic;
import org.lsst.ccs.drivers.reb.BoardDacs;
import org.lsst.ccs.drivers.reb.PowerAdcs;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.Sequencer;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.subsystem.rafts.config.DACS;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.REBDevice.AdcData;

/**
 * Implements control of the DACs on a REB.
 *
 * @author Owen Saxton
 */
public class DacControl extends Control implements ConfigurationBulkChangeHandler {

    /*
     * Constants.
     */
    public static final String SCLK_LOW_P = "sclkLowP", SCLK_HIGH_P = "sclkHighP", PCLK_LOW_P = "pclkLowP",
            PCLK_HIGH_P = "pclkHighP", RG_LOW_P = "rgLowP", RG_HIGH_P = "rgHighP",

            SCLK_LOW = "sclkLow", SCLK_LOW_SH = "sclkLowSh", SCLK_HIGH = "sclkHigh", SCLK_HIGH_SH = "sclkHighSh",
            PCLK_LOW = "pclkLow", PCLK_LOW_SH = "pclkLowSh", PCLK_HIGH = "pclkHigh", PCLK_HIGH_SH = "pclkHighSh",
            RG_LOW = "rgLow", RG_LOW_SH = "rgLowSh", RG_HIGH = "rgHigh", RG_HIGH_SH = "rgHighSh", CS_GATE = "csGate";

    private static final int SCLK_LOW_CH = 0x00001, SCLK_LOW_SH_CH = 0x00002, SCLK_HIGH_CH = 0x00004,
            SCLK_HIGH_SH_CH = 0x00008, PCLK_LOW_CH = 0x00010, PCLK_LOW_SH_CH = 0x00020, PCLK_HIGH_CH = 0x00040,
            PCLK_HIGH_SH_CH = 0x00080, RG_LOW_CH = 0x00100, RG_LOW_SH_CH = 0x00200, RG_HIGH_CH = 0x00400,
            RG_HIGH_SH_CH = 0x00800, CS_GATE_CH = 0x01000;

    private static final String[] checkList = { SCLK_LOW, SCLK_HIGH, PCLK_LOW, PCLK_HIGH, RG_LOW, RG_HIGH };

    // test parameters
    // TODO make configurables
    private static final double DAC_TOLERANCE = 0.4; // use shift and cal factor per dac
    private static final int DEFAULT_LOAD_WAIT = 160, DEFAULT_CLEAR_WAIT = 50;
    private static final int SLOW_ADCS_DELTA = 6, SLOW_ADCS_SAMPLE_DELAY = 34;
    private static final double TEST_VOLTS = 0.5, ZERO_ERROR = 0.2, VALUE_ERROR = 0.2, CHANGE_ERROR = 0.1,
            CLK_SHORTS_LIMIT = 0.002;

    private static final Map<Integer, Integer> seqLineMap = new HashMap<>();
    static {
        seqLineMap.put(REBDevice.ADC_PCLK_L,
                Sequencer.SEQ_LINE_P1 | Sequencer.SEQ_LINE_P2 | Sequencer.SEQ_LINE_P3 | Sequencer.SEQ_LINE_P4);
        seqLineMap.put(REBDevice.ADC_PCLK_U, 0);
        seqLineMap.put(REBDevice.ADC_SCLK_L, Sequencer.SEQ_LINE_S1 | Sequencer.SEQ_LINE_S2 | Sequencer.SEQ_LINE_S3);
        seqLineMap.put(REBDevice.ADC_SCLK_U, 0);
        seqLineMap.put(REBDevice.ADC_RG_L, Sequencer.SEQ_LINE_RG);
        seqLineMap.put(REBDevice.ADC_RG_U, 0);
    }

    /*
     * Data fields.
     */
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;
    @LookupField(strategy = LookupField.Strategy.ANCESTORS)
    private REBDevice rebDevc;
    @LookupPath
    String dacPath;

    // Supplied configurable fields - rail voltages
    @ConfigurationParameter(category = REBDevice.RAFTS, range = ("-6.5..-3.0"), description = "Serial clock low", units = "volts")
    private volatile double sclkLowP;
    @ConfigurationParameter(category = REBDevice.RAFTS, range = ("3.0..7.0"), description = "Serial clock high", units = "volts")
    private volatile double sclkHighP;
    @ConfigurationParameter(category = REBDevice.RAFTS, range = ("-9.0..-4.0"), description = "Parallel clock low", units = "volts")
    private volatile double pclkLowP;
    @ConfigurationParameter(category = REBDevice.RAFTS, range = ("1.0..6.0"), description = "Parallel clock high", units = "volts")
    private volatile double pclkHighP;
    @ConfigurationParameter(category = REBDevice.RAFTS, range = ("-5.0..0.0"), description = "Reset gate low", units = "volts")
    private volatile double rgLowP;
    @ConfigurationParameter(category = REBDevice.RAFTS, range = ("5.0..10.0"), description = "Reset gate high", units = "volts")
    private volatile double rgHighP;

    // Configurable rail voltage limits
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Serial clock low min", units = "volts")
    private volatile double sclkLowMin;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Serial clock low max", units = "volts")
    private volatile double sclkLowMax;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Serial clock high min", units = "volts")
    private volatile double sclkHighMin;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Serial clock high max", units = "volts")
    private volatile double sclkHighMax;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Parallel clock low min", units = "volts")
    private volatile double pclkLowMin;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Parallel clock low max", units = "volts")
    private volatile double pclkLowMax;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Parallel clock high min", units = "volts")
    private volatile double pclkHighMin;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Parallel clock high max", units = "volts")
    private volatile double pclkHighMax;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Reset gate low min", units = "volts")
    private volatile double rgLowMin;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Reset gate low max", units = "volts")
    private volatile double rgLowMax;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Reset gate high min", units = "volts")
    private volatile double rgHighMin;
    @ConfigurationParameter(category = REBDevice.RAFTS_LIMITS, description = "Reset gate high max", units = "volts")
    private volatile double rgHighMax;

    // Supplied configurable fields - raw DAC settings
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Serial clock low", units = "DAC counts")
    private volatile int sclkLow;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Serial clock low shifted", units = "DAC counts")
    private volatile int sclkLowSh;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Serial clock high", units = "DAC counts")
    private volatile int sclkHigh;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Serial clock high shifted", units = "DAC counts")
    private volatile int sclkHighSh;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Parallel clock low", units = "DAC counts")
    private volatile int pclkLow;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Parallel clock low shifted", units = "DAC counts")
    private volatile int pclkLowSh;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Parallel clock high", units = "DAC counts")
    private volatile int pclkHigh;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Parallel clock high shifted", units = "DAC counts")
    private volatile int pclkHighSh;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Reset gate low", units = "DAC counts")
    private volatile int rgLow;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Reset gate low shifted", units = "DAC counts")
    private volatile int rgLowSh;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Reset gate high", units = "DAC counts")
    private volatile int rgHigh;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Reset gate high shifted", units = "DAC counts")
    private volatile int rgHighSh;
    @ConfigurationParameter(category = REBDevice.RAFTS, description = "Current source", units = "DAC counts", maxLength = Asic.MAX_STRIPS)
    private final int[] csGate = new int[Asic.MAX_STRIPS];

    // Shorts test parameters
    // order is sclkLowP sclkHighP pclkLowP pclkHighP rgLowP rgHighP
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range="0.4..1.0", description = "Shorts Test Voltage for SClkL", units = "Volts")
    private volatile double sclkLowTestV;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for 0V w/SClkL test", units = "Volts")
    private volatile double sclkLowZeroErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for SClkL TestV", units = "Volts")
    private volatile double sclkLowValueErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range="0.4..1.0", description = "Shorts Test Voltage for SClkU", units = "Volts")
    private volatile double sclkHighTestV;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for 0V w/SClkU test", units = "Volts")
    private volatile double sclkHighZeroErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for SClkU TestV", units = "Volts")
    private volatile double sclkHighValueErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range="0.4..1.0", description = "Shorts Test Voltage for PClkL", units = "Volts")
    private volatile double pclkLowTestV;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for 0V w/PClkL test", units = "Volts")
    private volatile double pclkLowZeroErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for PClkL TestV", units = "Volts")
    private volatile double pclkLowValueErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range="0.4..1.0", description = "Shorts Test Voltage for PClkU", units = "Volts")
    private volatile double pclkHighTestV;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for 0V w/PClkU test", units = "Volts")
    private volatile double pclkHighZeroErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for PClkU TestV", units = "Volts")
    private volatile double pclkHighValueErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range="0.4..1.0", description = "Shorts Test Voltage for RGL", units = "Volts")
    private volatile double rgLowTestV;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for 0V w/RGL test", units = "Volts")
    private volatile double rgLowZeroErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for RGL TestV", units = "Volts")
    private volatile double rgLowValueErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range="0.4..1.0", description = "Shorts Test Voltage for RGU", units = "Volts")
    private volatile double rgHighTestV;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for 0V w/RGU test", units = "Volts")
    private volatile double rgHighZeroErr;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, description = "Tolerance for RGU TestV", units = "Volts")
    private volatile double rgHighValueErr;

    // Power on configurable fields - rail voltage tests
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.00..0.70"), description = "Serial low tolerance", units = "volts")
    private volatile double sclkLowTol;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.90..1.10"), description = "Serial low calib", units = "none")
    private volatile double sclkLowCal;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.00..0.70"), description = "Serial high tolerance", units = "volts")
    private volatile double sclkHighTol;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.90..1.15"), description = "Serial high calib", units = "none")
    private volatile double sclkHighCal;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.00..0.70"), description = "Parallel low tolerance", units = "volts")
    private volatile double pclkLowTol;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.90..1.10"), description = "Parallel low calib", units = "none")
    private volatile double pclkLowCal;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.00..0.70"), description = "Parallel high tolerance", units = "volts")
    private volatile double pclkHighTol;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.90..1.10"), description = "Parallel high calib", units = "none")
    private volatile double pclkHighCal;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.00..0.70"), description = "Reset low tolerance", units = "volts")
    private volatile double rgLowTol;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.90..1.10"), description = "Reset low calib", units = "none")
    private volatile double rgLowCal;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.00..0.70"), description = "Reset high tolerance", units = "volts")
    private volatile double rgHighTol;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.90..1.10"), description = "Reset high calib", units = "none")
    private volatile double rgHighCal;
    // current limits after powering clock rails for node: /Rxy/RebI/DAC parameters
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.100..0.120"), description = "Max ClkHI value", units = "Amps")
    private volatile double clkhIMax;
    @ConfigurationParameter(category = REBDevice.RAFTS_POWER, range = ("0.050..0.065"), description = "Max ClkLI value", units = "Amps")
    private volatile double clklIMax;

    // Set from Groovy file
    private boolean raw = false;

    // Other fields
    private static final Logger LOG = Logger.getLogger(DacControl.class.getName());
    private boolean cfgValid, isCurrent, isCorner;
    private int hwVersion;
    private BoardDacs dac;
    private int changed = 0;  // change by slm, set by powerState, clear, load*
    private final Map<String, Double> lowLimitMap = new HashMap<>();
    private final Map<String, Double> highLimitMap = new HashMap<>();
    private double railShiftConvH, railConvH, railShiftConvL, railConvL;
    private int adcType;

    /**
     * Constructor
     */
    public DacControl() {
        for (String pName : checkList) {
            lowLimitMap.put(pName, 0.0);
            highLimitMap.put(pName, 0.0);
        }
    }

    /**
     * Configures DAC description.
     *
     * @param mon  The associated monitor object
     * @param devc The associated device object
     */
    @Override
    protected void configure(Monitor mon, Device devc) {
        super.configure(mon, devc);
        dac = rebDevc.getBoardDacs();
    }

    /**
     * Checks configuration.
     *
     * @return Whether configuration is okay
     */
    public boolean checkConfig() {
        hwVersion = getHwVersion(dac);
        isCurrent = hwVersion >= REB.VERSION_1;
        isCorner = hwVersion == REB.VERSION_1 || hwVersion == REB.VERSION_3;

        if (!raw) {
            double shiftGainH = (hwVersion == REB.VERSION_1) ? 20.0 / 10.0 : 24.9 / 10.0;
            double shiftGainL = (hwVersion == REB.VERSION_2) ? 24.9 / 10.09 : shiftGainH;
            double fudge = (hwVersion == REB.VERSION_1) ? 1.0 : 1.04;
            railShiftConvH = fudge * REBDevice.DAC_CONV / shiftGainH;
            railConvH = fudge * REBDevice.DAC_CONV / (1 + shiftGainH);
            railShiftConvL = fudge * REBDevice.DAC_CONV / shiftGainL;
            railConvL = fudge * REBDevice.DAC_CONV / (1 + shiftGainL);
        } else {
            cfgValid = true;
        }
        adcType = isCorner ? REBDevice.TYPE_CR_VOLT : REBDevice.TYPE_BIAS_VOLT;
        changed = 0;

        return true;
    }

    /**
     * Gets the raw state.
     *
     * @return Whether data are raw ADC values
     */
    public boolean isRaw() {
        return raw;
    }

    /**
     * Gets the hardware version.
     *
     * @param dac The board DACs object
     * @return The hardware version
     */
    public static int getHwVersion(BoardDacs dac) {
        int version = dac.getVersion();
        if (version == BoardDacs.VERSION_REB0_0 || version == BoardDacs.VERSION_REB0_1) {
            version = REB.VERSION_0;
        } else if (version == BoardDacs.VERSION_WREB1 || version == BoardDacs.VERSION_WREB2
                || version == BoardDacs.VERSION_GREB1) {
            version = REB.VERSION_1;
        } else if (version == BoardDacs.VERSION_REB3 || version == BoardDacs.VERSION_REB5) {
            version = REB.VERSION_2;
        } else if (version == BoardDacs.VERSION_GREB2 || version == BoardDacs.VERSION_WREB4) {
            version = REB.VERSION_3;
        } else {
            version = REB.VERSION_UNSUPP;
        }
        return version;
    }

    /**
     * Gets the hardware version.
     *
     * @return The hardware version
     */
    @Deprecated
    public int getHwVersion() {
        return hwVersion;
    }

    /**
     * Validates configuration changes
     *
     * @param params Map of parameter values
     * @throws IllegalArgumentException
     */
    @Override
    public void validateBulkChange(Map<String, Object> params) throws IllegalArgumentException {
        String errMsg = null;
        if (!raw) {
            for (String pName : checkList) {
                double value = (Double) params.get(pName + "P");
                double max = (Double) params.get(pName + "Max");
                if (value > max) {
                    errMsg = String.format("%s: %sP (%5.5g) is above high limit (%5.5g)", getName(), pName, value, max);
                    break;
                }
                double min = (Double) params.get(pName + "Min");
                if (value < min) {
                    errMsg = String.format("%s: %sP (%5.5g) is below low limit (%5.5g)", getName(), pName, value, min);
                    break;
                }
            }
        }
        if (errMsg != null) {
            LOG.log(Level.SEVERE, "Configuration failure: {0}", errMsg);
            throw new IllegalArgumentException(errMsg);
        } else {
            cfgValid = true;
        }
    }


    /**
     * Augments setting of parameters to trigger CCDsPowerState update
     * Parameters are actually set by ConfigurationChanger methods
     *
     * @param params The map of configuration parameter values
     * @throws IllegalArgumentException
     */
    @Override
    public void setParameterBulk(Map<String, Object> params) throws IllegalArgumentException {
        StringBuilder changedString = new StringBuilder();
        Pattern changedPattern = Pattern.compile("(pclk|sclk|rg)(Low|High)(P|Cal|Tol)");
        if (!raw) {
            int matchCount = 0;
            for (String param: params.keySet()) {  // loop over changed parameters
                Matcher m = changedPattern.matcher(param);
                if (m.matches()) {
                    if        (param.startsWith("sclkLow")) {
                        changed |= SCLK_LOW_CH;
                    } else if (param.startsWith("sclkHigh")) {
                        changed |= SCLK_HIGH_CH;
                    } else if (param.startsWith("pclkLow")) {
                        changed |= PCLK_LOW_CH;
                    } else if (param.startsWith("pclkHigh")) {
                        changed |= PCLK_HIGH_CH;
                    } else if (param.startsWith("rgLow")) {
                        changed |= RG_LOW_CH;
                    } else if (param.startsWith("rgHigh")) {
                        changed |= RG_HIGH_CH;
                    } else {
                        LOG.log(Level.WARNING, "Programmer error: Unexpected parameter change match: "+param);
                    }
                    changedString.append(" "+param);
                    matchCount++;
                }
            }
            if (matchCount > 0) rebDevc.setUpdateCCDsPowerState();  // trigger call in REBDevice
            LOG.log(Level.INFO, dacPath+": "+matchCount+" parameters changed:"+changedString);
            LOG.log(Level.FINE, String.format("%s: changed = 0x%x", dacPath, changed));
        }
    }


    /**
     * Gets the changed variable value
     *
     * @return The value
     */
    @Command(type = CommandType.QUERY, description = "Get the changed variable value")
    public String getChanged() {
        return String.format("0x%x", changed);
    }

    /**
     * Sets the low parallel clock rail voltage.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setPclkLowP(double value) {
        pclkLowP = value;
        changed |= PCLK_LOW_CH;
    }

    /**
     * Gets the low parallel clock rail voltage.
     *
     * @return The value
     */
    public double getPclkLowP() {
        return pclkLowP;
    }

    /**
     * Sets the low parallel clock rail voltage maximum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setPclkLowMax(double value) {
        pclkLowMax = value;
        highLimitMap.put(PCLK_LOW, value);
    }

    /**
     * Sets the low parallel clock rail voltage minimum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setPclkLowMin(double value) {
        pclkLowMin = value;
        lowLimitMap.put(PCLK_LOW, value);
    }

    /**
     * Sets the low parallel clock rail DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setPclkLow(int value) {
        pclkLow = value;
        changed |= PCLK_LOW_CH;
    }

    /**
     * Gets the low parallel clock rail DAC value.
     *
     * @return value
     */
    public int getPclkLow() {
        return pclkLow;
    }

    /**
     * Sets the low parallel clock rail shift DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setPclkLowSh(int value) {
        pclkLowSh = value;
        changed |= PCLK_LOW_SH_CH;
    }

    /**
     * Gets the low parallel clock rail shift DAC value.
     *
     * @return value
     */
    public int getPclkLowSh() {
        return pclkLowSh;
    }

    /**
     * Sets the high parallel clock rail voltage.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setPclkHighP(double value) {
        pclkHighP = value;
        changed |= PCLK_HIGH_CH;
    }

    /**
     * Gets the high parallel clock rail voltage.
     *
     * @return The value
     */
    public double getPclkHighP() {
        return pclkHighP;
    }

    /**
     * Sets the high parallel clock rail voltage maximum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setPclkHighMax(double value) {
        pclkHighMax = value;
        highLimitMap.put(PCLK_HIGH, value);
    }

    /**
     * Sets the high parallel clock rail voltage minimum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setPclkHighMin(double value) {
        pclkHighMin = value;
        lowLimitMap.put(PCLK_HIGH, value);
    }

    /**
     * Sets the high parallel clock rail DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setPclkHigh(int value) {
        pclkHigh = value;
        changed |= PCLK_HIGH_CH;
    }

    /**
     * Gets the high parallel clock rail DAC value.
     *
     * @return value
     */
    public int getPclkHigh() {
        return pclkHigh;
    }

    /**
     * Sets the high parallel clock rail shift DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setPclkHighSh(int value) {
        pclkHighSh = value;
        changed |= PCLK_HIGH_SH_CH;
    }

    /**
     * Gets the high parallel clock rail shift DAC value.
     *
     * @return value
     */
    public int getPclkHighSh() {
        return pclkHighSh;
    }

    /**
     * Sets the low serial clock rail voltage.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setSclkLowP(double value) {
        sclkLowP = value;
        changed |= SCLK_LOW_CH;
    }

    /**
     * Gets the low serial clock rail voltage.
     *
     * @return The value
     */
    public double getSclkLowP() {
        return sclkLowP;
    }

    /**
     * Sets the low serial clock rail voltage maximum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setSclkLowMax(double value) {
        sclkLowMax = value;
        highLimitMap.put(SCLK_LOW, value);
    }

    /**
     * Sets the low serial clock rail voltage minimum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setSclkLowMin(double value) {
        sclkLowMin = value;
        lowLimitMap.put(SCLK_LOW, value);
    }

    /**
     * Sets the low serial clock rail DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setSclkLow(int value) {
        sclkLow = value;
        changed |= SCLK_LOW_CH;
    }

    /**
     * Gets the low serial clock rail DAC value.
     *
     * @return value
     */
    public int getSclkLow() {
        return sclkLow;
    }

    /**
     * Sets the low serial clock rail shift DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setSclkLowSh(int value) {
        sclkLowSh = value;
        changed |= SCLK_LOW_SH_CH;
    }

    /**
     * Gets the low serial clock rail shift DAC value.
     *
     * @return value
     */
    public int getSclkLowSh() {
        return sclkLowSh;
    }

    /**
     * Sets the high serial clock rail voltage.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setSclkHighP(double value) {
        sclkHighP = value;
        changed |= SCLK_HIGH_CH;
    }

    /**
     * Gets the high serial clock rail voltage.
     *
     * @return value
     */
    public double getSclkHighP() {
        return sclkHighP;
    }

    /**
     * Sets the high serial clock rail voltage maximum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setSclkHighMax(double value) {
        sclkHighMax = value;
        highLimitMap.put(SCLK_HIGH, value);
    }

    /**
     * Sets the high serial clock rail voltage minimum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setSclkHighMin(double value) {
        sclkHighMin = value;
        lowLimitMap.put(SCLK_HIGH, value);
    }

    /**
     * Sets the high serial clock rail DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setSclkHigh(int value) {
        sclkHigh = value;
        changed |= SCLK_HIGH_CH;
    }

    /**
     * Gets the high serial clock rail DAC value.
     *
     * @return value
     */
    public int getSclkHigh() {
        return sclkHigh;
    }

    /**
     * Sets the high serial clock rail shift DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setSclkHighSh(int value) {
        sclkHighSh = value;
        changed |= SCLK_HIGH_SH_CH;
    }

    /**
     * Gets the high serial clock rail shift DAC value.
     *
     * @return value
     */
    public int getSclkHighSh() {
        return sclkHighSh;
    }

    /**
     * Sets the low reset gate rail voltage.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRgLowP(double value) {
        rgLowP = value;
        changed |= RG_LOW_CH;
    }

    /**
     * Gets the low reset gate rail voltage.
     *
     * @return The value
     */
    public double getRgLowP() {
        return rgLowP;
    }

    /**
     * Sets the low reset gate rail voltage maximum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRgLowMax(double value) {
        rgLowMax = value;
        highLimitMap.put(RG_LOW, value);
    }

    /**
     * Sets the low reset gate rail voltage minimum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRgLowMin(double value) {
        rgLowMin = value;
        lowLimitMap.put(RG_LOW, value);
    }

    /**
     * Sets the low reset gate rail DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRgLow(int value) {
        rgLow = value;
        changed |= RG_LOW_CH;
    }

    /**
     * Gets the low reset gate rail DAC value.
     *
     * @return value
     */
    public int getRgLow() {
        return rgLow;
    }

    /**
     * Sets the low reset gate rail shift DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRgLowSh(int value) {
        rgLowSh = value;
        changed |= RG_LOW_SH_CH;
    }

    /**
     * Gets the low reset gate rail shift DAC value.
     *
     * @return value
     */
    public int getRgLowSh() {
        return rgLowSh;
    }

    /**
     * Sets the high reset gate rail voltage.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRgHighP(double value) {
        rgHighP = value;
        changed |= RG_HIGH_CH;
    }

    /**
     * Gets the high reset gate rail voltage.
     *
     * @return The value
     */
    public double getRgHighP() {
        return rgHighP;
    }

    /**
     * Sets the high reset gate rail voltage maximum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRgHighMax(double value) {
        rgHighMax = value;
        highLimitMap.put(RG_HIGH, value);
    }

    /**
     * Sets the high reset gate rail voltage minimum.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRgHighMin(double value) {
        rgHighMin = value;
        lowLimitMap.put(RG_HIGH, value);
    }

    /**
     * Sets the high reset gate rail DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRgHigh(int value) {
        rgHigh = value;
        changed |= RG_HIGH_CH;
    }

    /**
     * Gets the high reset gate rail DAC value.
     *
     * @return value
     */
    public int getRgHigh() {
        return rgHigh;
    }

    /**
     * Sets the high reset gate rail shift DAC value.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setRgHighSh(int value) {
        rgHighSh = value;
        changed |= RG_HIGH_SH_CH;
    }

    /**
     * Gets the high reset gate rail shift DAC value.
     *
     * @return value
     */
    public int getRgHighSh() {
        return rgHighSh;
    }

    /**
     * Sets/Gets the current source gates.
     *
     * @param value
     */
    @ConfigurationParameterChanger
    public void setCsGate(int[] value) {
        System.arraycopy(value, 0, csGate, 0, csGate.length);
        changed |= CS_GATE_CH;
    }

    public int[] getCsGate() {
        return csGate;
    }

    /**
     * Sets all the configuration data.
     * 
     * @param dacs The DAC values
     * @throws RaftException
     */
    void setConfig(DACS dacs) throws RaftException {
        String name = getName();
        if (!raw) {
            double[] values = dacs.getPValues();
            sce.submitChange(name, RG_HIGH_P, values[DACS.RG_HIGH]);
            sce.submitChange(name, RG_LOW_P, values[DACS.RG_LOW]);
            sce.submitChange(name, PCLK_HIGH_P, values[DACS.PCLK_HIGH]);
            sce.submitChange(name, PCLK_LOW_P, values[DACS.PCLK_LOW]);
            sce.submitChange(name, SCLK_HIGH_P, values[DACS.SCLK_HIGH]);
            sce.submitChange(name, SCLK_LOW_P, values[DACS.SCLK_LOW]);
        } else {
            int[] values = dacs.getValues();
            sce.submitChange(name, RG_HIGH, values[DACS.RG_HIGH]);
            sce.submitChange(name, RG_LOW, values[DACS.RG_LOW]);
            sce.submitChange(name, PCLK_HIGH, values[DACS.PCLK_HIGH]);
            sce.submitChange(name, PCLK_LOW, values[DACS.PCLK_LOW]);
            sce.submitChange(name, SCLK_HIGH, values[DACS.SCLK_HIGH]);
            sce.submitChange(name, SCLK_LOW, values[DACS.SCLK_LOW]);
            if (isCurrent) {
                sce.submitChange(name, RG_LOW_SH, values[DACS.RG_LOW_SH]);
                sce.submitChange(name, PCLK_LOW_SH, values[DACS.PCLK_LOW_SH]);
                sce.submitChange(name, SCLK_LOW_SH, values[DACS.SCLK_LOW_SH]);
                if (isCorner) {
                    sce.submitChange(name, RG_HIGH_SH, values[DACS.RG_HIGH_SH]);
                    sce.submitChange(name, PCLK_HIGH_SH, values[DACS.PCLK_HIGH_SH]);
                    sce.submitChange(name, SCLK_HIGH_SH, values[DACS.SCLK_HIGH_SH]);
                }
            } else {
                sce.submitChange(name, CS_GATE,
                        new int[] { values[DACS.CSGATE_1], values[DACS.CSGATE_2], values[DACS.CSGATE_3] });
            }
        }
        try {
            sce.commitBulkChange();
        } catch (BulkValidationException e) {
            sce.dropSubmittedChangesForComponent(name);
            throw new RaftException("Bulk configuration validation failed");
        }
    }

    /**
     * Gets all the configuration data.
     *
     * @param dacs The DAC values
     */
    void getConfig(DACS dacs) {
        if (!raw) {
            double[] values = dacs.getPValues();
            values[DACS.RG_HIGH] = rgHighP;
            values[DACS.RG_LOW] = rgLowP;
            values[DACS.PCLK_HIGH] = pclkHighP;
            values[DACS.PCLK_LOW] = pclkLowP;
            values[DACS.SCLK_HIGH] = sclkHighP;
            values[DACS.SCLK_LOW] = sclkLowP;
        } else {
            int[] values = dacs.getValues();
            values[DACS.RG_HIGH] = rgHigh;
            values[DACS.RG_LOW] = rgLow;
            values[DACS.PCLK_HIGH] = pclkHigh;
            values[DACS.PCLK_LOW] = pclkLow;
            values[DACS.SCLK_HIGH] = sclkHigh;
            values[DACS.SCLK_LOW] = sclkLow;
            if (isCurrent) {
                values[DACS.RG_LOW_SH] = rgLowSh;
                values[DACS.PCLK_LOW_SH] = pclkLowSh;
                values[DACS.SCLK_LOW_SH] = sclkLowSh;
                if (isCorner) {
                    values[DACS.RG_HIGH_SH] = rgHighSh;
                    values[DACS.PCLK_HIGH_SH] = pclkHighSh;
                    values[DACS.SCLK_HIGH_SH] = sclkHighSh;
                }
            } else {
                values[DACS.CSGATE_1] = csGate[0];
                values[DACS.CSGATE_2] = csGate[1];
                values[DACS.CSGATE_3] = csGate[2];
            }
        }
    }

    /**
     * Loads configured DAC values onto the REB.
     *
     * @param check    Whether to check the read-back value
     * @param dataList List to receive values from power & slow ADCs
     * @return The number of DACs loaded
     * @throws RaftException
     */
    public int loadConfig(boolean check, List<AdcData> dataList) throws RaftException {
        return loadConfig(DEFAULT_LOAD_WAIT, check, false, dataList);
    }

    /**
     * Loads configured DAC values onto the REB.
     *
     * @param wait     The time to wait after loading a DAC (ms)
     * @param check    Whether to check the read-back value
     * @param dataList List to receive read-back values
     * @return The number of DACs loaded
     * @throws RaftException
     *
     *                       Note: LCA-10064 order is SclkLow, SclkHigh, PclkLow,
     *                       PclkHigh, RgLow, RgHigh
     */
    public int loadConfig(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        if (!cfgValid) {
            throw new RaftException("DAC load error: invalid configuration");
        }
        if (!rebDevc.isSerialNumValid()) {
            throw new RaftException("DAC load error: invalid REB serial number");
        }
        int count = 0;
        rebDevc.readAllAdcs(dataList);
        count += loadSclkLow(wait, check, powerOn, dataList);
        count += loadSclkHigh(wait, check, powerOn, dataList);
        count += loadPclkLow(wait, check, powerOn, dataList);
        count += loadPclkHigh(wait, check, powerOn, dataList);
        count += loadRgLow(wait, check, powerOn, dataList);
        count += loadRgHigh(wait, check, powerOn, dataList);
        if (!isCurrent) {  // TODO: remove this
            setDac(BoardDacs.CHAN_CSGATE_1, csGate[0]);
            setDac(BoardDacs.CHAN_CSGATE_2, csGate[1]);
            setDac(BoardDacs.CHAN_CSGATE_3, csGate[2]);
            loadDac(0);
            count += 3;
        }
        rebDevc.readAllAdcs(dataList);
        changed = 0;
        return count;
    }

    /**
     * Loads changed configured DAC values onto the REB.
     *
     * @param check    Whether to check the read-back value
     * @param dataList List to receive read-back values
     * @return The number of DACs loaded
     * @throws RaftException
     */
    public int loadChanged(boolean check, List<AdcData> dataList) throws RaftException {
        return loadChanged(DEFAULT_LOAD_WAIT, check, dataList);
    }

    /**
     * Loads changed configured DAC values onto the REB.
     *
     * @param wait     The time to wait after loading a DAC (ms)
     * @param check    Whether to check the read-back value
     * @param dataList List to receive read-back values
     * @return The number of DACs loaded
     * @throws RaftException
     *
     *                       Note: LCA-10064 order is SclkLow, SclkHigh, PclkLow,
     *                       PclkHigh, RgLow, RgHigh
     */
    public int loadChanged(int wait, boolean check, List<AdcData> dataList) throws RaftException {
        if (!cfgValid) {
            throw new RaftException("DAC load error: invalid configuration");
        }
        if (!rebDevc.isSerialNumValid()) {
            throw new RaftException("DAC load error: invalid REB serial number");
        }
        int count = 0;
        rebDevc.readAllAdcs(dataList);
        if ((changed & SCLK_LOW_CH) != 0) {
            count += loadSclkLow(wait, check, false, dataList);
        }
        if ((changed & SCLK_HIGH_CH) != 0) {
            count += loadSclkHigh(wait, check, false, dataList);
        }
        if ((changed & PCLK_LOW_CH) != 0) {
            count += loadPclkLow(wait, check, false, dataList);
        }
        if ((changed & PCLK_HIGH_CH) != 0) {
            count += loadPclkHigh(wait, check, false, dataList);
        }
        if ((changed & RG_LOW_CH) != 0) {
            count += loadRgLow(wait, check, false, dataList);
        }
        if ((changed & RG_HIGH_CH) != 0) {
            count += loadRgHigh(wait, check, false, dataList);
        }
        if (!isCurrent) {
            if ((changed & CS_GATE_CH) != 0) {
                setDac(BoardDacs.CHAN_CSGATE_1, csGate[0]);
                setDac(BoardDacs.CHAN_CSGATE_2, csGate[1]);
                setDac(BoardDacs.CHAN_CSGATE_3, csGate[2]);
                loadDac(0);
                count += 3;
                rebDevc.readAllAdcs(dataList);
            }
        }
        changed = 0;
        return count;
    }

    /**
     * Clears (zeroes) all DAC values on the REB.
     *
     * @param dataList A list to receive read ADC values
     * @throws RaftException public void clear(List<AdcData> dataList) throws
     *                       RaftException { clear(DEFAULT_CLEAR_WAIT, dataList); }
     */

    /**
     * Clears (zeroes) all DAC values sequenced as pairs on the REB.
     *
     * @param wait The time to wait after loading a DAC (ms)
     * @throws RaftException
     */
    public void clear(int wait) throws RaftException {
        clearRgLow(0);
        clearRgHigh(wait);
        clearPclkLow(0);
        clearPclkHigh(wait);
        clearSclkLow(0);
        clearSclkHigh(wait);
        if (!isCurrent) {
            setDac(BoardDacs.CHAN_CSGATE_1, 0);
            setDac(BoardDacs.CHAN_CSGATE_2, 0);
            setDac(BoardDacs.CHAN_CSGATE_3, 0);
            loadDac(0);
        }
        changed = -1;
    }

    /**
     * Checks that 6 clock Dacs are within tolerance of zero (eg. OFF) or within
     * tolerance of the set value (eg ON)
     *
     * @param slowAdcs   Array holding the measured voltages to check
     * @param tmsgList   List to contain test message results
     * @param uSeqOffset Offset added to *HighZeroErr to account for high clock
     *                   rails delta (~1.0V) caused by sequencer output coupling
     *
     * @return state int indicating OFF = 0; ON = 1; DELTA = 2,3,... ,error = -1
     * @throws RaftException
     * 
     */
    public int getClockDacsPowerState(double[] slowAdcs, List<String> tmsgList, double uSeqOffset) {
        int expectedCnt = 6;
        int offCnt = 0;
        int onCnt = 0;
        int lowCnt = 0;
        int dltaCnt = 0;
        int changes = 0;
        double value;
        double delta;
        double maxDelta = rebDevc.getMaxDelta();

        // Serial Lower
        // Special care is needed because if sclkLowP is near 0.0 (e2v unipolar), it
        // passes
        // both ON and OFF tests, this is accounted for in final result
        lowCnt = 0;
        value = slowAdcs[REBDevice.ADC_SCLK_L];
        delta = sclkLowP - value;
        if (Math.abs(value) < sclkLowZeroErr) {
            offCnt += 1;
            lowCnt += 1;
            tmsgList.add(String.format("%s %s value %.2f is in OFF range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_SCLK_L), value));
        }
        if (Math.abs(delta) < sclkLowTol) {
            onCnt += 1;
            lowCnt += 1;
            changes |= (changed& SCLK_LOW_CH); // preserve if set
            tmsgList.add(String.format("%s %s value %.2f is in ON range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_SCLK_L), value));
        } else if (Math.abs(delta) < maxDelta) {
            onCnt += 1;
            lowCnt += 1;
            dltaCnt += 1;
            changes |= SCLK_LOW_CH;  // must be changed since out of tolerance
            tmsgList.add(String.format("%s %s value %.2f is in DELTA range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_SCLK_L), value));
        }
        if (lowCnt == 0) {
            tmsgList.add(String.format("%s %s value %.2f is NOT in OFF or ON allowed range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_SCLK_L), value));
        }
        if (lowCnt == 2) {
            tmsgList.add(String.format("%s %s value %.2f is in both OFF and ON allowed range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_SCLK_L), value));
        }
        // Serial Upper
        value = slowAdcs[REBDevice.ADC_SCLK_U];
        delta = sclkHighP - value;
        if (Math.abs(value) < sclkHighZeroErr + uSeqOffset) {
            offCnt += 1;
            tmsgList.add(String.format("%s %s value %.2f is in OFF range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_SCLK_U), value));
        } else if (Math.abs(delta) < sclkHighTol) {
            onCnt += 1;
            changes |= (changed& SCLK_HIGH_CH);
            tmsgList.add(String.format("%s %s value %.2f is in ON range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_SCLK_U), value));
        } else if (Math.abs(delta) < maxDelta) {
            onCnt += 1;
            dltaCnt += 1;
            changes |= SCLK_HIGH_CH;
            tmsgList.add(String.format("%s %s value %.2f is in DELTA range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_SCLK_U), value));
        } else {
            tmsgList.add(String.format("%s %s value %.2f is NOT in OFF or ON allowed range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_SCLK_U), value));
        }

        // Parallel Lower
        lowCnt = 0;
        value = slowAdcs[REBDevice.ADC_PCLK_L];
        delta = pclkLowP - value;
        if (Math.abs(value) < pclkLowZeroErr) {
            offCnt += 1;
            lowCnt += 1;
            tmsgList.add(String.format("%s %s value %.2f is in OFF range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_PCLK_L), value));
        }
        if (Math.abs(delta) < pclkLowTol) {
            onCnt += 1;
            lowCnt += 1;
            changes |= (changed & PCLK_LOW_CH);
            tmsgList.add(String.format("%s %s value %.2f is in ON range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_PCLK_L), value));
        } else if (Math.abs(delta) < maxDelta) {
            onCnt += 1;
            dltaCnt += 1;
            lowCnt += 1;
            changes |= PCLK_LOW_CH;
            tmsgList.add(String.format("%s %s value %.2f is in DELTA range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_PCLK_L), value));
        }
        if (lowCnt == 0) {
            tmsgList.add(String.format("%s %s value %.2f is NOT in OFF or ON allowed range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_PCLK_L), value));
        }
        if (lowCnt == 2) {
            tmsgList.add(String.format("%s %s value %.2f is in both OFF and ON allowed range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_PCLK_L), value));
        }

        // Parallel Upper
        lowCnt = 0;
        value = slowAdcs[REBDevice.ADC_PCLK_U];
        delta = pclkHighP - value;
        // TODO test is compat with dphi?
        if (Math.abs(value) < pclkHighZeroErr + uSeqOffset) {
            offCnt += 1;
            lowCnt += 1;
            tmsgList.add(String.format("%s %s value %.2f is in OFF range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_PCLK_U), value));
        }
        if (Math.abs(delta) < pclkHighTol) {
            onCnt += 1;
            lowCnt += 1;
            changes |= (changed & PCLK_HIGH_CH);
            tmsgList.add(String.format("%s %s value %.2f is in ON range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_PCLK_U), value));
        } else if (Math.abs(delta) < maxDelta) {
            onCnt += 1;
            dltaCnt += 1;
            lowCnt += 1;
            changes |= PCLK_HIGH_CH;
            tmsgList.add(String.format("%s %s value %.2f is in DELTA range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_PCLK_U), value));
        }
        if (lowCnt == 0) {
            tmsgList.add(String.format("%s %s value %.2f is NOT in OFF or ON allowed range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_PCLK_U), value));
        }
        if (lowCnt == 2) {
            tmsgList.add(String.format("%s %s value %.2f is in both OFF and ON allowed range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_PCLK_U), value));
        }

        // Reset Lower
        lowCnt = 0;
        value = slowAdcs[REBDevice.ADC_RG_L];
        delta = rgLowP - value;
        if (Math.abs(value) < rgLowZeroErr) {
            offCnt += 1;
            lowCnt += 1;
            tmsgList.add(String.format("%s %s value %.2f is in OFF range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_RG_L), value));
        }
        if (Math.abs(delta) < rgLowTol) {
            onCnt += 1;
            lowCnt += 1;
            changes |= (changed & RG_LOW_CH);
            tmsgList.add(String.format("%s %s value %.2f is in ON range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_RG_L), value));
        } else if (Math.abs(delta) < maxDelta) {
            onCnt += 1;
            lowCnt += 1;
            dltaCnt += 1;
            changes |= RG_LOW_CH;
            tmsgList.add(String.format("%s %s value %.2f is in DELTA range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_RG_L), value));
        }
        if (lowCnt == 0) {
            tmsgList.add(String.format("%s %s value %.2f is NOT in OFF or ON allowed range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_RG_L), value));
        }
        if (lowCnt == 2) {
            tmsgList.add(String.format("%s %s value %.2f is in both OFF and ON allowed range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_RG_L), value));
        }

        // Reset Upper
        value = slowAdcs[REBDevice.ADC_RG_U];
        delta = rgHighP - value;
        if (Math.abs(value) < rgHighZeroErr + uSeqOffset) {
            offCnt += 1;
            tmsgList.add(String.format("%s %s value %.2f is in OFF range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_RG_U), value));
        } else if (Math.abs(delta) < rgHighTol) {
            onCnt += 1;
            changes |= (changed & RG_HIGH_CH);
            tmsgList.add(String.format("%s %s value %.2f is in ON range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_RG_U), value));
        } else if (Math.abs(delta) < maxDelta) {
            onCnt += 1;
            dltaCnt += 1;
            changes |= RG_HIGH_CH;
            tmsgList.add(String.format("%s %s value %.2f is in DELTA range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_RG_U), value));
        } else {
            tmsgList.add(String.format("%s %s value %.2f is NOT in OFF or ON allowed range", dacPath,
                    REBDevice.getAdcName(REBDevice.ADC_RG_U), value));
        }
        tmsgList.add(String.format("%s: getClockDacsPowerState() found %d ON, %d OFF of %d", dacPath, onCnt,
                offCnt, expectedCnt));
        if (offCnt == expectedCnt) {
            changed = -1;  // sets all bits, forces subsequent loadChanged to load all
            return 0;
        } else if (onCnt == expectedCnt && dltaCnt == 0) {
            changed = changes;
            return 1;
        } else if (onCnt == expectedCnt && dltaCnt > 0) {
            changed = changes;
            return 1 + dltaCnt;
        } else {
            return -1;
        }
    }

    /**
     * Measure and Set ClockDac offsets to converge measured and configured voltages
     *
     * @param slowAdcs array of measured voltages for CCD biases
     * @param tmsgList List to contain results returned to caller
     * @param action   true|false to actually submit config changes returns 0 on
     *                 success or a count of voltages with out of range offsets
     * @return count of voltage that are out of range of correction
     */
    public int setClockDacsCalibration(double[] slowAdcs, List<String> tmsgList, boolean action) {
        int retval = 0;
        double value;
        double delta;
        double conv;
        double maxDelta = rebDevc.getMaxDelta();
        double minTol = rebDevc.getMinTol();

        // table format strings
        String formatStr0 = "%-8.8s %-8.8s %6.2f %6.2f  %6.3f --> %-6.3f"
                + "  %8.3f --> %-8.3f  %5d --> %5d  %6.2f --> %-6.2f";

        String formatStr1 = "%-8.8s %-8.8s %6.2f %6.2f  %6.3f --> ERROR Delta larger than %5.3f max";

        // Serial Lower
        double sclkLowCalNew = sclkLowCal; // nominal is 1.0
        double sclkLowTolNew = sclkLowTol;
        value = slowAdcs[REBDevice.ADC_SCLK_L];
        delta = sclkLowP - value;
        conv = sclkLowP >= 0.0 ? railConvL : railShiftConvL;

        if (Math.signum(sclkLowP) == Math.signum(value) && Math.abs(delta) < maxDelta) { // Small enough to correct

            sclkLowCalNew = sclkLowCal * sclkLowP / value;
            sclkLowTolNew = Math.abs(delta) / 4.0 > minTol ? Math.abs(delta) / 4.0 : minTol;

            tmsgList.add(String.format(formatStr0, dacPath, REBDevice.getAdcName(REBDevice.ADC_SCLK_L),
                    sclkLowP, slowAdcs[REBDevice.ADC_SCLK_L], sclkLowCal, sclkLowCalNew, sclkLowTol, sclkLowTolNew,
                    (int) (conv * sclkLowCal * sclkLowP), (int) (conv * sclkLowCalNew * sclkLowP),
                    conv * sclkLowCal, conv * sclkLowCalNew));
        } else { // invalid offset for this voltage, rd*New's are unchanged
            retval += 1;
            tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_SCLK_L),
                    sclkLowP, slowAdcs[REBDevice.ADC_SCLK_L], sclkLowCal, maxDelta));
        }

        // Serial Upper
        double sclkHighCalNew = sclkHighCal; // nominal is 1.0
        double sclkHighTolNew = sclkHighTol;
        value = slowAdcs[REBDevice.ADC_SCLK_U];
        delta = sclkHighP - value;
        conv = sclkHighP >= 0.0 ? railConvH : railShiftConvH;

        if (Math.signum(sclkHighP) == Math.signum(value) && Math.abs(delta) < maxDelta) { // Small enough to correct

            sclkHighCalNew = sclkHighCal * Math.abs(sclkHighP / slowAdcs[REBDevice.ADC_SCLK_U]);
            sclkHighTolNew = Math.abs(delta) / 4.0 > minTol ? Math.abs(delta) / 4.0 : minTol;

            tmsgList.add(String.format(formatStr0, dacPath, REBDevice.getAdcName(REBDevice.ADC_SCLK_U),
                    sclkHighP, slowAdcs[REBDevice.ADC_SCLK_U], sclkHighCal, sclkHighCalNew, sclkHighTol, sclkHighTolNew,
                    (int) (conv * sclkHighCal * sclkHighP), (int) (conv * sclkHighCalNew * sclkHighP),
                    conv * sclkHighCal, conv * sclkHighCalNew));
        } else { // invalid offset for this voltage, rd*New's are unchanged
            retval += 1;
            tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_SCLK_U),
                    sclkHighP, slowAdcs[REBDevice.ADC_SCLK_U], sclkHighCal, maxDelta));
        }

        // Parallel Lower
        double pclkLowCalNew = pclkLowCal; // nominal is 1.0
        double pclkLowTolNew = pclkLowTol;
        value = slowAdcs[REBDevice.ADC_PCLK_L];
        delta = pclkLowP - value;
        conv = pclkLowP >= 0.0 ? railConvL : railShiftConvL;

        if (Math.signum(pclkLowP) == Math.signum(value) && Math.abs(delta) < maxDelta) { // Small enough to correct

            pclkLowCalNew = pclkLowCal * Math.abs(pclkLowP / slowAdcs[REBDevice.ADC_PCLK_L]);
            pclkLowTolNew = Math.abs(delta) / 4.0 > minTol ? Math.abs(delta) / 4.0 : minTol;

            tmsgList.add(String.format(formatStr0, dacPath, REBDevice.getAdcName(REBDevice.ADC_PCLK_L),
                    pclkLowP, slowAdcs[REBDevice.ADC_PCLK_L], pclkLowCal, pclkLowCalNew, pclkLowTol, pclkLowTolNew,
                    (int) (conv * pclkLowCal * pclkLowP), (int) (conv * pclkLowCalNew * pclkLowP),
                    conv * pclkLowCal, conv * pclkLowCalNew));
        } else { // invalid offset for this voltage, rd*New's are unchanged
            retval += 1;
            tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_PCLK_L),
                    pclkLowP, slowAdcs[REBDevice.ADC_PCLK_L], pclkLowCal, maxDelta));
        }

        // Parallel Upper
        double pclkHighCalNew = pclkHighCal; // nominal is 1.0
        double pclkHighTolNew = pclkHighTol;
        value = slowAdcs[REBDevice.ADC_PCLK_U];
        delta = pclkHighP - value;
        conv = pclkHighP >= 0.0 ? railConvH : railShiftConvH;

        if (Math.signum(pclkHighP) == Math.signum(value) && Math.abs(delta) < maxDelta) { // Small enough to correct

            pclkHighCalNew = pclkHighCal * Math.abs(pclkHighP / slowAdcs[REBDevice.ADC_PCLK_U]);
            pclkHighTolNew = Math.abs(delta) / 4.0 > minTol ? Math.abs(delta) / 4.0 : minTol;

            tmsgList.add(String.format(formatStr0, dacPath, REBDevice.getAdcName(REBDevice.ADC_PCLK_U),
                    pclkHighP, slowAdcs[REBDevice.ADC_PCLK_U], pclkHighCal, pclkHighCalNew, pclkHighTol, pclkHighTolNew,
                    (int) (conv * pclkHighCal * pclkHighP), (int) (conv * pclkHighCalNew * pclkHighP),
                    conv * pclkHighCal, conv * pclkHighCalNew));
        } else { // invalid offset for this voltage, rd*New's are unchanged
            retval += 1;
            tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_PCLK_U),
                    pclkHighP, slowAdcs[REBDevice.ADC_PCLK_U], pclkHighCal, maxDelta));
        }

        // ResetGate Lower
        double rgLowCalNew = rgLowCal; // nominal is 1.0
        double rgLowTolNew = rgLowTol;
        value = slowAdcs[REBDevice.ADC_RG_L];
        delta = rgLowP - value;
        conv = rgLowP >= 0.0 ? railConvL : railShiftConvL;

        if (Math.signum(rgLowP) == Math.signum(value) && Math.abs(delta) < maxDelta) { // Small enough to correct

            rgLowCalNew = rgLowCal * Math.abs(rgLowP / slowAdcs[REBDevice.ADC_RG_L]);
            rgLowTolNew = Math.abs(delta) / 4.0 > minTol ? Math.abs(delta) / 4.0 : minTol;

            tmsgList.add(String.format(formatStr0, dacPath, REBDevice.getAdcName(REBDevice.ADC_RG_L), rgLowP,
                    slowAdcs[REBDevice.ADC_RG_L], rgLowCal, rgLowCalNew, rgLowTol, rgLowTolNew,
                    (int) (conv * rgLowCal * rgLowP), (int) (conv * rgLowCalNew * rgLowP),
                    conv * rgLowCal, conv * rgLowCalNew));
        } else { // invalid offset for this voltage, rd*New's are unchanged
            retval += 1;
            tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_RG_L), rgLowP,
                    slowAdcs[REBDevice.ADC_RG_L], rgLowCal, maxDelta));
        }

        // ResetGate Upper
        double rgHighCalNew = rgHighCal; // nominal is 1.0
        double rgHighTolNew = rgHighTol;
        value = slowAdcs[REBDevice.ADC_RG_U];
        delta = rgHighP - value;
        conv = rgHighP >= 0.0 ? railConvH : railShiftConvH;

        if (Math.signum(rgHighP) == Math.signum(value) && Math.abs(delta) < maxDelta) { // Small enough to correct

            rgHighCalNew = rgHighCal * Math.abs(rgHighP / slowAdcs[REBDevice.ADC_RG_U]);
            rgHighTolNew = Math.abs(delta) / 4.0 > minTol ? Math.abs(delta) / 4.0 : minTol;

            tmsgList.add(String.format(formatStr0, dacPath, REBDevice.getAdcName(REBDevice.ADC_RG_U), rgHighP,
                    slowAdcs[REBDevice.ADC_RG_U], rgHighCal, rgHighCalNew, rgHighTol, rgHighTolNew,
                    (int) (conv * rgHighCal * rgHighP), (int) (conv * rgHighCalNew * rgHighP),
                    conv * rgHighCal, conv * rgHighCalNew));
        } else { // invalid offset for this voltage, rd*New's are unchanged
            retval += 1;
            tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_RG_U), rgHighP,
                    slowAdcs[REBDevice.ADC_RG_U], rgHighCal, maxDelta));
        }

        // Submit if all offsets are valid
        if (action && retval == 0) {
            sce.submitChange(dacPath, "sclkLowCal", sclkLowCalNew);
            sce.submitChange(dacPath, "sclkLowTol", sclkLowTolNew);
            sce.submitChange(dacPath, "sclkHighCal", sclkHighCalNew);
            sce.submitChange(dacPath, "sclkHighTol", sclkHighTolNew);
            sce.submitChange(dacPath, "pclkLowCal", pclkLowCalNew);
            sce.submitChange(dacPath, "pclkLowTol", pclkLowTolNew);
            sce.submitChange(dacPath, "pclkHighCal", pclkHighCalNew);
            sce.submitChange(dacPath, "pclkHighTol", pclkHighTolNew);
            sce.submitChange(dacPath, "rgLowCal", rgLowCalNew);
            sce.submitChange(dacPath, "rgLowTol", rgLowTolNew);
            sce.submitChange(dacPath, "rgHighCal", rgHighCalNew);
            sce.submitChange(dacPath, "rgHighTol", rgHighTolNew);
        }
        return retval;
    }

    /**
     * Measure ClockDac offsets to prepare for a jump
     *
     * @param slowAdcs array of measured voltages for CCD biases
     * @param tmsgList List to contain results returned to caller
     *
     * @return true|false on okay to jump (step is less than maxStep)
     */
    public int checkClockStepLimits(double[] slowAdcs, List<String> tmsgList) {
        int retval = 0;
        boolean isOk;
        double delta;
        double maxStep = rebDevc.getMaxStep();

        String formatStr1 = "%-8.8s %-8.8s %6.2f --> %-6.2f: %s";

        // Serial Lower
        delta = sclkLowP - slowAdcs[REBDevice.ADC_SCLK_L];
        retval += (isOk = Math.abs(delta) < maxStep) ? 0 : 1;
        tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_SCLK_L),
                    slowAdcs[REBDevice.ADC_SCLK_L], sclkLowP, isOk ? "OK" : "STEPTOOLARGE"));
        // Serial Upper
        delta = sclkHighP - slowAdcs[REBDevice.ADC_SCLK_U];
        retval += (isOk = Math.abs(delta) < maxStep) ? 0 : 1;
        tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_SCLK_U),
                    slowAdcs[REBDevice.ADC_SCLK_U], sclkHighP, isOk ? "OK" : "STEPTOOLARGE"));
        // Parallel Lower
        delta = pclkLowP - slowAdcs[REBDevice.ADC_PCLK_L];
        retval += (isOk = Math.abs(delta) < maxStep) ? 0 : 1;
        tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_PCLK_L),
                    slowAdcs[REBDevice.ADC_PCLK_L], pclkLowP, isOk ? "OK" : "STEPTOOLARGE"));
        // Parallel Upper
        delta = pclkHighP - slowAdcs[REBDevice.ADC_PCLK_U];
        retval += (isOk = Math.abs(delta) < maxStep) ? 0 : 1;
        tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_PCLK_U),
                    slowAdcs[REBDevice.ADC_PCLK_U], pclkHighP, isOk ? "OK" : "STEPTOOLARGE"));
        // ResetGate Lower
        delta = rgLowP - slowAdcs[REBDevice.ADC_RG_L];
        retval += (isOk = Math.abs(delta) < maxStep) ? 0 : 1;
        tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_RG_L),
                    slowAdcs[REBDevice.ADC_RG_L], rgLowP, isOk ? "OK" : "STEPTOOLARGE"));
        // ResetGate Upper
        delta = rgHighP - slowAdcs[REBDevice.ADC_RG_U];
        retval += (isOk = Math.abs(delta) < maxStep) ? 0 : 1;
        tmsgList.add(String.format(formatStr1, dacPath, REBDevice.getAdcName(REBDevice.ADC_RG_U),
                    slowAdcs[REBDevice.ADC_RG_U], rgHighP, isOk ? "OK" : "STEPTOOLARGE"));
        return retval;
    }

    /**
     * Tests for clock lines shorts.
     *
     * @param dataList A list to receive read ADC values
     * @param tmsgList List to contain test messages
     * @return Whether the test failed
     * @throws RaftException
     */
    public boolean testShorts(List<AdcData> dataList, List<String> tmsgList) throws RaftException {
        boolean failed;
        failed = testShorts(BoardDacs.CHAN_SCLK_H, BoardDacs.CHAN_SCLK_L, REBDevice.ADC_SCLK_U, REBDevice.ADC_SCLK_L,
                sclkHighTestV, sclkHighValueErr, sclkHighZeroErr, sclkLowTestV, sclkLowValueErr, sclkLowZeroErr,
                dataList, tmsgList);
        //
        failed |= testShorts(BoardDacs.CHAN_PCLK_H, BoardDacs.CHAN_PCLK_L, REBDevice.ADC_PCLK_U, REBDevice.ADC_PCLK_L,
                pclkHighTestV, pclkHighValueErr, pclkHighZeroErr, pclkLowTestV, pclkLowValueErr, pclkLowZeroErr,
                dataList, tmsgList);
        //
        failed |= testShorts(BoardDacs.CHAN_RG_H, BoardDacs.CHAN_RG_L, REBDevice.ADC_RG_U, REBDevice.ADC_RG_L,
                rgHighTestV, rgHighValueErr, rgHighZeroErr, rgLowTestV, rgLowValueErr, rgLowZeroErr, dataList,
                tmsgList);
        return failed;
    }

    /**
     * Tests for clock lines shorts.
     *
     * @param dacUp         The upper rail DAC channel number
     * @param dacLo         The lower rail DAC channel number
     * @param adcUp         The upper rail ADC number
     * @param adcLo         The lower rail ADC number
     * @param adcUpTestV    The upper rail test voltage
     * @param adcUpValueErr The upper fail value tolerance
     * @param adcUpZeroErr  The tolerance for !adcUp @zero voltages
     * @param adcLoTestV    The upper rail test voltage
     * @param adcLoValueErr The upper fail value tolerance
     * @param adcLoZeroErr  The tolerance for !adcLo @zero voltages
     * @param dataList      A list to receive read slow ADC values
     * @param tmsgList      List to contain test messages
     * @return Whether check failed
     * @throws RaftException
     */
    private boolean testShorts(int dacUp, int dacLo, int adcUp, int adcLo, double adcUpTestV, double adcUpValueErr,
            double adcUpZeroErr, double adcLoTestV, double adcLoValueErr, double adcLoZeroErr, List<AdcData> dataList,
            List<String> tmsgList) throws RaftException {
        // Record original ADC values
        rebDevc.readSlowAdcs(dataList);
        double[] origPower = rebDevc.readPowerAdcs(dataList);
        long tstamp;

        // log the event
        tstamp = rebDevc.getTime();
        LOG.log(Level.FINE, String.format("%d initial reading done", tstamp));

        // Set clock rail voltages to zero; should already be there
        setDac(dacUp, 0);
        setDac(dacLo, 0);
        loadDac(DEFAULT_CLEAR_WAIT);

        // Record initial baseline ADC values
        rebDevc.readAllAdcs(dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.log(Level.FINE, String.format("%d set both rails to 0", tstamp));
        LOG.log(Level.FINE, String.format("%d %s,%s set to 0.0 with %d ms wait", tstamp, REBDevice.getAdcName(adcLo),
                REBDevice.getAdcName(adcUp), DEFAULT_CLEAR_WAIT));

        // give it a little extra time to settle
        try {
            Thread.sleep(DEFAULT_CLEAR_WAIT);
        } catch (InterruptedException e) {
        }

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.log(Level.FINE, String.format("%d settled for %d ms", tstamp, DEFAULT_CLEAR_WAIT));

        // Record "lower" ADC values, this is the baseline. Rails should be at 0V with
        // tolerance
        double[] lowerPower = rebDevc.readPowerAdcs(dataList);
        double[] lowerAdcs = rebDevc.readSlowAdcs(dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.log(Level.FINE, String.format("%d recorded lower values", tstamp));

        // Set both clock rail voltages to test value
        setDac(dacUp, (int) (railConvH * adcUpTestV));
        setDac(dacLo, (int) (railConvL * adcLoTestV));
        loadDac(DEFAULT_LOAD_WAIT, dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.log(Level.FINE, String.format("%d %s set to %.2f with %d ms wait", tstamp, REBDevice.getAdcName(adcLo),
                adcLoTestV, DEFAULT_LOAD_WAIT));
        LOG.log(Level.FINE, String.format("%d %s set to %.2f with %d ms wait", tstamp, REBDevice.getAdcName(adcUp),
                adcUpTestV, DEFAULT_LOAD_WAIT));

        // Record "upper" ADC values, this is at test voltage, Rails should be at test
        // volts with no extra current
        double[] upperPower = rebDevc.readPowerAdcs(dataList);
        double[] upperAdcs = rebDevc.readSlowAdcs(dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.log(Level.FINE, String.format("%d recorded at test voltage", tstamp));

        // give it a nice pedestal
        // try {
        //     Thread.sleep(DEFAULT_CLEAR_WAIT);
        // } catch (InterruptedException e) {
        // }

        // Record pedestal ADC values
        // rebDevc.readAllAdcs(dataList);

        // log the time and what happened
        // tstamp = rebDevc.getTime();
        // LOG.log(Level.FINE, String.format("%d pedestal reading", tstamp));

        // Set clock rail voltages back to zero
        setDac(dacUp, 0);
        setDac(dacLo, 0);
        loadDac(DEFAULT_CLEAR_WAIT, dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.log(Level.FINE, String.format("%d set both rails to 0", tstamp));
        LOG.log(Level.FINE, String.format("%d %s clock rails set to 0.0 with %d ms wait", tstamp,
                REBDevice.getAdcName(adcLo), DEFAULT_CLEAR_WAIT));

        // give it a nice pedestal
        try {
            Thread.sleep(DEFAULT_CLEAR_WAIT);
        } catch (InterruptedException e) {
        }

        // Record final ADC values
        rebDevc.readAllAdcs(dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.log(Level.FINE, String.format("%d final reading", tstamp));

        // Check the values
        return checkValues(adcUp, adcLo, adcUpTestV, adcUpValueErr, adcUpZeroErr, adcLoTestV, adcLoValueErr,
                adcLoZeroErr, lowerAdcs, upperAdcs, origPower, lowerPower, upperPower, tmsgList);
    }

    /**
     * Checks the voltages and currents.
     * 
     * @param adcUp         The upper rail ADC number
     * @param adcLo         The lower rail ADC number
     * @param adcUpTestV    Upper test voltage
     * @param adcUpValueErr Tolerance for test voltage
     * @param adcUpZeroErr  Tolerance for nominal zero
     * @param adcLoTestV    Upper test voltage
     * @param adcLoValueErr Tolerance for test voltage
     * @param adcLoZeroErr  Tolerance for nominal zero
     * @param baseAdcs      The slow ADC values at baseline
     * @param measAdcs      The slow ADC values at test voltage
     * @param origPower     The original board power ADC values
     * @param basePower     The board power ADC values at baseline
     * @param measPower     The board power ADC values at test voltage
     * @param tmsgList      List to contain result messages for logs or exceptions
     * @return Whether check failed
     */
    private boolean checkValues(int adcUp, int adcLo, double adcUpTestV, double adcUpValueErr, double adcUpZeroErr,
            double adcLoTestV, double adcLoValueErr, double adcLoZeroErr, double[] baseAdcs, double[] measAdcs,
            double[] origPower, double[] basePower, double[] measPower, List<String> tmsgList) {
        boolean failed = false;
        // first check measured and base voltage behavior
        for (int ad = 0; ad < baseAdcs.length; ad++) {
            double basevalue = baseAdcs[ad];
            double measvalue = measAdcs[ad];
            if (ad == adcUp) {
                if (Math.abs(basevalue) > adcUpZeroErr) { // the tested adcs baseline should be at zero
                    tmsgList.add(String.format("%s %s baseline value (%.2f) is out of bounds (%.2f)", dacPath,
                            REBDevice.getAdcName(ad), basevalue, adcUpZeroErr));
                    failed = true;
                }
                if (Math.abs(measvalue - adcUpTestV) > adcUpValueErr) { // tested measured value should be correct
                    tmsgList.add(String.format("%s %s measured value (%.2f) is out of bounds (%.2f +/- %.2f)",
                            dacPath, REBDevice.getAdcName(ad), measvalue, adcUpTestV, adcUpValueErr));
                    failed = true;
                }
            } else if (ad == adcLo) {
                if (Math.abs(basevalue) > adcLoZeroErr) { // the tested adcs baseline should be at zero
                    tmsgList.add(String.format("%s %s baseline value (%.2f) is out of bounds (%.2f)", dacPath,
                            REBDevice.getAdcName(ad), basevalue, adcLoZeroErr));
                    failed = true;
                }
                if (Math.abs(measvalue - adcLoTestV) > adcLoValueErr) { // tested measured value should be correct
                    tmsgList.add(String.format("%s %s measured value (%.2f) is out of bounds (%.2f +/- %.2f)",
                            dacPath, REBDevice.getAdcName(ad), measvalue, adcLoTestV, adcLoValueErr));
                    failed = true;
                }
            } else { // should be no change between the base and measured values for adcs not under
                     // test
                if (Math.abs(measvalue - basevalue) > adcLoZeroErr) {
                    tmsgList.add(String.format("%s %s change of (%.2f) is out of bounds (+/- %.2f)", dacPath,
                            REBDevice.getAdcName(ad), (measvalue - basevalue), CHANGE_ERROR));
                    failed = true;
                }
            }
        }
        // check for current increases in CLK[HL]I's0
        if (basePower[PowerAdcs.ADC_CLKH_CURRENT] > origPower[PowerAdcs.ADC_CLKH_CURRENT] + CLK_SHORTS_LIMIT) {
            tmsgList.add(String.format(
                    "%s ClkHi base current increased excessively from %.2f mA to %.2f mA while testing %s",
                    dacPath, 1000.0 * origPower[PowerAdcs.ADC_CLKH_CURRENT],
                    1000.0 * basePower[PowerAdcs.ADC_CLKH_CURRENT], REBDevice.getAdcName(adcUp)));
            failed = true;
        }
        if (measPower[PowerAdcs.ADC_CLKH_CURRENT] > basePower[PowerAdcs.ADC_CLKH_CURRENT] + CLK_SHORTS_LIMIT) {
            tmsgList.add(String.format(
                    "%s ClkHi meas current increased excessively from %.2f mA to %.2f mA while testing %s",
                    dacPath, 1000.0 * basePower[PowerAdcs.ADC_CLKH_CURRENT],
                    1000.0 * measPower[PowerAdcs.ADC_CLKH_CURRENT], REBDevice.getAdcName(adcUp)));
            failed = true;
        }
        if (basePower[PowerAdcs.ADC_CLKL_CURRENT] > origPower[PowerAdcs.ADC_CLKL_CURRENT] + CLK_SHORTS_LIMIT) {
            tmsgList.add(String.format(
                    "%s ClkLo base current increased excessively from %.2f mA to %.2f mA while testing %s",
                    dacPath, 1000.0 * origPower[PowerAdcs.ADC_CLKL_CURRENT],
                    1000.0 * basePower[PowerAdcs.ADC_CLKL_CURRENT], REBDevice.getAdcName(adcLo)));
            failed = true;
        }
        if (measPower[PowerAdcs.ADC_CLKL_CURRENT] > basePower[PowerAdcs.ADC_CLKL_CURRENT] + CLK_SHORTS_LIMIT) {
            tmsgList.add(String.format(
                    "%s ClkLo meas current increased excessively from %.2f mA to %.2f mA while testing %s",
                    dacPath, 1000.0 * basePower[PowerAdcs.ADC_CLKL_CURRENT],
                    1000.0 * measPower[PowerAdcs.ADC_CLKL_CURRENT], REBDevice.getAdcName(adcLo)));
            failed = true;
        }
        return failed;
    }

    /**
     * Loads the low parallel clock rail DACs.
     */
    private int loadPclkLow(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_PCLK_L, pclkLowP >= 0.0 ? (int) (railConvL * pclkLowCal * pclkLowP) : 0);
            setDac(BoardDacs.CHAN_PCLK_L_SH, pclkLowP <= 0.0 ? (int) (-railShiftConvL * pclkLowCal * pclkLowP) : 0);
            count++;
            loadDac(wait, dataList);
            try {
                checkAdc(REBDevice.ADC_PCLK_L, dataList, check, pclkLowP, pclkLowTol);
            } catch (RaftException e) {
                if (powerOn) clearPclkLow(0);  // only if sequencing from OFF state
                throw e;
            }
        } else {
            setDac(BoardDacs.CHAN_PCLK_L, pclkLow);
            if (isCurrent) {
                setDac(BoardDacs.CHAN_PCLK_L_SH, pclkLowSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }

    /**
     * Loads the high parallel clock rail DACs.
     */
    private int loadPclkHigh(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        int count = 1;
        if (!raw) { // new
            if (!isCorner) {
                setDac(BoardDacs.CHAN_PCLK_H, pclkHighP >= 0.0 ? (int) (railConvH * pclkHighCal * pclkHighP) : 0);
                loadDac(wait, dataList);
            } // check because pclkHighP should be set to match the expected (pclkLowP + dPhi) value
            try {
                checkAdc(REBDevice.ADC_PCLK_U, dataList, check, pclkHighP, pclkHighTol);
            } catch (RaftException e) {
                if (powerOn) clearPclkHigh(0);
                throw e;
            }
        }
        else {
            setDac(BoardDacs.CHAN_PCLK_H, pclkHigh);
            if (isCorner) {
                setDac(BoardDacs.CHAN_PCLK_H_SH, pclkHighSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }

    /**
     * Loads the low serial clock rail DACs.
     */
    private int loadSclkLow(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_SCLK_L, sclkLowP >= 0.0 ? (int) (railConvL * sclkLowCal * sclkLowP) : 0);
            setDac(BoardDacs.CHAN_SCLK_L_SH, sclkLowP <= 0.0 ? (int) (-railShiftConvL * sclkLowCal * sclkLowP) : 0);
            count++;
            loadDac(wait, dataList);
            try {
                checkAdc(REBDevice.ADC_SCLK_L, dataList, check, sclkLowP, sclkLowTol);
            } catch (RaftException e) {
                if (powerOn) clearSclkLow(0);
                throw e;
            }
        } else {
            setDac(BoardDacs.CHAN_SCLK_L, sclkLow);
            if (isCurrent) {
                setDac(BoardDacs.CHAN_SCLK_L_SH, sclkLowSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }

    /**
     * Loads the high serial clock rail DACs.
     */
    private int loadSclkHigh(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_SCLK_H, sclkHighP >= 0.0 ? (int) (railConvH * sclkHighCal * sclkHighP) : 0);
            if (isCorner) {
                setDac(BoardDacs.CHAN_SCLK_H_SH,
                        sclkHighP <= 0.0 ? (int) (-railShiftConvH * sclkHighCal * sclkHighP) : 0);
                count++;
            }
            loadDac(wait, dataList);
            try {
                checkAdc(REBDevice.ADC_SCLK_U, dataList, check, sclkHighP, sclkHighTol);
            } catch (RaftException e) {
                if (powerOn) clearSclkHigh(0);
                throw e;
            }
        } else {
            setDac(BoardDacs.CHAN_SCLK_H, sclkHigh);
            if (isCorner) {
                setDac(BoardDacs.CHAN_SCLK_H_SH, sclkHighSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }

    /**
     * Loads the low reset gate rail DACs.
     */
    private int loadRgLow(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_RG_L, rgLowP >= 0.0 ? (int) (railConvL * rgLowCal * rgLowP) : 0);
            setDac(BoardDacs.CHAN_RG_L_SH, rgLowP <= 0.0 ? (int) (-railShiftConvL * rgLowCal * rgLowP) : 0);
            count++;
            loadDac(wait, dataList);
            try {
                checkAdc(REBDevice.ADC_RG_L, dataList, check, rgLowP, rgLowTol);
            } catch (RaftException e) {
                if (powerOn) clearRgLow(0);
                throw e;
            }
        } else {
            setDac(BoardDacs.CHAN_RG_L, rgLow);
            if (isCurrent) {
                setDac(BoardDacs.CHAN_RG_L_SH, rgLowSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }

    /**
     * Loads the high reset gate rail DACs.
     */
    private int loadRgHigh(int wait, boolean check, boolean powerOn, List<AdcData> dataList) throws RaftException {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_RG_H, rgHighP >= 0.0 ? (int) (railConvH * rgHighCal * rgHighP) : 0);
            if (isCorner) {
                setDac(BoardDacs.CHAN_RG_H_SH, rgHighP <= 0.0 ? (int) (-railShiftConvH * rgHighCal * rgHighP) : 0);
                count++;
            }
            loadDac(wait, dataList);
            try {
                checkAdc(REBDevice.ADC_RG_U, dataList, check, rgHighP, rgHighTol);
            } catch (Exception e) {
                if (powerOn) clearRgHigh(0);
                throw e;
            }
        } else {
            setDac(BoardDacs.CHAN_RG_H, rgHigh);
            if (isCorner) {
                setDac(BoardDacs.CHAN_RG_H_SH, rgHighSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }

    /**
     * Obtains an ADC value and checks it.
     *
     * @param adc      index of Clock rail channel in SlowAdcs being checked
     * @param dataList for storing measured values
     * @param check    record and perform check (true) otherwise just record values
     * @param setPoint voltage set point for channel being checked
     * @param adcTol   Tolerance for comparison
     * @return null or a RaftException that the caller can throw
     */
    private void checkAdc(int adc, List<AdcData> dataList, boolean check, double setPoint, double adcTol)
            throws RaftException {
        StringBuilder exString = new StringBuilder();
        boolean failed = false;
        if (!check && dataList == null)
            return; // no action
        double[] adcValues = rebDevc.readSlowAdcs(dataList); // ~6ms
        double[] curPower = rebDevc.readPowerAdcs(dataList); // ~33ms
        // checking (3) failure modes and throw if "check"
        if (Math.abs(adcValues[adc] - setPoint) > adcTol) { // checks set value achieved
            exString.append(String.format("\n%s %s out of range: setpt=%.2f, reading: %.2f, tol: %.2f",
                    dacPath, REBDevice.getAdcName(adc), setPoint, adcValues[adc], adcTol));
            failed = true;
        }
        if (curPower[PowerAdcs.ADC_CLKH_CURRENT] > clkhIMax) { // this is RXY/RebN/DAC parameter
            exString.append(String.format("\n%s %s caused overcurrent on ClkHI: measured %.3f, limit: %.3f",
                    dacPath, REBDevice.getAdcName(adc), curPower[PowerAdcs.ADC_CLKH_CURRENT], clkhIMax));
            failed = true;
        }
        if (curPower[PowerAdcs.ADC_CLKL_CURRENT] > clklIMax) { // this is RXY/RebN/DAC parameter
            exString.append(String.format("\n%s %s caused overcurrent on ClkLI: measured %.3f, limit: %.3f",
                    dacPath, REBDevice.getAdcName(adc), curPower[PowerAdcs.ADC_CLKL_CURRENT], clklIMax));
            failed = true;
        }
        if (failed) { // caller to throw the returned exception
            LOG.log(Level.SEVERE, exString.toString());
            if (check) {
                throw new RaftException(exString.toString());
            }
        }
    }

    /**
     * Checks whether a read value is okay
     */
    private static boolean isValueOkay(double read, double set) {
        return Math.abs(read - set) <= DAC_TOLERANCE;
    }

    /**
     * Clears the low parallel clock rail DACs.
     */
    private void clearPclkLow(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_PCLK_L, 0);
        if (isCurrent) {
            setDac(BoardDacs.CHAN_PCLK_L_SH, 0);
        }
        loadDac(wait);
    }

    /**
     * Clears the high parallel clock rail DACs.
     */
    private void clearPclkHigh(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_PCLK_H, 0);
        if (isCorner) {
            setDac(BoardDacs.CHAN_PCLK_H_SH, 0);
        }
        loadDac(wait);
    }

    /**
     * Clears the low serial clock rail DACs.
     */
    private void clearSclkLow(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_SCLK_L, 0);
        if (isCurrent) {
            setDac(BoardDacs.CHAN_SCLK_L_SH, 0);
        }
        loadDac(wait);
    }

    /**
     * Clears the high serial clock rail DACs.
     */
    private void clearSclkHigh(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_SCLK_H, 0);
        if (isCorner) {
            setDac(BoardDacs.CHAN_SCLK_H_SH, 0);
        }
        loadDac(wait);
    }

    /**
     * Clears the low reset gate rail DACs.
     */
    private void clearRgLow(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_RG_L, 0);
        if (isCurrent) {
            setDac(BoardDacs.CHAN_RG_L_SH, 0);
        }
        loadDac(wait);
    }

    /**
     * Clears the high reset gate rail DACs.
     */
    private void clearRgHigh(int wait) throws RaftException {
        setDac(BoardDacs.CHAN_RG_H, 0);
        if (isCorner) {
            setDac(BoardDacs.CHAN_RG_H_SH, 0);
        }
        loadDac(wait);
    }

    /**
     * Sets a DAC value.
     *
     * @param chan  The channel number
     * @param value The value to set
     * @throws RaftException
     */
    private void setDac(int chan, int value) throws RaftException {
        try {
            dac.set(chan, Math.max(0, Math.min(value, REBDevice.DAC_LIMIT)));
        } catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }

    /**
     * Loads DAC values.
     *
     * @param wait The time to wait after loading the DAC (ms)
     * @throws RaftException
     */
    private void loadDac(int wait) throws RaftException {
        try {
            dac.loadGlobal();
        } catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
        try {
            Thread.sleep(wait);
        } catch (InterruptedException e) {
        }
    }

    /**
     * Loads DAC values.
     *
     * @param wait The time to wait after loading the DAC (ms)
     * @throws RaftException
     */
    private void loadDac(int wait, List<AdcData> dataList) throws RaftException {
        int delay = wait;
        try {
            dac.loadGlobal();
        } catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
        while (delay > (SLOW_ADCS_DELTA + SLOW_ADCS_SAMPLE_DELAY)) { // sample to get shape
            try {
                Thread.sleep(SLOW_ADCS_SAMPLE_DELAY);
            } catch (InterruptedException e) {
            }
            rebDevc.readSlowAdcs(dataList);
            delay -= (SLOW_ADCS_DELTA + SLOW_ADCS_SAMPLE_DELAY);
        }
        if (delay > 0) {
            try {
                Thread.sleep(delay);
            } catch (InterruptedException e) {
            }
        }
    }
}
