package org.lsst.ccs.subsystem.rafts;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.ConfigurationListener;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.daq.utilities.FitsHeaderKeywordData;
import org.lsst.ccs.drivers.reb.Asic;
import org.lsst.ccs.drivers.reb.Aspic;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.BoardDacs;
import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.drivers.reb.ImageClient;
import org.lsst.ccs.drivers.reb.PowerAdcs;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.REBTimeoutException;
import org.lsst.ccs.drivers.reb.Sequencer;
import org.lsst.ccs.drivers.reb.SlowAdcs;
import org.lsst.ccs.drivers.reb.StatusSet;
import org.lsst.ccs.drivers.reb.TempAdcs;
import org.lsst.ccs.drivers.reb.TempRtds;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorUpdateTask;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.rafts.config.ASPIC;
import org.lsst.ccs.subsystem.rafts.config.BiasDACS;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.RegisterData;
import org.lsst.ccs.subsystem.rafts.data.StatusData;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.states.HVBiasState;
import org.lsst.ccs.subsystem.rafts.states.CCDsPowerState;
import org.lsst.ccs.subsystem.rafts.states.RebDeviceState;
import org.lsst.ccs.subsystem.rafts.states.RebValidationState;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *  Implements access to a REB device.
 *
 *  @author Owen Saxton
 */
public class REBDevice extends Device implements ConfigurationListener, ConfigurationBulkChangeHandler {

    /*
     *  Class for holding read DAC settings
     */
    public static class AdcData {

        static final int
            TYPE_POWER = 0,
            TYPE_SLOW = 1,
            TYPE_CURR = 2;

        final int type;    // Power or slow ADC
        final int hwChan;
        final double value;
        final CCSTimeStamp timeStamp;
        Channel ch;

        public AdcData(int type, int hwChan, double value, CCSTimeStamp timeStamp)
        {
            this.type = type;
            this.hwChan = hwChan;
            this.value = value;
            this.timeStamp = timeStamp == null ? CCSTimeStamp.currentTime() : timeStamp;
            this.ch = null;
        }

        public AdcData( int type, Channel ch, double value, CCSTimeStamp timeStamp)
        {
            this(type, ch.getHwChan(), value, timeStamp);
            this.ch = ch;
        }
        
        
    }

    /*
     *  Constants.
     */
    private static final String
        SERIAL_NUM = "serialNum";

    public static final String
        RAFTS = "Rafts",
        RAFTS_LIMITS = "RaftsLimits",
        RAFTS_POWER = "RaftsPower";

    public static final int
        TYPE_BD_TEMP   = 0,
        TYPE_BD_POWER  = 1,
        TYPE_RTD       = 2,
        TYPE_ASP_TEMP  = 3,
        TYPE_BIAS_VOLT = 4,
        TYPE_CCD_CURR  = 5,
        TYPE_CR_VOLT   = 6,
        TYPE_HEATER    = 7,
        TYPE_HV_SWITCH = 8;

    public static final int
        CHAN_TOTAL_POWER = -1,
        CHAN_HTR_VOLTS = 0,
        CHAN_HTR_POWER = 1,
        NUM_HTR_CHANS  = 2,
        CHAN_RTD_INT_TEMP = TempRtds.NUM_RTD_TEMPS,
        CHAN_RTD_INT_VOLT = TempRtds.NUM_RTD_TEMPS + 1,
        NUM_RTD_CHANS     = TempRtds.NUM_RTD_TEMPS + 2;

    static final int
        REB_NUM_MASK = 0x03,
        RAFT_NUM_MASK = 0x3f,
        RAFT_NUM_SHIFT = 2;

    static final int
        POWER_ON_DELAY = 160,
        POWER_OFF_DELAY = 50;

    static final int
        DAC_LIMIT = 4095;

    static final double
        DAC_CONV = DAC_LIMIT / 5.0,
        HTR_CONV = DAC_CONV / (1.0 + 10.0 / 10.0),
        HTR_CONV_R5 = 2.0 * DAC_CONV / (1.0 + 30.0 / 10.0),
        HEATER_OHMS = 14.5,   // Estimated value at -100C
        HTR_LEAD_OHMS = 0.3;  // Lead resistance

    static final int
        ADC_PCLK_L = 0,
        ADC_PCLK_U = 1,
        ADC_SCLK_L = 2,
        ADC_SCLK_U = 3,
        ADC_RG_L = 4,
        ADC_RG_U = 5,
        ADC_GD_0 = 6,
        ADC_OD_0 = 7,
        ADC_OG_0 = 8,
        ADC_RD_0 = 9,
        NUM_WREB_ADCS = 10,
        ADC_GD_1 = 10,
        ADC_OD_1 = 11,
        ADC_OG_1 = 12,
        ADC_RD_1 = 13,
        NUM_GREB_ADCS = 14,
        ADC_GD_2 = 14,
        ADC_OD_2 = 15,
        ADC_OG_2 = 16,
        ADC_RD_2 = 17,
        NUM_SREB_ADCS = 18;

    private static enum ChannelGroup {
        rebTemp,
        aspicTemp,
        boardPower,
        crVolt,
        biasVolt,
        ccdCurrent;
    }

    private List<MonitorUpdateTask> tasksForDevice = null;
    private MonitorUpdateTask boardPowerMonitoringTask = null;
        
    /*
     *  Private lookup maps, etc
     */
    private static final Map<String,Integer> hdwTypeMap = new HashMap<>();
    static {
        hdwTypeMap.put("DAQ",  BaseSet.HDW_TYPE_DAQ);
        hdwTypeMap.put("DAQ0", BaseSet.HDW_TYPE_DAQ0);
        hdwTypeMap.put("DAQ1", BaseSet.HDW_TYPE_DAQ1);
        hdwTypeMap.put("DAQ2", BaseSet.HDW_TYPE_DAQ2);
        hdwTypeMap.put("DAQ4", BaseSet.HDW_TYPE_DAQ4);
        hdwTypeMap.put("PCI",  BaseSet.HDW_TYPE_PCI1);
        hdwTypeMap.put("PCI0", BaseSet.HDW_TYPE_PCI0);
        hdwTypeMap.put("PCI1", BaseSet.HDW_TYPE_PCI1);
    }
    private static final Map<String,Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("TEMP",   TYPE_BD_TEMP);
        typeMap.put("POWER",  TYPE_BD_POWER);
        typeMap.put("RTD",    TYPE_RTD);
        typeMap.put("ATEMP",  TYPE_ASP_TEMP);
        typeMap.put("BIAS",   TYPE_BIAS_VOLT);
        typeMap.put("CURR",   TYPE_CCD_CURR);
        typeMap.put("CRVOLT", TYPE_CR_VOLT);
        typeMap.put("HEATER", TYPE_HEATER);
        typeMap.put("HVSWCH", TYPE_HV_SWITCH);
    }
    private static final int[] adcChans = new int[NUM_SREB_ADCS];
    static {
        adcChans[ADC_PCLK_L] = SlowAdcs.CHAN_CKP_L;
        adcChans[ADC_PCLK_U] = SlowAdcs.CHAN_CKP_U;
        adcChans[ADC_SCLK_L] = SlowAdcs.CHAN_SCK_L;
        adcChans[ADC_SCLK_U] = SlowAdcs.CHAN_SCK_U;
        adcChans[ADC_RG_L] = SlowAdcs.CHAN_RG_L;
        adcChans[ADC_RG_U] = SlowAdcs.CHAN_RG_U;
        adcChans[ADC_GD_0] = SlowAdcs.CHAN_GD_0;
        adcChans[ADC_OD_0] = SlowAdcs.CHAN_OD_0;
        adcChans[ADC_RD_0] = SlowAdcs.CHAN_RD_0;
        adcChans[ADC_OG_0] = SlowAdcs.CHAN_OG_0;
        adcChans[ADC_GD_1] = SlowAdcs.CHAN_GD_1;
        adcChans[ADC_OD_1] = SlowAdcs.CHAN_OD_1;
        adcChans[ADC_RD_1] = SlowAdcs.CHAN_RD_1;
        adcChans[ADC_OG_1] = SlowAdcs.CHAN_OG_1;
        adcChans[ADC_GD_2] = SlowAdcs.CHAN_GD_2;
        adcChans[ADC_OD_2] = SlowAdcs.CHAN_OD_2;
        adcChans[ADC_RD_2] = SlowAdcs.CHAN_RD_2;
        adcChans[ADC_OG_2] = SlowAdcs.CHAN_OG_2;
    }
    private static final String[] adcNames = new String[NUM_SREB_ADCS];
    static {
        adcNames[ADC_PCLK_L] = "PclkLow";
        adcNames[ADC_PCLK_U] = "PclkHigh";
        adcNames[ADC_SCLK_L] = "SclkLow";
        adcNames[ADC_SCLK_U] = "SclkHigh";
        adcNames[ADC_RG_L] = "RGLow";
        adcNames[ADC_RG_U] = "RGHigh";
        adcNames[ADC_GD_0] = "GD0";
        adcNames[ADC_OD_0] = "OD0";
        adcNames[ADC_OG_0] = "OG0";
        adcNames[ADC_RD_0] = "RD0";
        adcNames[ADC_GD_1] = "GD1";
        adcNames[ADC_OD_1] = "OD1";
        adcNames[ADC_OG_1] = "OG1";
        adcNames[ADC_RD_1] = "RD1";
        adcNames[ADC_GD_2] = "GD2";
        adcNames[ADC_OD_2] = "OD2";
        adcNames[ADC_OG_2] = "OG2";
        adcNames[ADC_RD_2] = "RD2";
    }
    @SuppressWarnings("MismatchedQueryAndUpdateOfCollection")
    private static final Map<Integer,Integer> hwChanToAdcs = new HashMap<>();
    static {  // this is hwChan: adcChans[] array index map
        hwChanToAdcs.put(SlowAdcs.CHAN_CKP_L, ADC_PCLK_L);
        hwChanToAdcs.put(SlowAdcs.CHAN_CKP_U, ADC_PCLK_U);
        hwChanToAdcs.put(SlowAdcs.CHAN_SCK_L, ADC_SCLK_L);
        hwChanToAdcs.put(SlowAdcs.CHAN_SCK_U, ADC_SCLK_U);
        hwChanToAdcs.put(SlowAdcs.CHAN_RG_L, ADC_RG_L);
        hwChanToAdcs.put(SlowAdcs.CHAN_RG_U, ADC_RG_U);
        hwChanToAdcs.put(SlowAdcs.CHAN_GD_0, ADC_GD_0);
        hwChanToAdcs.put(SlowAdcs.CHAN_OD_0, ADC_OD_0);
        hwChanToAdcs.put(SlowAdcs.CHAN_RD_0, ADC_RD_0);
        hwChanToAdcs.put(SlowAdcs.CHAN_OG_0, ADC_OG_0);
        hwChanToAdcs.put(SlowAdcs.CHAN_GD_1, ADC_GD_1);
        hwChanToAdcs.put(SlowAdcs.CHAN_OD_1, ADC_OD_1);
        hwChanToAdcs.put(SlowAdcs.CHAN_RD_1, ADC_RD_1);
        hwChanToAdcs.put(SlowAdcs.CHAN_OG_1, ADC_OG_1);
        hwChanToAdcs.put(SlowAdcs.CHAN_GD_2, ADC_GD_2);
        hwChanToAdcs.put(SlowAdcs.CHAN_OD_2, ADC_OD_2);
        hwChanToAdcs.put(SlowAdcs.CHAN_RD_2, ADC_RD_2);
        hwChanToAdcs.put(SlowAdcs.CHAN_OG_2, ADC_OG_2);
    }

    private static final int SEQ_CLOCK_LINES = Sequencer.SEQ_LINE_P1 | Sequencer.SEQ_LINE_P2 | Sequencer.SEQ_LINE_P3
        | Sequencer.SEQ_LINE_P4 |Sequencer.SEQ_LINE_S1 | Sequencer.SEQ_LINE_S2
        | Sequencer.SEQ_LINE_S3 | Sequencer.SEQ_LINE_RG;


    /*
     *  Data fields.
     */
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    @SuppressWarnings("MismatchedQueryAndUpdateOfCollection")
    private final Map<String, Channel> childChannels = new HashMap<>();

    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    @SuppressWarnings("MismatchedQueryAndUpdateOfCollection")
    private final Map<String, Control> controlsMap = new LinkedHashMap<>();   // Control channels

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter = "S(\\d\\d)/Seg\\d\\d/I")
    @SuppressWarnings("MismatchedQueryAndUpdateOfCollection")
    private final Map<String, Channel> currentChannels = new HashMap<>();
    
    @LookupPath
    String rebPath;
    
    private final int id; 

    @ConfigurationParameter(name=SERIAL_NUM, category=RAFTS, units="unitless", description="The REB Serial Number")
    private volatile long serialNum;
    @ConfigurationParameter(category=RAFTS, isFinal=true, units="unitless", description="True if a WREB is being used with a science CCD (AuxTel only)")
    private volatile boolean useScienceCCD = false;
    
    @ConfigurationParameter(category=RAFTS, description = "Disables the RTD derived CCD type check.", units="unitless")
    private volatile boolean disableRTDHardwareCheck = false;

    // ODI
    @ConfigurationParameter(category=RAFTS_POWER, range="0.0..0.006", description="Max OFF_State ODI (A) per CCD", units="Amps")
    private volatile double  odiQmax;  // ~0.005
    @ConfigurationParameter(category=RAFTS_POWER, description="Minimum ON_State ODI (A) per CCD", units="Amps")
    private volatile double  odiAmin;  // ~0.023
    // CLKLI
    @ConfigurationParameter(category=RAFTS_POWER, range="0.0..0.050", description="Max OFF_State CLKLI (A) per REB", units="Amps")
    private volatile double  clkliQmax; // ~0.045
    @ConfigurationParameter(category=RAFTS_POWER, description="Minimum ON_State CLKLI (A) per REB", units="Amps")
    private volatile double  clkliAmin; // ~0.053
    // CLKHI
    @ConfigurationParameter(category=RAFTS_POWER, range="0.0..0.093", description="Max OFF_State CLKHI (A) per REB", units="Amps")
    private volatile double  clkhiQmax;  // ~0.093
    @ConfigurationParameter(category=RAFTS_POWER, description="Minimum ON_State CLKHI (A) per REB", units="Amps")
    private volatile double  clkhiAmin;  // ~0.103
    // Limits for voltage offset scheme
    @ConfigurationParameter(category=RAFTS_POWER, range="0.0..0.5", description="Max allowed SlowAdcs step", units="Volts")
    private volatile double  maxStep;        // expect ~0.5V or less
    @ConfigurationParameter(category=RAFTS_POWER, range="0.3..1.5", description="Max allowed SlowAdcs delta", units="Volts")
    private volatile double  maxDelta;        // expect ~1.0V or less
    @ConfigurationParameter(category=RAFTS_POWER, range="0.01..0.2", description="Min allowed SlowAdcs tolerance", units="Volts")
    private volatile double  minTol;         // expect to use 0.07

    // extra monitoring after trigger (reb goes online or ccds powered on
    @ConfigurationParameter(name="nPowerOnPub", category=RAFTS_POWER,
                                                description = "Number of consequtive monitor-update pubs after trigger", units="unitless")
    private volatile int nPowerOnPub = 15;

    //Variable used to disable/enable the back bias switch
    //When the back bias switch is disabled it cannot be closed;
    //it has to be enabled before.
    private volatile boolean isBackBiasSwitchDisabled = false;
    private volatile String backBiasDisabledReason = null;
    //This lock works in conjunction with the back bias switch enable variable above.
    private final Object backBiasEnableLock = new Object();
    
    // Set from the Groovy file
    private String hdwType = "daq";
    private String ifcName;   // Only if not using DAQ V2/4; normally set via a call
    private volatile int rebType = BaseSet.REB_TYPE_SCIENCE;

    private static final Logger LOG = Logger.getLogger(REBDevice.class.getName());
    private final BaseSet bss = new BaseSet();
    private final StatusSet sts = new StatusSet(bss);
    private final TempAdcs tmp = new TempAdcs(bss);
    private final PowerAdcs pwr = new PowerAdcs(bss);
    private final SlowAdcs slow = new SlowAdcs(bss);
    private final TempRtds rtd = new TempRtds(bss);
    private final BoardDacs dac = new BoardDacs(bss);
    private final Aspic asp = new Aspic(bss);
    private final Sequencer seq = new Sequencer(bss);
    private final ImageClient imc = new ImageClient(bss);
    private final BiasControl[] biases = new BiasControl[Asic.MAX_STRIPS];
    private final AspicControl[] aspics = new AspicControl[Asic.MAX_ASICS];
    private final SequencerProc sqp = new SequencerProc(bss);
    private volatile DacControl dacCtrl;
    private final Map<Integer, String> sadcChannelMap = new HashMap<>();
    private final Map<Integer, String> pwrChannelMap = new HashMap<>();
    private volatile long hwSerialNum = 0;
    private volatile boolean serialNumValid = false;
    
    protected int hdwTypeI, configCcdMask;
    
    protected volatile int realCcdMask, hwVersion;
    
    private volatile int maxTemp, maxPower, maxAtemp, maxCcdI, numBiasVolt, numCcdCurrents,
                numTemp, numPower, numAtemp, numCrVolt, numRtd;
    private volatile boolean dacRaw, biasRaw;
    private double[] tempValues, powerValues, atempValues, adcValues, ccdCurrents;
    
    private final double[] heaterValues = new double[NUM_HTR_CHANS];
    private volatile int errorCount;
    private volatile int dacLoadDelay = 0, dacClearDelay = 0;
    private volatile CCDType ccdType = null, hwCcdType = null;
    private Reb rebGeometry;
    private volatile boolean updateCCDsPowerState = false;
    private volatile int exceptionCount = 0;
    private volatile String exceptionText;
    
    private volatile boolean isReadDAQ = true;
    
    /**
     * Constructor
     * @param id The ID of this REB device, using Owens numbering scheme (e.g. R22 = 88)
     */
    public REBDevice(int id) {
        this.id = id;
    }
    
    public void setRebGeometry(Reb reb) {
        this.rebGeometry = reb;
        setCcdType(reb.getCCDType());
    }

    public Reb getRebGeometry() {
        return rebGeometry;
    }

    public void setIsRealDAQ(boolean isRealDAQ) {
        this.isReadDAQ = isRealDAQ;
    }
    
    @Override
    public void postStart() {
        sce.addConfigurationListener(this);
    }


    /**
     *
     */
    synchronized public void setUpdateCCDsPowerState()
    {
       updateCCDsPowerState = true;
       LOG.log(Level.INFO, "setting updateCCDsPowerState flag on {0}", rebPath);
    }

    /**
     *
     */
    @Command(type=CommandType.ACTION, description="Clear the updateCCDsPowerState flag")
    synchronized public void clearUpdateCCDsPowerState()
    {
       updateCCDsPowerState = false;
       LOG.log(Level.INFO, "clearing updateCCDsPowerState flag on {0}", rebPath);
    }


    @Command(type=CommandType.QUERY, description="Get the value of the updateCCDsPowerState flag")
    synchronized public boolean getUpdateCCDsPowerState()
    {
       updateCCDsPowerState = false;
       LOG.log(Level.INFO, String.format("updateCCDsPowerState= %s on %s",
                                         updateCCDsPowerState ? "TRUE" : "FALSE", rebPath));
       return updateCCDsPowerState;
    }

    /**
     *  Validates configuration parameter change.
     *
     *  @param  params  The map of configuration parameter values
     */
    @Override
    public void validateBulkChange(Map<String, Object> params)
    {
        long sNum = (Long)params.get(SERIAL_NUM);
        if (hwSerialNum != 0) {
            if (sNum != -1 && sNum != hwSerialNum) {
                // Note, this exception vetos the configuration change, so no state change should occur. 
                String msg = "Configuration has wrong REB s/n: " + sNum + "; " + rebPath + " s/n is " + hwSerialNum;
                throw new IllegalArgumentException(msg);
            }
            LOG.log(Level.INFO, String.format("Configured %s s/n (%d) is valid", rebPath, sNum));
            serialNumValid = true;
            stateService.updateAgentComponentState(this, RebValidationState.VALID);                        
        }
    }


    /**
     *  Triggers activities when configuration is changed
     *  1st use is to update CCDsPowerState if relevant parameters changed
     *
     * @param newConfigurationInfo
     * @param oldConfigurationInfo
     * @param configOperation
     */
    @Override
    public void configurationChanged(ConfigurationInfo newConfigurationInfo,
                   ConfigurationInfo oldConfigurationInfo, ConfigurationOperation configOperation) {
        // These are the changes between new and old configuration.
        List<ConfigurationParameterInfo> latestChanges = newConfigurationInfo.diff(oldConfigurationInfo);
        if (!latestChanges.isEmpty() && updateCCDsPowerState) {
            LOG.log(Level.INFO, String.format("%s:configurationChanged() found %d changes", rebPath, latestChanges.size()));
            List<AdcData> adcDataList = new ArrayList<>();
            try {
                setCCDsPowerState(adcDataList, true);
            }
            catch (RaftException e) { // force off since unknown state
                LOG.log(Level.SEVERE, String.format("FAILED to determine CCDsPowerState for %s", rebPath));
                LOG.log(Level.SEVERE, String.format("CCDsPowerState changed to %s",
                            stateService.getComponentState(rebPath, CCDsPowerState.class).name())); 
                LOG.log(Level.SEVERE, String.format("forcing powerCCDsOff() called for %s", rebPath));
                try {
                    powerCCDsOff(adcDataList);
                }
                catch (RaftException et) {
                    // TODO raise and alert here to get the REB shut off???
                    LOG.log(Level.SEVERE, String.format("powerCCDsOff() FAILED for %s", rebPath));
                    LOG.log(Level.SEVERE, String.format("CCD state undefined for %s", rebPath));
                    LOG.log(Level.SEVERE, String.format("powerCCDsOff failed, check/reboot %s", rebPath), et);
                }
            }
            publishAllAdcData(adcDataList);
            clearUpdateCCDsPowerState();
        }
    }

    /**
     *  Gets the REB ID.
     *
     *  @return  The REB ID
     */
    @Command(type=CommandType.QUERY, description="Get the REB ID")
    public int getId()
    {
        return id;
    }


    /**
     *  Gets the network interface name.
     *
     *  @return  The name of the network interface being used
     */
    @Command(type=CommandType.QUERY, description="Get the network interface name")
    public String getIfcName()
    {
        return ifcName;
    }
    
    public void setIfcName(String ifcName) {
        this.ifcName = ifcName;
    }


    /**
     *  Gets the mask of CCDs being used.
     *
     *  @return  The mask of active CCDs
     */
    @Command(type=CommandType.QUERY, description="Get the mask of CCDs being used")
    public int getCcdMask()
    {
        return realCcdMask;
    }


    /**
     *  Sets the CCD type.
     *
     *  @param  ccdType  The CCD type
     */
    // TODO resolve how ccdtype is getting set, power on will fail if
    // dacLoadDelay is not set except when no CCDs connected
    //@ConfigurationParameterChanger
    public void setCcdType(CCDType ccdType)
    {
        // TODO: We should check that ccdType is never set to null. but right now this
        // breaks some tests (specifically ATSFileWritingTest). In addtion we should insist
        // that ccdType has been set before initialization is complete  (LSSTCCSRAFTS-525)
//        if (ccdType == null) {
//            throw new IllegalArgumentException("ccdType cannot be null");
//        }
        LOG.log(Level.INFO, "setCcdType to {0} for {1}", new Object[]{ccdType, rebPath});
        this.ccdType = ccdType;
        dacLoadDelay = ccdType == null ? 0 : POWER_ON_DELAY;
        dacClearDelay = ccdType == null ? 0 : POWER_OFF_DELAY;
    }


    /**
     *  Gets the CCD type.
     *
     *  @return  The CCD type
     */
    @Command(type=CommandType.QUERY, description="Get the type of CCDs being used")
    public CCDType getCcdType()
    {
        return ccdType;
    }


    /**
     *  Gets the hardware CCD type.
     *
     *  @return  The hardware CCD type
     */
    @Command(type=CommandType.QUERY, description="Get the hardware type of CCDs being used")
    public CCDType getHwCcdType()
    {
        return hwCcdType;
    }


    /**
     *  Sets whether a science CCD is being used on a WREB.
     *
     *  @param  useScience  Whether it's a science CCD
     */
    @Command(type=CommandType.ACTION, description="Set whether a WREB is using a science CCD")
    public void setScienceCCD(@Argument(description="Whether a science CCD") boolean useScience)
    {
        rtd.setScienceCCD(useScience);
    }


    /**
     *  Sets the client factory.
     *
     *  @param  factory  The client factory
     */
    public void setClientFactory(ClientFactory factory)
    {
        if (factory != null) {
            bss.setClientFactory(factory);
            imc.setClientFactory(factory);
        }
    }


    /**
     *  Gets the REB type.
     *
     *  @return  The REB type
     */
    @Command(type=CommandType.QUERY, description="Gets the REB type")
    public int getRebType()
    {
        return rebType;
    }


    /**
     *  Gets the REB number.
     *
     *  @return  The REB number (0 - 2) within its raft
     */
    @Command(type=CommandType.QUERY, description="Gets the REB number")
    public int getRebNumber()
    {
        return id & REB_NUM_MASK;
    }


    /**
     *  Gets the raft number.
     *
     *  @return  The raft number (00 - 44) for the REB
     */
    @Command(type=CommandType.QUERY, description="Gets the raft number")
    public int getRaftNumber()
    {
        return (id >> RAFT_NUM_SHIFT) & RAFT_NUM_MASK;
    }


    /**
     *  Gets the hardware type.
     *
     *  @return  The hardware type
     */
    @Command(type=CommandType.QUERY, description="Gets the hardware type")
    public int getHdwType()
    {
        return hdwTypeI;
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version, or 0 if the REB is not connected
     */
    @Command(type=CommandType.QUERY, description="Get the hardware version")
    public int getHwVersion()
    {
        return hwVersion;
    }


    /**
     *  Gets the serial number.
     *
     *  @return  The serial number, or 0 if the REB is not connected
     */
    @Command(type=CommandType.QUERY, description="Get the serial number")
    public long getSerialNumber()
    {
        return hwSerialNum;
    }


    /**
     *  Gets whether the serial number is valid.
     *
     *  @return  The serial number validity
     */
    public boolean isSerialNumValid()
    {
        return serialNumValid;
    }


    /**
     *  Adds channel name or path to map.
     *
     *  @param  map  The map to add to
     *  @param  hwChan  The hw channel number
     *  @param  chName  The channel name
     */
    private void addToChannelMap(Map map, int hwChan, String chName)
    {
        Channel chan = childChannels.get(chName);
        if (chan == null) {
            chan = mon.getChannel(chName);
        }
        map.put(hwChan, chan.getPath());
    }


    /**
     *  Performs basic initialization.
     * 
     * Invoked when HasLifecycle::init is called on Device class.
     * 
     * Device fields that are initialized exclusively in this method don't need
     * to be marked as volatile.
     * 
     *  Checks that Groovy specifications are consistent
     */
    @Override
    protected void initDevice()
    {   
        stateService.registerState(RebDeviceState.class, "Reb Device ONLINE/OFFLINE State", this);
        stateService.registerState(RebValidationState.class, "Reb Device Validation State", this);        
        stateService.registerState(HVBiasState.class, "High Voltage Bias ON/OFF State", this);
        stateService.registerState(CCDsPowerState.class, "CCDs Power ON/OFF State", this);
        stateService.updateAgentComponentState(this, RebDeviceState.OFFLINE, RebValidationState.UNKNOWN,
                HVBiasState.UNKNOWN, CCDsPowerState.UNKNOWN);
        
        if (ifcName == null) {
            String exceptionError = "Fatal configuration error: "+rebPath+" config parameter ifcName has not been specified";
            LOG.log(Level.SEVERE, exceptionError);
            throw new RuntimeException(exceptionError);
        }
        Integer type = hdwTypeMap.get(hdwType.toUpperCase());
        if (type == null) {
            String exceptionError = "Fatal configuration error: "+rebPath+" config parameter hdwType is invalid";
            LOG.log(Level.SEVERE, exceptionError);
            throw new RuntimeException(exceptionError);
        }
        hdwTypeI = type;

        imc.setRebId("reb-" + id);

        /*
         *  Get masks of Bias DACs and ASPICs present
         */
        int biasMask = 0, aspMask = 0, aspicMask = 0;

        for (Control ctrl : controlsMap.values()) {
            if (ctrl instanceof BiasControl) {
                biasMask |= 1 << ctrl.getHwChan();
            }
            else if (ctrl instanceof AspicControl) {
                aspMask |= 1 << ctrl.getHwChan();
            }
        }

        /*
         *  Check that ASPICs come in pairs
         *  Generate the maximal mask of strips present
         */
        for (int j = 0; j < Asic.MAX_STRIPS; j++, aspMask >>= 2) {
            int pair = aspMask & 3;
            if (pair == 0) continue;
            if (pair == 3) {
                aspicMask |= (1 << j);
            }
            else {
                LOG.log(Level.SEVERE, "Unpaired configuration for {0} ASPIC {1} ignored",
                        new Object[]{rebPath, 2 * j + pair - 1});
            }
        }
        if (biasMask != 0 && biasMask != aspicMask) {
            LOG.log(Level.WARNING, "{0} ASPIC and bias configurations don't match", rebPath);
        }
        configCcdMask = biasMask | aspicMask;

        /*
         *  Fill the record of control devices
         */
        for (Control ctrl : controlsMap.values()) {
            if (ctrl instanceof DacControl dacControl) {
                dacCtrl = dacControl;
            }
            else if (ctrl instanceof BiasControl biasControl) {
                if (((1 << ctrl.getHwChan()) & configCcdMask) != 0) {
                    biases[ctrl.getHwChan()] = biasControl;
                }
            }
            else if (ctrl instanceof AspicControl aspicControl) {
                if (((1 << ctrl.getHwChan() / 2) & configCcdMask) != 0) {
                    aspics[ctrl.getHwChan()] = aspicControl;
                }
            }
        }
        
        tasksForDevice = mon.getMonitorTasksList();
        String taskName = rebPath+"/"+ChannelGroup.boardPower.name();
        for ( MonitorUpdateTask t : tasksForDevice) {
            if ( t.getName().equals(taskName) ) {
                boardPowerMonitoringTask = t;                
                break;
            }
        }
        if ( boardPowerMonitoringTask == null ) {
            throw new RuntimeException("Could not find task "+taskName);
        }

        /*
         *  Set initial value of maximum CCD mask
         */
        bss.setDefaultRebType(BaseSet.REB_TYPE_SCIENCE);
        try {
            realCcdMask = configCcdMask & ((1 << bss.getNumStrips()) - 1);
        }
        catch (REBException e) {
            LOG.log(Level.SEVERE, "Error getting expected {0} CCD count: {1}", new Object[]{rebPath, e}); // Shouldn't happen
        }
        fullName = name;
        
    }
    
    
    /**
     *  Performs full initialization.
     * 
     *  Opens connection to the REB and obtains information about it
     */
    @Override
    protected void initialize()
    {
        try {
            bss.open(hdwTypeI, id, ifcName);
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
            }
            bss.disable();
            bss.setTime();
            bss.enable();
            rebType = bss.getRebType();
            hwVersion = bss.getHwVersion();
            maxTemp = tmp.getNumRegs();
            maxPower = pwr.getNumRegs();
            maxAtemp = slow.getNumAspicTemps();
            maxCcdI = slow.getNumCcdCurrents();
            numTemp = numPower = numAtemp = numCrVolt = numRtd = numBiasVolt = numCcdCurrents = 0;
            tempValues = new double[maxTemp];
            powerValues = new double[maxPower];
            adcValues = new double[SlowAdcs.NUM_BIAS_CHAN];
            ccdCurrents = new double[maxCcdI];
            atempValues = new double[maxAtemp];
            realCcdMask = configCcdMask & ((1 << bss.getNumStrips()) - 1);
            /*
            try {
                seq.setCcdMask(maxCcdMask);
            }
            catch (REBException e) {
                log.severe("Error writing stripe selection");
            }
            */
            if (dacCtrl != null && !dacCtrl.checkConfig()) {
                dacCtrl = null;
            }
            dacRaw = dacCtrl != null ? dacCtrl.isRaw() : false;
            biasRaw = false;
            for (int j = 0; j < biases.length; j++) {
                BiasControl bias = biases[j];
                if (bias != null && !bias.checkConfig()) {
                    bias = null;
                }
                if (bias != null) {
                    biasRaw = bias.isRaw();
                }
                biases[j] = bias;
            }
            for (int j = 0; j < aspics.length; j++) {
                AspicControl aspic = aspics[j];
                if (aspic != null && !aspic.checkConfig()) {
                    aspic = null;
                }
                aspics[j] = aspic;
            }
            try {
                hwSerialNum = bss.getRebSerial();
            }
            catch (REBException e) {
                LOG.log(Level.SEVERE, "Error reading {0} serial number", rebPath);
                hwSerialNum = -1;
            }
            rtd.initialize();
            if ( ! disableRTDHardwareCheck ) {            
                try {
                    Thread.sleep(100);
                }
                catch (InterruptedException e) {
                    LOG.log(Level.SEVERE, "Thread.sleep(100) interrupted at rtd.initialize() for {0}", rebPath);
                }
                try {
                    int type = rtd.getCCDType();
                    hwCcdType = type == TempRtds.CCD_TYPE_E2V ? CCDType.getCCDType("e2v")
                            : type == TempRtds.CCD_TYPE_ITL ? CCDType.getCCDType("itl") : null;
                    if (RunMode.isSimulation()) {
                        hwCcdType = CCDType.getCCDType(ccdType.getName());
                    }
                    LOG.log(Level.INFO, "found hwCcdType {0} for {1}", new Object[]{hwCcdType, rebPath});
                    CCDType manufacturerType = CCDType.getCCDType(ccdType.getManufacturer());
                    if (hwCcdType == null || !hwCcdType.equals(manufacturerType)) {
                        throw new RuntimeException("CCD Hardware type " + hwCcdType + " is inconsistent with configured CCD Type " + manufacturerType);
                    }
                } catch (REBException e) {
                    LOG.log(Level.SEVERE, "Error getting CCD type for" + rebPath, e);
                    hwCcdType = null;
                }
            }
            
            rtd.setScienceCCD(useScienceCCD);
            errorCount = 0;
            initSensors();                     // cause initChannel() call on each monitored quantity
            imc.open(hdwTypeI, id, ifcName);

            logExceptionCount();
            LOG.log(Level.INFO, "Connected to {0}", rebPath);
            LOG.log(Level.INFO, String.format("%s has firmware version: 0x%x", rebPath, hwVersion));
            LOG.log(Level.INFO, String.format("%s has serialNumber: %d", rebPath, hwSerialNum));

            exceptionCount = 0;
            // determine state of sequencer
            try {
                if (sqp.isRunning()) { // Sequencer is running, just log for now
                    LOG.log(Level.INFO, "Sequencer is running for {0}", rebPath);
                } else {
                    LOG.log(Level.INFO, "Sequencer is idle for {0}", rebPath);
                }
            } catch (REBException e) {
                    LOG.log(Level.SEVERE, "Failed to determine sequencer state for "+rebPath, e);
            }

            // throw-away, OD0 value is always corrupted on first read at power up
            try {
                slow.readVoltages();
            } catch (REBException e) {
                LOG.log(Level.SEVERE, "SlowAdcs.readVoltages() exception for "+rebPath, e);
            }
            
            serialNumValid = serialNum == -1 || serialNum == hwSerialNum;
            if (!serialNumValid) {
                stateService.updateAgentComponentState(this, RebValidationState.INVALID);        
                LOG.log(Level.SEVERE, "{0} s/n ({1}) doesn''t match configuration value ({2})",
                        new Object[]{rebPath, hwSerialNum, serialNum});
            } else {
                stateService.updateAgentComponentState(this, RebValidationState.VALID);                        
            }            

            try {
                //TO-DO: is this the right place for it?
                //Check that the back bias switch is in a cosistent state
                //If we got this far then the device is online, so we pass "true" to the checkBackBiasDisabledState method.
                checkBackBiasDisabledState(true);
            } catch (RaftException re ) {
                //WHAT SHOULD HAPPEN HERE???
            }

            if ( inited ) {
                forceDataPublicationForNextUpdateIterations(nPowerOnPub);
            }
            
            setOnline(true);            
        }
        catch (REBException e) {
            close();
            String text = e.getMessage();
            if (exceptionCount == 0 || !exceptionText.equals(text)) {
                logExceptionCount();
                LOG.log(Level.SEVERE, "Error connecting to " + name, e);

                exceptionCount = 0;
                exceptionText = text;
            }
            exceptionCount++;
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            rebType = BaseSet.REB_TYPE_UNKNOWN;
            hwVersion = 0;
            hwSerialNum = 0;
            hwCcdType = null;
            bss.close();
        }
        catch (REBException e) {
            // Need to keep going
        }
        try {
            imc.close();
        }
        catch (REBException e) {
            // Need to keep going
        }
    }

    private Integer convertTypeStrToInt(String typeStr) throws Exception {
        Integer iType = typeMap.get(typeStr.toUpperCase());
        if (iType == null) {
            String exceptionError = "Fatal configuration error: "+rebPath+" config parameter channel type "+typeStr;
            LOG.log(Level.SEVERE, exceptionError);
            throw new RuntimeException(exceptionError);
        }
        return iType;        
    }
    

    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  ch     The Channel to check
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        Integer iType = convertTypeStrToInt(ch.getTypeStr());
        return new int[]{iType, 0};
    }


    /**
     *  Initializes a channel (in this case only checks HW channel number).
     *
     *  @param  ch     The channel to initialize
     */
    @Override
    protected void initChannel(Channel ch)
    {
        try {
            boolean chanOk;
            int type = ch.getType();
            int hwChan = ch.getHwChan();
            String channelName = ch.getName();
            if (type == TYPE_BIAS_VOLT || type == TYPE_CR_VOLT) {
                chanOk = slow.testChannel(hwChan);
                if (chanOk) {
                    addToChannelMap(sadcChannelMap, hwChan, channelName);
                }
            }
            else {
                int maxChan = type == TYPE_BD_TEMP ? maxTemp :
                              type == TYPE_BD_POWER ? maxPower :
                              type == TYPE_ASP_TEMP ? maxAtemp :
                              type == TYPE_CCD_CURR ? maxCcdI :
                              type == TYPE_RTD ? NUM_RTD_CHANS :
                              type == TYPE_HEATER ? NUM_HTR_CHANS : 
                              type == TYPE_HV_SWITCH ? 1 : 0;
                chanOk = (hwChan >= 0 || type == TYPE_BD_POWER) && hwChan < maxChan;
                if (chanOk && type == TYPE_BD_POWER && hwChan >= 0) {
                    addToChannelMap(pwrChannelMap, hwChan, channelName);
                }
            }
            if (!chanOk) {
                String exceptionError = "Fatal configuration error: "+rebPath+" config parameter hw channel number "+hwChan;
                LOG.log(Level.SEVERE, exceptionError);
                throw new RuntimeException(exceptionError);
            }
            switch (type) {
            case TYPE_BD_TEMP -> numTemp++;
            case TYPE_BD_POWER -> numPower++;
            case TYPE_ASP_TEMP -> numAtemp++;
            case TYPE_BIAS_VOLT -> numBiasVolt++;
            case TYPE_CR_VOLT -> numCrVolt++;
            case TYPE_CCD_CURR -> numCcdCurrents++;
            case TYPE_RTD -> numRtd++;
            }
        }
        catch (Exception e) {
            //LSSTCCSRAFTS-729 an exception is thrown only with channel initializxation fails
            //with the real DAQ. Simulated/Emulated DAQ still does not support
            //some RebG/W channels
            if ( isReadDAQ ) {
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    protected String getGroupForChannel(Channel ch) {
        try {
            String typeStr = ch.getTypeStr();
            int type = convertTypeStrToInt(typeStr);
            switch (type) {
                case TYPE_BD_TEMP -> {
                    return ChannelGroup.rebTemp.name();
                }
                case TYPE_BD_POWER -> {
                    return ChannelGroup.boardPower.name();
                }
                case TYPE_ASP_TEMP -> {
                    return ChannelGroup.aspicTemp.name();
                }
                case TYPE_CR_VOLT -> {
                    return ChannelGroup.crVolt.name();
                }
                case TYPE_BIAS_VOLT -> {
                    return ChannelGroup.biasVolt.name();
                }
                case TYPE_CCD_CURR -> {
                    return ChannelGroup.ccdCurrent.name();
                }
            }
            return null;
        } catch (Exception e) {
            throw new RuntimeException("Problem fetching group information for channel "+ch.getName(), e);
        }
    }

    private double[] getNaNArray(int length) {
        double[] allNaN = new double[length];
        for ( int i = 0; i < length; i++ ) {
            allNaN[i] = Double.NaN;
        }
        return allNaN;
    }
    
    // TODO Review exception handing, throw REB or RAFT Exceptions?
    @Override
    protected void readChannelGroup(String group) {
        try {
            ChannelGroup g = ChannelGroup.valueOf(group);
            if (!isOnline()) {
                return;
            }
            String item = "";
            try {

                switch (g) {
                    case aspicTemp -> {
                        if (numAtemp > 0) {  // slow.readAspicTemps() ~0.8 ms
                            item = "aspic temperatures";
                            atempValues = slow.readAspicTemps(0, maxAtemp);
                        }
                    }
                    case biasVolt, crVolt -> {
                        if (numBiasVolt + numCrVolt > 0) { // exclusive
                            item = "CCD voltages";
                            try {  // slow.readVoltages() ~5.3 ms
                                adcValues = slow.readVoltages();
                            } catch (REBException e) { // TODO -- fix
                                adcValues = getNaNArray(adcValues.length);
                                throw (e);
                            }
                        }
                    }
                    case ccdCurrent -> {
                        if (numCcdCurrents > 0) {
                            item = "CCD output currents";
                            try {  // ~12 ms
                                ccdCurrents = slow.readCurrents();
                            } catch (REBException e) { // TODO -- fix
                                ccdCurrents = getNaNArray(ccdCurrents.length);
                                throw (e);
                            }
                        }
                    }
                    case boardPower -> {
                        if (numPower > 0) {
                            try { // pwr.readAdcs() ~33 ms
                                powerValues = pwr.readAdcs();
                            } catch (REBException e) {
                                powerValues = getNaNArray(powerValues.length);
                                if ( ! (e instanceof REBTimeoutException) ) {
                                    item = "board power";
                                    throw (e);
                                }                                    
                            }
                        }
                    }
                    case rebTemp -> {
                        if (numTemp > 0) {
                            try { // tmp.readAdcs() ~1.7 ms
                                tempValues = tmp.readAdcs();
                            } catch (REBException e) {
                                tempValues = getNaNArray(tempValues.length);
                                if ( e instanceof REBTimeoutException ) {
                                    LOG.log(Level.FINE, "{0}tmp.readAdcs(): Completion wait", rebPath);
                                } else {
                                    item = "board temperatures";
                                    throw (e);
                                }
                            }
                        }
                    }
                    default -> throw new RuntimeException("Implementation is missing for group: " + g);                        
                }
                errorCount = 0;
            } catch (REBException e) {
                LOG.log(Level.SEVERE, "Error reading {0} {1}: {2}", new Object[]{name, item, e.getMessage()});
                checkErrorCount(e.getMessage());
            }

        } catch (Exception e) {
            throw new IllegalArgumentException("Could not read channels for group: " + group, e);
        }
    }

    
    /**
     *  Reads a channel.
     *
     *  @param  ch  The Channel to read.
     *  @return  The read value
     */
    @Override
    protected double readChannel(Channel ch)
    {
        double value = Double.NaN;
        if (isOnline()) {
            String item = "";
            try {
                int type = ch.getType();
                int hwChan = ch.getHwChan();
                switch (type) {
                    case TYPE_RTD -> {
                        item = "RTD";
                        if (hwChan < TempRtds.NUM_RTD_TEMPS) { // 7.4 ms
                            value = rtd.readTemperature(hwChan);
                        } else if (hwChan == CHAN_RTD_INT_TEMP) {
                            value = rtd.readIntTemperature();
                        } else {
                            value = rtd.readIntVoltage();
                        }
                        errorCount = 0;
                    }

                    case TYPE_HV_SWITCH -> {
                        item = "HV bias switch";
                        value = bss.isBackBiasOn() ? 1.0 : 0.0;
                        HVBiasState newState = value > 0 ? HVBiasState.ON : HVBiasState.OFF;
                        stateService.updateAgentComponentState(this, newState);
                        errorCount = 0;
                    }

                    case TYPE_BD_POWER -> {
                        if (hwChan >= 0) {
                            value = powerValues[hwChan];
                        } else {  // sum over I*V to get total power
                            value = 0.0;
                            for (int j = 0; j < powerValues.length; j += 2) {
                                value += powerValues[j] * powerValues[j + 1];
                            }
                            value += heaterValues[CHAN_HTR_POWER];
                        }
                    }

                    case TYPE_BD_TEMP -> value = tempValues[hwChan];

                    case TYPE_ASP_TEMP -> value = atempValues[hwChan];

                    case TYPE_HEATER -> value = heaterValues[hwChan];
                    case TYPE_CCD_CURR -> value = ccdCurrents[hwChan];

                    case TYPE_BIAS_VOLT, TYPE_CR_VOLT -> value = adcValues[hwChan];
                }
            }
            catch (REBException e) {
                LOG.log(Level.SEVERE, "Error reading {0} {1}: {2}", new Object[]{name, item, e.getMessage()});
                checkErrorCount(e.getMessage());
            }
        }
        return value;
    }

    /**
     *  Reads the board power ADCs.
     *
     *  @return  The array of board voltages and currents
     *  @throws  RaftException
     */
    public double[] readPowerAdcs() throws RaftException
    {
        try {
            return pwr.readAdcs();  // ~33 ms
        }
        catch (REBException e) {
            throw new RaftException(e);
        }
    }


    /**
     *  Reads the board power ADCs.
     *
     *  @return  The array of CCD output currents
     *  @throws  RaftException
     */
    public double[] readSlowCurrents() throws RaftException
    {
        try {
            return slow.readCurrents();  // ~12 ms
        }
        catch (REBException e) {
            throw new RaftException(e);
        }
    }

     
    public double[] readSlowCurrents(List<AdcData> dataList) throws RaftException {
        
        CCSTimeStamp tstamp = CCSTimeStamp.currentTime();
        double[] values = null;
        if ( maxCcdI > 0 ) {
            values = readSlowCurrents(); // 12 ms
        } 
        if (dataList != null) {
            for ( Channel ch : currentChannels.values() ) {
                dataList.add( new AdcData(AdcData.TYPE_CURR, ch, ch.convertRawValue(values[ch.getHwChan()]), tstamp));
            }
        }
        return values;
    }


    /**
     *  Reads the board power ADCs and adds them to data list.
     *  This takes ~33 ms reading the 5 (V,I) pairs
     *
     *  @param  dataList  List to receive the read values, etc
     *  @return  The read values
     *  @throws  RaftException 
     */
    public double[] readPowerAdcs(List<AdcData> dataList) throws RaftException
    {
        CCSTimeStamp tstamp = CCSTimeStamp.currentTime();
        double[] values = readPowerAdcs();  // 33 ms
        if (dataList != null) {
            for (int chan = 0; chan < values.length; chan++) {
                dataList.add(new AdcData(AdcData.TYPE_POWER, chan, values[chan], tstamp));
            }
        }
        return values;
    }


    /**
     * Reads the clock and bias voltage slow ADCs and adds them to data list.
     *
     * @param dataList List to receive the read values, etc
     * @return Array of read values
     * @throws RaftException
     */
    public double[] readSlowAdcs(List<AdcData> dataList) throws RaftException
    {
        int nAdc = rebType == BaseSet.REB_TYPE_SCIENCE ? NUM_SREB_ADCS :
                   rebType == BaseSet.REB_TYPE_GUIDER ? NUM_GREB_ADCS :
                   rebType == BaseSet.REB_TYPE_WAVEFRONT ? NUM_WREB_ADCS : 0;
        double[] data = new double[nAdc];
        double[] fullData;
        CCSTimeStamp tstamp = CCSTimeStamp.currentTime();

        if (nAdc == 0) return data;
        try {
            fullData = slow.readVoltages();  // 5.3 ms
        } catch (REBException e) {
            throw new RaftException(e);
        }
        for (int adc = 0; adc < nAdc; adc++) {
            data[adc] = fullData[adcChans[adc]];
            if (dataList != null) {
                dataList.add(new AdcData(AdcData.TYPE_SLOW, adcChans[adc], data[adc], tstamp));
            }
        }
        return data;
    }



    /**
     *  Reads all the ADCs and adds them to data list.
     *
     *  @param  dataList  List to receive the read values, etc
     *  @throws  RaftException
     */
    public void readAllAdcs(List<AdcData> dataList) throws RaftException
    {
        readSlowAdcs(dataList);
        readPowerAdcs(dataList);
    }

    
    /**
     * Gets the bit mask of DAC channels under threshold
     *
     * @return 
     * @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="get underMask protection register value")
    public int getUnders() throws RaftException
    {
        int underMask;
        try {
            underMask = dac.getUnders();
        }
        catch (REBException e) {
            throw new RaftException(e);
        }
        return underMask;
    }


    /**
     *  Gets the name of of a slow ADC.
     *
     *  @param  adc  The ADC ID
     *  @return  The ADC name
     */
    public static String getAdcName(int adc)
    {
        return adcNames[adc];
    }


    /**
     *  Reads the sequencer idle state lines.
     *
     *  @return  The lines value
     *  @throws  RaftException
     */
    public int readSeqIdleState() throws RaftException
    {
        try {
            return seq.readIdleState();
        }
        catch (REBException e) {
            throw new RaftException(e);
        }
    }


    /**
     *  Writes the sequencer idle state lines.
     *
     *  @param  value  The lines value
     *  @throws  RaftException
     */
    public void writeSeqIdleState(int value) throws RaftException
    {
        try {
            seq.writeIdleState(value);
        }
        catch (REBException e) {
            throw new RaftException(e);
        }
    }


    /**
     *  Sets the REB configuration data.
     *
     *  @param  reb  The REB configuration (object)
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Sets the REB configuration data")
    public void setREBConfig(@Argument(name="reb", description="REB configuration data")
                             REB reb) throws RaftException
    {
        dacCtrl.setConfig(reb.getDacs());
        BiasDACS[] biasConfig = reb.getBiases();
        for (int j = 0; j < biasConfig.length; j++) {
            BiasControl bias = biases[j];
            if (bias != null) {
                bias.setConfig(biasConfig[j]);
            }
        }
        ASPIC[] aspicConfig = reb.getAspics();
        for (int j = 0; j < aspicConfig.length; j++) {
            AspicControl aspic = aspics[j];
            if (aspic != null) {
                aspic.setConfig(aspicConfig[j]);
            }
        }
    }


    /**
     *  Gets the REB configuration data.
     *
     *  @return  The REB configuration (object)
     */
    @Command(type=CommandType.QUERY, description="Gets the REB configuration data")
    public REB getREBConfig() 
    {
        REB reb = new REB();
        reb.setId(id);
        reb.setIfcName(ifcName);
        reb.setCcdMask(realCcdMask);
        reb.setMaxCcdMask(realCcdMask);

        reb.setDacVersion(DacControl.getHwVersion(dac));
        reb.setDacRaw(dacRaw);
        reb.setBiasVersion(BiasControl.getHwVersion(dac));
        reb.setBiasRaw(biasRaw);
        reb.setAspicVersion(AspicControl.getHwVersion(asp));

        dacCtrl.getConfig(reb.getDacs());
        @SuppressWarnings("MismatchedReadAndWriteOfArray")
        BiasDACS[] biasConfig = reb.getBiases();
        for (int j = 0; j < biases.length; j++) {
            BiasControl bias = biases[j];
            if (bias != null) {
                biasConfig[j] = bias.getConfig();
            }
        }
        @SuppressWarnings("MismatchedReadAndWriteOfArray")
        ASPIC[] aspicConfig = reb.getAspics();
        for (int j = 0; j < aspics.length; j++) {
            AspicControl aspic = aspics[j];
            if (aspic != null) {
                aspicConfig[j] = aspic.getConfig();
            }
        }

        return reb;
    }


    /**
     * Transversal setting of all TM for the Aspics of this reb.
     * It internally uses the configuration service.
     * 
     * @param tm true for TM, false for normal
     */
    @Command(type=CommandType.CONFIGURATION, description="Set TM for all ASPICs on this REB")
    public void setAllAspicTM(boolean tm) {
        setAllAspic(AspicControl.TM, tm ? 1 : 0);
    }


    /**
     * Transversal setting of all Gains for the Aspics of this reb.
     * It internally uses the configuration service.
     * 
     * @param gain the new gain value
     */
    @Command(type=CommandType.CONFIGURATION, description="Set gain for all ASPICs on this REB")
    public void setAllAspicGain(int gain) {
        setAllAspic(AspicControl.GAIN, gain);
    }


    /**
     * Transversal setting of all Rcs for the Aspics of this reb.
     * It internally uses the configuration service.
     * 
     * @param rc the new rc value
     */
    @Command(type=CommandType.CONFIGURATION, description="Set RC for all ASPICs on this REB")
    public void setAllAspicRc(int rc) {
        setAllAspic(AspicControl.RC, rc);
    }


    /**
     * Transversal setting of a parameter for all the Aspics on the REB.
     * 
     * @param parmName
     * @param val 
     */
    private void setAllAspic(String parmName, int val) {
        setAllAspic(parmName, val, true);
    }
    
    public void setAllAspic(String parmName, int val, boolean commit) {
        for (AspicControl aspic : aspics) {
            if (aspic == null) continue;            
            sce.getComponentConfigurationEnvironment(aspic.aspicPath).submitChange(parmName, val);
        }
        if (commit) {
            sce.commitBulkChange();
        }
    }


    /**
     *  Performs CCD power on/off sequences.
     *
     *  @param  on  Whether to power on (true) or off (false)
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Perform CCD power on/off sequence", level = Command.ENGINEERING_EXPERT)
    public void powerCCDs(@Argument(description="Whether to power on") boolean on) throws RaftException
    {
        if (on) {
            powerCCDsOn();
        }
        else {
            powerCCDsOff();
        }
    }


   /**
     * measure power state of ccds, user version
     * turns off CCDs on error
     *
     * @return String representing CCDsPowerState
     * @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="get CCD Power State", level = Command.ENGINEERING_EXPERT)
    public String getCCDsPowerState() throws RaftException
    {
        List<AdcData> adcDataList = new ArrayList<>();
        CCDsPowerState powerState = getCCDsPowerState(adcDataList);
        publishAllAdcData(adcDataList);
        return powerState.name();
    }

 
    /**
     *  Returns CCDsPowerState
     *
     *  @throws  RaftException
     */
    private CCDsPowerState getCCDsPowerState(List<AdcData> adcDataList) throws RaftException
    {
        CCDsPowerState powerState = CCDsPowerState.UNKNOWN;
        try {
            powerState = setCCDsPowerState(adcDataList, true);
        }
        catch (RaftException e) { // force off since unknown state
            LOG.log(Level.SEVERE, "FAILED to determine CCDsPowerState for {0}", rebPath);
            LOG.log(Level.SEVERE, "CCDsPowerState changed to {0}", stateService.getComponentState(rebPath, CCDsPowerState.class).name()); 
            LOG.log(Level.SEVERE, "calling powerCCDsOff() on {0}", rebPath);
            try {
                powerCCDsOff(adcDataList);
            }
            catch (RaftException et) {
                // TODO raise and alert here to get the REB shut off???
                LOG.log(Level.SEVERE, "powerCCDsOff() FAILED for {0}", rebPath);
                LOG.log(Level.SEVERE, "CCDsPowerState changed to {0}", stateService.getComponentState(rebPath, CCDsPowerState.class).name()); 
                throw new RaftException("powerCCDsOff() FAILED, check/reboot "+rebPath, et);
            }
            publishAllAdcData(adcDataList);
            throw new RaftException("getCCDsPowerState() FAILED for "+rebPath, e);
        }
        return powerState;
    }


    /**
     *  Performs CCD power on sequence.
     *  CCDsPowerState is set to ON (success) or is set in
     *  powerCCDsOff() to OFF or FAULT 0
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Perform CCD power on sequence", level = Command.ENGINEERING_EXPERT)
    public void powerCCDsOn() throws RaftException
    {
        boolean all = true;
        boolean check = true;
        boolean powerOn = true;
        CCDsPowerState powerState;

        List<AdcData> adcDataList = new ArrayList<>();
        readSlowAdcs(adcDataList);
        readPowerAdcs(adcDataList);
        forceDataPublicationForNextUpdateIterations(nPowerOnPub);

        // satisfy prefrequisites
        try (@SuppressWarnings("unused") AutoCloseableReentrantLock lock = SequencerProc.lockSequencer(1L, TimeUnit.SECONDS)) {
            try {
                if (sqp.isRunning()) { // Sequencer is running, not appropriate to turn on CCDs
                    publishAllAdcData(adcDataList);
                    throw new RaftException("Sequencer is running, it must be off to powerCCDsOn()");
                }
            } catch (REBException e) {
                throw new RaftException("Failed to determine sequencer state for " + rebPath, e);
            }
            powerState = getCCDsPowerState(adcDataList); // throws on error
            if (!powerState.equals(CCDsPowerState.OFF)) {
                publishAllAdcData(adcDataList);
                throw new RaftException("found CCDsPowerState: " + powerState.name() + " != OFF, command aborted");
            }

            try {
                boardPowerMonitoringTask.pausePeriodicUpdate().get();
            } catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException("Could not pause boardPower monitoring task",e);
            }
            try {
                // good to go -- power up
                try {
                    if (isITLManufacturedCCD()) {
                        loadDacs(all, check, powerOn, adcDataList);
                        loadBiasDacs(all, check, powerOn, adcDataList);
                    } else {
                        loadBiasDacs(all, check, powerOn, adcDataList);
                        loadDacs(all, check, powerOn, adcDataList);
                    }
                    readSlowAdcs(adcDataList);
                    readPowerAdcs(adcDataList);
                } catch (RaftException e) { // power off upon exception
                    LOG.log(Level.SEVERE, "powerCCDsOn() FAILED for " + rebPath, e);
                    LOG.log(Level.SEVERE, "calling powerCCDsOff() on {0}", rebPath);
                    try {
                        powerCCDsOff(adcDataList);
                    } catch (RaftException et) {
                        publishAllAdcData(adcDataList);  // pub leftovers in case of exception
                        // TODO raise and alert here to get the REB shut off???
                        LOG.log(Level.SEVERE, "powerCCDsOff() FAILED for {0}", rebPath);
                        LOG.log(Level.SEVERE, "CCDsPowerState changed to {0}", stateService.getComponentState(rebPath, CCDsPowerState.class).name());
                        throw new RaftException("powerCCDsOn failed, powerCCDsOff failed, check/reboot " + rebPath, et);
                    }
                    publishAllAdcData(adcDataList);  // pub leftovers in case of exception
                    LOG.log(Level.SEVERE, "CCDsPowerState changed to {0}", stateService.getComponentState(rebPath, CCDsPowerState.class).name());
                    throw new RaftException("powerCCDsOn failed, CCS are powered OFF, check " + rebPath, e);
                }
                publishAllAdcData(adcDataList);  // pub leftovers in case of exception
                stateService.updateAgentComponentState(this, CCDsPowerState.ON);
                LOG.log(Level.INFO, "CCDsPowerState changed to {0}", stateService.getComponentState(rebPath, CCDsPowerState.class).name());
            } finally {
                boardPowerMonitoringTask.resumePeriodicUpdate();                
            }
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException ex) {
            throw new RaftException("Unable to obtain lock on sequencers, is idle_clear running?", ex);
        } 
    }


    /**
     *  Performs CCD power off sequence.
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Perform CCD power off sequence", level = Command.ENGINEERING_EXPERT)
    public void powerCCDsOff() throws RaftException
    {
        try ( AutoCloseableReentrantLock lock = SequencerProc.lockSequencerIfPossible() ) {
            List<AdcData> adcDataList = new ArrayList<>();
            forceDataPublicationForNextUpdateIterations(nPowerOnPub);
            try {
                powerCCDsOff(adcDataList);
            } catch (RaftException e) {
                publishAllAdcData(adcDataList);
                throw new RaftException("Exception occurred in powerCCDsOff() on" + rebPath, e);
            }
            publishAllAdcData(adcDataList);
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException ex ) {
            throw new RuntimeException("Problem invoking method close on the sequencer lock");
        }
    }


    /**
     *  Performs CCD power off sequence.
     *
     *  @throws  RaftException
     */
    @SuppressWarnings("SleepWhileHoldingLock")
    synchronized private void powerCCDsOff(List<AdcData> adcDataList) throws RaftException
    {
        try (@SuppressWarnings("unused") AutoCloseableReentrantLock lock = SequencerProc.lockSequencerIfPossible() ) {
            boolean backBiasState = true;
            try {
                backBiasState = bss.isBackBiasOn();
            } catch (REBException e) {  // TODO this should not fail, only recourse would be raiseAlert to power off REB
                LOG.log(Level.SEVERE, "Error checking back bias for " + rebPath, e);
            }
            try {
                bss.setBackBias(false);
            } catch (REBException e) {  // TODO Should we throw here? Hardware protection 
                LOG.log(Level.SEVERE, "Error setting back bias OFF for " + rebPath, e);
            }
            if (backBiasState) { // if it was on, add decay time
                try {
                    Thread.sleep(100);
                } catch (InterruptedException e) {
                    LOG.log(Level.SEVERE, "Thread.sleep(100) interrupted after setBackBias(false) for " + rebPath, e);
                }
            }
            try {
                if (isITLManufacturedCCD()) {
                    clearBiasDacs(adcDataList);
                    clearDacs(adcDataList);
                } else {
                    clearDacs(adcDataList);
                    clearBiasDacs(adcDataList);
                }
            } catch (RaftException e) {  // TODO: no recourse here but to log or raiseAlert(turn off REB)
                stateService.updateAgentComponentState(this, CCDsPowerState.FAULT);
                throw new RaftException("Exception occurred in powerCCDsOff() on" + rebPath, e);
            }
            readAllAdcs(adcDataList);
            publishAllAdcData(adcDataList);
            stateService.updateAgentComponentState(this, CCDsPowerState.OFF);  // should be definitive
            LOG.log(Level.INFO, String.format("powerCCDsOff(): CCDsPowerState set %s to %s", rebPath,
                    stateService.getComponentState(rebPath, CCDsPowerState.class).name()));
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException ex ) {
            throw new RuntimeException("Problem invoking method close on the sequencer lock");
        }
    }


    /**
     *  Sets Sequencer Clock lines to low state
     *  Corrupts the sequencer file -- needs strategy revision
     *  Needed in [WG]REB power management but incomplete
     *
     *  @throws  RaftException
     */
    // TODO Revise how this is used, if at all.  Since it corrupts the sequencer.
    @Command(type=CommandType.ACTION, description="Set CCD Clock lines to low state", level = Command.ENGINEERING_EXPERT)
    public void setCCDClocksLow() throws RaftException
    {
        try (@SuppressWarnings("unused") AutoCloseableReentrantLock lock = SequencerProc.lockSequencer(1L, TimeUnit.SECONDS)) {
            int seqState = readSeqIdleState();
            writeSeqIdleState(seqState & ~SEQ_CLOCK_LINES);
            LOG.log(Level.INFO, String.format("%s lowering sequencer clocks: 0x%x --> 0x%x",
                    rebPath, seqState, seqState & ~SEQ_CLOCK_LINES));
            LOG.log(Level.INFO, String.format("setCCDClocksLow() finished for %s", rebPath));
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException e) {
            throw new RaftException("Unable to obtain lock on sequencers, is idle_clear running?");
        }
    }


   /**
     * measure power state of ccds, user version
     *
     * @return String representing CCDsPowerState
     * @throws  RaftException
    @Command(type=CommandType.ACTION, description="Perform Test of CCD power state", level = Command.ENGINEERING_EXPERT)
    public String setCCDsPowerState() throws RaftException
    {
        List<AdcData> adcDataList = new ArrayList<>();
        CCDsPowerState powerState = setCCDsPowerState(adcDataList, true);
        publishAllAdcData(adcDataList);
        return powerState.name();
    }
     */
        
    /**
     * Get power state of ccds
     * Option to force fresh measurement
     * Intended to be called with forceMeas=false by monitoring with no
     * publication since it will use already published data.
     *
     * @return CCDsPowerState
     * @throws  RaftException
     */
    private CCDsPowerState setCCDsPowerState(boolean forceMeas) throws RaftException
    {
        List<AdcData> adcDataList = new ArrayList<>();
        return setCCDsPowerState(adcDataList, forceMeas);
    }
        
        
    /**
     * Check power state of ccds.
     *
     * CCD currents, voltages, etc. tested to determine powerState
     * If forceMeas take new measurements, else use monitored values
     * Failures are logged as SEVERE but testing continues.
     * Full test results are logged as FINE at the end.
     * 
     * The CCDsPowerState is set via stateService.updateAgent()
     * during the call and then returned.
     *
     * On failure set to FAULT and throws RaftException
     * Caller then should turn off CCDs
     *
     * @return CCDsPowerState {UNKNOWN, FAULT, OFF, ON, DELTA}
     * @throws  RaftException
     */
    synchronized private CCDsPowerState setCCDsPowerState(List<AdcData> adcDataList, boolean forceMeas) throws RaftException
    { 
        CCDsPowerState powerState;
        int onCnt = 0;
        int offCnt = 0;
        int dltaCnt = 0;
        int testCntTot = 0;  // counts the number of tests performed
        boolean seqLoaded = false;
        double uClockDacOffset;
        boolean failed = false;
        boolean testFailed = false;
        List<String> tmsgList = new ArrayList<>();
        int biasDacState;
        int clockDacState;
        double[] slowAdcs;
        double[] curPower;
        int underMask;

        // set number of CCDs to: 3 (REB), 2 (GREB), 1 (WREB) stripes
        int nccds = rebType == BaseSet.REB_TYPE_SCIENCE ? 3 :
                    rebType == BaseSet.REB_TYPE_GUIDER ? 2 :
                    rebType == BaseSet.REB_TYPE_WAVEFRONT ? 1 : 0;

        int nAdc = rebType == BaseSet.REB_TYPE_SCIENCE ? NUM_SREB_ADCS :
                   rebType == BaseSet.REB_TYPE_GUIDER ? NUM_GREB_ADCS :
                   rebType == BaseSet.REB_TYPE_WAVEFRONT ? NUM_WREB_ADCS : 0;

        if (!stateService.isComponentInState(rebPath, RebDeviceState.ONLINE)) {
            if (stateService.isComponentInState(rebPath, CCDsPowerState.UNKNOWN)) { // nominal case
                LOG.log(Level.INFO, "setCCDsPowerState() called in OFFLINE state on {0}", rebPath);
                return CCDsPowerState.UNKNOWN;
            } else { // FAULT case
                String pState = stateService.getComponentState(rebPath, CCDsPowerState.class).name();
                LOG.log(Level.SEVERE, "CCDsPowerState:{0} is inconsistent with OFFLINE {1}", new Object[]{pState, rebPath});
                stateService.updateAgentComponentState(this, CCDsPowerState.FAULT);
                // TODO Raise alert
                throw new RaftException("CCDsPowerState() not UNKNOW when called on OFFLINE "+rebPath);
            }
        }

        if (forceMeas) { // measure now, and populate adcDataList, time is ~40ms
            try {
                seqLoaded = readSeqIdleState() != 0;
                curPower = readPowerAdcs(adcDataList); // ~33ms
                slowAdcs = readSlowAdcs(adcDataList);  // ~5ms
                underMask = getUnders();
            }
            catch (RaftException e) {  // TODO implement raising alert
                stateService.updateAgentComponentState(this, CCDsPowerState.FAULT);
                throw new RaftException(String.format("Fault measuring CCDsPowerState for %s", rebPath), e);
            }
            // set bits in underSet for 3 (REB), 2 (GREB), 1 (WREB) stripes
            int underSet = rebType == BaseSet.REB_TYPE_SCIENCE ? 0x1ff :
                           rebType == BaseSet.REB_TYPE_GUIDER ? 0x3f :
                           rebType == BaseSet.REB_TYPE_WAVEFRONT ? 0x7: 0xffff;
            testCntTot += 1;
            if (underMask == underSet) { // All tested biases are below theshold and hence "off"
                tmsgList.add(String.format("%s: firmware protected biases test as OFF: 0x%x == 0x%x",
                                       rebPath, underMask, underSet));
                offCnt += 1;
            } else if (underMask == 0) {  // All tested biases are above theshold and hence "on"
                tmsgList.add(String.format("%s: firmware protected biases test as ON: 0x%x == 0",
                                       rebPath, underMask));
                onCnt += 1;
            } else {
                tmsgList.add(String.format("%s: firmware protection biases in UNDEFINED state: 0x%x != 0x%x",
                                          rebPath, underMask, underSet));
                testFailed = failed = true;
            }
            tmsgList.add(String.format("Test %d: %s firmware protection for %s",
                                       testCntTot, testFailed ? "FAILED": "PASSED", rebPath));
        }
        else { // TODO using monitoring values, not quite useable yet, fails at power off
            seqLoaded = true;   // assume true in absence of measurement
            slowAdcs = new double[nAdc];
            for (int adc = 0; adc < nAdc; adc++) {
                slowAdcs[adc] = adcValues[adcChans[adc]];
                }
            curPower = powerValues;
        }

        // Check board level currents
        testFailed = false;
        testCntTot += 1;
        if (odiQmax >= odiAmin) {
            tmsgList.add(String.format("Bad configuration: odiQmax:(%f) >= odiAmin:(%f)", odiQmax, odiAmin));
            testFailed = failed = true;
        } else if (curPower[PowerAdcs.ADC_OD_CURRENT] < nccds * odiQmax) {  // Maximum quiescent value
            tmsgList.add(String.format("ODI current is nominal for CCDs OFF State: %.2f mA",
                                   curPower[PowerAdcs.ADC_OD_CURRENT]*1000.0));
            offCnt += 1;
        } else if (curPower[PowerAdcs.ADC_OD_CURRENT] > nccds * odiAmin) {  // Minimum active value
            tmsgList.add(String.format("ODI current is nominal for CCDs ON State: %.2f mA",
                                   curPower[PowerAdcs.ADC_OD_CURRENT]*1000.0));
            onCnt += 1;
        } else { // in between, can't determine
            tmsgList.add(String.format("%s: mixed/undefined power state: %.2f < (ODI = %.2f) < %.2f",
                                     rebPath, nccds*odiQmax*1000.0, curPower[PowerAdcs.ADC_OD_CURRENT]*1000.0,
                                     nccds * odiAmin*1000.0) );
            tmsgList.add(String.format("%s: ODI current NOT in ON or OFF range", rebPath));
            testFailed = failed = true;
        }
        tmsgList.add(String.format("Test %d: %s REB ODI current test for  %s",
                                   testCntTot, testFailed ? "FAILED": "PASSED", rebPath));

        if (rebType == BaseSet.REB_TYPE_SCIENCE) { // [WG]REBs don't measure these

            // Clock Low Board Current
            testFailed = false;
            testCntTot += 1;
        if (clkliQmax >= clkliAmin) {
        tmsgList.add(String.format("Bad configuration: clkliQmax:(%f) >= clkliAmin:(%f)", clkliQmax, clkliAmin));
        testFailed = failed = true;
            } else if (curPower[PowerAdcs.ADC_CLKL_CURRENT] < clkliQmax) {  // Maximum quiescent value (~45mA)
                tmsgList.add(String.format("CLKLI current is nominal for CCDs OFF State %.2f mA",
                                       curPower[PowerAdcs.ADC_CLKL_CURRENT]*1000.0));
                offCnt += 1;
            } else if (curPower[PowerAdcs.ADC_CLKL_CURRENT] > clkliAmin) {  // Minimum active value (~50mA)
                tmsgList.add(String.format("CLKLI current is nominal for CCDs ON State %.2f mA",
                                       curPower[PowerAdcs.ADC_CLKL_CURRENT]*1000.0));
                onCnt += 1;
            } else { // can't determine
                tmsgList.add(String.format("%s: CCDs in mixed or undefined power state: %.2f < (CLKLI = %.2f) < %.2f",
                                         rebPath, clkliQmax*1000.0, curPower[PowerAdcs.ADC_CLKL_CURRENT]*1000.0,
                                         clkliAmin*1000.0) );
                tmsgList.add(String.format("%s: CLKLI current NOT in ON or OFF range", rebPath));
                testFailed = failed = true;
            }
            tmsgList.add(String.format("Test %d: %s REB CLKLI current test for  %s",
                                       testCntTot, testFailed ? "FAILED": "PASSED", rebPath));

            // Clock High Board Current
            testFailed = false;
            testCntTot += 1;
        if (clkhiQmax >= clkhiAmin) {
        tmsgList.add(String.format("Bad configuration: clkhiQmax:(%f) >= clkhiAmin:(%f)", clkhiQmax, clkhiAmin));
        testFailed = failed = true;
            } if (curPower[PowerAdcs.ADC_CLKH_CURRENT] < clkhiQmax) {  // Maximum quiescent value (~45mA)
                tmsgList.add(String.format("CLKHI current is nominal for CCDs OFF State %.2f mA",
                                       curPower[PowerAdcs.ADC_CLKH_CURRENT]*1000.0));
                offCnt += 1;
            } else if (curPower[PowerAdcs.ADC_CLKH_CURRENT] > clkhiAmin) {  // Minimum active value (~50mA)
                tmsgList.add(String.format("CLKHI current is nominal for CCDs ON State %.2f mA",
                                       curPower[PowerAdcs.ADC_CLKH_CURRENT]*1000.0));
                onCnt += 1;
            } else { // can't determine
                tmsgList.add(String.format("%s: CCDs in mixed or undefined power state: %.2f < (CLKHI = %.2f) < %.2f",
                                         rebPath, clkhiQmax*1000.0, curPower[PowerAdcs.ADC_CLKH_CURRENT]*1000.0,
                                         clkhiAmin*1000.0) );
                tmsgList.add(String.format("\n  %s: CLKHI current NOT in ON or OFF range", rebPath));
                testFailed = failed = true;
            }
            tmsgList.add(String.format("Test %d: %s REB CLKHI current test for  %s",
                                       testCntTot, testFailed ? "FAILED": "PASSED", rebPath));
        }

        // Check Bias levels for each CCD
        for (BiasControl bias : biases) {
            if (bias == null) continue;
            testFailed = false;
            testCntTot += 1;
            biasDacState = bias.getBiasDacsPowerState(slowAdcs, tmsgList);
            if (biasDacState == 0) {
                offCnt += 1;
            } else if (biasDacState == 1) {
                onCnt += 1;
            } else if (biasDacState > 1) {
                onCnt += 1;
                dltaCnt += (biasDacState - 1);  // tracks how many voltages are in DELTA state
            } else {
                tmsgList.add(String.format("%s: CCDs Bias voltage test FAILED", rebPath));
                testFailed = failed = true;
            }
            tmsgList.add(String.format("Test %d: %s Bias level test for  %s",
                                       testCntTot, testFailed ? "FAILED": "PASSED", bias.biasPath));
        }

        // Check Clock rails on the REB
        testFailed = false;
        testCntTot += 1;
        uClockDacOffset = seqLoaded ? 0.9: 0.0;

        clockDacState = dacCtrl.getClockDacsPowerState(slowAdcs, tmsgList, uClockDacOffset);
        if (clockDacState == 0) {
            offCnt += 1;
        } else if (clockDacState == 1) {
            onCnt += 1;
        } else if (clockDacState > 1) {
            onCnt += 1;
            dltaCnt += (clockDacState - 1);  // tracks how many voltages are in DELTA state
        } else {
            tmsgList.add(String.format("%s: CCDs clock voltage test FAILED", rebPath));
            testFailed = failed = true;
        }
        tmsgList.add(String.format("Test %d: %s Clock rails test for  %s",
                                   testCntTot, testFailed ? "FAILED": "PASSED", dacCtrl.dacPath));

        // Done with tests, determine outcome
        if (offCnt == testCntTot) {
            powerState = CCDsPowerState.OFF;
        } else if (onCnt == testCntTot && dltaCnt == 0) {
            powerState = CCDsPowerState.ON;
        } else if (onCnt == testCntTot && dltaCnt > 0) {
            powerState = CCDsPowerState.DELTA;
        } else {
            powerState = CCDsPowerState.FAULT;
            failed = true;
        }
        stateService.updateAgentComponentState(this, powerState);
        tmsgList.add(String.format("%s: setCCDsPowerState() %s: %d:%d:%d OFF:ON:DELTA of %d tests passed", 
                                 rebPath, failed ? "FAILED" : "PASSED", offCnt, onCnt, dltaCnt, testCntTot));
        tmsgList.add(String.format("%s CCDsPowerState updated to %s", rebPath, powerState.name()));
        for (String tmsg : tmsgList) {
            LOG.log(failed ? Level.SEVERE : Level.FINE, tmsg);
        }
        if (failed) {
            throw new RaftException(String.format("%s, see logfile for details",tmsgList.get(tmsgList.size() - 1)));
        }
        return powerState;
    }
        
   /**
     * Get voltage differences between setpoint and readback for Bias and Clock
     * rails to assist with calibration of setpoint to raw ADC conversion
     *
     * @return 
     * @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Report Bias, Clock lines set/read deltas and DAC calibration", level = Command.ENGINEERING_EXPERT)
    public String checkDacsCalibration() throws RaftException
    {
        List<AdcData> adcDataList = new ArrayList<>();
        String result = setDacsCalibration(adcDataList, false);
        publishAllAdcData(adcDataList);
        return result;
    }

        
   /**
     * Get voltage differences between setpoint and readback for Bias and Clock
     * rails to assist with calibration of setpoint to raw ADC conversion
     *
     * @return 
     * @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Report and update Bias, Clock lines set/read deltas and DAC calibration", level = Command.ENGINEERING_EXPERT)
    public String updateDacsCalibration() throws RaftException
    {
        List<AdcData> adcDataList = new ArrayList<>();
        String result = setDacsCalibration(adcDataList, true);
        publishAllAdcData(adcDataList);
        return result;
    }
        

    /**
    * Get voltage differences between setpoint and readback for Bias and Clock
    * rails to assist with configuration setup.
    *
    * Each bias, clock voltage offset from configured value (eg xxP) is measured.
    * Where delta = xxP - xxMeas.
    * The behavior depends on the boolean argument action: false/true for dry-run/make-changes
    * Two command forms:
    * 1: checkDacsCalibration(false): measure delta voltage, print table user and
    * logs: xxP, xxMeas, xxDelta, xxCal, xxCalNew
    *
    * 2: updateDacsCalibration(true): additionally adjust the corresponding
    * calibration parameter xxCal --> xxCalNew to reflect the measured
    * delta, where xxCalNew = xxCal * (1 + xxDelta/xxP). This changes
    * configuration and causes a transition to DELTA PowerState.  Also update the
    * tolerances (xxTol) to reflect the smaller delta (min of [minTol and delta/4)
    *
    * Issues:
    *   -On corner rafts, pclkHigh is set by pclkLowP + rebps/dphi so offset
    *    can be measured but not applied.
    *
    * ***********
    * Usage scenario is:
    * 0: Prerequisite that we start in an ON CCDsPowerState and CONFIGURED
    * 1: "check" to see what current offsets are and evaluate as sensible.
    * 2: "update" to establish a new xxCal parameter, changes state: DIRTY,DELTA
    * 3: call loadDeltaDacs() changes state: DIRTY:ON
    * 4: run tests etc.
    * 5: save RaftsPower category if desired to keep the config
    * 
    * @throws  RaftException
    *
    **/
    private String setDacsCalibration(List<AdcData> adcDataList, boolean action) throws RaftException
    { 
        double[] slowAdcs;
        List<String> tmsgList = new ArrayList<>();
        StringBuilder result = new StringBuilder();
        boolean doAction = false;   // action && conditions
        int errCnt = 0;
        int hdrStart = 0;

        readSlowAdcs(adcDataList);
        readPowerAdcs(adcDataList);

        // Prerequisite 1: sequencer idle
        try (@SuppressWarnings("unused") AutoCloseableReentrantLock lock = SequencerProc.lockSequencer(1L, TimeUnit.SECONDS) ) {
            try {
                if (sqp.isRunning()) { // Sequencer is running, need quiescent state
                    publishAllAdcData(adcDataList);
                    throw new RaftException("Sequencer is running, it must be idle");
                }
            }
            catch (REBException e) {
                throw new RaftException(String.format("failed to determine Sequencer state on %s", rebPath), e);
            }
            // Prerequisite 2: CCDs are on
            CCDsPowerState powerState = setCCDsPowerState(adcDataList, true);
            if (!powerState.equals(CCDsPowerState.ON)) {
                publishAllAdcData(adcDataList);
                throw new RaftException("command aborted, CCDsPowerState must be ON for this command");
            }
            // Prerequisite 3: backBias hvBias is off
            if (isBackBiasOn()) {  // TODO maybe we can/should do this with back bias on
                publishAllAdcData(adcDataList);
                throw new RaftException("command aborted, backBias must be off for this command");
            }
            // Prerequisite 4: setting the offsets only allowed if RAFTS, RAFTS_POWER are CONFIGURED
            if ( action && checkCategoriesForIsDirty(RAFTS, RAFTS_POWER) ) {
                throw new RaftException("command aborted: "+RAFTS+" or "+RAFTS_POWER+" categories are DIRTY");
            } else if (action) {
                doAction = true;
            }
            // Prerequisite 5: there are no submitted changes for the subsystem
            if ( action && !sce.getAllSubmittedChanges().isEmpty()) {
                throw new RaftException("command aborted: subsystem has uncommitted changes");
            } else if (action) {
                doAction = true;
            }
            // Prerequisites met, measure the voltages
            try {
                slowAdcs = readSlowAdcs(adcDataList);  // ~5ms
            }
            catch (RaftException e) {
                throw new RaftException("REB voltage measurements failed for "+rebPath, e);
            }
            // Check/Set Bias offsets for each CCD
            for (BiasControl bias : biases) {
            if (bias == null) continue;
                errCnt += bias.setBiasDacsCalibration(slowAdcs, tmsgList, doAction);
            }
            // Check/Set Clock rail offsets on the REB
            errCnt += dacCtrl.setClockDacsCalibration(slowAdcs, tmsgList, doAction);

            if (errCnt == 0 && doAction) { // commit the config changes to xxCal, xxTol
                try {
                    sce.commitBulkChange();
            }
            catch (Exception e) {
                    sce.dropAllSubmittedChanges();
                    throw new RaftException("Configuration change failed", e);
                }
            } else if (errCnt > 0 && doAction) {
                tmsgList.add(hdrStart++, String.format(
                        "Error: %d Offset adjustments were above limit: %5.2f", errCnt, maxStep));
            }
            // generate top line of table
            tmsgList.add(hdrStart++, String.format("\n%-8.8s %-8.8s %6.6s %6.6s"
                     +"  %6.6s --> %-6.6s  %8.7s --> %-8.7s"
                     +"  %5.5s --> %-5.5s  %6.6s --> %-6.6s",
                    "Raft/Reb", "Bias/Clk", "xxP", "xxMeas",
                    "xxCal", "xxCal", "xxTol", "xxTol",
                    "DAC", "DAC", "xxConv", "xxConv"));

            if (action && !doAction) {
                tmsgList.add(hdrStart++, String.format("\nError: Updating not allowed due to errors"));
            }
            for (String tmsg : tmsgList) {
                LOG.log(Level.FINE, tmsg);
                result.append(tmsg);
                result.append("\n");
            }
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException ex) {
            throw new RaftException("Unable to obtain lock on sequencers, is idle_clear running?", ex);
        }
        return result.toString();  // not sure I need the toString() here
    }

    /**
     * Checks to see if this component is dirty in any of the input categories
     *
     * @parameter Categories
     */
    private boolean checkCategoriesForIsDirty(String... categories) {
        boolean isDirty = false;
        ConfigurationInfo ci = sce.getConfigurationInfo();
        for ( String category : categories) {
            Map<String, ConfigurationParameterInfo> parsForCategory = ci.getCurrentParameterInfoForCategory(category);
            for ( ConfigurationParameterInfo par : parsForCategory.values() ) {
                if ( par.getComponentName().contains(rebPath) ) {
                    if ( par.isDirty() ) {
                        isDirty |= true;
                        LOG.log(Level.INFO, "{0}:{1}/{2} has unsaved changes", new Object[]{par.getCategoryName(), par.getComponentName(), par.getParameterName()});
                        break;
                    }
                }
            }
        }
        return isDirty;
    }


    /**
     *  Performs CCD shorts test.
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Perform CCD shorts test", level = Command.ENGINEERING_EXPERT)
    public void testCCDShorts() throws RaftException
    {
        List<AdcData> adcDataList = new ArrayList<>();
        boolean failed = false;
        CCDsPowerState powerState;
        int seqState = 0;
        List<String> tmsgList = new ArrayList<>();

        forceDataPublicationForNextUpdateIterations(5);
        // check prerequisites, abort if not met
        //
        try (@SuppressWarnings("unused") AutoCloseableReentrantLock lock = SequencerProc.lockSequencer(1L, TimeUnit.SECONDS) ) {
            try {
                if (sqp.isRunning()) {
                    throw new RaftException("Sequencer is running, it must be off to testCCDShorts()");
                } else {
                    seqState = readSeqIdleState();
                }
            }
            catch (REBException e) {
                throw new RaftException(e);
            }
            powerState = setCCDsPowerState(adcDataList, true);
            if (!powerState.equals(CCDsPowerState.OFF)) {
            throw new RaftException("CCDs must be OFF, found "+ powerState.name() +", testCCDShorts() aborted");
            }
            // force off to be sure, critical for CCDs to be off here
            powerCCDsOff(adcDataList);
            // lower sequencer clock lines before test and reset after
            try {
                writeSeqIdleState(seqState & ~SEQ_CLOCK_LINES);
                LOG.log(Level.INFO, String.format("%s lowering sequencer clocks for shorts test: 0x%x --> 0x%x",
                        rebPath, seqState, readSeqIdleState()));
                // allow the clocks to settle for 200ms
                try {
                    Thread.sleep(200);
                }
                catch (InterruptedException e) {
                    LOG.log(Level.SEVERE, "Thread.sleep(200) interrupted after lowering sequencer clocks for "+rebPath, e);
                }
                // now safe to run this test
                for (BiasControl bias : biases) {
                    if (bias == null) continue;
                    failed |= bias.testShorts(adcDataList, tmsgList);
                    publishAllAdcData(adcDataList);
                }
                failed |= dacCtrl.testShorts(adcDataList, tmsgList);
                publishAllAdcData(adcDataList);
                // reset sequencer clock lines after test
                LOG.log(Level.INFO, String.format("%s restoring sequencer clocks after shorts test: 0x%x --> 0x%x",
                        rebPath, readSeqIdleState(), seqState));
                writeSeqIdleState(seqState);
                // allow the clocks to settle for 20ms
                try {
                    Thread.sleep(20);
                }
                catch (InterruptedException e) {
                    LOG.log(Level.SEVERE, "Thread.sleep(20) interrupted after restoring sequencer clocks for "+rebPath, e);
                } 
                readSlowAdcs(adcDataList);
                readPowerAdcs(adcDataList);
                // allow the clocks to settle for 200ms
                try {
                    Thread.sleep(200);
                }
                catch (InterruptedException e) {
                    LOG.log(Level.SEVERE, "Thread.sleep(200) interrupted after restoring sequencer clocks for "+rebPath, e);
                } 
                readSlowAdcs(adcDataList);
                readPowerAdcs(adcDataList);
            }
            catch (RaftException e) {
                powerCCDsOff(adcDataList); // recover sensible state
                publishAllAdcData(adcDataList);
                writeSeqIdleState(seqState);
                throw new RaftException(String.format("testCCDShorts() FAILED for %s", rebPath), e);
            }
            // close out
            publishAllAdcData(adcDataList);
            if (failed) {
                tmsgList.add(String.format("%s shorts test FAILED", rebPath));
                for (String tmsg : tmsgList) {
                    LOG.log(Level.SEVERE, tmsg);
                }
                throw new RaftException(String.format("%s, see logfile for details", tmsgList.get(tmsgList.size() - 1)));
            }
            for (String tmsg : tmsgList) {
                LOG.log(Level.FINE, tmsg);
            }
            LOG.log(Level.INFO, "testCCDShorts() finished successfully");
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException ex) {
            throw new RaftException("Unable to obtain lock on sequencers, is idle_clear running?", ex);
        }
    }


    /**
     *  Publishes all ADC data.
     * 
     *  @param  adcDataList  The list of ADC data
     *
     *  The adcDataList is consumed and cleared upon return
     */
    private void publishAllAdcData(List<AdcData> adcDataList)
    {
        
        KeyValueDataList dataList = new KeyValueDataList();
        for (AdcData adcData : adcDataList) {
            if ( adcData.type == AdcData.TYPE_CURR ) {
                dataList.addData(adcData.ch.getPath(), adcData.value, adcData.timeStamp);
            } else {
                String chanName = (adcData.type == AdcData.TYPE_POWER ? pwrChannelMap : sadcChannelMap).get(adcData.hwChan);
                if (chanName != null) {  // convert to mA: Would be nice to get this from the Channel
                    double conv = adcData.type == AdcData.TYPE_POWER && (adcData.hwChan & 1) != 0 ? 1000.0 : 1.0;
                    dataList.addData(chanName, conv * adcData.value, adcData.timeStamp);
                }
            }
        }
        if (!dataList.getListOfKeyValueData().isEmpty()) {
            subsys.publishSubsystemDataOnStatusBus(dataList);
        }
        adcDataList.clear();  // flush out the stuff that was published
    }


    /**
     *  Loads configured bias and clcock DAC values.
     *
     *  @param  all  If true, all data are loaded; otherwise only changes
     *  @return  The number of Bias and Clock DACs loaded
     *  @throws  RaftException
     *           This is the user command, checks to verify CCDs are ON/DELTA and
     *           only changes allowed by configuration system can be used
     *
     *  Updates the CCDsPowerState on success (eg. DELTA -> ON)
     */
    @Command(type=CommandType.ACTION, description="Applies changed Bias and Clock DAC configuration values on Powered CCDs", level = Command.ENGINEERING_EXPERT)
    public int loadDeltaDacs(@Argument(description="reload all (true) or changes only (false)") boolean all) throws RaftException
    {
        try (@SuppressWarnings("unused") AutoCloseableReentrantLock lock = SequencerProc.lockSequencer(1L, TimeUnit.SECONDS)) {
            List<AdcData> adcDataList = new ArrayList<>();
            CCDsPowerState powerState = getCCDsPowerState(adcDataList);
            if (!powerState.equals(CCDsPowerState.ON) && !powerState.equals(CCDsPowerState.DELTA)) {
                throw new RaftException("CCDs must be ON|DELTA to use loadDeltaDacs command, command aborted");
            }
            int count = loadDeltaDacs(all, adcDataList);
            if (count < 0) {
                throw new RaftException(String.format("loadDeltaDacs() FAILED, %d params with step sizes > %.2f", -count, maxStep));
            }
            // update CCDsPowerState now that voltages have been adjusted
            powerState = getCCDsPowerState(adcDataList);
            LOG.log(Level.INFO, String.format("%s CCDsPowerState updated to %s", rebPath, powerState.name()));
            publishAllAdcData(adcDataList);
            return count;  // TODO provide a more informative return value
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException e) {
            throw new RaftException("Unable to obtain lock on sequencers, is idle_clear running?");
        }
    }


    /**
     *  Loads configured clock and bias DAC values on powered CCDs.
     *
     *  @param  adcDataList  Stores measured voltage, current values
     *  @return The number of bias and clock DACs loaded, -1 on config errors
     *  @throws RaftException
     *
     */
    private int loadDeltaDacs(boolean all, List<AdcData> adcDataList) throws RaftException
    {
        int count = 0;
        int errCnt = 0;
        double[] slowAdcs;
        List<String> tmsgList = new ArrayList<>();

        // Measure current value of voltages
        try {
            slowAdcs = readSlowAdcs(adcDataList);  // ~5ms
        }
        catch (RaftException e) {
            throw new RaftException("REB voltage and current measurements failed for "+rebPath, e);
        }
        // Verify voltage step is less than maxStep
        for (BiasControl bias : biases) {
            errCnt += bias.checkBiasStepLimits(slowAdcs, tmsgList);
        }
        errCnt += dacCtrl.checkClockStepLimits(slowAdcs, tmsgList);
        for (String tmsg : tmsgList) {
            LOG.log((errCnt > 0 ? Level.SEVERE : Level.FINE), tmsg);
        }
        if (errCnt > 0) { return -errCnt; }
        boolean check = true;     // verify new settings within tolerance
        boolean powerOn = false;  // not part of powerOn sequencing, CCDs must be on
        try {
            for (BiasControl bias : biases) {  // TODO: spawn threads, do these in parallel?
                if (bias == null) continue;
                count += all ? bias.load(dacLoadDelay, check, powerOn, adcDataList)
                             : bias.loadChanged(dacLoadDelay, check, adcDataList);
            }
            count += all ? dacCtrl.loadConfig(dacLoadDelay, check, powerOn, adcDataList)
                         : dacCtrl.loadChanged(dacLoadDelay, check, adcDataList);

            LOG.log(Level.INFO, "Loaded {0} {1} Bias and Clock DACs", new Object[]{count, rebPath});
        }
        catch (RaftException e) {
            try {  // things are sufficient borked that we should turn CCDs off
                powerCCDsOff(adcDataList);
            }
            catch (RaftException et) {
                // TODO raise and alert here to get the REB shut off???
                LOG.log(Level.SEVERE, "loadDeltaDacs() FAILED for "+rebPath, e);
                LOG.log(Level.SEVERE, "CCDsPowerState changed to {0}", stateService.getComponentState(rebPath, CCDsPowerState.class).name()); 
                publishAllAdcData(adcDataList);
                throw new RaftException("loadDeltaDacs() failed, powerCCDsOff failed, check/reboot "+rebPath, et);
            }
            publishAllAdcData(adcDataList);
            LOG.log(Level.SEVERE, "called powerCCDsOff() on {0}", rebPath);
            throw new RaftException("loadDeltaDacs() FAILED for "+rebPath, e);
        }
        publishAllAdcData(adcDataList);
        forceDataPublicationForNextUpdateIterations(5);
        return count;
    }


    /**
     *  Loads configured DAC values.
     *
     *  @param  all       If true, all data are loaded; otherwise only changes
     *  @param  check     Whether to check each loaded value
     *  @param  adcDataList  List to receive values from slowAdcs
     *  @return  The number of DACs loaded
     *  @throws  RaftException
     *
     *  N.B. Any caller of this method must call powerCCDsOff() upon an exception
     *
     *  This is the version called in powerCCDsOn()
     */
    private int loadDacs(boolean all, boolean check, boolean powerOn, List<AdcData> adcDataList) throws RaftException
    {
        int count = all ? dacCtrl.loadConfig(dacLoadDelay, check, powerOn, adcDataList)
                        : dacCtrl.loadChanged(dacLoadDelay, check, adcDataList);
        LOG.log(Level.INFO, "Loaded {0} {1} DACs", new Object[]{count, rebPath});
        return count;
    }


    /**
     *  Clears DAC values.
     *
     *  @throws  RaftException
     */
    private void clearDacs(List<AdcData> adcDataList) throws RaftException
    {
        dacCtrl.clear(dacClearDelay, adcDataList);
    }


    /**
     *  Loads configured bias DAC values.
     *
     *  @param  all       If true, all data are loaded; otherwise only changes
     *  @param  check     Whether to check each loaded value
     *  @return The number of bias DACs loaded
     *  @throws RaftException
     *
     *  N.B. Any caller of this method must call powerCCDsOff() upon an exception
     *
     *  This is the version called during powerCCDsOn()
     */
    private int loadBiasDacs(boolean all, boolean check,
                             boolean powerOn, List<AdcData> adcDataList) throws RaftException
    {
        int count = 0;
        for (BiasControl bias : biases) {
            if (bias == null) continue;
            count += all ? bias.load(dacLoadDelay, check, powerOn, adcDataList)
                         : bias.loadChanged(dacLoadDelay, check, adcDataList);
        }
        LOG.log(Level.INFO, "Loaded {0} {1} bias DACs", new Object[]{count, rebPath});
        return count;
    }


    /**
     *  Clears bias DAC values.
     *
     *  @throws  RaftException
     */
    private void clearBiasDacs(List<AdcData> adcDataList) throws RaftException
    {
        for (BiasControl bias : biases) {
            if (bias == null) continue;
            bias.clear(dacClearDelay, adcDataList);
        }
    }


    /**
     *  Loads configuration data to the ASPICs.
     *
     *  @param  all  If true, all data are loaded; otherwise only changes
     *  @return  The number of ASPICs loaded
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Load configuration data to the ASPICs")
    public int loadAspics(boolean all) throws Exception
    {
        int count = 0;
        for (AspicControl aspic : aspics) {
            if (aspic != null) {
                count += all ? aspic.load() : aspic.loadChanged();
            }
        }
        LOG.log(Level.INFO, "Loaded {0} {1} ASPICs", new Object[]{count, rebPath});
        return count;
    }


    /**
     *  Resets the ASPICs.
     *
     *  @return  The number of ASPICs reset
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Reset the ASPICs")
    public int resetAspics() throws Exception
    {
        int count = 0, mask = 0;
        for (AspicControl aspic : aspics) {
            if (aspic != null) {
                int strip = aspic.getHwChan() / 2;
                if ((mask & (1 << strip)) == 0) {
                    count += aspic.reset();
                    mask |= (1 << strip);
                }
            }
        }
        LOG.log(Level.INFO, "Reset {0} {1} ASPICs", new Object[]{count, rebPath});
        return count;
    }


    /**
     *  Checks loaded ASPIC configuration data.
     *
     *  @return  The mask of mismatched fields
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Check loaded ASPIC configuration")
    public int checkAspics() throws Exception
    {
        int mask = 0, index = 0;
        for (AspicControl aspic : aspics) {
            if (aspic != null) {
                mask |= (aspic.check() << index);
                index += 4;
            }
        }
        return mask;
    }


    /**
     *  Gets the back bias on state.
     *
     *  @return  Whether back bias is turned on
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Get the back bias on state")
    public boolean isBackBiasOn() throws RaftException
    {
        try {
            return bss.isBackBiasOn();
        }
        catch (REBException e) {
            throw new RaftException("Error getting back bias for"+rebPath, e);
        }
    }


    /**
     *  Turns back bias on or off.
     *
     *  @param  value  The back bias on state to set (true or false)
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Turn back bias on or off")
    public void setBackBias(@Argument(description="The on state")
                            boolean value) throws RaftException
    {
        synchronized(backBiasEnableLock) {
            if ( isBackBiasSwitchDisabled && value == true ) {
                throw new RaftException(getPath()+": the back bias switch cannot be closed because it's disabled ("+backBiasDisabledReason+")");
            }
            try {
                bss.setBackBias(value);
            } catch (REBException e) {
                throw new RaftException("Error setting back bias for " + rebPath, e);
            }
        }
    }

    /**
     * Disable/Enable the back bias switch.
     * 
     * When disabling, if the back bias is currently closed it will be opened.
     * Future attempts at closing the back bias switch will fail for as long as
     * it's disabled.
     * 
     * When enabling the back bias switch its state will remain unchanged, i.e.
     * it will remain open. Future attempts at closing the switch will be
     * successful.
     * 
     *
     *  @param  disable True/False to disable/enable the back bias switch.
     *  @param  reason  The reason for enabling/disabling the back bias switch.
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Disable/Enable the back bias switch", category = Command.CommandCategory.USER)
    public void disableBackBias(@Argument(description="True/False to disable/enable the back bias switch")      
                            boolean disable, @Argument(description="The reason for enabling/disabling the back bias switch")    
                            String reason) throws RaftException {
        disableBackBias(disable,reason,isOnline());
    }

    private void disableBackBias(boolean disable, String reason, boolean isOnline) throws RaftException {
        synchronized(backBiasEnableLock) {
            try {
                if ( ! isOnline ) {
                    LOG.log(Level.INFO, "{0}: is offline. The back bias flag has been set to {1} ({2})", new Object[]{getPath(), disable ? "disabled" : "enabled", reason});                    
                } else {
                    if (disable && isBackBiasOn()) {
                        setBackBias(false);
                        LOG.log(Level.WARNING, "{0}: the back bias has been disabled ({1}) while the switch was closed. The switch is being opened now.", new Object[]{getPath(), reason});
                    } else {
                        LOG.log(Level.INFO, "{0}: the back bias has been {1} ({2})", new Object[]{getPath(), disable ? "disabled" : "enabled", reason});
                    }
                }
                
            } finally {                
                isBackBiasSwitchDisabled = disable;
                backBiasDisabledReason = reason;                
            }
        }        
    }
    
    
    //This method is used to verify that the back bias switch is in a consistent
    //state with respect to the disabled flag.
    //If the back bias switch has been disabled and the switch is closed the
    //switch will be opened immediately.
    //The method is invoked from the device initialization code and is meant to 
    //cover the case in which the back bias switch was disabled    
    private void checkBackBiasDisabledState(boolean isOnline) throws RaftException {
        synchronized(backBiasEnableLock) {
            disableBackBias(isBackBiasSwitchDisabled, backBiasDisabledReason, isOnline);
        }
    }
    
    /**
     *  Check if the back bias switch has been disabled.
     *
     *  @return  true if the back bias is disabled, false otherwise.
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Check if the back bias switch has been disabled")
    public boolean isBackBiasDisabled() throws RaftException
    {
        return isBackBiasSwitchDisabled;
    }
    
    /**
     *  Sets a heater voltage.
     *
     *  @param  heater  The heater number: 0 or 1
     *  @param  volts   The voltage to set
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Set heater voltage")
    public void setHeater(@Argument(description="Heater number")
                          int heater,
                          @Argument(description="The voltage")
                          double volts) throws RaftException
    {
        double amps = volts / (HEATER_OHMS + HTR_LEAD_OHMS);
        heaterValues[CHAN_HTR_VOLTS] = volts;
        heaterValues[CHAN_HTR_POWER] = amps * amps * HEATER_OHMS;
        int value = (int)((dac.getVersion() == BoardDacs.VERSION_REB5 ? HTR_CONV_R5 : HTR_CONV) * volts);
        try {
            dac.set(heater == 0 ? BoardDacs.CHAN_HEATER_1 : BoardDacs.CHAN_HEATER_2,
                    value < 0 ? 0 : value > DAC_LIMIT ? DAC_LIMIT : value, true);
        }
        catch (REBException e) {
            throw new RaftException("Error setting heater for"+rebPath, e);
        }
    }


    /**
     *  Sets heater power.
     *
     *  @param  heater  The heater number: 0 or 1
     *  @param  power   The power value to set
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Set heater power")
    public void setHeaterPower(@Argument(description="Heater number")
                               int heater,
                               @Argument(description="The power value")
                               double power) throws RaftException
    {
        setHeater(heater, (HEATER_OHMS + HTR_LEAD_OHMS) * Math.sqrt(power / HEATER_OHMS));
    }


    /**
     *  Sets the time base to the current system time.
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Sets the time base to the current system time")
    public void setTime() throws RaftException
    {
        try {
            bss.disable();
            bss.setTime();
            bss.enable();
        }
        catch (REBException e) {
            throw new RaftException("setTime() failed"+rebPath, e);
        }
    }


    /**
     *  Gets the time base as Unix time.
     *
     *  @return  The time base converted to Unix time
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Gets the time base as Unix time")
    public long getTime() throws RaftException
    {
        try {
            return bss.getTime();
        }
        catch (REBException e) {
            return 0;
        }
    }


    /**
     *  Gets a trigger time as Unix time.
     *
     *  @param  rSet  The name of the trigger time's register set: "stat",
     *                "time" or "seq:.
     *  @return  The trigger time converted to Unix time
     *  @throws  RaftException
     */
    private static final Map<String,Integer> rsetNames = new HashMap();
    static {
        rsetNames.put("stat", BaseSet.RSET_STATUS);
        rsetNames.put("time", BaseSet.RSET_TIME_BASE);
        rsetNames.put("seq",  BaseSet.RSET_SEQUENCER);
        rsetNames.put("tadc", BaseSet.RSET_TEMP_ADCS);
        rsetNames.put("padc", BaseSet.RSET_POWER_ADCS);
    }

    @Command(type=CommandType.QUERY, description="Gets a trigger time as Unix time")
    public long getTime(@Argument(name="regset", description="Register set name")
                        String rSet) throws RaftException
    {
        Integer regSet = rsetNames.get(rSet.toLowerCase());
        if (regSet == null) {
            throw new RaftException("Invalid register set name");
        }
        try {
            return bss.getTriggerTime(regSet);
        }
        catch (REBException e) {
            return 0;
        }
    }


    /**
     *  Sets the FITS file test conditions.
     *
     *  @param  filter       The filter name
     *  @param  temperature  The temperature set point
     *  @param  wavelength   The monochromator wavelength
     */
    @Command(type=CommandType.ACTION, description="Sets the FITS file test conditions")
    public void setFitsConditions(@Argument(description="Filter name")
                                  String filter, 
                                  @Argument(description="The temperature")
                                  double temperature,
                                  @Argument(description="The wavelength")
                                  double wavelength)
    {        
        FitsHeaderKeywordData data = new FitsHeaderKeywordData(rebGeometry.getUniqueId());
        data.addHeaderKeywordValue("primary", "FilterName", filter, true);
        data.addHeaderKeywordValue("primary", "TemperatureSetPoint", temperature, true);
        data.addHeaderKeywordValue("primary", "MonochromatorWavelength", wavelength, true);                        
        subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());
    }


    /**
     *  Resets the front end.
     *
     *  This is the only way to stop a running sequencer program
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Resets the front end")
    public void resetFrontEnd() throws RaftException
    {
        try {
            bss.reset(0);
            bss.enable();
            bss.setTime();
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Gets the contents of REB registers.
     *
     *  @param  address  The address of the first register to read
     *  @param  count    The number of registers to read
     *  @return  The block of register values
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Gets the contents of REB registers")
    public RegisterData getRegister(@Argument(description="Address of the first register")
                                    int address,
                                    @Argument(description="Number of registers to read")
                                    int count) throws RaftException
    {
        int[] values = new int[count];
        try {
            bss.read(address, values);
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }

        return new RegisterData(address, count, values);
    }


    /**
     *  Sets the contents of REB registers.
     *
     *  @param  address  The address of the first register to write
     *  @param  values   The array of values to write to consecutive registers
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Sets the contents of REB registers")
    public void setRegister(@Argument(description="Address of the first register")
                            int address,
                            @Argument(description="Values to write")
                            int[] values) throws RaftException
    {
        
        try (@SuppressWarnings("unused") AutoCloseableReentrantLock lock = SequencerProc.lockSequencer(1L, TimeUnit.SECONDS)) {
            try {
                bss.write(address, values);
            } catch (REBException e) {
                throw new RaftException(e.getMessage());
            }
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException ex) {
            throw new RaftException("Unable to obtain lock on sequencers, is idle_clear running?", ex);            
        }
    }


    /**
     *  Gets the REB status block.
     *
     *  @return  The REB status block
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Gets the REB status block")
    public StatusData getRebStatus() throws RaftException
    {
        try {
            return new StatusData(sts.readStatus());
        }
        catch (REBException e) {
            //checkTimeout(e);
            return null;
        }
    }


    /**
     *  Gets the associated board DACs object.
     *
     *  @return  The DAC object
     */
    BoardDacs getBoardDacs()
    {
        return dac;
    }


    /**
     *  Gets the associated ASPIC object.
     *
     *  @return  The ASPIC object
     */
    Aspic getAspic()
    {
        return asp;
    }


    /**
     *  Gets the associated sequencer.
     *
     *  @return  The sequencer
     */
    public SequencerProc getSequencer()
    {
        return sqp;
    }


    /**
     *  Gets a temperature sensor value.
     *
     *  @return  The value
     */
    @SuppressWarnings("unused")
    double getTempValue(int hwChan)
    {
        return isOnline() ? tempValues[hwChan] : 0;
    }


    /**
     *  Gets a power sensor value.
     *
     *  @return  The value
     */
    @SuppressWarnings("unused")
    double getPowerValue(int hwChan)
    {
        return isOnline() ? powerValues[hwChan] : 0;
    }


    /**
     * Get config param maxDelta
     * 
     * @return  The value
     */
    public double getMaxDelta()
    {
        return maxDelta;
    }

    /**
     * Get config param maxStep
     * 
     * @return  The value
     */
    public double getMaxStep()
    {
        return maxStep;
    }

    /**
     * Get config param minTol 
     * 
     * @return  The value
     */
    public double getMinTol()
    {
        return minTol;
    }


    /**
     * Simulate a disconnect
     */
    @Command(simulation = true)
    public void simulateDisconnection() {
        setOnline(false);
    }


    /**
     * Checks error count.
     * 
     * Increments consecutive error count and sets offline if excessive
     */
    private void checkErrorCount(String excpnText) {
        if (++errorCount > 4) {
            exceptionText = excpnText;
            exceptionCount = 1;
            setOnline(false);
        }
    }
    

    /**
     * Logs the exception count
     */
    private void logExceptionCount() {
        if (exceptionCount > 1) {
            LOG.log(Level.INFO, "Previous {0} exception occurred {1} times", new Object[]{name, exceptionCount});
        }
    }

    
    private void determineCCDsPowerState() {
        try {
            CCDsPowerState powerState = setCCDsPowerState(true);
            clearUpdateCCDsPowerState();
            LOG.log(Level.INFO, "CCDsPowerState for {0} is {1}", new Object[]{rebPath, powerState.name()});
        } catch (RaftException e) { // first reading can be spurious on OD0
            LOG.log(Level.SEVERE, "Error getting CCDsPowerState for " + rebPath, e);
            LOG.log(Level.SEVERE, "CCDsPowerState changed to {0}",
                    stateService.getComponentState(rebPath, CCDsPowerState.class).name());
            LOG.log(Level.SEVERE, "calling powerCCDsOff() for {0}", rebPath);
            try {
                powerCCDsOff();
            } catch (RaftException et) {
                // TODO raise and alert here to get the REB shut off???
                // alertService.raiseAlert(...)
                LOG.log(Level.SEVERE, String.format("powerCCDsOff() FAILED for %s", rebPath));
                LOG.log(Level.SEVERE, String.format("CCDsPowerState changed to %s",
                        stateService.getComponentState(rebPath, CCDsPowerState.class).name()));
            }
        }
        
    }

    @Override
    protected void setOnline(boolean online) {
        super.setOnline(online);
        if (online) {            
            stateService.updateAgentComponentState(this, RebDeviceState.ONLINE);
            determineCCDsPowerState();
        } else {
           stateService.updateAgentComponentState(this, RebDeviceState.OFFLINE, RebValidationState.UNKNOWN,
                   HVBiasState.UNKNOWN, CCDsPowerState.UNKNOWN);
        }
    }
    

    /**
     * Determines whether CCD is manufactured by ITL
     * @return <code>true</code> iff ccd manufacturer is ITL
     */
    boolean isITLManufacturedCCD()
    {
        return "itl".equalsIgnoreCase(getCcdType().getManufacturer());
    }

    private void forceDataPublicationForNextUpdateIterations(int count) {
        for (MonitorUpdateTask task: tasksForDevice) {
            task.forceDataPublicationOnNextUpdates(count);
        }
    }

    @Command(type=CommandType.QUERY, description="Show list of monitor-update task names")
    public String getMonitorTasks()
    {
       return getMonitorTasksAsString();
    }

    private String getMonitorTasksAsString()
    {
        StringBuilder result = new StringBuilder();
        for (MonitorUpdateTask task: tasksForDevice) {
            result.append(task.getName()).append(" ");
        }
        return result.toString();
    }

    @Command(type=CommandType.QUERY, description="Pause \"all\" or selected monitoring tasks")
    public void pauseMonitorTasks(@Argument(description = "tasks...") String... taskNames) {
        Set<MonitorUpdateTask> tasksToPause = new HashSet<>();
        for (MonitorUpdateTask task : tasksForDevice) {
            for (String taskName : taskNames) {
                if (taskName.equals("all") || taskName.equals(task.getName())) {
                    tasksToPause.add(task);
                }
            }
        }
        for (MonitorUpdateTask task : tasksToPause) {
            task.pausePeriodicUpdate();
            LOG.log(Level.FINE, "monitor-update pausing task {0}", task.getName());
        }
    }

    @Command(type=CommandType.QUERY, description="Resume \"all\" or selected monitoring tasks")
    public void resumeMonitorTasks(@Argument(description="tasks...") String... taskNames)
    {
        Set<MonitorUpdateTask> tasksToResume = new HashSet<>();
        for (MonitorUpdateTask task: tasksForDevice) {
            for (String taskName : taskNames) {
                if (taskName.equals("all") || taskName.equals(task.getName())) {
                    tasksToResume.add(task);
                }
            }
        }
        for (MonitorUpdateTask task: tasksToResume) {
            task.resumePeriodicUpdate();
                LOG.log(Level.FINE, "monitor-update resuming task {0}", task.getName());
            }
    }

    @Command(type = CommandType.QUERY, description = "Trigger \"all\" or selected monitoring tasks")
    public void triggerMonitorTasks(@Argument(description = "tasks...") String... taskNames) {
        Set<MonitorUpdateTask> tasksToTrigger = new HashSet<>();
        for (MonitorUpdateTask task : tasksForDevice) {
            for (String taskName : taskNames) {
                if (taskName.equals("all") || taskName.equals(task.getName())) {
                    tasksToTrigger.add(task);
                }
            }
        }
        for (MonitorUpdateTask task : tasksToTrigger) {
            task.scheduleUpdateAndPublishNow();
            LOG.log(Level.FINE, "monitor-update triggered task {0}", task.getName());
        }
    }   
}
