package org.lsst.ccs.subsystem.rafts;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import java.util.logging.Level;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.rafts.data.ImageData;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.data.RaftState;
import org.lsst.ccs.subsystem.rafts.data.RaftsAgentProperties;
import org.lsst.ccs.subsystem.rafts.states.SequencerMainState;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.FocalPlane;
import org.lsst.ccs.utilities.ccd.Raft;

/**
 *  Implements the rafts test subsystem.
 *
 *  @author Owen Saxton
 */
public class RaftsMain implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final List<REBDevice> rebDevices = new ArrayList();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    protected final Map<String, REBDevice> rebDevicesMap = new HashMap();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private TempControl tempCtrl;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService aptService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService apService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService asService;

    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private ImageNameService imageNameService;
        
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private GlobalProc globalProc;

    @ConfigurationParameter(category=REBDevice.RAFTS)
    private volatile CCDType ccdType = null;

    private static final Logger LOG = Logger.getLogger(RaftsMain.class.getName());
    private RaftsCommands raftsCommands;

    private final Raft raftGeometry = FocalPlane.createFocalPlane().getChild(2, 2);
    private boolean running = false;

    @Override
    public void postBuild() {
        //Add a periodic task to check if Sequencers are running
        //This will be replaced once the DAQ v2 will be able to call us back
        //when the sequencer is done running
        aptService.scheduleAgentPeriodicTask( new AgentPeriodicTask("sequencerMonitor",
                () -> { raftsCommands.checkSequencerState(); }).withIsFixedRate(false).withPeriod(Duration.ofSeconds(5)));        
    }

    
    /**
     *  Initializes the rafts subsystem.
     */
    @Override
    public void postInit() {
        // Add AgentInfo properties
        apService.setAgentProperty(RaftsAgentProperties.RAFT_TYPE_AGENT_PROPERTY, RaftsMain.class.getCanonicalName());

        // Check for temperature control object
        if (tempCtrl == null) {
            LOG.log(Level.WARNING, "No temperature control available");
        }

        // Create RaftsCommands object and add to the CommandSet
        raftsCommands = new RaftsCommands(subsys, rebDevices, globalProc, raftGeometry, tempCtrl, imageNameService);
        subsys.getAgentService(AgentCommandDictionaryService.class).addCommandSetToObject(raftsCommands, "");
    }


    /**
     *  Starts the subsystem.
     *
     */
    @Override
    public void postStart() 
    {
        // Propagete the CCD type everywhere
        raftsCommands.setCcdType(ccdType);

        running = true;
        LOG.log(Level.INFO, "Rafts subsystem started");
        //At start we don't know what is the main state of the Sequencer
        //so we set it to UNKNOWN
        asService.updateAgentState(SequencerMainState.UNKNOWN);
        publishState();
    }


    /**
     *  Sets the CCD type
     *
     *  @param  type  The enumerated CCD type
     */
    @ConfigurationParameterChanger
    public void setCcdType(CCDType type)
    {
        ccdType = type;
        raftsCommands.setCcdType(ccdType);
    }


    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(name="value", description="The tick period (ms)")
                                int value)
    {
        setTickPeriod(value);
        publishState();
    }


    /**
     *  Saves a named rafts configuration.
     *
     *  @param  name  The configuration name
     *  @throws  IOException
     */
    @Command(type=CommandType.ACTION, description="Save the current configuration")
    public void saveNamedConfig(String name) throws IOException
    {
        if (name.isEmpty()) {
            sce.saveChangesForCategories(REBDevice.RAFTS);
        }
        else {
            sce.saveChangesForCategories(REBDevice.RAFTS + ":" + name);
        }
    }


    /**
     *  Loads a named rafts configuration.
     *
     *  @param  name  The configuration name
     *  @throws  IOException
     */
    @Command(type=CommandType.ACTION, description="Load a named rafts configuration")
    public void loadNamedConfig(String name) throws IOException
    {
        sce.loadCategories(REBDevice.RAFTS_LIMITS + ":" + name, REBDevice.RAFTS + ":" + name);
    }


    /**
     *  Gets the full state of the rafts module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The full Raft state
     */
    @Command(type=CommandType.QUERY, description="Get the full Raft module state")
    public RaftState getFullState()
    {
        return new RaftState(getTickPeriod());
    }    


    /**
     *  Gets a portion of a saved raw image.
     *
     *  @param  fileName  The name of the file containing the image
     *  @param  offset    The offset (in pixels) to the first pixel data to obtain.
     *  @param  count     The number of data pixels to obtain.
     *  @return  The returned pixel data
     *  @throws  RaftException
     *  @throws  IOException
     */
    @Command(type=CommandType.QUERY, description="Get a portion of a saved raw image")
    public static ImageData getImage(@Argument(name="filename", description="The image file name")
                                     String fileName,
                                     @Argument(name="offset", description="The offset to the first pixel")
                                     int offset,
                                     @Argument(name="count", description="The number of pixels to get")
                                     int count)
        throws RaftException, IOException
    {
        return ImageProc.getImage(fileName, offset, count);
    }


    /**
     *  Publishes the state of the raft module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    void publishState()
    {
        if (!running) return;
        KeyValueData data = new KeyValueData(RaftState.KEY, new RaftState(getTickPeriod()));
        subsys.publishSubsystemDataOnStatusBus(data);
    }


    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        aptService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the tick period
     */
    private int getTickPeriod()
    {
        return (int)aptService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }


    /**
     *  Protected methods. Used only by tests.
     */
    RaftsCommands getRaftsCommands() {
        return raftsCommands;
    }
    
}
