package org.lsst.ccs.subsystem.rafts;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.PIController;

/**
 *  Implements a temperature controller for the rafts system.
 *
 *  @author Owen Saxton
 */
public class TempControl implements HasLifecycle {

    private static final String TIMER_NAME = "temp-loop-timer";

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupName
    private String name;

    @LookupPath
    private String tempPath;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String,Channel> allChannels = new HashMap<>();
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String,REBDevice> allRebs = new HashMap<>();
    
    private static final Logger LOG = Logger.getLogger(TempControl.class.getName());
    private Double  gain;              // loop gain
    private Double  timeConst;         // integration time constant (secs)
    private Double  smoothTime;        // input smoothing time (secs)   
    private Double  maxOutput;         // maximum PID output (watts)
    private Double  awGain;            // anti-windup gain
    private Double  basePower;         // base power input
    private Double  tolerance;         // maximum on-target error (%)
    private double  minOutput = 0.0;   // minimum PID output (watts)
    private double  maxInput = 100.0;  // maximum input
    private double  minInput = -200.0; // minimum input
    private Integer updateTime;        // The update time interval (msec)
    private String[] tempChans;        // Temperature channels to use
    private String[] rebs;             // The REBs with heaters to use

    private PIController pic;          // The PI controller
    private final List<Channel> tempChansL = new ArrayList<>();  // Temperature channels to use
    private final List<REBDevice> rebsL = new ArrayList<>();     // REBs with heaters to use

    private double lastPower;          // The last power value set
    private boolean active;            // True if loop is active
    private double setTemp;            // Temperature set point


    /**
     *  Start temperature control task.
     */
    @Override
    public void build() {
        pts.scheduleAgentPeriodicTask(new AgentPeriodicTask(TIMER_NAME, () -> {
            iterateLoop();
        }).withPeriod(Duration.ofMillis(updateTime)));
    }


    /**
     *  Checks parameters and initializes the PI controller.
     */
    @Override
    public void init()
    {
        if (gain == null) {
            LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{tempPath, "gain", "is missing"});
            throw new RuntimeException("Fatal configuration error");
        }
        if (timeConst == null) {
            LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{tempPath, "timeConst", "is missing"});
            throw new RuntimeException("Fatal configuration error");
        }
        if (smoothTime == null) {
            LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{tempPath, "smoothTime", "is missing"});
            throw new RuntimeException("Fatal configuration error");
        }
        if (maxOutput == null) {
            LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{tempPath, "maxOutput", "is missing"});
            throw new RuntimeException("Fatal configuration error");
        }
        if (awGain == null) {
            LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{tempPath, "awGain", "is missing"});
            throw new RuntimeException("Fatal configuration error");
        }
        if (basePower == null) {
            LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{tempPath, "basePower", "is missing"});
            throw new RuntimeException("Fatal configuration error");
        }
        if (tolerance == null) {
            LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{tempPath, "tolerance", "is missing"});
            throw new RuntimeException("Fatal configuration error");
        }
        if (updateTime == null) {
            LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{tempPath, "updateTime", "is missing"});
            throw new RuntimeException("Fatal configuration error");
        }
        if (tempChans == null) {
            LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{tempPath, "tempChans", "is missing"});
            throw new RuntimeException("Fatal configuration error");
        }
        else {
            for (String cName : tempChans) {
                Channel chan = allChannels.get(cName);
                if (chan != null) {
                    tempChansL.add(chan);
                }
                else {
                    LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}",
                                              new Object[]{tempPath, "tempChans", "contains non-Channel item"});
                    throw new RuntimeException("Fatal configuration error");
                }
            }
        }
        if (rebs == null) {
            LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{tempPath, "rebs", "is missing"});
            throw new RuntimeException("Fatal configuration error");
        }
        else {
            for (String cName : rebs) {
                REBDevice reb = allRebs.get(cName);
                if (reb != null) {
                    rebsL.add(reb);
                }
                else {
                    LOG.log(Level.SEVERE,
                               "{0} configuration parameter {1} {2}", new Object[]{tempPath, "rebs", "contains non-REBDevice item"});
                    throw new RuntimeException("Fatal configuration error");
                }
            }
        }
        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(basePower);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
    }


    /**
     *  Sets the heater power.
     *
     *  @param  power  The power to set
     */
    void setPower(double power)
    {
        for (REBDevice reb : rebsL) {
            try {
                reb.setHeaterPower(0, power / rebs.length);
            }
            catch (Exception e) {
                LOG.log(Level.SEVERE, "Error setting {0} heater: {1}", new Object[]{reb.getName(), e.getMessage()});
            }
        }
    }


    /**
     *  Sets the target temperature.
     *
     *  @param  temp  The temperature to set
     */
    public void setTemperature(double temp)
    {
        setTemp = temp;
        pic.setSetpoint(setTemp);
    }


    /**
     *  Gets the target temperature.
     *
     *  @return  The set temperature
     */
    public double getTemperature()
    {
        return setTemp;
   }


    /**
     *  Starts the control loop.
     *
     *  @param  power  The initial power value
     */
    public void startLoop(double power)
    {
        if (active) return;
        lastPower = power;
        startLoop();
    }


    /**
     *  Starts the control loop.
     */
    public void startLoop()
    {
        if (active) return;
        pic.reset();
        pic.setIntegral(lastPower - basePower);
        active = true;
    }


    /**
     *  Stops the control loop.
     */
    public void stopLoop()
    {
        if (!active) return;
        active = false;
        setPower(0.0);
    }


    /**
     *  Gets the control loop state.
     *
     *  @return  Whether the control loop is active
     */
    public boolean isLoopActive()
    {
        return active;
    }


    /**
     *  Resets the controller.
     */
    public void reset()
    {
        pic.reset();
    }


    /**
     *  Timer method for control loop iteration.
     */
    private void iterateLoop()
    {
        if (!active) return;
        int count = 0;
        double temp = 0.0;
        for (Channel tempChan : tempChansL) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                temp += value;
                count++;
            }
        }
        if (count > 0) {
            double tod = (double)System.currentTimeMillis() / 1000;
            lastPower = pic.performPI(new double[]{temp / count}, tod);
            setPower(lastPower);
        }
        else {
            LOG.log(Level.SEVERE, "Control loop iteration failed: no valid temperature values available");
        }
    }

}
