package org.lsst.ccs.subsystem.rafts;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.reb.Aspic;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.subsystem.rafts.config.ASPIC;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.RaftException;

/**
 *  Implements control of an ASPIC.
 *
 *  @author Owen Saxton
 */
public class AspicControl extends Control {

    /*
     *  Constants.
     */
    public static final String
        RC     = "rc",
        GAIN   = "gain",
        CLAMP  = "clamp",
        AF1    = "af1",
        TM     = "tm";

    private static final int
        GAIN_RC_CH = 0x01,
        CLAMP_CH   = 0x02,
        MODE_CH    = 0x04;

    /*
     *  Data fields.
     */
    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;
    @LookupPath
    String aspicPath;

    // Supplied configurable fields
    @ConfigurationParameter(name=GAIN, category=REBDevice.RAFTS, description="Gain Selector", units="DAC counts")
    private volatile int gain;
    @ConfigurationParameter(name=RC, category=REBDevice.RAFTS, description="RC Selector", units="DAC counts")
    private volatile int rc;
    @ConfigurationParameter(name=CLAMP, category=REBDevice.RAFTS, description="Clamp bits", units= "DAC counts")
    private volatile int clamp;
    @ConfigurationParameter(name=AF1, category=REBDevice.RAFTS, description="First Stage Gain", units= "DAC counts")
    private volatile int af1;
    @ConfigurationParameter(name=TM, category=REBDevice.RAFTS, description="Transparent mode", units= "DAC counts")
    private volatile int tm;

    // Other fields
    private static final Logger LOG = Logger.getLogger(AspicControl.class.getName());
    private Aspic asp;
    private int hwVersion;
    private int change = -1;


    /**
     *  Configures channel description.
     *
     *  @param  mon   The associated monitor object
     *  @param  devc  The associated device object
     */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        asp = ((REBDevice)devc).getAspic();
    }


    /**
     *  Checks configuration.
     *
     *  @return  Whether configuration is okay
     */
    public boolean checkConfig()
    {
        hwVersion = getHwVersion(asp);
        if (hwVersion == REB.VERSION_UNSUPP) {
            LOG.log(Level.SEVERE, "ASPIC {0} configuration not supported by the firmware", hwChan);
            return false;
        }

        try {
            if (hwChan >= 0 && hwChan < Aspic.NUM_SIDES * asp.getNumStrips()) return true;
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error getting number of strips");
            return false;
        }
        try {
	    LOG.log(Level.SEVERE, "{0} configuration parameter {1} {2}", new Object[]{aspicPath, "HW channel", hwChan});
	    throw new RuntimeException("Fatal configuration error");
        }
        catch (Exception e) {
            /* Can be safely ignored */
        }
        return false;
    }


    /**
     *  Gets the hardware version.
     *
     *  @param  asp  The ASPIC object
     *  @return  The hardware version
     */
    public static int getHwVersion(Aspic asp)
    {
        return asp.getVersion() == Aspic.VERSION_UNSUPP
                 ? REB.VERSION_UNSUPP : REB.VERSION_0;
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version
     */
    @Deprecated
    public int getHwVersion()
    {
        return hwVersion;
    }


    /**
     *  Sets/gets the gain.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGain(int value)
    {
        gain = value;
        change |= GAIN_RC_CH;
    }

    @Command(type=CommandType.QUERY, description="Gets the gain")
    public int getGain()
    {
        return gain;
    }


    /**
     *  Sets/gets the rc value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRc(int value)
    {
        rc = value;
        change |= GAIN_RC_CH;
    }

    @Command(type=CommandType.QUERY, description="Gets the rc value")
    public int getRc()
    {
        return rc;
    }


    /**
     *  Sets/gets the clamp mask.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setClamp(int value)
    {
        clamp = value;
        change |= CLAMP_CH;
    }

    @Command(type=CommandType.QUERY, description="Gets the clamp")
    public int getClamp()
    {
        return clamp;
    }


    /**
     *  Sets/gets the af1 (force first-stage gain to 1) value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setAf1(int value)
    {
        af1 = value;
        change |= MODE_CH;
    }

    @Command(type=CommandType.QUERY, description="Gets the af1 value")
    public int getAf1()
    {
        return af1;
    }


    /**
     *  Sets/gets the tm (transparent mode) value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setTm(int value)
    {
        tm = value;
        change |= MODE_CH;
    }

    @Command(type=CommandType.QUERY, description="Gets the tm")
    public int getTm()
    {
        return tm;
    }


    /**
     *  Sets all the configuration data.
     */
    void setConfig(ASPIC aspic)
    {
        sce.change(aspicPath, GAIN, aspic.getGain());
        sce.change(aspicPath, RC, aspic.getRc());
        sce.change(aspicPath, CLAMP, aspic.getClamp());
        sce.change(aspicPath, AF1, aspic.isAf1() ? 1 : 0);
        sce.change(aspicPath, TM, aspic.isTm() ? 1 : 0);
    }


    /**
     *  Gets all the configuration data.
     */
    ASPIC getConfig()
    {
        ASPIC aspic = new ASPIC();
        aspic.setGain(gain);
        aspic.setRc(rc);
        aspic.setClamp(clamp);
        aspic.setAf1(af1 != 0);
        aspic.setTm(tm != 0);

        return aspic;
    }


    /**
     *  Loads configuration data to an ASPIC.
     *
     *  @return  The number of ASPICs loaded: 0 or 1
     *  @throws  Exception
     */
    int load() throws Exception
    {
        if (asp.getVersion() == Aspic.VERSION_UNSUPP) return 0;
        try {
            int strips = 1 << hwChan / 2;
            int side = (hwChan & 1) == 0 ? Aspic.SIDE_TOP : Aspic.SIDE_BOTTOM;
            asp.writeGainRc(strips, 1 << side, gain, rc);
            asp.writeClamp(strips, 1 << side, clamp);
            int mode = (af1 != 0 ? Aspic.MODE_AF1 : 0) | (tm != 0 ? Aspic.MODE_TM : 0);
            asp.writeModes(strips, 1 << side, mode);
            change = 0;
            return 1;
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
            return 0;
        }
    }


    /**
     *  Loads changed configuration data to an ASPIC.
     *
     *  @return  The number of ASPICs loaded
     *  @throws  Exception
     */
    int loadChanged() throws Exception
    {
        if (asp.getVersion() == Aspic.VERSION_UNSUPP) return 0;
        int count = 0;
        if (change != 0) {
            try {
                int strips = 1 << hwChan / 2;
                int side = (hwChan & 1) == 0 ? Aspic.SIDE_TOP : Aspic.SIDE_BOTTOM;
                if ((change & GAIN_RC_CH) != 0) {
                    asp.writeGainRc(strips, 1 << side, gain, rc);
                }
                if ((change & CLAMP_CH) != 0) {
                    asp.writeClamp(strips, 1 << side, clamp);
                }
                if ((change &= MODE_CH) != 0){
                    int mode = (af1 != 0 ? Aspic.MODE_AF1 : 0) | (tm != 0 ? Aspic.MODE_TM : 0);
                    asp.writeModes(strips, 1 << side, mode);
                }
                count = 1;
                change = 0;
            }
            catch (REBException e) {
                checkTimeout(e, RaftException.class);
            }
        }
        return count;
    }


    /**
     *  Checks loaded ASPIC data against its configuration.
     *
     *  @return  A mask of field numbers that didn't match
     *  @throws  Exception
     */
    int check() throws Exception
    {
        if (asp.getVersion() == Aspic.VERSION_UNSUPP) return 0;
        try {
            int mask = 0;
            int strips = 1 << hwChan / 2;
            int side = (hwChan & 1) == 0 ? Aspic.SIDE_TOP : Aspic.SIDE_BOTTOM;
            int[][] values = asp.readGainRc(strips, side);
            if (values[0][0] != gain) {
                mask |= 1;
            }
            if (values[0][1] != rc) {
                mask |= 2;
            }
            values[0] = asp.readClamp(strips, side);
            if (values[0][0] != clamp) {
                mask |= 4;
            }
            values[0] = asp.readModes(strips, side);
            int value = values[0][0] & (Aspic.MODE_AF1 | Aspic.MODE_TM);
            if (value != ((af1 != 0 ? Aspic.MODE_AF1 : 0) | (tm != 0 ? Aspic.MODE_TM : 0))) {
                mask |= 8;
            }
            return mask;
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
            return 0;
        }
    }


    /**
     *  Resets an ASPIC and its strip companion.
     *
     *  @return  The number of ASPICs reset: 0 or 2
     *  @throws  Exception
     */
    int reset() throws Exception
    {
        if (asp.getVersion() == Aspic.VERSION_UNSUPP) return 0;
        try {
            asp.reset(hwChan / 2);
            return 2;
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
            return 0;
        }
    }

}
