package org.lsst.ccs.subsystem.rafts;

import java.util.Arrays;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.daq.utilities.FitsHeaderKeywordData;
import org.lsst.ccs.daq.utilities.FitsService;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.drivers.reb.Asic;
import org.lsst.ccs.drivers.reb.Aspic;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.BoardDacs;
import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.drivers.reb.ImageClient;
import org.lsst.ccs.drivers.reb.ImageMetadata;
import org.lsst.ccs.drivers.reb.PowerAdcs;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.Sequencer;
import org.lsst.ccs.drivers.reb.SlowAdcs;
import org.lsst.ccs.drivers.reb.StatusSet;
import org.lsst.ccs.drivers.reb.TempAdcs;
import org.lsst.ccs.drivers.reb.TempRtds;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.rafts.alerts.RaftAlert;
import org.lsst.ccs.subsystem.rafts.config.ASPIC;
import org.lsst.ccs.subsystem.rafts.config.BiasDACS;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.ImageData;
import org.lsst.ccs.subsystem.rafts.data.RegisterData;
import org.lsst.ccs.subsystem.rafts.data.StatusData;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.states.HVBiasState;
import org.lsst.ccs.subsystem.rafts.states.RebDeviceState;
import org.lsst.ccs.subsystem.rafts.states.RebValidationState;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *  Implements access to a REB device.
 *
 *  @author Owen Saxton
 */
public class REBDevice extends Device implements ClearAlertHandler, ConfigurationBulkChangeHandler {

    /*
     *  Class for holding read DAC settings
     */
    public static class AdcData {

        static final int
            TYPE_POWER = 0,
            TYPE_SLOW = 1;

        final int type;    // Power or slow ADC
        final int hwChan;
        final double value;
        final CCSTimeStamp timeStamp;

        public AdcData(int type, int hwChan, double value, CCSTimeStamp timeStamp)
        {
            this.type = type;
            this.hwChan = hwChan;
            this.value = value;
            this.timeStamp = timeStamp == null ? CCSTimeStamp.currentTime() : timeStamp;
        }

    }

    /*
     *  Constants.
     */
    private static final String
        SERIAL_NUM = "serialNum";

    public static final String
        RAFTS = "Rafts",
        RAFTS_LIMITS = "RaftsLimits",
        RAFTS_POWER = "RaftsPower";

    public static final int
        TYPE_BD_TEMP   = 0,
        TYPE_BD_POWER  = 1,
        TYPE_RTD       = 2,
        TYPE_ASP_TEMP  = 3,
        TYPE_BIAS_VOLT = 4,
        TYPE_CCD_CURR  = 5,
        TYPE_CR_VOLT   = 6,
        TYPE_HEATER    = 7,
        TYPE_HV_SWITCH = 8;

    public static final int
        CHAN_TOTAL_POWER = -1,
        CHAN_HTR_VOLTS = 0,
        CHAN_HTR_POWER = 1,
        NUM_HTR_CHANS  = 2,
        CHAN_RTD_INT_TEMP = TempRtds.NUM_RTD_TEMPS,
        CHAN_RTD_INT_VOLT = TempRtds.NUM_RTD_TEMPS + 1,
        NUM_RTD_CHANS     = TempRtds.NUM_RTD_TEMPS + 2;

    static final int
        REB_NUM_MASK = 0x03,
        RAFT_NUM_MASK = 0x3f,
        RAFT_NUM_SHIFT = 2;

    static final int
        POWER_ON_DELAY = 200,
        POWER_OFF_DELAY = 50;

    static final int
        DAC_LIMIT = 4095;

    static final double
        DAC_CONV = DAC_LIMIT / 5.0,
        HTR_CONV = DAC_CONV / (1.0 + 10.0 / 10.0),
        HTR_CONV_R5 = 2.0 * DAC_CONV / (1.0 + 30.0 / 10.0),
        HEATER_OHMS = 14.5,   // Estimated value at -100C
        HTR_LEAD_OHMS = 0.3;  // Lead resistance

    static final int
        ADC_PCLK_L = 0,
        ADC_PCLK_U = 1,
        ADC_SCLK_L = 2,
        ADC_SCLK_U = 3,
        ADC_RG_L = 4,
        ADC_RG_U = 5,
        ADC_GD_0 = 6,
        ADC_OD_0 = 7,
        ADC_OG_0 = 8,
        ADC_RD_0 = 9,
        NUM_WREB_ADCS = 10,
        ADC_GD_1 = 10,
        ADC_OD_1 = 11,
        ADC_OG_1 = 12,
        ADC_RD_1 = 13,
        NUM_GREB_ADCS = 14,
        ADC_GD_2 = 14,
        ADC_OD_2 = 15,
        ADC_OG_2 = 16,
        ADC_RD_2 = 17,
        NUM_SREB_ADCS = 18;

    private static enum ChannelGroup {
        rebTemp,
        aspicTemp,
        boardPower,
        crVolt;
    }
        
    /*
     *  Private lookup maps, etc
     */
    private static final Map<String,Integer> hdwTypeMap = new HashMap<>();
    static {
        hdwTypeMap.put("DAQ",  BaseSet.HDW_TYPE_DAQ);
        hdwTypeMap.put("DAQ0", BaseSet.HDW_TYPE_DAQ0);
        hdwTypeMap.put("DAQ1", BaseSet.HDW_TYPE_DAQ1);
        hdwTypeMap.put("DAQ2", BaseSet.HDW_TYPE_DAQ2);
        hdwTypeMap.put("DAQ4", BaseSet.HDW_TYPE_DAQ4);
        hdwTypeMap.put("PCI",  BaseSet.HDW_TYPE_PCI1);
        hdwTypeMap.put("PCI0", BaseSet.HDW_TYPE_PCI0);
        hdwTypeMap.put("PCI1", BaseSet.HDW_TYPE_PCI1);
    }
    private static final Map<String,Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("TEMP",   TYPE_BD_TEMP);
        typeMap.put("POWER",  TYPE_BD_POWER);
        typeMap.put("RTD",    TYPE_RTD);
        typeMap.put("ATEMP",  TYPE_ASP_TEMP);
        typeMap.put("BIAS",   TYPE_BIAS_VOLT);
        typeMap.put("CURR",   TYPE_CCD_CURR);
        typeMap.put("CRVOLT", TYPE_CR_VOLT);
        typeMap.put("HEATER", TYPE_HEATER);
        typeMap.put("HVSWCH", TYPE_HV_SWITCH);
    }
    private static final int[] adcChans = new int[NUM_SREB_ADCS];
    static {
        adcChans[ADC_PCLK_L] = SlowAdcs.CHAN_CKP_L;
        adcChans[ADC_PCLK_U] = SlowAdcs.CHAN_CKP_U;
        adcChans[ADC_SCLK_L] = SlowAdcs.CHAN_SCK_L;
        adcChans[ADC_SCLK_U] = SlowAdcs.CHAN_SCK_U;
        adcChans[ADC_RG_L] = SlowAdcs.CHAN_RG_L;
        adcChans[ADC_RG_U] = SlowAdcs.CHAN_RG_U;
        adcChans[ADC_GD_0] = SlowAdcs.CHAN_GD_0;
        adcChans[ADC_OD_0] = SlowAdcs.CHAN_OD_0;
        adcChans[ADC_RD_0] = SlowAdcs.CHAN_RD_0;
        adcChans[ADC_OG_0] = SlowAdcs.CHAN_OG_0;
        adcChans[ADC_GD_1] = SlowAdcs.CHAN_GD_1;
        adcChans[ADC_OD_1] = SlowAdcs.CHAN_OD_1;
        adcChans[ADC_RD_1] = SlowAdcs.CHAN_RD_1;
        adcChans[ADC_OG_1] = SlowAdcs.CHAN_OG_1;
        adcChans[ADC_GD_2] = SlowAdcs.CHAN_GD_2;
        adcChans[ADC_OD_2] = SlowAdcs.CHAN_OD_2;
        adcChans[ADC_RD_2] = SlowAdcs.CHAN_RD_2;
        adcChans[ADC_OG_2] = SlowAdcs.CHAN_OG_2;
    }
    private static final String[] adcNames = new String[NUM_SREB_ADCS];
    static {
        adcNames[ADC_PCLK_L] = "PclkLow";
        adcNames[ADC_PCLK_U] = "PclkHigh";
        adcNames[ADC_SCLK_L] = "SclkLow";
        adcNames[ADC_SCLK_U] = "SclkHigh";
        adcNames[ADC_RG_L] = "RGLow";
        adcNames[ADC_RG_U] = "RGHigh";
        adcNames[ADC_GD_0] = "GD0";
        adcNames[ADC_OD_0] = "OD0";
        adcNames[ADC_OG_0] = "OG0";
        adcNames[ADC_RD_0] = "RD0";
        adcNames[ADC_GD_1] = "GD1";
        adcNames[ADC_OD_1] = "OD1";
        adcNames[ADC_OG_1] = "OG1";
        adcNames[ADC_RD_1] = "RD1";
        adcNames[ADC_GD_2] = "GD2";
        adcNames[ADC_OD_2] = "OD2";
        adcNames[ADC_OG_2] = "OG2";
        adcNames[ADC_RD_2] = "RD2";
    }

    private static final int seqClockLines = Sequencer.SEQ_LINE_P1 | Sequencer.SEQ_LINE_P2 | Sequencer.SEQ_LINE_P3
        | Sequencer.SEQ_LINE_P4 |Sequencer.SEQ_LINE_S1 | Sequencer.SEQ_LINE_S2
        | Sequencer.SEQ_LINE_S3 | Sequencer.SEQ_LINE_RG;


    /*
     *  Data fields.
     */
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private GlobalProc globalProc;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private FitsService fitsService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, Channel> childChannels = new HashMap<>();

    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private final Map<String, Control> controlsMap = new LinkedHashMap<>();   // Control channels
    
    
    @ConfigurationParameter(name=SERIAL_NUM, category=RAFTS)
    private long serialNum;
    @ConfigurationParameter(category=RAFTS, isFinal=true)
    private int id = -1;
    @ConfigurationParameter(category=RAFTS, isFinal=true)
    private boolean useScienceCCD = false;
    
    @ConfigurationParameter(category=RAFTS, description = "Disables the RTD derived CCD type check.")
    private boolean disableRTDHardwareCheck = false;

    boolean processImages = true;
    
    // ODI
    @ConfigurationParameter(category=RAFTS_POWER, description="Maximum OFF_State ODI (A) per CCD", units="Amps")
    private double  odiQmax;  // ~0.005
    @ConfigurationParameter(category=RAFTS_POWER, description="Minimum ON_State ODI (A) per CCD", units="Amps")
    private double  odiAmin;  // ~0.023
    // CLKLI
    @ConfigurationParameter(category=RAFTS_POWER, description="Maximum OFF_State CLKLI (A) per REB", units="Amps")
    private double  clkliQmax; // ~0.045
    @ConfigurationParameter(category=RAFTS_POWER, description="Minimum ON_State CLKLI (A) per REB", units="Amps")
    private double  clkliAmin; // ~0.053
    // CLKHI
    @ConfigurationParameter(category=RAFTS_POWER, description="Maximum OFF_State CLKHI (A) per REB", units="Amps")
    private double  clkhiQmax;  // ~0.093
    @ConfigurationParameter(category=RAFTS_POWER, description="Minimum ON_State CLKHI (A) per REB", units="Amps")
    private double  clkhiAmin;  // ~0.103

    // Set from the Groovy file
    private String hdwType = "daq";
    private String ifcName;   // Only if not using DAQ V2/4; normally set via a call
    private int rebType = BaseSet.REB_TYPE_SCIENCE;
    private int[] dataSegmentMap = null;

    private static final Logger LOG = Logger.getLogger(REBDevice.class.getName());
    private final BaseSet bss = new BaseSet();
    private final StatusSet sts = new StatusSet(bss);
    private final TempAdcs tmp = new TempAdcs(bss);
    private final PowerAdcs pwr = new PowerAdcs(bss);
    private final SlowAdcs slow = new SlowAdcs(bss);
    private final TempRtds rtd = new TempRtds(bss);
    private final BoardDacs dac = new BoardDacs(bss);
    private final Aspic asp = new Aspic(bss);
    private final Sequencer seq = new Sequencer(bss);
    private final ImageClient imc = new ImageClient(bss);
    private final BiasControl[] biases = new BiasControl[Asic.MAX_STRIPS];
    private final AspicControl[] aspics = new AspicControl[Asic.MAX_ASICS];
    private ImageProc img;
    private final SequencerProc sqp = new SequencerProc(bss);
    private DacControl dacCtrl;
    private final Map<Integer, String> sadcChannelMap = new HashMap<>();
    private final Map<Integer, String> pwrChannelMap = new HashMap<>();
    private long hwSerialNum = 0;
    private boolean serialNumValid = false;
    protected int hdwTypeI, configCcdMask, realCcdMask, hwVersion;
    private int maxTemp, maxPower, maxAtemp, maxCcdI,
                numTemp, numPower, numAtemp, numCrVolt, numRtd;
    private boolean dacRaw, biasRaw;
    private double[] tempValues, powerValues, atempValues;
    private final double[] heaterValues = new double[NUM_HTR_CHANS];
    private int errorCount;
    private int dacLoadDelay = 0, dacClearDelay = 0;  // TODO make config
    private Boolean useFullPath;
    private CCDType ccdType = null, hwCcdType = null;
    private Reb rebGeometry;


    /**
     *  Early initialization (build phase).
     */
    @Override
    public void build()
    {
        super.build();

        ComponentLookup lookupService = subsys.getComponentLookup();
        ComponentNode rebNode = lookupService.getComponentNodeForObject(this);


        if ( processImages ) {
            img = new ImageProc(imc);
            //Add an image processing node
            ComponentNode imageProcNode = new ComponentNode("imageProc", img);
            lookupService.addComponentNodeToLookup(rebNode, imageProcNode);
            if (dataSegmentMap != null) {
                img.setDataSegmentMap(dataSegmentMap);
            }
            if (fitsService == null) {
                LOG.log(Level.WARNING, "No fits service for {0} is defined in groovy. For now using internal instance.", name);
                FitsService fitsSrvc = new FitsService();
                //Since the fits service was not defined in groovy we add it here.
                ComponentNode fitsServiceNode = new ComponentNode(name + ".FitsService", fitsSrvc);
                lookupService.addComponentNodeToLookup(rebNode, fitsServiceNode);
                fitsSrvc.setHeaderFilesList(Arrays.asList(new String[]{"primary", "extended", "Fits_primary_header.spec:primary"}));
            }
        } else {
            LOG.log(Level.INFO, "No images will be processed by this device: {0}", name);            
        }
                
    }
    
    public void setRebGeometry(Reb reb) {
        this.rebGeometry = reb;
        if ( img != null ) {
            img.setRebGeometry(reb);
        }
        setCcdType(reb.getCCDType());
    }

    public Reb getRebGeometry() {
        return rebGeometry;
    }

    /**
     *  Validates configuration parameter change.
     *
     *  @param  params  The map of configuration parameter values
     */
    @Override
    public void validateBulkChange(Map<String, Object> params)
    {
        long sNum = (Long)params.get(SERIAL_NUM);
        if (hwSerialNum != 0) {
            if (sNum != -1 && sNum != hwSerialNum) {
                // Note, this exception vetos the configuration change, so no state change should occur. 
                String msg = "Configuration has wrong REB s/n: " + sNum + "; " + name + " s/n is " + hwSerialNum;
                throw new IllegalArgumentException(msg);
            }
            LOG.log(Level.INFO, "Configured {0} s/n ({1}) is valid", new Object[]{name, sNum});
            serialNumValid = true;
            stateService.updateAgentComponentState(this, RebValidationState.VALID);                        
        }
    }


    /**
     *  Gets the REB ID.
     *
     *  @return  The REB ID
     */
    @Command(type=CommandType.QUERY, description="Get the REB ID")
    public int getId()
    {
        return id;
    }


    /**
     *  Gets the network interface name.
     *
     *  @return  The name of the network interface being used
     */
    @Command(type=CommandType.QUERY, description="Get the network interface name")
    public String getIfcName()
    {
        return ifcName;
    }


    /**
     *  Gets the mask of CCDs being used.
     *
     *  @return  The mask of active CCDs
     */
    @Command(type=CommandType.QUERY, description="Get the mask of CCDs being used")
    public int getCcdMask()
    {
        return realCcdMask;
    }


    /**
     *  Sets the CCD type.
     *
     *  @param  ccdType  The CCD type
     */
    //@ConfigurationParameterChanger
    public void setCcdType(CCDType ccdType)
    {
        LOG.log(Level.INFO, "setCcdType to {0} for {1}", new Object[]{ccdType, name});
        this.ccdType = ccdType;
        dacLoadDelay = ccdType == null ? 0 : POWER_ON_DELAY;
        dacClearDelay = ccdType == null ? 0 : POWER_OFF_DELAY;
    }


    /**
     *  Gets the CCD type.
     *
     *  @return  The CCD type
     */
    @Command(type=CommandType.QUERY, description="Get the type of CCDs being used")
    public CCDType getCcdType()
    {
        return ccdType;
    }


    /**
     *  Gets the hardware CCD type.
     *
     *  @return  The hardware CCD type
     */
    @Command(type=CommandType.QUERY, description="Get the hardware type of CCDs being used")
    public CCDType getHwCcdType()
    {
        return hwCcdType;
    }


    /**
     *  Sets whether a science CCD is being used on a WREB.
     *
     *  @param  useScience  Whether it's a science CCD
     */
    @Command(type=CommandType.ACTION, description="Set whether a WREB is using a science CCD")
    public void setScienceCCD(@Argument(description="Whether a science CCD") boolean useScience)
    {
        rtd.setScienceCCD(useScience);
    }


    /**
     *  Sets the client factory.
     *
     *  @param  factory  The client factory
     */
    public void setClientFactory(ClientFactory factory)
    {
        if (factory != null) {
            bss.setClientFactory(factory);
            imc.setClientFactory(factory);
        }
    }


    /**
     *  Gets the REB type.
     *
     *  @return  The REB type
     */
    @Command(type=CommandType.QUERY, description="Gets the REB type")
    public int getRebType()
    {
        return rebType;
    }


    /**
     *  Gets the REB number.
     *
     *  @return  The REB number (0 - 2) within its raft
     */
    @Command(type=CommandType.QUERY, description="Gets the REB number")
    public int getRebNumber()
    {
        return id & REB_NUM_MASK;
    }


    /**
     *  Gets the raft number.
     *
     *  @return  The raft number (00 - 44) for the REB
     */
    @Command(type=CommandType.QUERY, description="Gets the raft number")
    public int getRaftNumber()
    {
        return (id >> RAFT_NUM_SHIFT) & RAFT_NUM_MASK;
    }


    /**
     *  Gets the hardware type.
     *
     *  @return  The hardware type
     */
    @Command(type=CommandType.QUERY, description="Gets the hardware type")
    public int getHdwType()
    {
        return hdwTypeI;
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version, or 0 if the REB is not connected
     */
    @Command(type=CommandType.QUERY, description="Get the hardware version")
    public int getHwVersion()
    {
        return hwVersion;
    }


    /**
     *  Gets the serial number.
     *
     *  @return  The serial number, or 0 if the REB is not connected
     */
    @Command(type=CommandType.QUERY, description="Get the serial number")
    public long getSerialNumber()
    {
        return hwSerialNum;
    }


    /**
     *  Gets whether the serial number is valid.
     *
     *  @return  The serial number validity
     */
    public boolean isSerialNumValid()
    {
        return serialNumValid;
    }


    /**
     * Callback to clear an {@code Alert} instance.
     * Return a ClearAlertCode for the provided Alert.
     * 
     * @param alert The Alert instance to clear.
     * @param alertState The AlertState for the provided Alert.
     * @return A ClearAlertCode to indicate which action is to be taken
     *         by the framework.
     * 
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState)
    {
        if ( alert.getAlertId().equals(RaftAlert.REB_NOT_CONNECTED.getAlertId()) ) {
            return ClearAlertCode.CLEAR_ALERT;
        }
        return ClearAlertCode.UNKNOWN_ALERT;
    }
    

    /**
     *  Adds channel name or path to map.
     *
     *  @param  map  The map to add to
     *  @param  hwChan  The hw channel number
     *  @param  chName  The channel name
     */
    private void addToChannelMap(Map map, int hwChan, String chName)
    {
        if (useFullPath == null) {
            String fpProperty = subsys.getAgentInfo().getAgentProperty("org.lsst.ccs.use.full.paths", "false");
            useFullPath = "true".equals(fpProperty.toLowerCase());
        }
        if (useFullPath) {
            Channel chan = childChannels.get(chName);
            if (chan == null) {
                chan = mon.getChannel(chName);
            }
            map.put(hwChan, chan.getPath());
        }
        else {
            map.put(hwChan, chName);
        }
    }


    /**
     *  Performs basic initialization.
     * 
     * Invoked when HasLifecycle::init is called on Device class.
     * 
     *  Checks that Groovy specifications are consistent
     */
    @Override
    protected void initDevice()
    {   
        stateService.registerState(RebDeviceState.class, "Reb Device ONLINE/OFFLINE State", this);
        stateService.registerState(RebValidationState.class, "Reb Device Validation State", this);        
        stateService.registerState(HVBiasState.class, "High Voltage Bias ON/OFF State", this);
        stateService.updateAgentComponentState(this, RebDeviceState.OFFLINE, RebValidationState.UNKNOWN, HVBiasState.UNKNOWN);
        
        if (id < 0) {
            ErrorUtils.reportConfigError(LOG, name, "id", "has not been specified");
        }
        String partition = globalProc.getPartition();
        if (partition != null && !partition.isEmpty()) {
            ifcName = partition;
        }
        if (ifcName == null) {
            ErrorUtils.reportConfigError(LOG, name, "ifcName", "has not been specified");
        }
        Integer type = hdwTypeMap.get(hdwType.toUpperCase());
        if (type == null) {
            ErrorUtils.reportConfigError(LOG, name, "hdwType", "is invalid");
        }
        hdwTypeI = type;

        imc.setRebId("reb-" + id);

        /*
         *  Configure the image processor
         */
        if ( img != null ) {
            img.configure(null);
        }

        /*
         *  Get masks of Bias DACs and ASPICs present
         */
        int biasMask = 0, aspMask = 0, aspicMask = 0;

        for (Control ctrl : controlsMap.values()) {
            if (ctrl instanceof BiasControl) {
                biasMask |= 1 << ctrl.getHwChan();
            }
            else if (ctrl instanceof AspicControl) {
                aspMask |= 1 << ctrl.getHwChan();
            }
        }

        /*
         *  Check that ASPICs come in pairs
         *  Generate the maximal mask of strips present
         */
        for (int j = 0; j < Asic.MAX_STRIPS; j++, aspMask >>= 2) {
            int pair = aspMask & 3;
            if (pair == 0) continue;
            if (pair == 3) {
                aspicMask |= (1 << j);
            }
            else {
                LOG.log(Level.SEVERE, "Unpaired configuration for {0} ASPIC {1} ignored",
                        new Object[]{name, 2 * j + pair - 1});
            }
        }
        if (biasMask != 0 && biasMask != aspicMask) {
            LOG.log(Level.WARNING, "{0} ASPIC and bias configurations don't match", name);
        }
        configCcdMask = biasMask | aspicMask;

        /*
         *  Fill the record of control devices
         */
        for (Control ctrl : controlsMap.values()) {
            if (ctrl instanceof DacControl) {
                dacCtrl = (DacControl)ctrl;
            }
            else if (ctrl instanceof BiasControl) {
                if (((1 << ctrl.getHwChan()) & configCcdMask) != 0) {
                    biases[ctrl.getHwChan()] = (BiasControl)ctrl;
                }
            }
            else if (ctrl instanceof AspicControl) {
                if (((1 << ctrl.getHwChan() / 2) & configCcdMask) != 0) {
                    aspics[ctrl.getHwChan()] = (AspicControl)ctrl;
                }
            }
        }

        /*
         *  Set initial value of maximum CCD mask
         */
        bss.setDefaultRebType(BaseSet.REB_TYPE_SCIENCE);
        try {
            realCcdMask = configCcdMask & ((1 << bss.getNumStrips()) - 1);
        }
        catch (REBException e) {
            LOG.log(Level.SEVERE, "Error getting expected {0} CCD count: {1}", new Object[]{name, e}); // Shouldn't happen
        }
        fullName = "REB " + id + (ifcName != null ? " (" + ifcName + ")" : "");
    }
    
    
    /**
     *  Performs full initialization.
     * 
     *  Opens connection to the REB and obtains information about it
     */
    @Override
    protected void initialize()
    {
        try {
            bss.open(hdwTypeI, id, ifcName);
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
            }
            bss.disable();
            bss.setTime();
            bss.enable();
            rebType = bss.getRebType();
            hwVersion = bss.getHwVersion();
            maxTemp = tmp.getNumRegs();
            maxPower = pwr.getNumRegs();
            maxAtemp = slow.getNumAspicTemps();
            maxCcdI = slow.getNumCcdCurrents();
            numTemp = numPower = numAtemp = numCrVolt = numRtd = 0;
            tempValues = new double[maxTemp];
            powerValues = new double[maxPower];
            atempValues = new double[maxAtemp];
            realCcdMask = configCcdMask & ((1 << bss.getNumStrips()) - 1);
            if ( img != null ) {
                img.setCcdMask(realCcdMask);
                img.setNumRebCcds(bss.getNumStrips());
                img.setDataInversion(bss.getRebType() == BaseSet.REB_TYPE_SCIENCE);
            }
            /*
            try {
                seq.setCcdMask(maxCcdMask);
            }
            catch (REBException e) {
                log.severe("Error writing stripe selection");
            }
            */
            if (dacCtrl != null && !dacCtrl.checkConfig()) {
                dacCtrl = null;
            }
            dacRaw = dacCtrl != null ? dacCtrl.isRaw() : false;
            biasRaw = false;
            for (int j = 0; j < biases.length; j++) {
                BiasControl bias = biases[j];
                if (bias != null && !bias.checkConfig()) {
                    bias = null;
                }
                if (bias != null) {
                    biasRaw = bias.isRaw();
                }
                biases[j] = bias;
            }
            for (int j = 0; j < aspics.length; j++) {
                AspicControl aspic = aspics[j];
                if (aspic != null && !aspic.checkConfig()) {
                    aspic = null;
                }
                aspics[j] = aspic;
            }
            try {
                hwSerialNum = bss.getRebSerial();
            }
            catch (REBException e) {
                LOG.log(Level.SEVERE, "Error reading {0} serial number", name);
                hwSerialNum = -1;
            }
            serialNumValid = serialNum == -1 || serialNum == hwSerialNum;
            if (!serialNumValid) {
                stateService.updateAgentComponentState(this, RebValidationState.INVALID);        
                LOG.log(Level.SEVERE, "{0} s/n ({1}) doesn''t match configuration value ({2})",
                        new Object[]{name, hwSerialNum, serialNum});
            } else {
                stateService.updateAgentComponentState(this, RebValidationState.VALID);                        
            }
            rtd.initialize();
            try {
                Thread.sleep(globalProc.getSleepAfterRtdInitialize());
            }
            catch (InterruptedException e) {
                LOG.log(Level.SEVERE, "Thread.sleep(100) interrupted at rtd.initialize() for {0}", name);
            }
            if ( ! disableRTDHardwareCheck ) {            
                try {
                    int type = rtd.getCCDType();
                    hwCcdType = type == TempRtds.CCD_TYPE_E2V ? CCDType.getCCDType("e2v")
                            : type == TempRtds.CCD_TYPE_ITL ? CCDType.getCCDType("itl") : null;
                    if (RunMode.isSimulation()) {
                        hwCcdType = CCDType.getCCDType(ccdType.getName());
                    }
                    LOG.log(Level.INFO, "found hwCcdType {0} for {1}", new Object[]{hwCcdType, name});
                    if (hwCcdType == null || !hwCcdType.equals(ccdType)) {
                        throw new RuntimeException("CCD Hardware type " + hwCcdType + " is inconsistent with configured CCD Type " + ccdType);
                    }
                } catch (REBException e) {
                    LOG.log(Level.SEVERE, "Error getting CCD type for" + name, e);
                    hwCcdType = null;
                }
            }
            
            rtd.setScienceCCD(useScienceCCD);
            errorCount = 0;
            initSensors();                     // cause initChannel() call on each monitored quantity
            imc.open(hdwTypeI, id, ifcName);

            if ( stateService.isInState(PhaseState.OPERATIONAL) ) {
                try {
                    boolean CCDsPowerState = getCCDsPowerState();
                    if (CCDsPowerState) { // CCDs are on -- what to do? what not to do?
                        LOG.log(Level.INFO, "getCCDsPowerState() for {0} found CCDs are ON ", name);
                    } else { // CCDs are off
                        LOG.log(Level.INFO, "getCCDsPowerState() for {0} found CCDs are OFF", name);
                    }
                } catch (RaftException e) {
                    LOG.log(Level.SEVERE, "Error getting CCDsPowerState for "+name,e);
                }
                try {
                    if (sqp.isRunning()) { // Sequencer is running, just log for now
                        LOG.log(Level.INFO, "Sequencer is running for {0}", name);
                    } else {
                        LOG.log(Level.INFO, "Sequencer is idle for {0}", name);
                    }
                } catch (REBException e) {
                        LOG.log(Level.SEVERE, "Failed to determine sequencer state for "+name,e);
                }
            }
            setOnline(true);
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            LOG.log(Level.INFO, String.format("%s has firmware version: 0x%x", name, hwVersion));
            LOG.log(Level.INFO, String.format("%s has serialNumber: %d", name, hwSerialNum));
            if (inited) {                    
                alertService.raiseAlert(RaftAlert.REB_NOT_CONNECTED.getAlert(), AlertState.NOMINAL,
                                        fullName + " is connected");
            }

        }
        catch (REBException e) {
            if (!inited) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                close("could not connect");
            }
            else {
                close(null);
            }
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        close("was disconnected");
    }


    private void close(String alertText)
    {
        try {
            rebType = BaseSet.REB_TYPE_UNKNOWN;
            hwVersion = 0;
            hwSerialNum = 0;
            hwCcdType = null;
            bss.close();
        }
        catch (REBException e) {
        }
        try {
            imc.close();
        }
        catch (REBException e) {
        }
        if (alertText != null) {
            alertService.raiseAlert(RaftAlert.REB_NOT_CONNECTED.getAlert(), AlertState.ALARM,
                                    fullName + " " + alertText);
        }
    }

    private Integer convertTypeStrToInt(String typeStr) throws Exception {
        Integer iType = typeMap.get(typeStr.toUpperCase());
        if (iType == null) {
            ErrorUtils.reportChannelError(LOG, name, "channel type", typeStr);
        }
        return iType;        
    }
    

    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer iType = convertTypeStrToInt(type);
        return new int[]{iType, 0};
    }


    /**
     *  Initializes a channel (in this case only checks HW channel number).
     *
     *  @param  name     The channel name
     *  @param  id       The channel ID
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type
     *  @param  subtype  The channel subtype
     */
    @Override
    protected void initChannel(String name, int id, int hwChan, int type, int subtype)
    {
        try {
            boolean chanOk;
            if (type == TYPE_BIAS_VOLT || type == TYPE_CR_VOLT) {
                chanOk = slow.testChannel(hwChan);
                if (chanOk) {
                    addToChannelMap(sadcChannelMap, hwChan, name);
                }
            }
            else {
                int maxChan = type == TYPE_BD_TEMP ? maxTemp :
                              type == TYPE_BD_POWER ? maxPower :
                              type == TYPE_ASP_TEMP ? maxAtemp :
                              type == TYPE_CCD_CURR ? maxCcdI :
                              type == TYPE_RTD ? NUM_RTD_CHANS :
                              type == TYPE_HEATER ? NUM_HTR_CHANS : 
                              type == TYPE_HV_SWITCH ? 1 : 0;
                chanOk = (hwChan >= 0 || type == TYPE_BD_POWER) && hwChan < maxChan;
                if (chanOk && type == TYPE_BD_POWER && hwChan >= 0) {
                    addToChannelMap(pwrChannelMap, hwChan, name);
                }
            }
            if (!chanOk) {
                ErrorUtils.reportChannelError(LOG, name, "hw channel number", hwChan);
            }
            switch (type) {
            case TYPE_BD_TEMP:
                numTemp++;
                break;
            case TYPE_BD_POWER:
                numPower++;
                break;
            case TYPE_ASP_TEMP:
                numAtemp++;
                break;
            case TYPE_CR_VOLT:
                numCrVolt++;
                break;
            case TYPE_RTD:
                numRtd++;
                break;
            }
        }
        catch (Exception e) {
            dropChannel(id);
        }
    }

    @Override
    protected String getGroupForChannel(Channel ch) {
        try {
            String typeStr = ch.getTypeStr();
            int type = convertTypeStrToInt(typeStr).intValue();
            switch (type) {
                case TYPE_BD_TEMP:
                    return ChannelGroup.rebTemp.name();
                case TYPE_BD_POWER:
                    return ChannelGroup.boardPower.name();
                case TYPE_ASP_TEMP:
                    return ChannelGroup.aspicTemp.name();
                case TYPE_CR_VOLT:
                    return ChannelGroup.crVolt.name();
            }
            return null;
        } catch (Exception e) {
            throw new RuntimeException("Problem fetching group information for channel "+ch.getName(),e);
        }
    }

    
    
    @Override
    protected void readChannelGroup(String group) {
        try {
            ChannelGroup g = ChannelGroup.valueOf(group);
            if (!online) {
                return;
            }
            String item = "";
            try {

                switch (g) {
                    case aspicTemp:
                        if (numAtemp > 0) {
                            item = "aspic temperatures";
                            atempValues = slow.readAspicTemps(0, maxAtemp);
                        }
                        break;
                    case crVolt:
                        if (numCrVolt > 0) {
                            item = "CCD voltages";
                            slow.fetchVoltages();
                        }
                        break;
                    case boardPower:
                        if (numPower > 0) {
                            try {
                                powerValues = pwr.readAdcs();
                            } catch (REBException e) {
                                if (e.getMessage().contains("Completion wait")) {
                                    for (int j = 0; j < powerValues.length; j++) {
                                        powerValues[j] = Double.NaN;
                                    }
                                } else {
                                    item = "board power";
                                    throw (e);
                                }
                            }
                        }
                        break;
                    case rebTemp:
                        if (numTemp > 0) {
                            try {
                                tempValues = tmp.readAdcs();
                            } catch (REBException e) {
                                if (e.getMessage().contains("Completion wait")) {
                                    for (int j = 0; j < tempValues.length; j++) {
                                        tempValues[j] = Double.NaN;
                                    }
                                } else {
                                    item = "board temperatures";
                                    throw (e);
                                }
                            }
                        }
                        break;
                    default:
                        throw new RuntimeException("Implementation is missing for group: " + g);                        
                }
                errorCount = 0;
            } catch (REBException e) {
                LOG.log(Level.SEVERE, "Error reading {0} {1}: {2}", new Object[]{name, item, e.getMessage()});
                checkErrorCount();
            }

        } catch (Exception e) {
            throw new IllegalArgumentException("Could not read channles for group: " + group,e);
        }
    }



    
    /**
     *  Reads grouped channels.
     */
    @Override
    protected void readChannelGroup()
    {
        for (ChannelGroup g : ChannelGroup.values() ) {
            readChannelGroup(g.name());
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        String item = "";
        try {
            switch (type) {
            case TYPE_CCD_CURR:
                item = "CCD current";
                value = slow.readCurrent(hwChan);
                errorCount = 0;
                break;

            case TYPE_BIAS_VOLT:
            case TYPE_CR_VOLT:
                item = "CCD voltage";
                value = slow.readVoltage(hwChan);
                errorCount = 0;
                break;

            case TYPE_RTD:
                item = "RTD";
                if (hwChan < TempRtds.NUM_RTD_TEMPS) {
                    value = rtd.readTemperature(hwChan);
                }
                else if (hwChan == CHAN_RTD_INT_TEMP) {
                    value = rtd.readIntTemperature();
                }
                else {
                    value = rtd.readIntVoltage();
                }
                errorCount = 0;
                break;

            case TYPE_HV_SWITCH:
                item = "HV bias switch";
                value = bss.isBackBiasOn() ? 1.0 : 0.0;
                HVBiasState newState = value > 0 ? HVBiasState.ON : HVBiasState.OFF;
                if ( ! stateService.isComponentInState(getName(), newState) ) {
                    stateService.updateAgentComponentState(this, newState );                                        
                }
                errorCount = 0;
                break;

            case TYPE_BD_POWER:
                if (hwChan >= 0) {
                    value = powerValues[hwChan];
                }
                else {
                    value = 0.0;
                    for (int j = 0; j < powerValues.length; j += 2) {
                        value += powerValues[j] * powerValues[j + 1];
                    }
                    value += heaterValues[CHAN_HTR_POWER];
                }
                break;

            case TYPE_BD_TEMP:
                value = tempValues[hwChan];
                break;

            case TYPE_ASP_TEMP:
                value = atempValues[hwChan];
                break;

            case TYPE_HEATER:
                value = heaterValues[hwChan];
                break;
            }
        }
        catch (REBException e) {
            LOG.log(Level.SEVERE, "Error reading {0} {1}: {2}", new Object[]{name, item, e.getMessage()});
            checkErrorCount();
        }
        return value;
    }


    /**
     *  Reads a channel immediately.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    public double readChannelNow(int hwChan, int type)
    {
        double value = Double.NaN;
        String item = "";
        try {
            switch (type) {
            case TYPE_BD_TEMP:
                item = "board temperature";
                value = tmp.readAdc(hwChan);
                errorCount = 0;
                break;

            case TYPE_BD_POWER:
                item = "board power";
                value = pwr.readAdc(hwChan);
                errorCount = 0;
                break;

            case TYPE_ASP_TEMP:
                item = "aspic temperature";
                value = slow.readAspicTemp(hwChan / Asic.NUM_SIDES, hwChan % Asic.NUM_SIDES);
                errorCount = 0;
                break;

            case TYPE_CR_VOLT:
                item = "CCD voltage";
                value = slow.readVoltageNow(hwChan);
                errorCount = 0;
                break;

            default:
                value = super.readChannelNow(hwChan, type);
            }
        }
        catch (REBException e) {
            LOG.log(Level.SEVERE, "Error reading {0} {1}: {2}", new Object[]{name, item, e.getMessage()});
            checkErrorCount();
        }

        return value;
    }


    /**
     *  Reads the board power ADCs.
     *  This takes ~33 ms reading the 5 (V,I) pairs
     *
     *  @return  The array of board voltages and currents
     *  @throws  RaftException
     */
    public double[] readPowerAdcs() throws RaftException
    {
        try {
            return pwr.readAdcs();
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Reads the board power ADCs and adds them to data list.
     *
     *  @param  dataList  List to receive the read values, etc
     *  @return  The read values
     *  @throws  RaftException 
     */
    public double[] readPowerAdcs(List<AdcData> dataList) throws RaftException
    {
        CCSTimeStamp tstamp = CCSTimeStamp.currentTime();
        double[] values = readPowerAdcs();
        if (dataList != null) {
            for (int chan = 0; chan < values.length; chan++) {
                dataList.add(new AdcData(AdcData.TYPE_POWER, chan, values[chan], tstamp));
            }
        }
        return values;
    }


    /**
     *  Reads the clock and bias voltage slow ADCs and adds them to data list.
     *
     *  @param  dataList  List to receive the read values, etc
     *  @return  Array of read values
     *  @throws  RaftException
     */
    public double[] readSlowAdcs(List<AdcData> dataList) throws RaftException
    {
        int nAdc = rebType == BaseSet.REB_TYPE_SCIENCE ? NUM_SREB_ADCS :
                   rebType == BaseSet.REB_TYPE_GUIDER ? NUM_GREB_ADCS :
                   rebType == BaseSet.REB_TYPE_WAVEFRONT ? NUM_WREB_ADCS : 0;
        double[] data = new double[nAdc];
        if (nAdc == 0) return data;
        int adcType = rebType == BaseSet.REB_TYPE_SCIENCE ? TYPE_BIAS_VOLT : TYPE_CR_VOLT;
        if (rebType != BaseSet.REB_TYPE_SCIENCE) {
            try {
                slow.fetchVoltages();
            }
            catch (REBException e) {
                throw new RaftException(e.getMessage());
            }
        }
        for (int adc = 0; adc < nAdc; adc++) {
            data[adc] = readChannel(adcChans[adc], adcType);
            if (dataList != null) {
                dataList.add(new AdcData(AdcData.TYPE_SLOW, adcChans[adc], data[adc], null));
            }
        }
        return data;
    }


    /**
     *  Reads all the ADCs and adds them to data list.
     *
     *  @param  dataList  List to receive the read values, etc
     *  @throws  RaftException
     */
    public void readAllAdcs(List<AdcData> dataList) throws RaftException
    {
        readSlowAdcs(dataList);
        readPowerAdcs(dataList);
    }


    /**
     *  Gets the name of of a slow ADC.
     *
     *  @param  adc  The ADC ID
     *  @return  The ADC name
     */
    public static String getAdcName(int adc)
    {
        return adcNames[adc];
    }


    /**
     *  Reads the sequencer idle state lines.
     *
     *  @return  The lines value
     *  @throws  RaftException
     */
    public int readSeqIdleState() throws RaftException
    {
        try {
            return seq.readIdleState();
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Writes the sequencer idle state lines.
     *
     *  @param  value  The lines value
     *  @throws  RaftException
     */
    public void writeSeqIdleState(int value) throws RaftException
    {
        try {
            seq.writeIdleState(value);
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Sets the REB configuration data.
     *
     *  @param  reb  The REB configuration (object)
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Sets the REB configuration data")
    public void setREBConfig(@Argument(name="reb", description="REB configuration data")
                             REB reb) throws RaftException
    {
        dacCtrl.setConfig(reb.getDacs());
        BiasDACS[] biasConfig = reb.getBiases();
        for (int j = 0; j < biasConfig.length; j++) {
            BiasControl bias = biases[j];
            if (bias != null) {
                bias.setConfig(biasConfig[j]);
            }
        }
        ASPIC[] aspicConfig = reb.getAspics();
        for (int j = 0; j < aspicConfig.length; j++) {
            AspicControl aspic = aspics[j];
            if (aspic != null) {
                aspic.setConfig(aspicConfig[j]);
            }
        }
    }


    /**
     *  Gets the REB configuration data.
     *
     *  @return  The REB configuration (object)
     */
    @Command(type=CommandType.QUERY, description="Gets the REB configuration data")
    public REB getREBConfig() 
    {
        REB reb = new REB();
        reb.setId(id);
        reb.setIfcName(ifcName);
        reb.setCcdMask(realCcdMask);
        reb.setMaxCcdMask(realCcdMask);

        reb.setDacVersion(DacControl.getHwVersion(dac));
        reb.setDacRaw(dacRaw);
        reb.setBiasVersion(BiasControl.getHwVersion(dac));
        reb.setBiasRaw(biasRaw);
        reb.setAspicVersion(AspicControl.getHwVersion(asp));

        dacCtrl.getConfig(reb.getDacs());
        BiasDACS[] biasConfig = reb.getBiases();
        for (int j = 0; j < biases.length; j++) {
            BiasControl bias = biases[j];
            if (bias != null) {
                biasConfig[j] = bias.getConfig();
            }
        }
        ASPIC[] aspicConfig = reb.getAspics();
        for (int j = 0; j < aspics.length; j++) {
            AspicControl aspic = aspics[j];
            if (aspic != null) {
                aspicConfig[j] = aspic.getConfig();
            }
        }

        return reb;
    }


    /**
     * Transversal setting of all TM for the Aspics of this reb.
     * It internally uses the configuration service.
     * 
     * @param tm true for TM, false for normal
     */
    @Command(type=CommandType.CONFIGURATION, description="Set TM for all ASPICs on this REB")
    public void setAllAspicTM(boolean tm) {
        setAllAspic(AspicControl.TM, tm ? 1 : 0);
    }


    /**
     * Transversal setting of all Gains for the Aspics of this reb.
     * It internally uses the configuration service.
     * 
     * @param gain the new gain value
     */
    @Command(type=CommandType.CONFIGURATION, description="Set gain for all ASPICs on this REB")
    public void setAllAspicGain(int gain) {
        setAllAspic(AspicControl.GAIN, gain);
    }


    /**
     * Transversal setting of all Rcs for the Aspics of this reb.
     * It internally uses the configuration service.
     * 
     * @param rc the new rc value
     */
    @Command(type=CommandType.CONFIGURATION, description="Set RC for all ASPICs on this REB")
    public void setAllAspicRc(int rc) {
        setAllAspic(AspicControl.RC, rc);
    }


    /**
     * Transversal setting of a parameter for all the Aspics on the REB.
     * 
     * @param parmName
     * @param val 
     */
    private void setAllAspic(String parmName, int val) {
        for (AspicControl aspic : aspics) {
            if (aspic == null) continue;
            subsys.getComponentConfigurationEnvironment(aspic).submitChange(parmName, val);
        }
        sce.commitBulkChange();
    }


    /**
     *  Performs CCD power on/off sequences.
     *
     *  @param  on  Whether to power on (true) or off (false)
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Perform CCD power on/off sequence")
    public void powerCCDs(@Argument(description="Whether to power on") boolean on) throws RaftException
    {
        if (on) {
            powerCCDsOn();
        }
        else {
            powerCCDsOff();
        }
    }


    /**
     *  Performs CCD power on sequence.
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Perform CCD power on sequence")
    public void powerCCDsOn() throws RaftException
    {
        RaftException re = null;
        boolean all = true;
        boolean check = true;
        boolean odStep = true;
        boolean powerState = true;
        int underMask = 0;

        List<AdcData> adcDataList = new ArrayList<>();
        readAllAdcs(adcDataList);

        try {
            if (sqp.isRunning()) { // Sequencer is running, not appropriate to turn on CCDs
                publishAllAdcData(adcDataList);
                throw new RaftException("Sequencer is running, it must be off to powerCCDsOn()");
            }
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
        try {
            powerState = getCCDsPowerState(adcDataList);
        }
        catch (RaftException e) { // force off since unknown state
            try {
                powerCCDsOff(adcDataList);
            }
            catch (RaftException et) {}
            publishAllAdcData(adcDataList);
            LOG.severe(String.format("powerCCDsOn() FAILED for %s", name));
            throw new RaftException(e.getMessage());
        }
        if (powerState) { // CCDs are on, abort this command
            publishAllAdcData(adcDataList);
            throw new RaftException("CCDs are already on, command aborted");
        }
        try {
            if (ccdType.getName().equals("itl")) {
                loadDacs(all, check, adcDataList);
                loadBiasDacs(all, check, odStep, adcDataList);
            }
            else {
                loadBiasDacs(all, check, odStep, adcDataList);
                loadDacs(all, check, adcDataList);
            }
        }
        catch (RaftException e) {
            re = e;
            try {
                powerCCDsOff(adcDataList);
            }
            catch (RaftException et) {}
        }
        publishAllAdcData(adcDataList);  // pub leftovers in case of exception
        if (re != null) {
            throw re;
        }
        LOG.info(String.format("powerCCDsOn() success for %s", name));
    }


    /**
     *  Performs CCD power off sequence.
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Perform CCD power off sequence")
    public void powerCCDsOff() throws RaftException
    {
        List<AdcData> adcDataList = new ArrayList<>();
        powerCCDsOff(adcDataList);
    }


    /**
     *  Performs CCD power off sequence.
     *
     *  @throws  RaftException
     */
    private void powerCCDsOff(List<AdcData> adcDataList) throws RaftException
    {
        boolean backBiasState = true;
        try {
            backBiasState = bss.isBackBiasOn();
        }
        catch (REBException e) {
            LOG.log(Level.SEVERE, "Error checking back bias for "+name,e);
        }
        setBackBias(false);
        if (backBiasState) { // give voltage decay a head start
            try {
                Thread.sleep(100);
            }
            catch (InterruptedException e) {
                LOG.log(Level.SEVERE, "Thread.sleep(100) interrupted after setBackBias(false) for "+name,e);
            }
        }
        if (ccdType.getName().equals("itl")) {
            clearBiasDacs(adcDataList);
            clearDacs(adcDataList);
        }
        else {
            clearDacs(adcDataList);
            clearBiasDacs(adcDataList);
        }
        publishAllAdcData(adcDataList);
        LOG.info(String.format("powerCCDsOff() finished for %s", name));
    }


    /**
     *  Sets Sequencer Clock lines to low state
     *  Corrupts the sequencer file -- needs strategy revision
     *  Needed in [WG]REB power management but incomplete
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Set CCD Clock lines to low state")
    public void setCCDClocksLow() throws RaftException
    {
        int seqState = readSeqIdleState();
        writeSeqIdleState(seqState & ~seqClockLines);
        LOG.info(String.format("setCCDClocksLow() finished for %s", name));
    }


   /**
     * check power state of ccds
     * return exception to caller if not safe to power on
     *
     * @return  Whether power state is on
     * @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Perform Test of CCD power state")
    public boolean getCCDsPowerState() throws RaftException
    {
        List<AdcData> adcDataList = new ArrayList<>();
        boolean result = getCCDsPowerState(adcDataList);
        publishAllAdcData(adcDataList);
        return result;
    }
        
        
    /**
     * check power state of ccds
     * return exception to caller if not safe to power on
     *
     * @return  Whether power is on
     * @throws  RaftException
     */
    private boolean getCCDsPowerState(List<AdcData> adcDataList) throws RaftException
    { 
        int CCDsPowerState = -1;
        int onCnt = 0;
        int offCnt = 0;
        int testCntTot = 0;  // counts the number of tests
        StringBuilder fcause = new StringBuilder();  // used to build exception message
        RaftException re = null;
        boolean seqLoaded = true;
        double uClockDacOffset;
        boolean failed = false;
        boolean testFailed = false;

        // find out state of sequencer
        try {
            if (sqp.isRunning()) { // Sequencer is running, it must be loaded
                seqLoaded = true; 
            } else {
                seqLoaded = readSeqIdleState() != 0 ? true : false;
            }
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
        // measure
        double[] adcValues = readSlowAdcs(adcDataList);
        double[] curPower = readPowerAdcs(adcDataList);

        // set number of CCDs to: 3 (REB), 2 (GREB), 1 (WREB) stripes
        int nccds = rebType == BaseSet.REB_TYPE_SCIENCE ? 3 :
                    rebType == BaseSet.REB_TYPE_GUIDER ? 2 :
                    rebType == BaseSet.REB_TYPE_WAVEFRONT ? 1 : 0;

        // set bits in underSet for 3 (REB), 2 (GREB), 1 (WREB) stripes
        int underSet = rebType == BaseSet.REB_TYPE_SCIENCE ? 0x1ff :
                       rebType == BaseSet.REB_TYPE_GUIDER ? 0x3f :
                       rebType == BaseSet.REB_TYPE_WAVEFRONT ? 0x7: 0xffff;

        // evaluate the protection register stuff (bias bits)
        try {
            int underMask = dac.getUnders();
            testCntTot += 1;
            if (underMask == underSet) { // All tested biases are below theshold and hence "off"
                LOG.info(String.format("%s: firmware protected biases test as OFF: 0x%x => %s",
                                       name, underMask, dac.getUndersString()));
                offCnt += 1;
            } else if (underMask == 0) {  // All tested biases are above theshold and hence "on"
                LOG.info(String.format("%s: firmware protected biases test as ON: 0x%x => %s",
                                       name, underMask, dac.getUndersString()));
                onCnt += 1;
            } else {
                LOG.warning(String.format("%s: firmware protection biases in mixed state: 0x%x => %s",
                                          name, underMask, dac.getUndersString()));
                LOG.severe(String.format("%s: CCDs in mixed or UNDEFINED power state", name));
                fcause.append(String.format("\n  %s: firmware protection biases in UNDEFINED state: 0x%x => %s",
                                            name, underMask, dac.getUndersString()));
                testFailed = failed = true;
            }
        }
        catch (REBException e) {
            fcause.append(String.format("\n  Could not get state of %s protection register: getUnders() FAILED\n", name));
            fcause.append(String.format("\n  %s\n", e.getMessage()));
            testFailed = failed = true;
        }
        LOG.info(String.format("Test %d: firmware protection %s", testCntTot, testFailed ? "FAILED": "PASSED"));

        // Check board level currents
        testFailed = false;
        testCntTot += 1;
        if (curPower[PowerAdcs.ADC_OD_CURRENT] < nccds * odiQmax) {  // Maximum quiescent value (15mA)
            LOG.info(String.format("ODI current is nominal for CCDs OFF State: %.2f mA",
                                   curPower[PowerAdcs.ADC_OD_CURRENT]*1000.0));
            offCnt += 1;
        } else if (curPower[PowerAdcs.ADC_OD_CURRENT] > nccds * odiAmin) {  // Minimum active value
            LOG.info(String.format("ODI current is nominal for CCDs ON State: %.2f mA",
                                   curPower[PowerAdcs.ADC_OD_CURRENT]*1000.0));
            onCnt += 1;
        } else { // in between, can't determine off or on
            LOG.severe(String.format("%s: CCDs in mixed or undefined power state: %.2f < (ODI = %.2f) < %.2f",
                                     name, nccds*odiQmax*1000.0, curPower[PowerAdcs.ADC_OD_CURRENT]*1000.0,
                                     nccds * odiAmin*1000.0) );
            fcause.append(String.format("\n  %s: ODI current NOT in ON or OFF range", name));
            testFailed = failed = true;
        }
        LOG.info(String.format("Test %d: REB ODI current test %s", testCntTot, testFailed ? "FAILED": "PASSED"));

        if (rebType == BaseSet.REB_TYPE_SCIENCE) { // skip clock currents on corner W/G REBs

            // Clock Low Board Current
            testFailed = false;
            testCntTot += 1;
            if (curPower[PowerAdcs.ADC_CLKL_CURRENT] < clkliQmax) {  // Maximum quiescent value (45mA)
                LOG.info(String.format("CLKLI current is nominal for CCDs OFF State %.2f mA",
                                       curPower[PowerAdcs.ADC_CLKL_CURRENT]*1000.0));
                offCnt += 1;
            } else if (curPower[PowerAdcs.ADC_CLKL_CURRENT] > clkliAmin) {  // Minimum active value (50mA)
                LOG.info(String.format("CLKLI current is nominal for CCDs ON State %.2f mA",
                                       curPower[PowerAdcs.ADC_CLKL_CURRENT]*1000.0));
                onCnt += 1;
            } else { // can't determine
                LOG.severe(String.format("%s: CCDs in mixed or undefined power state: %.2f < (CLKLI = %.2f) < %.2f",
                                         name, clkliQmax*1000.0, curPower[PowerAdcs.ADC_CLKL_CURRENT]*1000.0,
                                         clkliAmin*1000.0) );
                fcause.append(String.format("\n  %s: CLKLI current NOT in ON or OFF range", name));
                testFailed = failed = true;
            }
            LOG.info(String.format("Test %d: REB CLKLI current test %s", testCntTot, testFailed ? "FAILED": "PASSED"));

            // Clock High Board Current
            testFailed = false;
            testCntTot += 1;
            if (curPower[PowerAdcs.ADC_CLKH_CURRENT] < clkhiQmax) {  // Maximum quiescent value (45mA)
                LOG.info(String.format("CLKHI current is nominal for CCDs OFF State %.2f mA",
                                       curPower[PowerAdcs.ADC_CLKH_CURRENT]*1000.0));
                offCnt += 1;
            } else if (curPower[PowerAdcs.ADC_CLKH_CURRENT] > clkhiAmin) {  // Minimum active value (50mA)
                LOG.info(String.format("CLKHI current is nominal for CCDs ON State %.2f mA",
                                       curPower[PowerAdcs.ADC_CLKH_CURRENT]*1000.0));
                onCnt += 1;
            } else { // can't determine
                LOG.severe(String.format("%s: CCDs in mixed or undefined power state: %.2f < (CLKHI = %.2f) < %.2f",
                                         name, clkhiQmax*1000.0, curPower[PowerAdcs.ADC_CLKH_CURRENT]*1000.0,
                                         clkhiAmin*1000.0) );
                fcause.append(String.format("\n  %s: CLKHI current NOT in ON or OFF range", name));
                testFailed = failed = true;
            }
            LOG.info(String.format("Test %d: REB CLKHI current test %s", testCntTot, testFailed ? "FAILED": "PASSED"));
        }

        // Check Bias levels for each CCD
        testFailed = false;
        for (BiasControl bias : biases) {
            if (bias == null) continue;
            testCntTot += 1;
            int biasDacState = bias.getBiasDacsPowerState(adcDataList, fcause);
            if (biasDacState == 0) {
                offCnt += 1;
            } else if (biasDacState == 1) {
                onCnt += 1;
            } else {
                LOG.severe(String.format("%s: CCDs Bias voltage test FAILED", name));
                fcause.append(String.format("\n  %s: CCDs Bias voltage test FAILED", name));
                testFailed = failed = true;
            }
            LOG.info(String.format("Test %d: Bias level test %s", testCntTot, testFailed ? "FAILED": "PASSED"));
        }

        // Check Clock rails on the REB
        testFailed = false;
        testCntTot += 1;
        if (seqLoaded) { // if sequencer is loaded, high rail has offset when CCDs are off
            uClockDacOffset = 0.9;
        } else {
            uClockDacOffset = 0.0;
        }
        int clockDacState = dacCtrl.getClockDacsPowerState(adcDataList, fcause, uClockDacOffset);
        if (clockDacState == 0) {
            offCnt += 1;
        } else if (clockDacState == 1) {
            onCnt += 1;
        } else {
            LOG.severe(String.format("%s: CCDs clock voltage test FAILED", name));
            fcause.append(String.format("\n  %s: CCDs clock voltage test FAILED", name));
            testFailed = failed = true;
        }
        LOG.info(String.format("Test %d: Clock rails test %s", testCntTot, testFailed ? "FAILED": "PASSED"));

        // Done with tests, determine outcome
        if (onCnt == testCntTot) {
            CCDsPowerState = 1;
        } else if (offCnt == testCntTot) {
            CCDsPowerState = 0;
        } else {
            LOG.severe(String.format("%s: getCCDsPowerState() FAILED: %d(%d) ON(OFF) of %d tests passed", 
                                     name, onCnt, offCnt, testCntTot));
            fcause.append(String.format("\n  %s: getCCDsPowerState() FAILED: %d(%d) ON(OFF) of %d tests passed",
                                        name, onCnt, offCnt, testCntTot));
            failed = true;
        }
        if (failed) {
            LOG.severe(fcause.toString());
            throw new RaftException(fcause.toString());
        }
        return CCDsPowerState == 1;
    }

    /**
     *  Performs CCD shorts test.
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Perform CCD shorts test")
    public void testCCDShorts() throws RaftException
    {
        List<AdcData> adcDataList = new ArrayList<>();
        RaftException re = null;
        boolean failed = false;
        boolean powerState = true;
        int seqState = 0;

        // check prerequisites, abort if not met
        //
        try {
            if (sqp.isRunning()) { // Sequencer is running, not appropriate to test shorts
                publishAllAdcData(adcDataList);
                throw new RaftException("Sequencer is running, it must be off to testCCDShorts()");
            } else {
                seqState = readSeqIdleState();
            }
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
        try {
            powerState = getCCDsPowerState(adcDataList);
        }
        catch (RaftException e) {  // exception thrown getting PowerState: turn off
            try {
                powerCCDsOff(adcDataList);
            }
            catch (RaftException et) {}
            publishAllAdcData(adcDataList);
            LOG.severe(String.format("testCCDShorts() FAILED for %s", name));
            throw new RaftException(e.getMessage());
        }
        if (powerState) {
            LOG.severe("testCCDShorts() called with CCDs on, aborting...");
            throw new RaftException("CCDs are on, testCCDShorts() aborted");
        }
        // force off to be sure, critical for CCDs to be off here
        powerCCDsOff(adcDataList);
        // lower sequencer clock lines before test and reset after
        writeSeqIdleState(seqState & ~seqClockLines);
        LOG.info(String.format("%s lowering sequencer clocks for shorts test: 0x%x --> 0x%x", 
                               name, seqState, readSeqIdleState()));
        // allow the clocks to settle for 200ms
        try {
            Thread.sleep(200);
        }
        catch (InterruptedException e) {
            LOG.log(Level.SEVERE, "Thread.sleep(200) interrupted after lowering sequencer clocks for "+name,e);
        }
        // now safe to run this test
        StringBuilder fcause = new StringBuilder();
        for (BiasControl bias : biases) {
            if (bias == null) continue;
            failed |= bias.testShorts(adcDataList, fcause);
            publishAllAdcData(adcDataList);
        }
        failed |= dacCtrl.testShorts(adcDataList, fcause);
        // reset sequencer clock lines after test
        LOG.info(String.format("%s restoring sequencer clocks after shorts test: 0x%x --> 0x%x", 
                               name, readSeqIdleState(), seqState));
        writeSeqIdleState(seqState);
        // allow the clocks to settle for 100ms
        try {
            Thread.sleep(100);
        }
        catch (InterruptedException e) {
            LOG.log(Level.SEVERE, "Thread.sleep(100) interrupted after restoring sequencer clocks for "+name,e);
        }
        // close out
        publishAllAdcData(adcDataList);
        if (failed) {
            throw new RaftException(name + " shorts test FAILED" + fcause.toString());
        }
        LOG.info("testCCDShorts() finished successfully");
    }


    /**
     *  Publishes all ADC data.
     * 
     *  @param  adcDataList  The list of ADC data
     *
     *  The adcDataList is consumed and cleared upon return
     */
    private void publishAllAdcData(List<AdcData> adcDataList)
    {
        
        KeyValueDataList dataList = new KeyValueDataList();
        for (AdcData adcData : adcDataList) {
            String chanName = (adcData.type == AdcData.TYPE_POWER ? pwrChannelMap : sadcChannelMap).get(adcData.hwChan);
            if (chanName != null) {
                double conv = adcData.type == AdcData.TYPE_POWER && (adcData.hwChan & 1) != 0 ? 1000.0 : 1.0;  // mA: Would be nice to get this from the Channel
                dataList.addData(chanName, conv * adcData.value, adcData.timeStamp);
            }
        }
        if (!dataList.getListOfKeyValueData().isEmpty()) {
            subsys.publishSubsystemDataOnStatusBus(dataList);
        }
        adcDataList.clear();  // flush out the stuff that was published
    }


    /**
     *  Loads configured DAC values.
     *
     *  @param  all       If true, all data are loaded; otherwise only changes
     *  @return  The number of DACs loaded
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Loads configured DAC values")
    public int loadDacs(@Argument(description="Whether all DACs are loaded, or only changed ones")
                        boolean all) throws RaftException
    {
        boolean powerState = true;
        List<AdcData> adcDataList = new ArrayList<>();
        readAllAdcs(adcDataList);
        try {
            powerState = getCCDsPowerState(adcDataList);
        }
        catch (RaftException e) {  // exception thrown getting State
            try {  // things are sufficient borked that we should turn CCDs off
                powerCCDsOff(adcDataList);
            }
            catch (RaftException et) {}
            publishAllAdcData(adcDataList);
            LOG.severe(String.format("loadDacs() FAILED for %s", name));
            throw new RaftException(e.getMessage());
        }
        publishAllAdcData(adcDataList);
        if (!powerState) { // CCDs are off, abort this command
            throw new RaftException("CCDs must be ON to use loadDacs() command, command aborted");
        }
        return loadDacs(all, false);
    }


    /**
     *  Loads configured DAC values.
     *
     *  @param  all       If true, all data are loaded; otherwise only changes
     *  @param  check     Whether to check each loaded value
     *  @return  The number of DACs loaded
     *  @throws  RaftException
     */
    private int loadDacs(boolean all, boolean check) throws RaftException
    {
        List<AdcData> adcDataList = new ArrayList<>();
        int count = all ? dacCtrl.loadConfig(dacLoadDelay, check, adcDataList)
                        : dacCtrl.loadChanged(dacLoadDelay, check, adcDataList);
        LOG.log(Level.INFO, "Loaded {0} {1} DACs", new Object[]{count, name});
        publishAllAdcData(adcDataList);
        return count;
    }


    /**
     *  Loads configured DAC values.
     *
     *  @param  all       If true, all data are loaded; otherwise only changes
     *  @param  check     Whether to check each loaded value
     *  @param  adcDataList  List to receive values from slowAdcs
     *  @return  The number of DACs loaded
     *  @throws  RaftException
     */
    private int loadDacs(boolean all, boolean check, List<AdcData> adcDataList) throws RaftException
    {
        int count = all ? dacCtrl.loadConfig(dacLoadDelay, check, adcDataList)
                        : dacCtrl.loadChanged(dacLoadDelay, check, adcDataList);
        LOG.log(Level.INFO, "Loaded {0} {1} DACs", new Object[]{count, name});
        publishAllAdcData(adcDataList);
        return count;
    }


    /**
     *  Clears DAC values.
     *
     *  Should be removed -- only use powerCCDsOff
     *
     *  @throws  RaftException
     */
    /*
     *  @Command(type=CommandType.ACTION, description="Clear DAC values")
     *  public void clearDacs() throws RaftException
     *  {
     *      List<AdcData> adcDataList = new ArrayList<>();
     *      dacCtrl.clear(dacClearDelay, adcDataList);
     *      publishAllAdcData(adcDataList);
     *  }
     */


    /**
     *  Clears DAC values.
     *
     *  @throws  RaftException
     */
    private void clearDacs(List<AdcData> adcDataList) throws RaftException
    {
        dacCtrl.clear(dacClearDelay, adcDataList);
    }


    /**
     *  Loads configured bias DAC values.
     *
     *  @param  all  If true, all data are loaded; otherwise only changes
     *  @return  The number of bias DACs loaded
     *  @throws  RaftException
     *           This is the user command, checks to verify CCDs are on and
     *           only changes allowed by configuration system can be used
     */
    @Command(type=CommandType.ACTION, description="Loads configured bias DAC values")
    public int loadBiasDacs(@Argument(description="Whether all DACs are loaded, or only changed ones")
                            boolean all) throws RaftException
    {
        boolean powerState = true;
        List<AdcData> adcDataList = new ArrayList<>();
        readAllAdcs(adcDataList);
        try {
            powerState = getCCDsPowerState(adcDataList);
        }
        catch (RaftException e) {  // exception thrown getting State
            try {  // things are sufficient borked that we should turn CCDs off
                powerCCDsOff(adcDataList);
            }
            catch (RaftException et) {}
            publishAllAdcData(adcDataList);
            LOG.severe(String.format("loadBiasDacs() FAILED for %s", name));
            throw new RaftException(e.getMessage());
        }
        publishAllAdcData(adcDataList);
        if (!powerState) { // CCDs are off, abort this command
            throw new RaftException("CCDs must be ON to use loadBiasDacs command, command aborted");
        }
        return loadBiasDacs(all, false);
    }


    /**
     *  Loads configured bias DAC values.
     *
     *  @param  all       If true, all data are loaded; otherwise only changes
     *  @param  check     Whether to check each loaded value
     *  @return  The number of bias DACs loaded
     *  @throws  RaftException
     */
    private int loadBiasDacs(boolean all, boolean check) throws RaftException
    {
        return loadBiasDacs(all, check, false);
    }


    /**
     *  Loads configured bias DAC values.
     *
     *  @param  all       If true, all data are loaded; otherwise only changes
     *  @param  check     Whether to check each loaded value
     *  @return  The number of bias DACs loaded
     *  @throws  RaftException
     *
     *  This is the "standalone" version not called during powerCCDsOn()
     */
    private int loadBiasDacs(boolean all, boolean check, boolean odStep) throws RaftException
    {
        int count = 0;
        List<AdcData> adcDataList = new ArrayList<>();
        return loadBiasDacs(all, check, false, adcDataList);
    }


    /**
     *  Loads configured bias DAC values.
     *
     *  @param  all       If true, all data are loaded; otherwise only changes
     *  @param  check     Whether to check each loaded value
     *  @return  The number of bias DACs loaded
     *  @throws  RaftException
     *
     *  This is the version called during powerCCDsOn()
     */
    private int loadBiasDacs(boolean all, boolean check,
                             boolean odStep, List<AdcData> adcDataList) throws RaftException
    {
        int count = 0;
        for (BiasControl bias : biases) {
            if (bias == null) continue;
                count += all ? bias.load(dacLoadDelay, check, odStep, adcDataList)
                             : bias.loadChanged(dacLoadDelay, check, odStep, adcDataList);
            publishAllAdcData(adcDataList);
        }
        LOG.log(Level.INFO, "Loaded {0} {1} bias DACs", new Object[]{count, name});
        return count;
    }


    /**
     *  Clears bias DAC values.
     *
     *  Seems like this should be removed
     *
     *  @throws  RaftException
     */
    /*
     *  @Command(type=CommandType.ACTION, description="Clear bias DAC values")
     *  public void clearBiasDacs() throws RaftException
     *  {
     *      List<AdcData> adcDataList = new ArrayList<>();
     *      for (BiasControl bias : biases) {
     *          if (bias == null) continue;
     *          bias.clear(dacClearDelay, adcDataList);
     *          publishAllAdcData(adcDataList);
     *      }
     *  }
     */


    /**
     *  Clears bias DAC values.
     *
     *  @throws  RaftException
     *  records values for powerCCDsOff usage
     */
    private void clearBiasDacs(List<AdcData> adcDataList) throws RaftException
    {
        for (BiasControl bias : biases) {
            if (bias == null) continue;
            bias.clear(dacClearDelay, adcDataList);
        }
    }


    /**
     *  Loads configuration data to the ASPICs.
     *
     *  @param  all  If true, all data are loaded; otherwise only changes
     *  @return  The number of ASPICs loaded
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Load configuration data to the ASPICs")
    public int loadAspics(boolean all) throws Exception
    {
        int count = 0;
        for (AspicControl aspic : aspics) {
            if (aspic != null) {
                count += all ? aspic.load() : aspic.loadChanged();
            }
        }
        LOG.log(Level.INFO, "Loaded {0} {1} ASPICs", new Object[]{count, name});
        return count;
    }


    /**
     *  Resets the ASPICs.
     *
     *  @return  The number of ASPICs reset
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Reset the ASPICs")
    public int resetAspics() throws Exception
    {
        int count = 0, mask = 0;
        for (AspicControl aspic : aspics) {
            if (aspic != null) {
                int strip = aspic.getHwChan() / 2;
                if ((mask & (1 << strip)) == 0) {
                    count += aspic.reset();
                    mask |= (1 << strip);
                }
            }
        }
        LOG.log(Level.INFO, "Reset {0} {1} ASPICs", new Object[]{count, name});
        return count;
    }


    /**
     *  Checks loaded ASPIC configuration data.
     *
     *  @return  The mask of mismatched fields
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Check loaded ASPIC configuration")
    public int checkAspics() throws Exception
    {
        int mask = 0, index = 0;
        for (AspicControl aspic : aspics) {
            if (aspic != null) {
                mask |= (aspic.check() << index);
                index += 4;
            }
        }
        return mask;
    }


    /**
     *  Gets the back bias on state.
     *
     *  @return  Whether back bias is turned on
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Get the back bias on state")
    public boolean isBackBiasOn() throws RaftException
    {
        try {
            return bss.isBackBiasOn();
        }
        catch (REBException e) {
            LOG.log(Level.SEVERE, "Error getting back bias for"+name,e);
            return false;
        }
    }


    /**
     *  Turns back bias on or off.
     *
     *  @param  value  The back bias on state to set (true or false)
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Turn back bias on or off")
    public void setBackBias(@Argument(description="The on state")
                            boolean value) throws RaftException
    {
        try {
            bss.setBackBias(value);
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Sets a heater voltage.
     *
     *  @param  heater  The heater number: 0 or 1
     *  @param  volts   The voltage to set
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Set heater voltage")
    public void setHeater(@Argument(description="Heater number")
                          int heater,
                          @Argument(description="The voltage")
                          double volts) throws RaftException
    {
        double amps = volts / (HEATER_OHMS + HTR_LEAD_OHMS);
        heaterValues[CHAN_HTR_VOLTS] = volts;
        heaterValues[CHAN_HTR_POWER] = amps * amps * HEATER_OHMS;
        int value = (int)((dac.getVersion() == BoardDacs.VERSION_REB5 ? HTR_CONV_R5 : HTR_CONV) * volts);
        try {
            dac.set(heater == 0 ? BoardDacs.CHAN_HEATER_1 : BoardDacs.CHAN_HEATER_2,
                    value < 0 ? 0 : value > DAC_LIMIT ? DAC_LIMIT : value, true);
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Sets heater power.
     *
     *  @param  heater  The heater number: 0 or 1
     *  @param  power   The power value to set
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Set heater power")
    public void setHeaterPower(@Argument(description="Heater number")
                               int heater,
                               @Argument(description="The power value")
                               double power) throws RaftException
    {
        setHeater(heater, (HEATER_OHMS + HTR_LEAD_OHMS) * Math.sqrt(power / HEATER_OHMS));
    }


    /**
     *  Sets the time base to the current system time.
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Sets the time base to the current system time")
    public void setTime() throws RaftException
    {
        try {
            bss.disable();
            bss.setTime();
            bss.enable();
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Gets the time base as Unix time.
     *
     *  @return  The time base converted to Unix time
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Gets the time base as Unix time")
    public long getTime() throws RaftException
    {
        try {
            return bss.getTime();
        }
        catch (REBException e) {
            //checkTimeout(e);
            return 0;
        }
    }


    /**
     *  Gets a trigger time as Unix time.
     *
     *  @param  rSet  The name of the trigger time's register set: "stat",
     *                "time" or "seq:.
     *  @return  The trigger time converted to Unix time
     *  @throws  RaftException
     */
    private static final Map<String,Integer> rsetNames = new HashMap();
    static {
        rsetNames.put("stat", BaseSet.RSET_STATUS);
        rsetNames.put("time", BaseSet.RSET_TIME_BASE);
        rsetNames.put("seq",  BaseSet.RSET_SEQUENCER);
        rsetNames.put("tadc", BaseSet.RSET_TEMP_ADCS);
        rsetNames.put("padc", BaseSet.RSET_POWER_ADCS);
    }

    @Command(type=CommandType.QUERY, description="Gets a trigger time as Unix time")
    public long getTime(@Argument(name="regset", description="Register set name")
                        String rSet) throws RaftException
    {
        Integer regSet = rsetNames.get(rSet.toLowerCase());
        if (regSet == null) {
            throw new RaftException("Invalid register set name");
        }
        try {
            return bss.getTriggerTime(regSet);
        }
        catch (REBException e) {
            //checkTimeout(e);
            return 0;
        }
    }


    /**
     *  Sets the FITS file test conditions.
     *
     *  @param  filter       The filter name
     *  @param  temperature  The temperature set point
     *  @param  wavelength   The monochromator wavelength
     */
    @Command(type=CommandType.ACTION, description="Sets the FITS file test conditions")
    public void setFitsConditions(@Argument(description="Filter name")
                                  String filter, 
                                  @Argument(description="The temperature")
                                  double temperature,
                                  @Argument(description="The wavelength")
                                  double wavelength)
    {        
        FitsHeaderKeywordData data = new FitsHeaderKeywordData(rebGeometry.getUniqueId());
        data.addHeaderKeywordValue("primary", "FilterName", filter, true);
        data.addHeaderKeywordValue("primary", "TemperatureSetPoint", temperature, true);
        data.addHeaderKeywordValue("primary", "MonochromatorWavelength", wavelength, true);                        
        subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());
    }


    /**
     *  Gets the DAQ metadata for the current image.
     *
     *  @return  The image metadata
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Get the DAQ metadata for the current image")
    public ImageMetadata getImageMetadata() throws RaftException
    {
        if ( img == null ) {
            throw new RuntimeException("This REB Device is not configured to process images");
        }
        return img.getImageMetadata();
    }


    /**
     *  Gets a portion of the current image.
     *
     *  @param  offset  The offset (in pixels) to the first pixel data to obtain.
     *  @param  count   The number of data pixels to obtain.  If zero, all
     *                  the data, starting at offset, is obtained.
     *  @return  The returned pixel data
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Gets a portion of the current image")
    public ImageData getImage(@Argument(description="Offset to the pixel data")
                              int offset,
                              @Argument(description="Number of pixels")
                              int count) throws RaftException
    {
        if ( img == null ) {
            throw new RuntimeException("This REB Device is not configured to process images");
        }
        return img.getImage(0, offset, count);
    }


    /**
     *  Gets a portion of the current image.
     *
     *  @param  ccd     The CCD number
     *  @param  offset  The offset (in pixels) to the first pixel data to obtain.
     *  @param  count   The number of data pixels to obtain.  If zero, all
     *                  the data, starting at offset, is obtained.
     *  @return  The returned pixel data
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Gets a portion of the current image")
    public ImageData getImage(@Argument(description="CCD number")
                              int ccd,
                              @Argument(description="Offset to the pixel data")
                              int offset,
                              @Argument(description="Number of pixels")
                              int count) throws RaftException
    {
        if ( img == null ) {
            throw new RuntimeException("This REB Device is not configured to process images");
        }
        return img.getImage(ccd, offset, count);
    }


    /**
     *  Resets the front end.
     *
     *  This is the only way to stop a running sequencer program
     *
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Resets the front end")
    public void resetFrontEnd() throws RaftException
    {
        try {
            bss.reset(0);
            bss.enable();
            bss.setTime();
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Gets the contents of REB registers.
     *
     *  @param  address  The address of the first register to read
     *  @param  count    The number of registers to read
     *  @return  The block of register values
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Gets the contents of REB registers")
    public RegisterData getRegister(@Argument(description="Address of the first register")
                                    int address,
                                    @Argument(description="Number of registers to read")
                                    int count) throws RaftException
    {
        int[] values = new int[count];
        try {
            bss.read(address, values);
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }

        return new RegisterData(address, count, values);
    }


    /**
     *  Sets the contents of REB registers.
     *
     *  @param  address  The address of the first register to write
     *  @param  values   The array of values to write to consecutive registers
     *  @throws  RaftException
     */
    @Command(type=CommandType.ACTION, description="Sets the contents of REB registers")
    public void setRegister(@Argument(description="Address of the first register")
                            int address,
                            @Argument(description="Values to write")
                            int[] values) throws RaftException
    {
        try {
            bss.write(address, values);
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Gets the REB status block.
     *
     *  @return  The REB status block
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Gets the REB status block")
    public StatusData getRebStatus() throws RaftException
    {
        try {
            return new StatusData(sts.readStatus());
        }
        catch (REBException e) {
            //checkTimeout(e);
            return null;
        }
    }


    /**
     *  Gets the associated board DACs object.
     *
     *  @return  The DAC object
     */
    BoardDacs getBoardDacs()
    {
        return dac;
    }


    /**
     *  Gets the associated ASPIC object.
     *
     *  @return  The ASPIC object
     */
    Aspic getAspic()
    {
        return asp;
    }


    /**
     *  Gets the associated image processor.
     *
     *  @return  The image
     */
    public ImageProc getImageProc()
    {
        return img;
    }


    /**
     *  Gets the associated sequencer.
     *
     *  @return  The sequencer
     */
    public SequencerProc getSequencer()
    {
        return sqp;
    }


    /**
     *  Gets a temperature sensor value.
     *
     *  @return  The value
     */
    double getTempValue(int hwChan)
    {
        return online ? tempValues[hwChan] : 0;
    }


    /**
     *  Gets a power sensor value.
     *
     *  @return  The value
     */
    double getPowerValue(int hwChan)
    {
        return online ? powerValues[hwChan] : 0;
    }


    /**
     * Simulate a disconnect
     */
    @Command(simulation = true)
    public void simulateDisconnection() {
        setOnline(false);
    }


    /**
     * Checks error count.
     * 
     * Increments consecutive error count and sets offline if excessive
     */
    private void checkErrorCount() {
        if (++errorCount > 4) {
            setOnline(false);
        }
    }
    
    @Override
    protected void setOnline(boolean online) {
        super.setOnline(online);
        if (isOnline()) {
           stateService.updateAgentComponentState(this, RebDeviceState.ONLINE);
        } else {
           stateService.updateAgentComponentState(this, RebDeviceState.OFFLINE, RebValidationState.UNKNOWN, HVBiasState.UNKNOWN);
        }
    }
    
}
