package org.lsst.ccs.subsystem.rafts;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TimeZone;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import nom.tam.fits.FitsException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.utilities.FitsService;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.Image;
import org.lsst.ccs.drivers.reb.ImageClient;
import org.lsst.ccs.drivers.reb.ImageMetadata;

import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.rafts.data.ImageData;
import org.lsst.ccs.subsystem.rafts.data.ImageState;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.image.FitsFileWriter;
import org.lsst.ccs.utilities.image.FitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.image.MetaDataSet;
import org.lsst.ccs.utilities.ccd.Segment;
import org.lsst.ccs.utilities.image.ImageSet;
import org.lsst.ccs.utilities.pattern.FileNamePatternUtils;
import org.lsst.ccs.utilities.readout.DaqImageSet;
import org.lsst.ccs.utilities.readout.GeometryFitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.readout.ReadOutImageSet;
import org.lsst.ccs.utilities.readout.ReadOutParameters;
import org.lsst.ccs.utilities.readout.ReadOutParametersBuilder;
import org.lsst.ccs.utilities.readout.ReadOutParametersOld;

/**
 * Routines for handling image data
 *
 * @author Owen Saxton
 */
public class ImageProc {

    /**
     * The order of the Segments in the fits files is: S10->...
     * ->S17->S07->...->S00
     *
     * while the order of the segments coming from the DAQ is
     * S00->...->S07->S10->...->S17
     *
     * So we introduce the array below to describe the needed mapping. Note, the
     * ATS appears to have a different mapping, so this needs to be made
     * settable.
     */
    private int[] dataSegmentMap = {15, 14, 13, 12, 11, 10, 9, 8, 0, 1, 2, 3, 4, 5, 6, 7};
    private int[] invertedDataSegmentMap;

    /*
     *  Constants.
     */
    private static final int NUM_ADCS = 16,
            DATA_MASK_STRAIGHT = 0x20000,
            DATA_MASK_INVERTED = 0x1ffff;

    @LookupField(strategy = LookupField.Strategy.ANCESTORS)
    private REBDevice reb;
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy = LookupField.Strategy.SIBLINGS)
    private FitsService fitsService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private GlobalVisualizationClient globalVisualizationClient;

    private static final Logger LOG = Logger.getLogger(ImageProc.class.getName());
    private int ccdMask, numCcds, numRebCcds;
    private Image currImage = new Image();
    private ByteBuffer[][] currImageBuffer;
    private RaftException currImageExcp = new RaftException("No image data available");
    private int fitsSeqnum;
    private boolean externalSequenceNumber = false;
    private long imageTime;

    private String defaultDirectory = ".";
    private String fitsFileNamePattern = "Image_${rebName}_${sensorId}_${imageName}.fits";
    private String imageDataFileNamePattern = "Image_${rebName}_${imageName}.dat";
    private final Properties fileNamePatternProperties = new Properties();
    private Reb rebGeometry;
    private volatile ReadOutParameters readOutParameters;
    private int dataInversionMask = DATA_MASK_INVERTED;
    private final BlockingQueue<Integer> imageQueue = new ArrayBlockingQueue<>(1);

    private final List<ImageListener> imageListeners = new CopyOnWriteArrayList<>();

    /**
     * Inner class for receiving an image.
     */
    private class GetImage implements ImageClient.Listener {

        /**
         * Processes received images.
         *
         * Broadcasts a message on the status bus containing the image metadata.
         *
         */
        @Override
        public void processImage(Image image) {
            currImage = image;
            LOG.log(Level.INFO, "Received image: Reb = {0}, Name = {1}, Length = {2}, Status = {3}", new Object[]{reb.getName(), currImage.getName(), currImage.getLength(), currImage.getStatus()});
            try {
                currImageBuffer = splitImage();
                currImageExcp = null;
            } catch (RaftException e) {
                LOG.severe(e.getMessage());
                currImageBuffer = null;
                currImageExcp = e;
                return;
            }
            imageTime = currImage.getTimestamp() / 1_000_000;
            final String timestampString = timestampString(currImage.getTimestamp());
            fileNamePatternProperties.setProperty("timestamp", timestampString);
            fileNamePatternProperties.setProperty("imageName", currImage.getName() == null ? timestampString : currImage.getName());
            readOutParameters = createReadOutParametersFromRegistersMetadata(image.getRegisters());
            imageQueue.offer(0);
            
            for (ImageListener l : imageListeners) {
                l.imageReceived(ImageProc.this, image);
            }

            ImageState iState = new ImageState(currImage.getTimestamp(), currImage.getLength());
            KeyValueData kvd = new KeyValueData(ImageState.KEY, iState);
            subsys.publishSubsystemDataOnStatusBus(kvd);

            try {
                if (globalVisualizationClient != null && globalVisualizationClient.isConnected()) {
                    for (int ccd = 0; ccd < numCcds; ccd++) {
                        CCD ccdGeometry = rebGeometry.getCCDs().get(ccd);

                        //For now let's take all the data including pre/over scan regions
                        ByteBuffer ccdData = ByteBuffer.allocate(readOutParameters.getTotalParallelSize() * readOutParameters.getTotalSerialSize() * NUM_ADCS * 4);

                        for (int adc = 0; adc < NUM_ADCS; adc++) {
                            ByteBuffer buff = currImageBuffer[ccd][adc];
                            ccdData.put(buff);
                            buff.rewind();
                        }
                        ccdData.rewind();
                        LOG.log(Level.FINE, "Sending image data {0}", image.getName());
                        globalVisualizationClient.sendImageData(image.getMetadata(), ccdData, ccdGeometry, readOutParameters);
                    }
                }
            } catch (Exception e) {
                //Do nothing and continue
                e.printStackTrace();
            }

            /*
            double [][] stats;
            try {
                stats = getImageStats();
            }
            catch (RaftException e) {
                LOG.severe(e.getMessage());
                return;
            }
            for (int j = 0; j < stats.length / 3; j++) {
                for (int k = 0; k < 3; k++) {
                    int ix = k * stats.length / 3 + j;
                    System.out.format("    %2s-%s: %7.1f %5.1f", j, k,
                                      stats[ix][0], stats[ix][1]);
                }
                System.out.println();
            }
             */
        }
    }
    
    public static interface ImageListener {
        void imageReceived(ImageProc proc, Image image);
    }
    
    public void addImageListener(ImageListener l) {
        imageListeners.add(l);
    }
    
    public void removeImageListener(ImageListener l) {
        imageListeners.remove(l);
    }
    /**
     * Constructor.
     *
     * @param imc The image client object
     */
    public ImageProc(ImageClient imc) {
        imc.setListener(new GetImage(), currImage);
        this.invertedDataSegmentMap = invertDataSegmentMap(dataSegmentMap);
        fileNamePatternProperties.setProperty("raftLoc", "99");
    }

    private int[] invertDataSegmentMap(int[] dataSegmentMap) {
        int[] result = new int[dataSegmentMap.length];
        for (int j = 0; j < result.length; j++) {
            result[dataSegmentMap[j]] = j;
        }
        return result;
    }

    void setDataSegmentMap(int[] map) {
        if (map.length != dataSegmentMap.length) {
            throw new IllegalArgumentException("Invalid map length");
        }
        int[] inverted = invertDataSegmentMap(map);
        // Sanity check
        if (!Arrays.equals(invertDataSegmentMap(inverted), map)) {
            throw new IllegalArgumentException("Invalue dataSegmentMap " + Arrays.toString(dataSegmentMap));
        }
        this.dataSegmentMap = map;
        this.invertedDataSegmentMap = inverted;
    }

    /**
     * Performs configuration.
     *
     * @param log The associated logger (not used)
     */
    public void configure(Logger log) {
        fileNamePatternProperties.setProperty("rebName", reb.getName());
    }

    /**
     * Performs configuration.
     *
     * @param rebName The REB device name
     * @param rebId The REB ID (address)
     * @param subsys The associated subsystem
     * @param log The associated logger (not used)
     */
    @Deprecated
    public void configure(String rebName, int rebId, Subsystem subsys, Logger log) {
//        this.rebName = rebName;
//        this.rebId = rebId;
        this.subsys = subsys;
        fileNamePatternProperties.setProperty("rebName", rebName);
    }

    /**
     * Set a property for resolving file name and output directory
     * @param propertyName
     * @param propertyValue
     */
    public void setPatternProperty(String propertyName, String propertyValue) {
        fileNamePatternProperties.setProperty(propertyName, propertyValue);
    }

    /**
     * Sets the mask of CCDs being used.
     *
     * @param ccdMask The mask of active CCDs
     */
    public void setCcdMask(int ccdMask) {
        this.ccdMask = ccdMask;
        numCcds = Integer.bitCount(this.ccdMask);
    }

    /**
     * Sets the number of CCDs on the REB.
     *
     * @param numRebCcds The number of CCDs
     */
    public void setNumRebCcds(int numRebCcds) {
        this.numRebCcds = numRebCcds;
    }

    /**
     * Set the Reb Geometry corresponding to this ImageProc instance.
     *
     * @param reb The Reb Geometry.
     */
    public void setRebGeometry(Reb reb) {
        this.rebGeometry = reb;
        fitsService.setGeometry(rebGeometry);
    }

    /**
     * Sets whether image data values are inverted
     *
     * @param invert Whether data values are to be inverted
     */
    public void setDataInversion(boolean invert) {
        dataInversionMask = invert ? DATA_MASK_INVERTED : DATA_MASK_STRAIGHT;
    }

    /**
     * Initializes for a new set of images
     */
    public void initImageWait() {
        imageQueue.clear();
    }

    /**
     * Waits for an image to arrive
     *
     * @param timeout The timeout (msec)
     * @return The number of images received: 0 (if timed out) or 1
     */
    public int waitForImage(int timeout) {
        Integer value = null;
        try {
            value = imageQueue.poll(timeout, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
        }
        return value != null ? 1 : 0;
    }

    /**
     * Saves the current image as raw bytes.
     *
     * The default name of the file is "Image_[reb]_[imagename].dat", where
     * [imagename] is the name provided in the DAQ meta-data or if not specified
     * 16-digit hexadecimal representation of the image timestamp, and [reb] is
     * the name of the REB that produced the image.
     *
     * @param dName The name of the directory where the image file is to saved.
     * @return The name of the saved file
     * @throws RaftException
     * @throws IOException
     */
    public String saveImage(String dName) throws RaftException, IOException {
        if ((currImage.isInterleaved() ? currImage.getData() : currImage.getData(0)) == null) {
            throw new RaftException("Image contains no valid data");
        }

        String dirName = ((dName == null || dName.isEmpty()) ? defaultDirectory : dName) + "/";
        String filName = FileNamePatternUtils.resolveFileName(imageDataFileNamePattern, fileNamePatternProperties);
        String shortFileName = Paths.get(filName).getFileName().toString();
        setFitsFileName(shortFileName);

        try (FileOutputStream ofs = new FileOutputStream(dirName + filName)) {
            if (currImage.isInterleaved()) {
                ofs.getChannel().write(currImage.getData());
            } else {
                for (int ccd = 0; ccd < numCcds; ccd++) {
                    ofs.getChannel().write(currImage.getData(ccd));
                }
            }
        }

        return filName;
    }

    void setExternalSequenceNumber(int sequenceNumber) {
        fitsSeqnum = sequenceNumber;
        externalSequenceNumber = true;
    }

    /**
     * Saves the current image data as one or more FITS files.
     *
     * Each active CCD on the REB produces a separate file. The default name of
     * each file is "Image_[reb]_[ccd]_[imagename].fits", where [imagename] is
     * the name provided in the DAQ meta-data or if not specified the 16-digit
     * hexadecimal representation of the image timestamp, [reb] is the name of
     * the REB, and [ccd] is the CCD number.
     *
     * @param dName The name of the directory where the FITS file is to be
     * saved.
     * @param provider An external FitsHeaderMetadataProvider.
     * @return A list containing the names of the saved files
     * @throws RaftException
     * @throws IOException
     */
    public List<String> saveFitsImage(String dName, FitsHeaderMetadataProvider provider)
            throws IOException, RaftException {
        return saveFitsImage(dName, provider, false);
    }
    
    /**
     * Saves the current image data as one or more FITS files.
     *
     * Each active CCD on the REB produces a separate file. The default name of
     * each file is "Image_[reb]_[ccd]_[imagename].fits", where [imagename] is
     * the name provided in the DAQ meta-data or if not specified the 16-digit
     * hexadecimal representation of the image timestamp, [reb] is the name of
     * the REB, and [ccd] is the CCD number.
     * 
     * @param dName The name of the directory where the FITS file is to be
     * saved.
     * @param provider An external FitsHeaderMetadataProvider.
     * @param returnFullPaths If <code>true</code> then the returned list will include full paths
     * to the generated FITS files, otherwise just the file names.
     * @return A list containing the names of the saved files
     * @throws RaftException
     * @throws IOException
     */    
    
    public List<String> saveFitsImage(String dName, FitsHeaderMetadataProvider provider, boolean returnFullPaths)
            throws IOException, RaftException {
        if (currImageExcp != null) {
            throw currImageExcp;
        }
        int numberOfColumns = readOutParameters.getSerialReadPixels();
        int numberOfRows = readOutParameters.getParallelReadPixels();

        if (currImageBuffer[0][0].limit() != 4 * numberOfColumns * numberOfRows) {
            throw new RaftException("Image size is inconsistent with slice count. Size: "
                    + (currImageBuffer[0][0].limit() / 4) + ", expected: "
                    + (numberOfColumns * numberOfRows)
                    + " (" + numberOfColumns + "x" + numberOfRows + ")");
        }
        List<String> names = new ArrayList<>();
        String dirPatt = ((dName == null || dName.isEmpty()) ? defaultDirectory : dName) + "/";
        for (int ccd = 0; ccd < numCcds; ccd++) {
            CCD ccdGeometry = rebGeometry.getCCDs().get(ccd);

            //TO-DO: can this be stored/built more efficiently? When the image is changed?
            ImageSet imageSet = new ReadOutImageSet(ccdGeometry, readOutParameters);

            fitsService.setHeaderKeywordValue("SequenceNumber", externalSequenceNumber ? fitsSeqnum : ++fitsSeqnum);
            fitsService.setHeaderKeywordValue("FileCreationTime", new Date(System.currentTimeMillis()));
            fitsService.setHeaderKeywordValue("ObservationDate", new Date(imageTime));
            fitsService.setHeaderKeywordValue("Tag", String.valueOf(imageTime));

            try {
                ImageName in = new ImageName(currImage.getName());
                fitsService.setHeaderKeywordValue("ImageName", in.toString());
                fitsService.setHeaderKeywordValue("ImageDate", in.getDateString());
                fitsService.setHeaderKeywordValue("ImageNumber", in.getNumber());
                fitsService.setHeaderKeywordValue("ImageController", in.getController().getCode());
                fitsService.setHeaderKeywordValue("ImageSource", in.getSource().getCode());

                fileNamePatternProperties.setProperty("imageDate", in.getDateString());
                fileNamePatternProperties.setProperty("imageNumber", in.getNumberString());
                fileNamePatternProperties.setProperty("imageController", in.getController().getCode());
                fileNamePatternProperties.setProperty("imageSource", in.getSource().getCode());
            } catch (IllegalArgumentException x) {
                LOG.log(Level.FINE, "Error parsing {0}: {1}", new Object[]{currImage.getName(), x});
            }

            FitsFileWriter writer;
            List<FitsHeaderMetadataProvider> providers = new ArrayList<>();
            providers.add( new ImageProcFitsHeaderMetadataProvider(ccdGeometry) );
            providers.add(fitsService.getFitsHeaderMetadataProvider(ccdGeometry.getUniqueId()));
            providers.add(new GeometryFitsHeaderMetadataProvider(ccdGeometry));
            if (provider != null) {
                providers.add(provider);
            }

            //Convert the metadata providers to properties
            Properties resolverProperties = getPropertiesFromMetadataProviders(providers);

            //Resolve the fileName and output directory pattern.
            fileNamePatternProperties.setProperty("sensorId", String.valueOf(ccd));
            fileNamePatternProperties.setProperty("sensorLoc", String.valueOf(reb.getRebNumber()) + String.valueOf(ccd));
            try {
                fileNamePatternProperties.setProperty("testType", resolverProperties.getProperty("TestType").toLowerCase());
            } catch (RuntimeException ex) {
            }
            try {
                fileNamePatternProperties.setProperty("imageType", resolverProperties.getProperty("ImageType").toLowerCase());
            } catch (RuntimeException ex) {
            }

            resolverProperties.putAll(fileNamePatternProperties);

            String fileName = FileNamePatternUtils.resolveFileName(fitsFileNamePattern, resolverProperties);
            String shortFileName = Paths.get(fileName).getFileName().toString();
            setFitsFileName(shortFileName);

            String dirName = FileNamePatternUtils.resolveFileName(dirPatt, resolverProperties);
            File file = new File(dirName, fileName);
            file.getParentFile().mkdirs();
            try {
                writer = new FitsFileWriter(file, imageSet, fitsService.getHeaderSpecificationMap(), providers);
            } catch (FitsException e) {
                throw new RaftException("FITS error: " + e);
            }
            for (int adc = 0; adc < NUM_ADCS; adc++) {
                ByteBuffer buff = currImageBuffer[ccd][adc];
                buff.rewind();
                writer.write(dataSegmentMap[adc], buff);
            }
            writer.close();
            names.add(returnFullPaths ? file.getAbsolutePath() : file.getName());
        }
        fitsService.clearNonStickyHeaderKeywordValues();
        return names;
    }

    public List<String> saveFitsImage(String dName)
            throws IOException, RaftException {
        return saveFitsImage(dName, null);
    }

    private Properties getPropertiesFromMetadataProviders(List<FitsHeaderMetadataProvider> providers) {
        Properties p = new Properties();
        for (FitsHeaderMetadataProvider provider : providers) {
            MetaDataSet md = provider.getPrimaryHeaderMetadata();
            if (md != null) {
                p.putAll(md.convertToProperties());
            }
        }
        return p;
    }

    /**
     * Gets the DAQ metadata for the current image.
     *
     * @return The image metadata
     * @throws RaftException
     */
    public ImageMetadata getImageMetadata() throws RaftException {
        return currImage.getMetadata();
    }

    /**
     * Gets a portion of the current image.
     *
     * @param ccd The CCD number
     * @param offset The offset (in pixels) to the first pixel data to obtain.
     * @param count The number of data pixels to obtain. If zero, all the data,
     * starting at offset, is obtained.
     * @return The returned pixel data
     * @throws RaftException
     */
    public ImageData getImage(int ccd, int offset, int count) throws RaftException {

        if ((currImage.isInterleaved() ? currImage.getData() : currImage.getData(0)) == null) {
            throw new RaftException("Image contains no valid data");
        }

        int length = count;
        int iLength = currImage.getLength();
        if (length < 0 || offset < 0 || offset >= iLength) {
            throw new RaftException("Invalid length or offset");
        }
        if (length == 0 || length + offset > iLength) {
            length = iLength - offset;
        }
        int[] data = new int[length];
        ByteBuffer buff = currImage.isInterleaved() ? currImage.getData() : currImage.getData(ccd);
        buff.order(ByteOrder.LITTLE_ENDIAN);
        buff.position(4 * offset);
        buff.asIntBuffer().get(data);

        return new ImageData(currImage.getTimestamp(), offset, data);
    }

    /**
     * Gets a portion of a saved raw image.
     *
     * @param fileName The name of the file containing the image
     * @param offset The offset (in pixels) to the first pixel data to obtain.
     * @param count The number of data pixels to obtain.
     * @return The returned pixel data
     * @throws RaftException
     * @throws IOException
     */
    static ImageData getImage(String fileName, int offset, int count)
            throws RaftException, IOException {

        offset *= 4;
        int length = 4 * count;
        FileInputStream ifs = new FileInputStream(fileName);
        ByteBuffer buff;
        try {
            FileChannel chan = ifs.getChannel();
            long iLength = chan.size();
            if (length < 0 || offset < 0 || offset >= iLength) {
                throw new RaftException("Invalid length or offset");
            }
            if (length == 0 || length + offset > iLength) {
                length = (int) (iLength - offset);
            }
            buff = chan.map(FileChannel.MapMode.READ_ONLY, offset, length);
        } finally {
            ifs.close();
        }
        int[] data = new int[length / 4];
        buff.order(ByteOrder.LITTLE_ENDIAN);
        buff.asIntBuffer().get(data);

        return new ImageData(0, offset / 4, data);
    }

    /**
     * Gets pixel value statistics for the current image.
     *
     * @return The array of (average, stddev) pairs, one per image segment
     * @throws RaftException
     */
    public double[][] getImageStats() throws RaftException {
        if (currImageExcp != null) {
            throw currImageExcp;
        }
        double[][] stats = new double[numCcds * NUM_ADCS][2];
        for (int ccd = 0; ccd < numCcds; ccd++) {
            for (int adc = 0; adc < NUM_ADCS; adc++) {
                double sum = 0.0, sumSq = 0.0;
                ByteBuffer buff = currImageBuffer[ccd][adc];
                int count = buff.limit() / 4;
                for (int k = 0; k < count; k++) {
                    double value = buff.getInt();
                    sum += value;
                    sumSq += value * value;
                }
                double avg = sum / count;
                int j = ccd * NUM_ADCS + adc;
                stats[j][0] = avg;
                stats[j][1] = Math.sqrt(sumSq / count - avg * avg);
            }
        }

        return stats;
    }

    /**
     * Splits the current image into its segments.
     *
     * Each active CCD on the REB produces 16 segments.
     *
     * @return The array of image segment buffers
     * @throws RaftException
     */
    public ByteBuffer[][] splitImage() throws RaftException {
        ByteBuffer data = currImage.isInterleaved() ? currImage.getData() : currImage.getData(0);
        if (data == null || currImage.getStatus() != Image.STATUS_GOOD) {
            throw new RaftException("Image contains no valid data");
        }
        int numSeg = numCcds * NUM_ADCS;
        int buffSegs = currImage.isInterleaved() ? numSeg : NUM_ADCS;
        if (data.limit() % (4 * buffSegs) != 0) {
            throw new RaftException(("Invalid image size: " + data.limit()));
        }
        int segSize = data.limit() / buffSegs;
        ByteBuffer[][] adata = new ByteBuffer[numCcds][NUM_ADCS];
        for (int ccd = 0; ccd < numCcds; ccd++) {
            for (int adc = 0; adc < NUM_ADCS; adc++) {
                adata[ccd][adc] = ByteBuffer.allocate(segSize);
            }
        }
        if (currImage.isInterleaved()) {
            data.order(ByteOrder.nativeOrder());
            for (int j = 0; j < segSize / 4; j++) {
                for (int ccd = 0; ccd < numCcds; ccd++) {
                    int actCcd = reb.getRebType() != BaseSet.REB_TYPE_SCIENCE || numCcds < 3 ? ccd : numRebCcds - 1 - ccd;
                    for (int adc = 0; adc < NUM_ADCS; adc++) {
                        adata[actCcd][adc].putInt(data.getInt() ^ dataInversionMask);  // Raw values are usually inverted
                    }
                }
            }
        } else {
            ByteBuffer[] cdata = new ByteBuffer[numCcds];
            for (int ccd = 0; ccd < numCcds; ccd++) {
                cdata[ccd] = currImage.getData(ccd);
                cdata[ccd].order(ByteOrder.nativeOrder());
            }
            for (int j = 0; j < segSize / 4; j++) {
                for (int ccd = 0; ccd < numCcds; ccd++) {
                    for (int adc = 0; adc < NUM_ADCS; adc++) {
                        adata[ccd][adc].putInt(cdata[ccd].getInt() ^ dataInversionMask);  // Raw values are usually inverted
                    }
                }
            }
        }
        for (int ccd = 0; ccd < numCcds; ccd++) {
            for (int adc = 0; adc < NUM_ADCS; adc++) {
                adata[ccd][adc].flip();
            }
        }
        return adata;
    }

    /**
     * Sets constant FITS metadata.
     *
     * @param serial The REB serial number
     */
    private void setFitsFileName(String fitsFileName) {
        fitsService.setHeaderKeywordValue("OriginalFileName", fitsFileName);
    }

    /**
     * Sets the default image directory.
     *
     * @param dirName The directory name
     */
    public void setDefaultImageDirectory(String dirName) {
        defaultDirectory = dirName == null || dirName.isEmpty() ? "." : dirName;
    }

    /**
     * Sets the FITS image file name pattern.
     *
     * @param pattern The file name pattern to set
     */
    public void setFitsFileNamePattern(String pattern) {
        fitsFileNamePattern = pattern;
    }

    /**
     * Sets the raw image data file name pattern.
     *
     * @param pattern The file name pattern to set
     */
    public void setImageDataFileNamePattern(String pattern) {
        imageDataFileNamePattern = pattern;
    }

    /**
     * Get the corresponding FitsService
     * @return The FitsService
     */
    public FitsService getFitsService() {
        return fitsService;
    }

    /**
     * Converts image timestamp (ns) to canonical string (YYYYMMDDHHMMSS).
     */
    private static String timestampString(long tstamp) {
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(tstamp / 1_000_000);
        cal.setTimeZone(TimeZone.getTimeZone("GMT"));
        return String.format("%tY%<tm%<td%<tH%<tM%<tS", cal);
    }

    private class ImageProcFitsHeaderMetadataProvider implements FitsHeaderMetadataProvider {

        private final CCD ccd;
        
        ImageProcFitsHeaderMetadataProvider(CCD ccd) {
            this.ccd = ccd;
        }
        
        @Override
        public MetaDataSet getPrimaryHeaderMetadata() {
            CCDType ccdType = ccd.getType();
            MetaDataSet m = new MetaDataSet();
            m.addMetaData("primary", "CCDManufacturer", ccdType.getName().toUpperCase());
            m.addMetaData("primary", "CCDModel", ccdType.getType());
            return m;
        }

        @Override
        public MetaDataSet getDataExtendedHeaderMetadata(int extendedIndex) {
            MetaDataSet segmentMetaDataSet = new MetaDataSet();
            Map<String, Object> metadata = new HashMap<>();
            try {
                analyzeImage(ccd.getSegments().get(extendedIndex),
                        ccd.getSerialPosition(), extendedIndex, metadata);
            } catch (RaftException ex) {
                LOG.log(Level.SEVERE, ex.getMessage(), ex);
            }
            segmentMetaDataSet.addMetaDataMap("seg", metadata);
            return segmentMetaDataSet;
        }

        @Override
        public MetaDataSet getAdditionalExtendedHeaderMetadata(String extendedKeyword) {
            return null;
        }
    }

    /*
     * This map determines the DAQ CCD index (0, 1 or 2) within the image data
     * for a specified CCD number.  Since it's possible that not all CCDs were
     * read out, the value of the mask of read CCDs (0 - 7) is needed.
     *
     * The three rows of the map correspond to the three CCDs.  Each row contains
     * eight entries, one for each possible value of the CCD mask.  Negative
     * values correspond to invalid combinations.  Non-negative values provide
     * the desired mapping.
     *
     * So a given column in the table below formed by the initialization values
     * contains the index within the data for each CCD, numbering from the top.
     * E.g. the "5" column (0, -1, 1) shows that CCD 0 is at index 0, CCD 1 is
     * not present, and CCD 2 is at index 1.
     */
    private static final int[][] CCD_MAP = {{-1, 0, -1, 0, -1, 0, -1, 0},
    {-1, -1, 0, 1, -1, -1, 0, 1},
    {-1, -1, -1, -1, 0, 1, 1, 2}};

    /**
     * Calculates image statistics for a given segment for the active and
     * overscan regions separately. The results are added to the metadata for
     * the segment header.
     */
    private void analyzeImage(Segment segment, int ccdNum, int segNum, Map<String, Object> imageMetaData)
            throws RaftException {

        if (currImageExcp != null) {
            throw currImageExcp;
        }

        int daqCcdNum = CCD_MAP[ccdNum][ccdMask];
        if (daqCcdNum < 0) {
            throw new RaftException("Invalid CCD number: " + ccdNum + "; CCD mask = " + ccdMask);
        }
        ByteBuffer buff = currImageBuffer[daqCcdNum][invertedDataSegmentMap[segNum]];
        buff.rewind();

        double pix_sum_active = 0.;
        double pix_sum_sq_active = 0.;
        int npix_active = 0;
        double pix_sum_bias = 0.;
        double pix_sum_sq_bias = 0.;
        int npix_bias = 0;

        //Total serial read pixels: readCols + readCols2 + overCols
        int numberOfColumns = readOutParameters.getSerialReadPixels();
        //Total parallel read pixels: readRows + overRows
        int numberOfRows = readOutParameters.getParallelReadPixels();

        //The total serial active size. These are the active pixels and don't
        //include the serial prescan pixels.
        int segmentSerialActiveSize = segment.getSegmentSerialActiveSize();

        //The serial prescan. This depends on the CCD type
        int serialPrescan = readOutParameters.getSerialPrescan();

        for (int row = 1; row <= numberOfRows; row++) {
            for (int col = 1; col <= numberOfColumns; col++) {
                double pixval = buff.getInt();
//                int pixval = segment.getImagePixelData().getPixelData(row, col);
                if (col > serialPrescan && col <= (serialPrescan + segmentSerialActiveSize)) {
                    pix_sum_active += pixval;
                    pix_sum_sq_active += (double) pixval * pixval;
                    npix_active++;
                } else if (col > (serialPrescan + segmentSerialActiveSize)) {
                    pix_sum_bias += pixval;
                    pix_sum_sq_bias += (double) pixval * pixval;
                    npix_bias++;
                }
                //if (row == 100) {
                //    System.out.println(preCols + " " + segmentSerialActiveSize + " : " + col + " " + pixval);
                //}
            }
        }

        LOG.log(Level.FINE, "Active {0} Bias {1} PreCols {2} ActiveSize {3} NCols {4} NRows {5}", new Object[]{npix_active, npix_bias, serialPrescan, segmentSerialActiveSize, numberOfColumns, numberOfRows});

        if (npix_active > 0) {
            double average = (double) pix_sum_active / npix_active;
            double stdev = Math.sqrt(pix_sum_sq_active / npix_active - (average * average));
            LOG.log(Level.FINE, "Adding AVERAGE {0} STDEV {1}", new Object[]{average, stdev});
            imageMetaData.put("AVERAGE", average);
            imageMetaData.put("STDEV", stdev);
        }
        if (npix_bias > 0) {
            double average = (double) pix_sum_bias / npix_bias;
            double stdev = Math.sqrt(pix_sum_sq_bias / npix_bias - (average * average));
            LOG.log(Level.FINE, "Adding AVGBIAS {0} STDVBIAS {1}", new Object[]{average, stdev});
            imageMetaData.put("AVGBIAS", average);
            imageMetaData.put("STDVBIAS", stdev);
        }

    }

    private ReadOutParameters createReadOutParametersFromRegistersMetadata(int[] registers) {
        return ReadOutParametersBuilder.create()
                .ccdType(rebGeometry.getCCDs().get(0).getType())
                .readoutParameterValues(registers)
                .readoutParameterNames(ReadOutParametersOld.DEFAULT_NAMES)
                .build();
    }
}
