package org.lsst.ccs.subsystem.rafts;

import java.awt.Point;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.reb.ImageMetadata;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.Geometry;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.image.ReadOutParameters;
import org.lsst.ccs.visualization.client.DataMessage;
import org.lsst.ccs.visualization.client.EndMessage;
import org.lsst.ccs.visualization.client.IngestClient;
import org.lsst.ccs.visualization.client.StartMessage;

/**
 *
 * @author The LSST CCS Team
 */
public class GlobalVisualizationClient implements HasLifecycle {
    
    private int clientPort;
    private String clientHost;

    private IngestClient ingestClient;
    
    private final ReentrantLock sendDataLock = new ReentrantLock();
    
    private Map<String,Integer> startedImages = new ConcurrentHashMap<>();
    
    private String currentImageName;
    
    private static final Logger LOG = Logger.getLogger(GlobalVisualizationClient.class.getName());
    
    private boolean isConnected = false;
    
    @Override
    public void postInit() {
        try {
            ingestClient = new IngestClient(clientHost, clientPort);
            isConnected = true;
        } catch (IOException ioe) {
            LOG.log(Level.SEVERE, "Failed to initialize the Visualization client {0}:{1}: {2}",
                    new Object[]{clientHost, clientPort, ioe});
        }
    }

    @Override
    public void shutdown() {
        try {
            if (ingestClient != null) {                
                ingestClient.close();                
            }
            isConnected = false;
        } catch (IOException ioe) {
            throw new RuntimeException("Failed to close the Visualization client "+clientHost+":"+clientPort,ioe);
        }
    }


    public boolean isConnected() {
        return isConnected;
    }
    
    public void sendImageData(ImageMetadata imageMetadata, ByteBuffer data, CCD ccd, ReadOutParameters readOutParameters) {
        
        LOG.log(Level.FINE, "Sending image data for {0} {1}", new Object[]{ccd.getUniqueId(), imageMetadata.getName()});
        
        sendDataLock.lock();
        try {
            
            
            //The name of a simulated image is null so we cannot use it.
            //For now we just assume that there we send the first 9 we receive.            
//            String imageName = imageMetadata.getName(); 
            String imageName = ccd.getUniqueId();
            //Make a guess on the dimensions
            int height = readOutParameters.getSerialReadPixels() * 3;
            int width = readOutParameters.getParallelReadPixels() * 3;
                        
//            if ( ! startedImages.containsKey(imageName) ) {            
              if ( startedImages.isEmpty() ) {            
                Geometry reb = ccd.getParent();
                Raft raft = null;
                if ( reb != null ) {
                    raft = (Raft)reb.getParent();
                }
                
                //If we have a raft, then get the dimensions;
                if ( raft != null) {
                    height = raft.getHeight();
                    width = raft.getWidth();                    
                }
                LOG.log(Level.FINE, "Sending start message {0}x{1}", new Object[]{width, height});
                
                //This will have to removed when we sort out the name being null for simulation
                currentImageName = imageName;
                ingestClient.send(new StartMessage(currentImageName, width,height, 0, 9));
                startedImages.put(imageName, 0);
            }
            
            //Calculate offsets
            //offset to beginning of data
            //Needs to be corrected with readout parametres information
            Point absoluteOrigin = ccd.getGeometryAbsolutePosition();
            int offset = absoluteOrigin.getLocation().y*width + absoluteOrigin.getLocation().x;
            
            //The width of a row of data
            int stepLength = readOutParameters.getTotalParallelSize();
            
            //The step to the next begin of the data
            //Needs correction from readoutParameters
            int stepOffset = width - stepLength;
            
            ingestClient.send(new DataMessage(currentImageName, offset, stepLength, stepOffset, data));
            
            //For now the data is complete, so send end of image message
            ingestClient.send(new EndMessage(currentImageName));
            
//            startedImages.put(imageName, startedImages.get(imageName)+1);
            startedImages.put(imageName, 1);
            
//            if ( startedImages.get(imageName) == 9) {
            if ( startedImages.size() == 9) {
                LOG.log(Level.FINE, "**** Done with image {0}", currentImageName);
//                startedImages.remove(imageName);
                startedImages.clear();
            }
            
            
        } catch (IOException ioe) {
            //We have to decide how to handle exceptions when we cannot communicate with the Visualization Server
            ioe.printStackTrace();
        } finally {
            sendDataLock.unlock();
        }
        
    }
    
}