package org.lsst.ccs.subsystem.rafts;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.config.BulkValidationException;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.drivers.reb.Asic;
import org.lsst.ccs.drivers.reb.BoardDacs;
import org.lsst.ccs.drivers.reb.PowerAdcs;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.Sequencer;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.subsystem.rafts.config.DACS;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.REBDevice.AdcData;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements control of the DACs on a REB.
 *
 *  @author Owen Saxton
 */
public class DacControl extends Control implements ConfigurationBulkChangeHandler {

    /*
     *  Constants.
     */
    public static final String
        SCLK_LOW_P   = "sclkLowP",
        SCLK_HIGH_P  = "sclkHighP",
        PCLK_LOW_P   = "pclkLowP",
        PCLK_HIGH_P  = "pclkHighP",
        RG_LOW_P     = "rgLowP",
        RG_HIGH_P    = "rgHighP",

        SCLK_LOW     = "sclkLow",
        SCLK_LOW_SH  = "sclkLowSh",
        SCLK_HIGH    = "sclkHigh",
        SCLK_HIGH_SH = "sclkHighSh",
        PCLK_LOW     = "pclkLow",
        PCLK_LOW_SH  = "pclkLowSh",
        PCLK_HIGH    = "pclkHigh",
        PCLK_HIGH_SH = "pclkHighSh",
        RG_LOW       = "rgLow",
        RG_LOW_SH    = "rgLowSh",
        RG_HIGH      = "rgHigh",
        RG_HIGH_SH   = "rgHighSh",
        CS_GATE      = "csGate";

    private static final int
        SCLK_LOW_CH     = 0x00001,
        SCLK_LOW_SH_CH  = 0x00002,
        SCLK_HIGH_CH    = 0x00004,
        SCLK_HIGH_SH_CH = 0x00008,
        PCLK_LOW_CH     = 0x00010,
        PCLK_LOW_SH_CH  = 0x00020,
        PCLK_HIGH_CH    = 0x00040,
        PCLK_HIGH_SH_CH = 0x00080,
        RG_LOW_CH       = 0x00100,
        RG_LOW_SH_CH    = 0x00200,
        RG_HIGH_CH      = 0x00400,
        RG_HIGH_SH_CH   = 0x00800,
        CS_GATE_CH      = 0x01000;

    private static final String[]
        checkList = {SCLK_LOW, SCLK_HIGH, PCLK_LOW, PCLK_HIGH, RG_LOW, RG_HIGH};

    // test parameters
    // TODO make configurables
    private static final double DAC_TOLERANCE = 0.4;  // use shift and offset per dac
    private static final int DEFAULT_LOAD_WAIT = 200, DEFAULT_CLEAR_WAIT = 50;
    private static final double TEST_VOLTS = 0.5, ZERO_ERROR = 0.2, VALUE_ERROR = 0.2, 
                                CHANGE_ERROR = 0.1, CLK_SHORTS_LIMIT = 0.002;

    private static final Map<Integer, Integer> seqLineMap = new HashMap<>();
    static {
        seqLineMap.put(REBDevice.ADC_PCLK_L, Sequencer.SEQ_LINE_P1 | Sequencer.SEQ_LINE_P2 | Sequencer.SEQ_LINE_P3
                                              | Sequencer.SEQ_LINE_P4);
        seqLineMap.put(REBDevice.ADC_PCLK_U, 0);
        seqLineMap.put(REBDevice.ADC_SCLK_L, Sequencer.SEQ_LINE_S1 | Sequencer.SEQ_LINE_S2 | Sequencer.SEQ_LINE_S3);
        seqLineMap.put(REBDevice.ADC_SCLK_U, 0);
        seqLineMap.put(REBDevice.ADC_RG_L, Sequencer.SEQ_LINE_RG);
        seqLineMap.put(REBDevice.ADC_RG_U, 0);
    }

    /*
    *  Data fields.
    */
    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    // Supplied configurable fields - rail voltages
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock low", units="volts")
    private double sclkLowP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock high", units="volts")
    private double sclkHighP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock low", units="volts")
    private double pclkLowP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock high", units="volts")
    private double pclkHighP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate low", units="volts")
    private double rgLowP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate high", units="volts")
    private double rgHighP;

    // Configurable rail voltage limits
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Serial clock low min", units="volts")
    private double sclkLowMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Serial clock low max", units="volts")
    private double sclkLowMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Serial clock high min", units="volts")
    private double sclkHighMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Serial clock high max", units="volts")
    private double sclkHighMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Parallel clock low min", units="volts")
    private double pclkLowMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Parallel clock low max", units="volts")
    private double pclkLowMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Parallel clock high min", units="volts")
    private double pclkHighMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Parallel clock high max", units="volts")
    private double pclkHighMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Reset gate low min", units="volts")
    private double rgLowMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Reset gate low max", units="volts")
    private double rgLowMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Reset gate high min", units="volts")
    private double rgHighMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Reset gate high max", units="volts")
    private double rgHighMax;

    // Supplied configurable fields - raw DAC settings
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock low", units= "DAC counts")
    private int    sclkLow;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock low shifted", units= "DAC counts")
    private int    sclkLowSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock high", units= "DAC counts")
    private int    sclkHigh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock high shifted", units= "DAC counts")
    private int    sclkHighSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock low", units= "DAC counts")
    private int    pclkLow;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock low shifted", units= "DAC counts")
    private int    pclkLowSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock high", units= "DAC counts")
    private int    pclkHigh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock high shifted", units= "DAC counts")
    private int    pclkHighSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate low", units= "DAC counts")
    private int    rgLow;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate low shifted", units= "DAC counts")
    private int    rgLowSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate high", units= "DAC counts")
    private int    rgHigh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate high shifted", units= "DAC counts")
    private int    rgHighSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Current source", units= "DAC counts")
    private final int[]  csGate = new int[Asic.MAX_STRIPS];

    // Shorts test parameters
    // order is sclkLowP sclkHighP pclkLowP pclkHighP rgLowP rgHighP
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Shorts Test Voltage for SClkL", units="Volts")
    private double  sclkLowTestV;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for 0V w/SClkL test", units="Volts")
    private double  sclkLowZeroErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for SClkL TestV", units="Volts")
    private double  sclkLowValueErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Shorts Test Voltage for SClkU", units="Volts")
    private double  sclkHighTestV;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for 0V w/SClkU test", units="Volts")
    private double  sclkHighZeroErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for SClkU TestV", units="Volts")
    private double  sclkHighValueErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Shorts Test Voltage for PClkL", units="Volts")
    private double  pclkLowTestV;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for 0V w/PClkL test", units="Volts")
    private double  pclkLowZeroErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for PClkL TestV", units="Volts")
    private double  pclkLowValueErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Shorts Test Voltage for PClkU", units="Volts")
    private double  pclkHighTestV;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for 0V w/PClkU test", units="Volts")
    private double  pclkHighZeroErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for PClkU TestV", units="Volts")
    private double  pclkHighValueErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Shorts Test Voltage for RGL", units="Volts")
    private double  rgLowTestV;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for 0V w/RGL test", units="Volts")
    private double  rgLowZeroErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for RGL TestV", units="Volts")
    private double  rgLowValueErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Shorts Test Voltage for RGU", units="Volts")
    private double  rgHighTestV;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for 0V w/RGU test", units="Volts")
    private double  rgHighZeroErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for RGU TestV", units="Volts")
    private double  rgHighValueErr;

    // Power on configurable fields - rail voltage tests
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Serial clock low tolerance", units="volts")
    private double sclkLowTol;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Serial clock low offset", units="volts")
    private double sclkLowOff;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Serial clock high tolerance", units="volts")
    private double sclkHighTol;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Serial clock high offset", units="volts")
    private double sclkHighOff;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Parallel clock low tolerance", units="volts")
    private double pclkLowTol;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Parallel clock low offset", units="volts")
    private double pclkLowOff;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Parallel clock high tolerance", units="volts")
    private double pclkHighTol;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Parallel clock high offset", units="volts")
    private double pclkHighOff;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Reset gate low tolerance", units="volts")
    private double rgLowTol;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Reset gate low offset", units="volts")
    private double rgLowOff;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Reset gate high tolerance", units="volts")
    private double rgHighTol;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Reset gate high offset", units="volts")
    private double rgHighOff;
    // current limits after powering clock rails for node: /Rxy/RebI/DAC parameters
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Max ClkHI value", units="Amps")
    private double clkhIMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Max ClkLI value", units="Amps")
    private double clklIMax;


    // Set from Groovy file
    private boolean raw = false;

    // Other fields
    private static final Logger LOG = Logger.getLogger(DacControl.class.getName());
    private boolean cfgValid, isCurrent, isCorner;
    private int hwVersion;
    private REBDevice rebDevc;
    private BoardDacs dac;
    private int changed = -1;
    private final Map<String, Double> lowLimitMap = new HashMap<>();
    private final Map<String, Double> highLimitMap = new HashMap<>();
    private double railShiftConvH, railConvH, railShiftConvL, railConvL;
    private int adcType;


    /**
     *  Constructor
     */
    public DacControl()
    {
        for (String pName : checkList) {
            lowLimitMap.put(pName, 0.0);
            highLimitMap.put(pName, 0.0);
        }
    }


    /**
     *  Configures DAC description.
     *
     *  @param  mon   The associated monitor object
     *  @param  devc  The associated device object
     */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        rebDevc = (REBDevice)devc;
        dac = rebDevc.getBoardDacs();
    }


    /**
     *  Checks configuration.
     *
     *  @return  Whether configuration is okay
     */
    public boolean checkConfig()
    {
        hwVersion = getHwVersion(dac);
        isCurrent = hwVersion >= REB.VERSION_1;
        isCorner = hwVersion == REB.VERSION_1 || hwVersion == REB.VERSION_3;

        if (!raw) {
            double shiftGainH = (hwVersion == REB.VERSION_1) ? 20.0 / 10.0 : 24.9 / 10.0;
            double shiftGainL = (hwVersion == REB.VERSION_2) ? 24.9 / 10.09 : shiftGainH;
            double fudge = (hwVersion == REB.VERSION_1) ? 1.0 : 1.04;
            railShiftConvH = fudge * REBDevice.DAC_CONV / shiftGainH;
            railConvH = fudge * REBDevice.DAC_CONV / (1 + shiftGainH);
            railShiftConvL = fudge * REBDevice.DAC_CONV / shiftGainL;
            railConvL = fudge * REBDevice.DAC_CONV / (1 + shiftGainL);
        }
        else {
            cfgValid = true;
        }
        adcType = isCorner ? REBDevice.TYPE_CR_VOLT : REBDevice.TYPE_BIAS_VOLT;

        return true;
    }


    /**
     *  Gets the raw state.
     *
     *  @return  Whether data are raw ADC values
     */
    public boolean isRaw()
    {
        return raw;
    }


    /**
     *  Gets the hardware version.
     *
     *  @param  dac  The board DACs object
     *  @return  The hardware version
     */
    public static int getHwVersion(BoardDacs dac)
    {
        int version = dac.getVersion();
        if (version == BoardDacs.VERSION_REB0_0
              || version == BoardDacs.VERSION_REB0_1) {
            version = REB.VERSION_0;
        }
        else if (version == BoardDacs.VERSION_WREB1
                  || version == BoardDacs.VERSION_WREB2
                  || version == BoardDacs.VERSION_GREB1) {
            version = REB.VERSION_1;
        }
        else if (version == BoardDacs.VERSION_REB3
                  || version == BoardDacs.VERSION_REB5) {
            version = REB.VERSION_2;
        }
        else if (version == BoardDacs.VERSION_GREB2
                  || version == BoardDacs.VERSION_WREB4) {
            version = REB.VERSION_3;
        }
        else {
            version = REB.VERSION_UNSUPP;
        }
        return version;
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version
     */
    @Deprecated
    public int getHwVersion()
    {
        return hwVersion;
    }


    /**
     *  Validates configuration changes
     *
     *  @param  params  Map of parameter values
     *  @throws IllegalArgumentException 
     */
    @Override
    public void validateBulkChange(Map<String, Object> params) throws IllegalArgumentException
    {
        //System.out.println(getName() + " validateBulkChange called");
        String errMsg = null;
        if (!raw) {
            for (String pName : checkList) {
                double value = (Double)params.get(pName + "P");
                double max = (Double)params.get(pName + "Max");
                if (value > max) {
                    errMsg = String.format("%s: %sP (%5.5g) is above high limit (%5.5g)", getName(), pName, value, max);
                    break;
                }
                double min = (Double)params.get(pName + "Min");
                if (value < min) {
                    errMsg = String.format("%s: %sP (%5.5g) is below low limit (%5.5g)", getName(), pName, value, min);
                    break;
                }
            }
        }
        if (errMsg != null) {
            LOG.severe("Configuration failure: " + errMsg);
            throw new IllegalArgumentException(errMsg);
        }
        else {
            cfgValid = true;
        }
    }


    /**
     *  Sets the low parallel clock rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkLowP(double value)
    {
        pclkLowP = value;
        changed |= PCLK_LOW_CH;
    }


    /**
     *  Gets the low parallel clock rail voltage.
     *
     *  @return  The value
     */
    public double getPclkLowP()
    {
        return pclkLowP;
    }


    /**
     *  Sets the low parallel clock rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkLowMax(double value)
    {
        pclkLowMax = value;
        highLimitMap.put(PCLK_LOW, value);
    }


    /**
     *  Sets the low parallel clock rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkLowMin(double value)
    {
        pclkLowMin = value;
        lowLimitMap.put(PCLK_LOW, value);
    }


    /**
     *  Sets the low parallel clock rail DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkLow(int value)
    {
        pclkLow = value;
        changed |= PCLK_LOW_CH;
    }


    /**
     *  Gets the low parallel clock rail DAC value.
     *
     *  @return  value
     */
    public int getPclkLow()
    {
        return pclkLow;
    }


    /**
     *  Sets the low parallel clock rail shift DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkLowSh(int value)
    {
        pclkLowSh = value;
        changed |= PCLK_LOW_SH_CH;
    }


    /**
     *  Gets the low parallel clock rail shift DAC value.
     *
     *  @return  value
     */
    public int getPclkLowSh()
    {
        return pclkLowSh;
    }


    /**
     *  Sets the high parallel clock rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkHighP(double value)
    {
        pclkHighP = value;
        changed |= PCLK_HIGH_CH;
    }


    /**
     *  Gets the high parallel clock rail voltage.
     *
     *  @return  The value
     */
    public double getPclkHighP()
    {
        return pclkHighP;
    }


    /**
     *  Sets the high parallel clock rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkHighMax(double value)
    {
        pclkHighMax = value;
        highLimitMap.put(PCLK_HIGH, value);
    }


    /**
     *  Sets the high parallel clock rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkHighMin(double value)
    {
        pclkHighMin = value;
        lowLimitMap.put(PCLK_HIGH, value);
    }


    /**
     *  Sets the high parallel clock rail DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkHigh(int value)
    {
        pclkHigh = value;
        changed |= PCLK_HIGH_CH;
    }


    /**
     *  Gets the high parallel clock rail DAC value.
     *
     *  @return  value
     */
    public int getPclkHigh()
    {
        return pclkHigh;
    }


    /**
     *  Sets the high parallel clock rail shift DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkHighSh(int value)
    {
        pclkHighSh = value;
        changed |= PCLK_HIGH_SH_CH;
    }


    /**
     *  Gets the high parallel clock rail shift DAC value.
     *
     *  @return  value
     */
    public int getPclkHighSh()
    {
        return pclkHighSh;
    }


    /**
     *  Sets the low serial clock rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkLowP(double value)
    {
        sclkLowP = value;
        changed |= SCLK_LOW_CH;
    }


    /**
     *  Gets the low serial clock rail voltage.
     *
     *  @return  The value
     */
    public double getSclkLowP()
    {
        return sclkLowP;
    }


    /**
     *  Sets the low serial clock rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkLowMax(double value)
    {
        sclkLowMax = value;
        highLimitMap.put(SCLK_LOW, value);
    }


    /**
     *  Sets the low serial clock rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkLowMin(double value)
    {
        sclkLowMin = value;
        lowLimitMap.put(SCLK_LOW, value);
    }


    /**
     *  Sets the low serial clock rail DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkLow(int value)
    {
        sclkLow = value;
        changed |= SCLK_LOW_CH;
    }


    /**
     *  Gets the low serial clock rail DAC value.
     *
     *  @return  value
     */
    public int getSclkLow()
    {
        return sclkLow;
    }


    /**
     *  Sets the low serial clock rail shift DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkLowSh(int value)
    {
        sclkLowSh = value;
        changed |= SCLK_LOW_SH_CH;
    }


    /**
     *  Gets the low serial clock rail shift DAC value.
     *
     *  @return  value
     */
    public int getSclkLowSh()
    {
        return sclkLowSh;
    }


    /**
     *  Sets the high serial clock rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkHighP(double value)
    {
        sclkHighP = value;
        changed |= SCLK_HIGH_CH;
    }


    /**
     *  Gets the high serial clock rail voltage.
     *
     *  @return  value
     */
    public double getSclkHighP()
    {
        return sclkHighP;
    }


    /**
     *  Sets the high serial clock rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkHighMax(double value)
    {
        sclkHighMax = value;
        highLimitMap.put(SCLK_HIGH, value);
    }


    /**
     *  Sets the high serial clock rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkHighMin(double value)
    {
        sclkHighMin = value;
        lowLimitMap.put(SCLK_HIGH, value);
    }


    /**
     *  Sets the high serial clock rail DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkHigh(int value)
    {
        sclkHigh = value;
        changed |= SCLK_HIGH_CH;
    }


    /**
     *  Gets the high serial clock rail DAC value.
     *
     *  @return  value
     */
    public int getSclkHigh()
    {
        return sclkHigh;
    }


    /**
     *  Sets the high serial clock rail shift DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkHighSh(int value)
    {
        sclkHighSh = value;
        changed |= SCLK_HIGH_SH_CH;
    }


    /**
     *  Gets the high serial clock rail shift DAC value.
     *
     *  @return  value
     */
    public int getSclkHighSh()
    {
        return sclkHighSh;
    }


    /**
     *  Sets the low reset gate rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgLowP(double value)
    {
        rgLowP = value;
        changed |= RG_LOW_CH;
    }


    /**
     *  Gets the low reset gate rail voltage.
     *
     *  @return  The value
     */
    public double getRgLowP()
    {
        return rgLowP;
    }


    /**
     *  Sets the low reset gate rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgLowMax(double value)
    {
        rgLowMax = value;
        highLimitMap.put(RG_LOW, value);
    }


    /**
     *  Sets the low reset gate rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgLowMin(double value)
    {
        rgLowMin = value;
        lowLimitMap.put(RG_LOW, value);
    }


    /**
     *  Sets the low reset gate rail DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgLow(int value)
    {
        rgLow = value;
        changed |= RG_LOW_CH;
    }


    /**
     *  Gets the low reset gate rail DAC value.
     *
     *  @return  value
     */
    public int getRgLow()
    {
        return rgLow;
    }


    /**
     *  Sets the low reset gate rail shift DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgLowSh(int value)
    {
        rgLowSh = value;
        changed |= RG_LOW_SH_CH;
    }


    /**
     *  Gets the low reset gate rail shift DAC value.
     *
     *  @return  value
     */
    public int getRgLowSh()
    {
        return rgLowSh;
    }


    /**
     *  Sets the high reset gate rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgHighP(double value)
    {
        rgHighP = value;
        changed |= RG_HIGH_CH;
    }


    /**
     *  Gets the high reset gate rail voltage.
     *
     *  @return  The value
     */
    public double getRgHighP()
    {
        return rgHighP;
    }


    /**
     *  Sets the high reset gate rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgHighMax(double value)
    {
        rgHighMax = value;
        highLimitMap.put(RG_HIGH, value);
    }


    /**
     *  Sets the high reset gate rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgHighMin(double value)
    {
        rgHighMin = value;
        lowLimitMap.put(RG_HIGH, value);
    }


    /**
     *  Sets the high reset gate rail DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgHigh(int value)
    {
        rgHigh = value;
        changed |= RG_HIGH_CH;
    }


    /**
     *  Gets the high reset gate rail DAC value.
     *
     *  @return  value
     */
    public int getRgHigh()
    {
        return rgHigh;
    }


    /**
     *  Sets the high reset gate rail shift DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgHighSh(int value)
    {
        rgHighSh = value;
        changed |= RG_HIGH_SH_CH;
    }


    /**
     *  Gets the high reset gate rail shift DAC value.
     *
     *  @return  value
     */
    public int getRgHighSh()
    {
        return rgHighSh;
    }


   /**
     *  Sets/Gets the current source gates.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setCsGate(int[] value)
    {
        System.arraycopy(value, 0, csGate, 0, csGate.length);
        changed |= CS_GATE_CH;
    }

    public int[] getCsGate()
    {
        return csGate;
    }


    /**
     *  Sets all the configuration data.
     * 
     *  @param  dacs  The DAC values
     *  @throws  RaftException
     */
    void setConfig(DACS dacs) throws RaftException
    {
        String name = getName();
        if (!raw) {
            double[] values = dacs.getPValues();
            sce.submitChange(name, RG_HIGH_P, values[DACS.RG_HIGH]);
            sce.submitChange(name, RG_LOW_P, values[DACS.RG_LOW]);
            sce.submitChange(name, PCLK_HIGH_P, values[DACS.PCLK_HIGH]);
            sce.submitChange(name, PCLK_LOW_P, values[DACS.PCLK_LOW]);
            sce.submitChange(name, SCLK_HIGH_P, values[DACS.SCLK_HIGH]);
            sce.submitChange(name, SCLK_LOW_P, values[DACS.SCLK_LOW]);
        }
        else {
            int[] values = dacs.getValues();
            sce.submitChange(name, RG_HIGH, values[DACS.RG_HIGH]);
            sce.submitChange(name, RG_LOW, values[DACS.RG_LOW]);
            sce.submitChange(name, PCLK_HIGH, values[DACS.PCLK_HIGH]);
            sce.submitChange(name, PCLK_LOW, values[DACS.PCLK_LOW]);
            sce.submitChange(name, SCLK_HIGH, values[DACS.SCLK_HIGH]);
            sce.submitChange(name, SCLK_LOW, values[DACS.SCLK_LOW]);
            if (isCurrent) {
                sce.submitChange(name, RG_LOW_SH, values[DACS.RG_LOW_SH]);
                sce.submitChange(name, PCLK_LOW_SH, values[DACS.PCLK_LOW_SH]);
                sce.submitChange(name, SCLK_LOW_SH, values[DACS.SCLK_LOW_SH]);
                if (isCorner) {
                    sce.submitChange(name, RG_HIGH_SH, values[DACS.RG_HIGH_SH]);
                    sce.submitChange(name, PCLK_HIGH_SH, values[DACS.PCLK_HIGH_SH]);
                    sce.submitChange(name, SCLK_HIGH_SH, values[DACS.SCLK_HIGH_SH]);
                }
            }
            else {
                sce.submitChange(name, CS_GATE, new int[]{values[DACS.CSGATE_1],
                                 values[DACS.CSGATE_2], values[DACS.CSGATE_3]});
            }
        }
        try {
            sce.commitBulkChange();
        }
        catch (BulkValidationException e) {
            sce.dropSubmittedChangesForComponent(name);
            throw new RaftException("Bulk configuration validation failed");
        }
    }


    /**
     *  Gets all the configuration data.
     *
     *  @param  dacs  The DAC values
     */
    void getConfig(DACS dacs)
    {
        if (!raw) {
            double[] values = dacs.getPValues();
            values[DACS.RG_HIGH] = rgHighP;
            values[DACS.RG_LOW] = rgLowP;
            values[DACS.PCLK_HIGH] = pclkHighP;
            values[DACS.PCLK_LOW] = pclkLowP;
            values[DACS.SCLK_HIGH] = sclkHighP;
            values[DACS.SCLK_LOW] = sclkLowP;
        }
        else {
            int[] values = dacs.getValues();
            values[DACS.RG_HIGH] = rgHigh;
            values[DACS.RG_LOW] = rgLow;
            values[DACS.PCLK_HIGH] = pclkHigh;
            values[DACS.PCLK_LOW] = pclkLow;
            values[DACS.SCLK_HIGH] = sclkHigh;
            values[DACS.SCLK_LOW] = sclkLow;
            if (isCurrent) {
                values[DACS.RG_LOW_SH] = rgLowSh;
                values[DACS.PCLK_LOW_SH] = pclkLowSh;
                values[DACS.SCLK_LOW_SH] = sclkLowSh;
                if (isCorner) {
                    values[DACS.RG_HIGH_SH] = rgHighSh;
                    values[DACS.PCLK_HIGH_SH] = pclkHighSh;
                    values[DACS.SCLK_HIGH_SH] = sclkHighSh;
                }
            }
            else {
                values[DACS.CSGATE_1] = csGate[0];
                values[DACS.CSGATE_2] = csGate[1];
                values[DACS.CSGATE_3] = csGate[2];
            }
        }
    }


    /**
     *  Loads configured DAC values onto the REB.
     *
     *  @param  check     Whether to check the read-back value
     *  @param  dataList  List to receive values from power & slow ADCs
     *  @return  The number of DACs loaded
     *  @throws  RaftException
     */
    public int loadConfig(boolean check, List<AdcData> dataList) throws RaftException
    {
        return loadConfig(DEFAULT_LOAD_WAIT, check, dataList);
    }


    /**
     *  Loads configured DAC values onto the REB.
     *
     *  @param  wait      The time to wait after loading a DAC (ms)
     *  @param  check     Whether to check the read-back value
     *  @param  dataList  List to receive read-back values  
     *  @return  The number of DACs loaded
     *  @throws  RaftException
     *
     *  Note: LCA-10064 order is SclkLow, SclkHigh, PclkLow, PclkHigh, RgLow, RgHigh
     */
    public int loadConfig(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        if (!cfgValid) {
            throw new RaftException("DAC load error: invalid configuration");
        }
        if (!rebDevc.isSerialNumValid()) {
            throw new RaftException("DAC load error: invalid REB serial number");
        }
        int count = 0;
        rebDevc.readAllAdcs(dataList);
        count += loadSclkLow(wait, check, dataList);
        count += loadSclkHigh(wait, check, dataList);
        count += loadPclkLow(wait, check, dataList);
        count += loadPclkHigh(wait, check, dataList);
        count += loadRgLow(wait, check, dataList);
        count += loadRgHigh(wait, check, dataList);
        if (!isCurrent) {
            setDac(BoardDacs.CHAN_CSGATE_1, csGate[0]);
            setDac(BoardDacs.CHAN_CSGATE_2, csGate[1]);
            setDac(BoardDacs.CHAN_CSGATE_3, csGate[2]);
            loadDac(0);
            count += 3;
            rebDevc.readAllAdcs(dataList);
        }
        changed = 0;
        return count;
    }


    /**
     *  Loads changed configured DAC values onto the REB.
     *
     *  @param  check     Whether to check the read-back value
     *  @param  dataList  List to receive read-back values  
     *  @return  The number of DACs loaded
     *  @throws  RaftException
     */
    public int loadChanged(boolean check, List<AdcData> dataList) throws RaftException
    {
        return loadChanged(DEFAULT_LOAD_WAIT, check, dataList);
    }


    /**
     *  Loads changed configured DAC values onto the REB.
     *
     *  @param  wait      The time to wait after loading a DAC (ms)
     *  @param  check     Whether to check the read-back value
     *  @param  dataList  List to receive read-back values  
     *  @return  The number of DACs loaded
     *  @throws  RaftException
     *
     *  Note: LCA-10064 order is SclkLow, SclkHigh, PclkLow, PclkHigh, RgLow, RgHigh
     */
    public int loadChanged(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        if (!cfgValid) {
            throw new RaftException("DAC load error: invalid configuration");
        }
        if (!rebDevc.isSerialNumValid()) {
            throw new RaftException("DAC load error: invalid REB serial number");
        }
        int count = 0;
        rebDevc.readAllAdcs(dataList);
        if ((changed & SCLK_LOW_CH) != 0) {
            count += loadSclkLow(wait, check, dataList);
        }
        if ((changed & SCLK_HIGH_CH) != 0) {
            count += loadSclkHigh(wait, check, dataList);
        }
        if ((changed & PCLK_LOW_CH) != 0) {
            count += loadPclkLow(wait, check, dataList);
        }
        if ((changed & PCLK_HIGH_CH) != 0) {
            count += loadPclkHigh(wait, check, dataList);
        }
        if ((changed & RG_LOW_CH) != 0) {
            count += loadRgLow(wait, check, dataList);
        }
        if ((changed & RG_HIGH_CH) != 0) {
            count += loadRgHigh(wait, check, dataList);
        }
        if (!isCurrent) {
            if ((changed & CS_GATE_CH) != 0) {
                setDac(BoardDacs.CHAN_CSGATE_1, csGate[0]);
                setDac(BoardDacs.CHAN_CSGATE_2, csGate[1]);
                setDac(BoardDacs.CHAN_CSGATE_3, csGate[2]);
                loadDac(0);
                count += 3;
                rebDevc.readAllAdcs(dataList);
            }
        }
        changed = 0;
        return count;
    }


    /**
     *  Clears (zeroes) all DAC values on the REB.
     *
     *  @param  dataList  A list to receive read ADC values
     *  @throws  RaftException
     */
    public void clear(List<AdcData> dataList) throws RaftException
    {
        clear(DEFAULT_CLEAR_WAIT, dataList);
    }


    /**
     *  Clears (zeroes) all DAC values on the REB.
     *
     *  @param  wait  The time to wait after loading a DAC (ms)
     *  @param  dataList  A list to receive read ADC values
     *  @throws  RaftException
     */
    public void clear(int wait, List<AdcData> dataList) throws RaftException
    {
        clearRgLow(wait);
        clearRgHigh(wait);
        clearPclkLow(wait);
        clearPclkHigh(wait);
        clearSclkLow(wait);
        clearSclkHigh(wait);
        if (!isCurrent) {
            setDac(BoardDacs.CHAN_CSGATE_1, 0);
            setDac(BoardDacs.CHAN_CSGATE_2, 0);
            setDac(BoardDacs.CHAN_CSGATE_3, 0);
            loadDac(0);
        }
        changed = -1;
        rebDevc.readAllAdcs(dataList);
    }


    /**
     *  Checks that 6 clock Dacs are within tolerance of zero (eg. OFF)
     *  or within tolerance of the set value (eg ON)
     *
     *  @param dataList  List to hold measure values for publication
     *  @param fcause    StringBuilder array to return error messages
     *  @return state    int indicating OFF = 0; ON = 1; error = -1
     *  
     */
    public int getClockDacsPowerState(List<AdcData> dataList, StringBuilder fcause) throws RaftException
    {
        int expectedCnt = 6;
        int offCnt = 0;
        int onCnt = 0;
        double[] origPower = rebDevc.readPowerAdcs(dataList);
        double[] slowAdcs = rebDevc.readSlowAdcs(dataList);
        double value;

        // Serial Lower
        value = slowAdcs[REBDevice.ADC_SCLK_L];
        if (Math.abs(value) < sclkLowZeroErr) {
            offCnt += 1;
            fcause.append(String.format("\n  %s %s value %.2f is in OFF range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_SCLK_L),
                                        value));
        } else if (Math.abs(value + sclkLowOff - sclkLowP) < sclkLowTol) {
            onCnt += 1;
            fcause.append(String.format("\n  %s %s value %.2f is in ON range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_SCLK_L),
                                        value));
        } else {
            fcause.append(String.format("\n  %s %s value %.2f is not in OFF or ON allowed range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_SCLK_L),
                                        value));
        }
        // Serial Upper
        // (has extra 0.8 added to tolerance on "uppers" to account for
        // sequencer output bit high effect on clocked high rails
        // only needed when seq bit is high but in this context CCD is still off
        value = slowAdcs[REBDevice.ADC_SCLK_U];
        if (Math.abs(value) < sclkHighZeroErr + 0.8) {
            offCnt += 1;
            fcause.append(String.format("\n  %s %s value %.2f is in OFF range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_SCLK_U),
                                        value));
        } else if (Math.abs(value + sclkHighOff - sclkHighP) < sclkHighTol) {
            onCnt += 1;
            fcause.append(String.format("\n  %s %s value %.2f is in ON range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_SCLK_U),
                                        value));
        } else {
            fcause.append(String.format("\n  %s %s value %.2f is not in OFF or ON allowed range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_SCLK_U),
                                        value));
        }

        // Parallel Lower
        value = slowAdcs[REBDevice.ADC_PCLK_L];
        if (Math.abs(value) < pclkLowZeroErr) {
            offCnt += 1;
            fcause.append(String.format("\n  %s %s value %.2f is in OFF range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_PCLK_L),
                                        value));
        } else if (Math.abs(value + pclkLowOff - pclkLowP) < pclkLowTol) {
            onCnt += 1;
            fcause.append(String.format("\n  %s %s value %.2f is in ON range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_PCLK_L),
                                        value));
        } else {
            fcause.append(String.format("\n  %s %s value %.2f is not in OFF or ON allowed range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_PCLK_L),
                                        value));
        }

        // Parallel Upper
        value = slowAdcs[REBDevice.ADC_PCLK_U];
        if (!isCorner) {
            if (Math.abs(value) < pclkHighZeroErr + 0.8) {
                offCnt += 1;
                fcause.append(String.format("\n  %s %s value %.2f is in OFF range",
                                            rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_PCLK_U),
                                            value));
            } else if (Math.abs(value + pclkHighOff - pclkHighP) < pclkHighTol) {
                onCnt += 1;
                fcause.append(String.format("\n  %s %s value %.2f is in ON range",
                                            rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_PCLK_U),
                                            value));
            } else {
                fcause.append(String.format("\n  %s %s value %.2f is not in OFF or ON allowed range",
                                            rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_PCLK_U),
                                            value));
            }
        } else { // skip this test
            expectedCnt = 5;
        }

        // Reset Lower
        value = slowAdcs[REBDevice.ADC_RG_L];
        if (Math.abs(value) < rgLowZeroErr) {
            offCnt += 1;
            fcause.append(String.format("\n  %s %s value %.2f is in OFF range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_RG_L),
                                        value));
        } else if (Math.abs(value + rgLowOff - rgLowP) < rgLowTol) {
            onCnt += 1;
            fcause.append(String.format("\n  %s %s value %.2f is in ON range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_RG_L),
                                        value));
        } else {
            fcause.append(String.format("\n  %s %s value %.2f is not in OFF or ON allowed range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_RG_L),
                                        value));
        }

        // Reset Upper
        value = slowAdcs[REBDevice.ADC_RG_U];
        if (Math.abs(value) < rgHighZeroErr + 0.8) {
            offCnt += 1;
            fcause.append(String.format("\n  %s %s value %.2f is in OFF range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_RG_U),
                                        value));
        } else if (Math.abs(value + rgHighOff - rgHighP) < rgHighTol) {
            onCnt += 1;
            fcause.append(String.format("\n  %s %s value %.2f is in ON range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_RG_U),
                                        value));
        } else {
            fcause.append(String.format("\n  %s %s value %.2f is not in OFF or ON allowed range",
                                        rebDevc.getName(), REBDevice.getAdcName(REBDevice.ADC_RG_U),
                                        value));
        }
        fcause.append(String.format("\n  %s: getClockDacsPowerState() found %d ON, %d OFF of 6",
                    rebDevc.getName(), onCnt, offCnt));
        if (offCnt == expectedCnt) {
            return 0;
        } else if (onCnt == expectedCnt) {
            return 1;
        } else {
            return -1;
        }
    }


    /**
     *  Tests for clock lines shorts.
     *
     *  @param  dataList  A list to receive read ADC values
     *  @param  fcause  A StringBuilder to append any failure cause to
     *  @return  Whether the test failed
     *  @throws  RaftException
     */
    public boolean testShorts(List<AdcData> dataList, StringBuilder fcause) throws RaftException
    {
        boolean failed = false;
        failed =  testShorts(BoardDacs.CHAN_SCLK_H, BoardDacs.CHAN_SCLK_L,
                             REBDevice.ADC_SCLK_U, REBDevice.ADC_SCLK_L, 
                             sclkHighTestV, sclkHighValueErr, sclkHighZeroErr,
                             sclkLowTestV, sclkLowValueErr, sclkLowZeroErr, dataList, fcause);
        //
        failed |= testShorts(BoardDacs.CHAN_PCLK_H, BoardDacs.CHAN_PCLK_L,
                             REBDevice.ADC_PCLK_U, REBDevice.ADC_PCLK_L,
                             pclkHighTestV, pclkHighValueErr, pclkHighZeroErr,
                             pclkLowTestV, pclkLowValueErr, pclkLowZeroErr, dataList, fcause);
        //
        failed |= testShorts(BoardDacs.CHAN_RG_H, BoardDacs.CHAN_RG_L,
                             REBDevice.ADC_RG_U, REBDevice.ADC_RG_L,
                             rgHighTestV, rgHighValueErr, rgHighZeroErr,
                             rgLowTestV, rgLowValueErr, rgLowZeroErr, dataList, fcause);
        return failed;
    }


    /**
     *  Tests for clock lines shorts.
     *
     *  @param  dacUp   The upper rail DAC channel number
     *  @param  dacLo   The lower rail DAC channel number
     *  @param  adcUp   The upper rail ADC number
     *  @param  adcLo   The lower rail ADC number
     *  @param  adcUpTestV The upper rail test voltage
     *  @param  adcUpValueErr The upper fail value tolerance
     *  @param  adcUpZeroErr The tolerance for !adcUp @zero voltages
     *  @param  adcLoTestV The upper rail test voltage
     *  @param  adcLoValueErr The upper fail value tolerance
     *  @param  adcLoZeroErr The tolerance for !adcLo @zero voltages
     *  @param  dataList  A list to receive read slow ADC values
     *  @param  fcause  A StringBuilder to append any failure cause to
     *  @return  Whether check failed
     *  @throws  RaftException
     */
    private boolean testShorts(int dacUp, int dacLo, int adcUp, int adcLo,
                               double adcUpTestV, double adcUpValueErr, double adcUpZeroErr, 
                               double adcLoTestV, double adcLoValueErr, double adcLoZeroErr, 
                               List<AdcData> dataList, StringBuilder fcause) throws RaftException
    {
        // Record original ADC values
        rebDevc.readSlowAdcs(dataList);
        double[] origPower = rebDevc.readPowerAdcs(dataList);
        long tstamp;

        // log the event
        tstamp = rebDevc.getTime();
        LOG.fine(String.format("%d initial reading done", tstamp));

        // Set clock rail voltages to zero; should already be there
        setDac(dacUp, 0);
        setDac(dacLo, 0);
        loadDac(DEFAULT_CLEAR_WAIT);

        // Record initial baseline ADC values
        rebDevc.readAllAdcs(dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.fine(String.format("%d set both rails to 0", tstamp));
        LOG.fine(String.format("%d %s,%s set to 0.0 with %d ms wait", tstamp, REBDevice.getAdcName(adcLo),
                               REBDevice.getAdcName(adcUp), DEFAULT_CLEAR_WAIT));

        // give it a little extra time to settle
        try {
           Thread.sleep(DEFAULT_CLEAR_WAIT);
        }
        catch (InterruptedException e) {}

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.fine(String.format("%d settled for %d ms", tstamp, DEFAULT_CLEAR_WAIT));

        // Record "lower" ADC values, this is the baseline. Rails should be at 0V with tolerance
        double[] lowerPower = rebDevc.readPowerAdcs(dataList);
        double[] lowerAdcs = rebDevc.readSlowAdcs(dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.fine(String.format("%d recorded lower values", tstamp));

        // Set both clock rail voltages to test value
        setDac(dacUp, (int)(railConvH * adcUpTestV));
        setDac(dacLo, (int)(railConvL * adcLoTestV));
        loadDac(DEFAULT_LOAD_WAIT);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.fine(String.format("%d %s set to %.2f with %d ms wait", tstamp, REBDevice.getAdcName(adcLo),
                               adcLoTestV, DEFAULT_LOAD_WAIT));
        LOG.fine(String.format("%d %s set to %.2f with %d ms wait", tstamp, REBDevice.getAdcName(adcUp),
                               adcUpTestV, DEFAULT_LOAD_WAIT));

        // Record "upper" ADC values, this is at test voltage, Rails should be at test volts with no extra current
        double[] upperPower = rebDevc.readPowerAdcs(dataList);
        double[] upperAdcs = rebDevc.readSlowAdcs(dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.fine(String.format("%d recorded at test voltage", tstamp));

        // give it a nice pedestal
        try {
           Thread.sleep(DEFAULT_CLEAR_WAIT);
        }
        catch (InterruptedException e) {}

        // Record pedestal ADC values
        rebDevc.readAllAdcs(dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.fine(String.format("%d pedestal reading", tstamp));

        // Set clock rail voltages back to zero
        setDac(dacUp, 0);
        setDac(dacLo, 0);
        loadDac(DEFAULT_CLEAR_WAIT);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.fine(String.format("%d set both rails to 0", tstamp));
        LOG.fine(String.format("%d %s clock rails set to 0.0 with %d ms wait", tstamp,
                               REBDevice.getAdcName(adcLo), DEFAULT_CLEAR_WAIT));

        // Record final ADC values
        rebDevc.readAllAdcs(dataList);

        // log the time and what happened
        tstamp = rebDevc.getTime();
        LOG.fine(String.format("%d final reading", tstamp));

        // Check the values
        return checkValues(adcUp, adcLo, 
                           adcUpTestV, adcUpValueErr, adcUpZeroErr,
                           adcLoTestV, adcLoValueErr, adcLoZeroErr,
                           lowerAdcs, upperAdcs, origPower,
                           lowerPower, upperPower, fcause);
    }


    /**
     *  Checks the voltages and currents.
     * 
     *  @param  adcUp   The upper rail ADC number
     *  @param  adcLo   The lower rail ADC number
     *  @param  adcUpTestV Upper test voltage
     *  @param  adcUpValueErr Tolerance for test voltage
     *  @param  adcUpZeroErr  Tolerance for nominal zero
     *  @param  adcLoTestV Upper test voltage
     *  @param  adcLoValueErr Tolerance for test voltage
     *  @param  adcLoZeroErr  Tolerance for nominal zero
     *  @param  baseAdcs  The slow ADC values at baseline
     *  @param  measAdcs  The slow ADC values at test voltage
     *  @param  origPower  The original board power ADC values
     *  @param  basePower  The board power ADC values at baseline
     *  @param  measPower  The board power ADC values at test voltage
     *  @param  fcause  A StringBuilder to append any failure cause to
     *  @return  Whether check failed
     */
    private boolean checkValues(int adcUp, int adcLo, 
                                double adcUpTestV, double adcUpValueErr, double adcUpZeroErr, 
                                double adcLoTestV, double adcLoValueErr, double adcLoZeroErr, 
                                double[] baseAdcs, double[] measAdcs, double[] origPower,  
                                double[] basePower, double[] measPower, StringBuilder fcause)
   {
        boolean failed = false;
        // first check measured and base voltage behavior
        for (int ad = 0; ad < baseAdcs.length; ad++) {
            double basevalue = baseAdcs[ad];
            double measvalue = measAdcs[ad];
            if (ad == adcUp) {  
                if (Math.abs(basevalue) > adcUpZeroErr) {  // the tested adcs baseline should be at zero
                    fcause.append(String.format("%s %s baseline value (%.2f) is out of bounds (%.2f)\n",
                                                rebDevc.getName(), REBDevice.getAdcName(ad), basevalue, adcUpZeroErr));
                    failed = true;
                }
                if (Math.abs(measvalue - adcUpTestV) > adcUpValueErr) {  // tested measured value should be correct
                    fcause.append(String.format("%s %s measured value (%.2f) is out of bounds (%.2f +/- %.2f)\n",
                                                rebDevc.getName(), REBDevice.getAdcName(ad), measvalue, adcUpTestV, adcUpValueErr));
                    failed = true;
                }
            }
            else if (ad == adcLo) {
                if (Math.abs(basevalue) > adcLoZeroErr) {  // the tested adcs baseline should be at zero
                    fcause.append(String.format("%s %s baseline value (%.2f) is out of bounds (%.2f)\n",
                                                rebDevc.getName(), REBDevice.getAdcName(ad), basevalue, adcLoZeroErr));
                    failed = true;
                }
                if (Math.abs(measvalue - adcLoTestV) > adcLoValueErr) {  // tested measured value should be correct
                    fcause.append(String.format("%s %s measured value (%.2f) is out of bounds (%.2f +/- %.2f)\n",
                                                rebDevc.getName(), REBDevice.getAdcName(ad), measvalue, adcLoTestV, adcLoValueErr));
                    failed = true;
                }
            }
            else { // should be no change between the base and measured values for adcs not under test
                if (Math.abs(measvalue - basevalue) > adcLoZeroErr) {
                    fcause.append(String.format("%s %s change of (%.2f) is out of bounds (+/- %.2f)\n",
                                                rebDevc.getName(), REBDevice.getAdcName(ad), (measvalue - basevalue), CHANGE_ERROR));
                    failed = true;
                }
            }
        }
        // check for current increases in CLK[HL]I's0
        if (basePower[PowerAdcs.ADC_CLKH_CURRENT] > origPower[PowerAdcs.ADC_CLKH_CURRENT] + CLK_SHORTS_LIMIT) {
            fcause.append(String.format("%s ClkHi base current increased excessively from %.2f mA to %.2f mA while testing %s\n",
                                        rebDevc.getName(), 1000.0 * origPower[PowerAdcs.ADC_CLKH_CURRENT],
                                        1000.0 * basePower[PowerAdcs.ADC_CLKH_CURRENT], REBDevice.getAdcName(adcUp)));
            failed = true;
        }
        if (measPower[PowerAdcs.ADC_CLKH_CURRENT] > basePower[PowerAdcs.ADC_CLKH_CURRENT] + CLK_SHORTS_LIMIT) {
            fcause.append(String.format("%s ClkHi meas current increased excessively from %.2f mA to %.2f mA while testing %s\n",
                                        rebDevc.getName(), 1000.0 * basePower[PowerAdcs.ADC_CLKH_CURRENT],
                                        1000.0 * measPower[PowerAdcs.ADC_CLKH_CURRENT], REBDevice.getAdcName(adcUp)));
            failed = true;
        }
        if (basePower[PowerAdcs.ADC_CLKL_CURRENT] > origPower[PowerAdcs.ADC_CLKL_CURRENT] + CLK_SHORTS_LIMIT) {
            fcause.append(String.format("%s ClkLo base current increased excessively from %.2f mA to %.2f mA while testing %s\n",
                                        rebDevc.getName(), 1000.0 * origPower[PowerAdcs.ADC_CLKL_CURRENT],
                                        1000.0 * basePower[PowerAdcs.ADC_CLKL_CURRENT], REBDevice.getAdcName(adcLo)));
            failed = true;
        }
        if (measPower[PowerAdcs.ADC_CLKL_CURRENT] > basePower[PowerAdcs.ADC_CLKL_CURRENT] + CLK_SHORTS_LIMIT) {
            fcause.append(String.format("%s ClkLo meas current increased excessively from %.2f mA to %.2f mA while testing %s\n",
                                        rebDevc.getName(), 1000.0 * basePower[PowerAdcs.ADC_CLKL_CURRENT],
                                        1000.0 * measPower[PowerAdcs.ADC_CLKL_CURRENT], REBDevice.getAdcName(adcLo)));
            failed = true;
        }
        return failed;
    }


    /**
     *  Loads the low parallel clock rail DACs.
     */
    private int loadPclkLow(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_PCLK_L, pclkLowP >= 0.0 ? (int)(railConvL * pclkLowP) : 0);
            setDac(BoardDacs.CHAN_PCLK_L_SH, pclkLowP <= 0.0 ? (int)(-railShiftConvL * pclkLowP) : 0);
            count++;
            loadDac(wait);
            RaftException ex = checkAdc(REBDevice.ADC_PCLK_L, dataList, check, pclkLowP, pclkLowOff, pclkLowTol);
            if (ex != null) {
                clearPclkLow(0);
                throw ex;
            }
        }
        else {
            setDac(BoardDacs.CHAN_PCLK_L, pclkLow);
            if (isCurrent) {
                setDac(BoardDacs.CHAN_PCLK_L_SH, pclkLowSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }


    /**
     *  Loads the high parallel clock rail DACs.
     */
    private int loadPclkHigh(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        int count = 1;
        if (!raw) { // new
            if (!isCorner) {
                setDac(BoardDacs.CHAN_PCLK_H, pclkHighP >= 0.0 ? (int)(railConvH * pclkHighP) : 0);
                loadDac(wait);
            } // else do nothing for corner raft
            RaftException ex = checkAdc(REBDevice.ADC_PCLK_U, dataList, check, pclkHighP, pclkHighOff, pclkHighTol);
            if (ex != null) {
                clearPclkHigh(0);
                throw ex;
            }
        }

        else {
            setDac(BoardDacs.CHAN_PCLK_H, pclkHigh);
            if (isCorner) {
                setDac(BoardDacs.CHAN_PCLK_H_SH, pclkHighSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }


    /**
     *  Loads the low serial clock rail DACs.
     */
    private int loadSclkLow(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_SCLK_L, sclkLowP >= 0.0 ? (int)(railConvL * sclkLowP) : 0);
            setDac(BoardDacs.CHAN_SCLK_L_SH, sclkLowP <= 0.0 ? (int)(-railShiftConvL * sclkLowP) : 0);
            count++;
            loadDac(wait);
            RaftException ex = checkAdc(REBDevice.ADC_SCLK_L, dataList, check, sclkLowP, sclkLowOff, sclkLowTol);
            if (ex != null) {
                clearSclkLow(0);
                throw ex;
            }
        }
        else {
            setDac(BoardDacs.CHAN_SCLK_L, sclkLow);
            if (isCurrent) {
                setDac(BoardDacs.CHAN_SCLK_L_SH, sclkLowSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }


    /**
     *  Loads the high serial clock rail DACs.
     */
    private int loadSclkHigh(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_SCLK_H, sclkHighP >= 0.0 ? (int)(railConvH * sclkHighP) : 0);
            if (isCorner) {
                setDac(BoardDacs.CHAN_SCLK_H_SH, sclkHighP <= 0.0 ? (int)(-railShiftConvH * sclkHighP) : 0);
                count++;
            }
            loadDac(wait);
            RaftException ex = checkAdc(REBDevice.ADC_SCLK_U, dataList, check, sclkHighP, sclkHighOff, sclkHighTol);
            if (ex != null) {
                clearSclkHigh(0);
                throw ex;
            }
        }
        else {
            setDac(BoardDacs.CHAN_SCLK_H, sclkHigh);
            if (isCorner) {
                setDac(BoardDacs.CHAN_SCLK_H_SH, sclkHighSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }


    /**
     *  Loads the low reset gate rail DACs.
     */
    private int loadRgLow(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_RG_L, rgLowP >= 0.0 ? (int)(railConvL * rgLowP) : 0);
            setDac(BoardDacs.CHAN_RG_L_SH, rgLowP <= 0.0 ? (int)(-railShiftConvL * rgLowP) : 0);
            count++;
            loadDac(wait);
            RaftException ex = checkAdc(REBDevice.ADC_RG_L, dataList, check, rgLowP, rgLowOff, rgLowTol);
            if (ex != null) {
                clearRgLow(0);
                throw ex;
            }
        }
        else {
            setDac(BoardDacs.CHAN_RG_L, rgLow);
            if (isCurrent) {
                setDac(BoardDacs.CHAN_RG_L_SH, rgLowSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }


    /**
     *  Loads the high reset gate rail DACs.
     */
    private int loadRgHigh(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_RG_H, rgHighP >= 0.0 ? (int)(railConvH * rgHighP) : 0);
            if (isCorner) {
                setDac(BoardDacs.CHAN_RG_H_SH, rgHighP <= 0.0 ? (int)(-railShiftConvH * rgHighP) : 0);
                count++;
            }
            loadDac(wait);
            RaftException ex = checkAdc(REBDevice.ADC_RG_U, dataList, check, rgHighP, rgHighOff, rgHighTol);
            if (ex != null) {
                clearRgHigh(0);
                throw ex;
            }
        }
        else {
            setDac(BoardDacs.CHAN_RG_H, rgHigh);
            if (isCorner) {
                setDac(BoardDacs.CHAN_RG_H_SH, rgHighSh);
                count++;
            }
            loadDac(wait);
        }
        return count;
    }


    /**
     *  Obtains an ADC value and checks it.
     *
     *  @param adc       index of Clock rail channel in SlowAdcs being checked
     *  @param dataList  for storing measured values
     *  @param check     record and perform check (true) otherwise just record values
     *  @param setPoint  voltage set point for channel being checked
     *  @param adcOff    added to measured voltage before comparing to tolerance
     *  @param adcTol    Tolerance for comparison
     *  @return          null or a RaftException that the caller can throw
     */
    private RaftException checkAdc(int adc, List<AdcData> dataList, boolean check,
                                   double setPoint, double adcOff, double adcTol) throws RaftException
    {
        StringBuilder exString = new StringBuilder();
        boolean failed = false;
        if (!check && dataList == null) return null;
        double[] adcValues = rebDevc.readSlowAdcs(dataList);  // ~18ms
        double[] curPower = rebDevc.readPowerAdcs(dataList);  // ~33ms
        // checking for (3) failure modes to allow caller to throw the returned exception
        if (Math.abs(adcValues[adc] + adcOff - setPoint) > adcTol) {  // checks set value achieved
            exString.append(String.format("\n%s %s out of range: setpt=%.2f, reading: %.2f, offset: %.2f, tol: %.2f",
                            rebDevc.getName(), REBDevice.getAdcName(adc),    
                             setPoint, adcValues[adc], adcOff, adcTol));
            failed = true;
        }
        if (curPower[PowerAdcs.ADC_CLKH_CURRENT] > clkhIMax) {  // this is RXY/RebN/DAC parameter
            exString.append(String.format("\n%s %s caused overcurrent on ClkHI: measured %.3f, limit: %.3f",
                        rebDevc.getName(), REBDevice.getAdcName(adc),    
                        curPower[PowerAdcs.ADC_CLKH_CURRENT], clkhIMax));
            failed = true;
        }
        if (curPower[PowerAdcs.ADC_CLKL_CURRENT] > clklIMax) {  // this is RXY/RebN/DAC parameter
            exString.append(String.format("\n%s %s caused overcurrent on ClkLI: measured %.3f, limit: %.3f",
                        rebDevc.getName(), REBDevice.getAdcName(adc),    
                        curPower[PowerAdcs.ADC_CLKL_CURRENT], clklIMax));
            failed = true;
        }
        if (failed) {
            if (check) {
                return new RaftException(exString.toString());
            }
            else {
                LOG.warn(exString.toString());
            }
        }
        return null;
    }


    /**
     *  Checks whether a read value is okay
     */
    private static boolean isValueOkay(double read, double set)
    {
        return Math.abs(read - set) <= DAC_TOLERANCE;
    }


    /**
     *  Clears the low parallel clock rail DACs.
     */
    private void clearPclkLow(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_PCLK_L, 0);
        if (isCurrent) {
            setDac(BoardDacs.CHAN_PCLK_L_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Clears the high parallel clock rail DACs.
     */
    private void clearPclkHigh(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_PCLK_H, 0);
        if (isCorner) {
            setDac(BoardDacs.CHAN_PCLK_H_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Clears the low serial clock rail DACs.
     */
    private void clearSclkLow(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_SCLK_L, 0);
        if (isCurrent) {
            setDac(BoardDacs.CHAN_SCLK_L_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Clears the high serial clock rail DACs.
     */
    private void clearSclkHigh(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_SCLK_H, 0);
        if (isCorner) {
            setDac(BoardDacs.CHAN_SCLK_H_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Clears the low reset gate rail DACs.
     */
    private void clearRgLow(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_RG_L, 0);
        if (isCurrent) {
            setDac(BoardDacs.CHAN_RG_L_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Clears the high reset gate rail DACs.
     */
    private void clearRgHigh(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_RG_H, 0);
        if (isCorner) {
            setDac(BoardDacs.CHAN_RG_H_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Sets a DAC value.
     *
     *  @param  chan   The channel number
     *  @param  value  The value to set
     *  @throws  RaftException
     */
    private void setDac(int chan, int value) throws RaftException
    {
        testOnline();
        try {
            dac.set(chan, Math.max(0, Math.min(value, REBDevice.DAC_LIMIT)));
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Loads DAC values.
     *
     *  @param  wait  The time to wait after loading the DAC (ms)
     *  @throws  RaftException
     */
    private void loadDac(int wait) throws RaftException
    {
        testOnline();
        try {
            dac.loadGlobal();
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
        try {
            Thread.sleep(wait);
        }
        catch (InterruptedException e) {}
    }

}
