package org.lsst.ccs.subsystem.rafts;

import java.io.IOException;
import java.io.Serializable;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.daq.utilities.FitsHeaderKeywordData;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.rafts.alerts.RaftAlert;
import org.lsst.ccs.subsystem.rafts.data.CcdType;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.data.SequencerData;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo;
import org.lsst.ccs.subsystem.rafts.states.SequencerMainState;
import org.lsst.ccs.subsystem.rafts.states.SequencerState;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.CCDTypeUtils;
import org.lsst.ccs.utilities.ccd.E2VCCDType;
import org.lsst.ccs.utilities.ccd.Geometry;
import org.lsst.ccs.utilities.ccd.ITLCCDType;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.image.FitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.image.ReadOutParameters;

/**
 *  A Class containing all the Rafts level commands.
 * 
 *  @author  Owen Saxton
 */
public class RaftsCommands {
       
    // CCD types
    private static final Map<String, CcdType> ccdTypes = new HashMap<>();
    static {
        for (CcdType type : CcdType.values()) {
            ccdTypes.put(type.name().toLowerCase(), type);
        }
    }

    private final List<REBDevice> rebDevices;
    private final List<SequencerProc> sequencers = new ArrayList<>();
    private final List<ImageProc> imageProcs = new ArrayList<>();
    private final GlobalProc globalProc;
    private final TempControl tempCtrl;
    private final Subsystem subsys;
    private CCDType ccdType = new ITLCCDType();
    private Geometry geometry;
    private String sequencerStartMain;
    private ReadOutParameters readOutParameters;
    private ImageNameService imageNameService;

    /**
     * Public constructor for the Raft level commands object.
     * 
     * @param  subsys     The associated subsystem
     * @param  rebDevices The list of REBDevices for this raft.
     * @param  globalProc The global proc for the raft
     * @param  geometry   The geometry for this raft
     * @param  tempCtrl   The temperature controller
     */
    public RaftsCommands(Subsystem subsys, List<REBDevice> rebDevices,
                         GlobalProc globalProc, Geometry geometry,
                         TempControl tempCtrl) {
        this(subsys, rebDevices, globalProc, geometry, tempCtrl, null);
    }
    /**
     * Public constructor for the Raft level commands object.
     * 
     * @param subsys     The associated subsystem
     * @param rebDevices The list of REBDevices for this raft.
     * @param globalProc The global proc for the raft
     * @param geometry   The geometry for this raft
     * @param tempCtrl   The temperature controller
     * @param imageNameService The image naming service
     */    
    public RaftsCommands(Subsystem subsys, List<REBDevice> rebDevices,
                         GlobalProc globalProc, Geometry geometry,
                         TempControl tempCtrl, ImageNameService imageNameService) {
        this.subsys = subsys;
        this.rebDevices = rebDevices;
        this.globalProc = globalProc;
        this.geometry = geometry;
        this.tempCtrl = tempCtrl;
        this.imageNameService = imageNameService;
        
        // Initialize lists of sequencers and image processors
        for (REBDevice devc : rebDevices) {
            sequencers.add(devc.getSequencer());
            imageProcs.add(devc.getImageProc());
        }
        
        updateImageProcGeometry();
    }


    /**
     *  Sets the CCD type.
     *
     *  @param  type    The (enumerated) CCD type
     */
    public void setCcdType(CcdType type)
    {
        for (REBDevice reb : rebDevices) {
            reb.setCcdType(type);
        }

        ccdType = type == CcdType.E2V ? new E2VCCDType() : type == CcdType.ITL ? new ITLCCDType() : null;
        if ( ccdType == null ) return;
        
        geometry = CCDTypeUtils.changeCCDTypeForGeometry(geometry, ccdType);
        updateImageProcGeometry();
        if (readOutParameters == null) {
            readOutParameters = new ReadOutParameters(ccdType);
        } else {
            readOutParameters.setCCDType(ccdType);
        }
    }


    /**
     * This is a temporary fix to propagate the new child Geometry information
     * when the CCDType of the whole Raft is changed. See LSSTCCSRAFTS-164.
     * 
     * The Geometry is set up to handle only the three REBs belonging to a
     * single raft.  Each REB is identified to the geometry by the two low-order
     * bits of its ID.  The higher-order bits of the REB ID designate which raft
     * the REB belongs to and are masked out for the purpose of obtaining the
     * correct child geometry.
     * 
     * We will have to review the whole geometry implementation when DAQ 1.9 will
     * be available.
     */
    private void updateImageProcGeometry() {
        for (REBDevice rebDev : rebDevices) {
            rebDev.getImageProc().setRebGeometry((Reb) geometry.getChild(rebDev.getRebNumber(), 0));
        }
    }


    /**
     *  Sets the CCD type.
     *
     *  @param  type    The CCD type
     */
    public void setCcdType(String type)
    {
        CcdType typeE = ccdTypes.get(type);
        if ( typeE == null ) {
            throw new IllegalArgumentException("Illegal CCD type: "+type+". Must be one of : "+ccdTypes.keySet());
        }
        setCcdType(typeE);
    }


    /**
     *  Changes the CCD type.
     *
     *  @param  type    The CCD type
     */
    public void changeCcdType(String type)
    {
        setCcdType(type);
    }


    /**
     *  Sets the control temperature
     *
     *  @param  temp  The temperature to set
     */
    @Command(type=Command.CommandType.ACTION, description="Set the control temperature")
    public void setControlTemp(@Argument(name="temp", description="Temperature value")
                               double temp)
    {
        tempCtrl.setTemperature(temp);
    }


    /**
     *  Gets the control temperature
     *
     *  @return  The set temperature
     */
    @Command(type=Command.CommandType.QUERY, description="Get the control temperature")
    public double getControlTemp()
    {
        return tempCtrl.getTemperature();
    }


    /**
     *  Starts temperature control
     */
     @Command(type=Command.CommandType.ACTION, description="Start temperature control")
    public void startTempControl()
    {
        tempCtrl.startLoop(0.0);
    }


    /**
     *  Stops temperature control
     */
    @Command(type=Command.CommandType.ACTION, description="Stop temperature control")
    public void stopTempControl()
    {
        tempCtrl.stopLoop();
    }


    /**
     *  Gets the temperature control state
     *
     *  @return  Whether temperature control is active
     */
    @Command(type=Command.CommandType.QUERY, description="Get temperature control state")
    public boolean isTempControlActive()
    {
        return tempCtrl.isLoopActive();
    }


    /**
     *  Gets the CCD type
     *
     *  @return  The CCD type
     */
    @Command(type=Command.CommandType.QUERY, description="Get the CCD type")
    public String getCcdType()
    {        
        return ccdType != null ? ccdType.getName() : "Not defined";
    }


    /**
     *  Gets the list of REB device names.
     *
     *  @return  The list of REB device names
     */
    @Command(type=Command.CommandType.QUERY, description="Get the list of REB devices")
    public List<String> getREBDeviceNames()
    {
        List<String> devList = new ArrayList<>();
        for (Device devc : rebDevices) {
            devList.add(devc.getName());
        }
        return devList;
    }


    /**
     *  Gets the list of REB device names.
     *
     *  @return  The list of REB device names
     */
    @Deprecated
    @Command(type=Command.CommandType.QUERY, description="Get the list of REB devices")
    public List<String> getREBDevices()
    {
        List<String> devList = new ArrayList<>();
        for (Device devc : rebDevices) {
            devList.add(devc.getName());
        }
        return devList;
    }


    /**
     *  Gets the list of REB hardware versions.
     *
     *  @return  The list of REB hardware versions
     */
    @Command(type=Command.CommandType.QUERY, description="Get the list of REB hardware versions")
    public List<Integer> getREBHwVersions()
    {
        List<Integer> devList = new ArrayList<>();
        for (REBDevice devc : rebDevices) {
            devList.add(devc.getHwVersion());
        }
        return devList;
    }


    /**
     *  Gets the list of REB IDs.
     *
     *  @return  The list of REB IDs
     */
    @Command(type=Command.CommandType.QUERY, description="Get the list of REB IDs")
    public List<Integer> getREBIds()
    {
        List<Integer> devList = new ArrayList<>();
        for (REBDevice devc : rebDevices) {
            devList.add(devc.getId());
        }
        return devList;
    }


    /**
     *  Gets the list of REB serial numbers.
     *
     *  @return  The list of REB serial numbers
     */
    @Command(type=Command.CommandType.QUERY, description="Get the list of REB serial numbers")
    public List<Long> getREBSerialNumbers()
    {
        List<Long> devList = new ArrayList<>();
        for (REBDevice devc : rebDevices) {
            devList.add(devc.getSerialNumber());
        }
        return devList;
    }


    /**
     *  Resets the front ends.
     *
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Reset the front end")
    public void resetFirmware() throws Exception
    {
        for (REBDevice reb : rebDevices) {
            reb.resetFrontEnd();
        }
    }


    /**
     *  Gets the data sources from all REBs.
     *
     *  @return  A list of the encoded data sources (CCD or simulation) for each REB
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.QUERY, description="Get the data source")
    public List<Integer> getDataSource() throws Exception
    {
        List<Integer> source = new ArrayList<>();
        for (REBDevice reb : rebDevices) {
            source.add(reb.getSequencer().getDataSource());
        }
        return source;
    }


    /**
     *  Sets the data sources in all REBs.
     *
     *  @param  value  The encoded data source: CCD or simulation
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Set the data sources")
    public void setDataSource(@Argument(name="value", description="The data source value")
                              int value) throws Exception
    {
        for (REBDevice reb : rebDevices) {
            reb.getSequencer().setDataSource(value);
        }
    }


    /**
     *  Gets the scan mode enabled states.
     *
     *  @return  A list of the scan mode states for each REB
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.QUERY, description="Get the scan mode enabled states")
    public List<Boolean> isScanEnabled() throws Exception
    {
        List<Boolean> enabled = new ArrayList<>();
        for (REBDevice reb : rebDevices) {
            boolean enab = reb.getSequencer().isScanEnabled();
            reb.getImageProc().enableScan(enab);
            enabled.add(enab);
        }
        return enabled;
    }


    /**
     *  Sets the scan mode enables.
     *
     *  @param  value  The enabled state to set, true or false
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Set the scan mode enables")
    public void enableScan(@Argument(name="value", description="The scan mode enabled state")
                           boolean value) throws Exception
    {
        for (REBDevice reb : rebDevices) {
            reb.getSequencer().enableScan(value);
            reb.getImageProc().enableScan(value);
        }
    }


    /**
     *  Resets the scan modes.
     *
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Reset the scan modes")
    public void resetScan() throws Exception
    {
        for (REBDevice reb : rebDevices) {
            reb.getSequencer().resetScan();
        }
    }


    /**
     *  Powers on/off the REB CCDs.
     *
     *  @param  on  Whether to power on (true) or off (false)
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Power on/off the CCDs")
    public void powerCCDs(@Argument(description="Whether to power on") boolean on) throws Exception
    {
        for (REBDevice reb : rebDevices) {
            reb.powerCCDs(on);
        }
    }


    /**
     *  Loads DACs with their configured values.
     *
     *  @param  all  If true, all data are loaded; otherwise only changes
     *  @return  The number of DACs loaded
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Load DACs")
    public int loadDacs(@Argument(name="all", description="If true, load all DACs")
                        boolean all) throws Exception
    {
        int count = 0;
        for (REBDevice reb : rebDevices) {
            count += reb.loadDacs(all);
        }
        return count;
    }


    /**
     *  Clears DAC values.
     *
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Clear DACs")
    public void clearDacs() throws Exception
    {
        for (REBDevice reb : rebDevices) {
            reb.clearDacs();
        }
    }


    /**
     *  Loads bias DACs with their configured values.
     *
     *  @param  all  If true, all data are loaded; otherwise only changes
     *  @return  The number of DACs loaded
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Load bias DACs")
    public int loadBiasDacs(@Argument(name="all", description="If true, load all bias DACs")
                            boolean all) throws Exception
    {
        int count = 0;
        for (REBDevice reb : rebDevices) {
            count += reb.loadBiasDacs(all);
        }
        return count;
    }
    

    /**
     *  Clears bias DAC values.
     *
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Clear bias DACs")
    public void clearBiasDacs() throws Exception
    {
        for (REBDevice reb : rebDevices) {
            reb.clearBiasDacs();
        }
    }
    

    /**
     *  Loads ASPICs with their configured values.
     *
     *  @param  all  If true, all data are loaded; otherwise only changes
     *  @return  The number of ASPICs loaded
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Load ASPICs")
    public int loadAspics(@Argument(name="all", description="If true, load all ASPICs")
                          boolean all) throws Exception
    {
        int count = 0;
        for (REBDevice reb : rebDevices) {
            count += reb.loadAspics(all);
        }
        return count;
    }

    /**
     *  Checks all loaded ASPIC values against the configuration.
     *
     *  @return  A list of mismatch masks, one per REB
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.QUERY, description="Check all ASPICs")
    public List<Integer> checkAspics() throws Exception
    {
        List<Integer> masks = new ArrayList<>();
        for (REBDevice reb : rebDevices) {
            masks.add(reb.checkAspics());
        }
        return masks;
    }

    /**
     *  Resets ASPICs.
     *
     *  @return  The number of ASPICs loaded
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Reset ASPICs")
    public int resetAspics() throws Exception
    {
        int count = 0;
        for (REBDevice reb : rebDevices) {
            count += reb.resetAspics();
        }
        return count;
    }

    /**
     *  Sets the time base to the current system time.
     *
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Set the REBs' system time")
    public void setTime() throws Exception
    {
        for (REBDevice rebd : rebDevices) {
            rebd.setTime();
        }
    }


    /**
     *  Gets all time base values.
     *
     *  @return  The map of REB name vs. time value
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.QUERY, description="Get the REBs' system times")
    public Map<String, Long> getTimes() throws Exception
    {
        Map<String, Long> times = new HashMap<>();
        for (REBDevice rebd : rebDevices) {
            times.put(rebd.getName(), rebd.getTime());
        }
        return times;
    }

    /**
     *  Gets all time base values.
     *
     *  @return  The list of (REB name, time values) pairs
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.QUERY, description="Get the REBs' system times")
    public List getTime() throws Exception
    {
        List times = new ArrayList();
        for (REBDevice rebd : rebDevices) {
            times.add(rebd.getName());
            times.add(rebd.getTime());
        }
        return times;
    }
     
    /**
     *  Gets the back bias on states.
     *
     *  @return  A list of the back bias on states for each REB
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.QUERY, description="Get the back bias on states")
    public List<Boolean> isBackBiasOn() throws Exception
    {
        List<Boolean> state = new ArrayList<>();
        for (REBDevice reb : rebDevices) {
            state.add(reb.isBackBiasOn());
        }
        return state;
    }

    /**
     *  Sets the back bias on states.
     *
     *  @param  value  The on state to set, true or false
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Set the back bias on states")
    public void setBackBias(@Argument(name="value", description="The back bias on state")
                            boolean value) throws Exception
    {
        for (REBDevice reb : rebDevices) {
            reb.setBackBias(value);
        }
    }
    
    /**
     *  Set the value of Primary Header Keyword
     *
     *  @param  name The name of the header keyword
     *  @param  value The value of the header keyword
     */
    @Command(type=Command.CommandType.QUERY, description="Set a primary header keyword value")
    public void setPrimaryHeaderKeyword(String name, Serializable value) {
        setHeaderKeyword("primary", name, value);
    }

    /**
     *  Set the values of Primary Header Keywords
     *
     *  @param  headersMap The map of header keyword name, value pairs to be set
     */
    @Command(type=Command.CommandType.QUERY, description="Set primary header keyword values")
    public void setPrimaryHeaderKeyword(Map<String, Serializable> headersMap) {
        setHeaderKeyword("primary", headersMap);
    }

    /**
     *  Set the value of a named HDU Header Keyword
     *
     *  @param  hduName The name of the HDU on which to set the header keyword
     *  @param  name The name of the header keyword
     *  @param  value The value of the header keyword
     */
    @Command(type=Command.CommandType.QUERY, description="Set a header keyword value on a named HDU")
    public void setHeaderKeyword(String hduName, String name, Serializable value) {
        Map<String,Serializable> map = new HashMap<>();
        map.put(name, value);
        setHeaderKeyword(hduName, map);
    }

    /**
     *  Set the value of a header keyword for a given named HDU.
     *
     *  @param  hduName The name of the HDU on which to set the header keyword
     *  @param  name The name of the header keyword
     *  @param  value The value of the header keyword
     *  @param  sticky If this value is sticky
     */
    public void setHeaderKeyword(String hduName, String name, Serializable value, boolean sticky) {
        FitsHeaderKeywordData data = new FitsHeaderKeywordData();
        data.addHeaderKeywordValue(hduName,name, value, sticky);
        subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());
    }

    /**
     *  Set the value of a header keyword for a given named HDU.
     *
     *  @param  ccdId The Id of the CCD for which this keyword is published
     *  @param  name The name of the header keyword
     *  @param  value The value of the header keyword
     *  @param  sticky If this value is sticky
     */
    public void setHeaderKeywordForCCD(String ccdId, String name, Serializable value, boolean sticky) {
        FitsHeaderKeywordData data = new FitsHeaderKeywordData(ccdId);
        data.addHeaderKeywordValue(ccdId,name, value, sticky);
        subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());
    }

    /**
     *  Set the values of a named HDU Header Keywords
     *
     *  @param  hduName The name of the HDU on which to set the header keyword
     *  @param  headersMap The map of header keyword name, value pairs to be set
     */
    @Command(type=Command.CommandType.QUERY, description="Set header keyword values on a named HDU")
    public void setHeaderKeyword(String hduName, Map<String, Serializable> headersMap) {
        FitsHeaderKeywordData data = new FitsHeaderKeywordData();
        for ( Entry<String,Serializable> e : headersMap.entrySet() ) {
            data.addHeaderKeywordValue(hduName, e.getKey(), e.getValue(), true);
        }
        subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());
    }

    /**
     *  Loads all sequencers from a file.
     *
     *  @param  fileName  The name of the file containing sequencer instructions
     *  @return  A list of the number of slices expected to be read from each REB
     *  @throws  Exception
     *  @throws  IOException
     */
    @Command(type=Command.CommandType.ACTION, description="Load all sequencers from a file")
    public List<Integer> loadSequencer(@Argument(name="filename", description="Sequencer file name")
                                       String fileName)
        throws Exception, IOException
    {
        List<Integer> sliceCount = new ArrayList<>();
        for (int j = 0; j < sequencers.size(); j++) {
            SequencerProc seq = sequencers.get(j);
            // TODO: We use deprecated method to maintain existing behaviour
            int nSlice = seq.loadSequencerFromBootstrapOrFile(fileName);
            String shortFileName = Paths.get(fileName).getFileName().toString();

            FitsHeaderKeywordData data = new FitsHeaderKeywordData(imageProcs.get(j).getRebGeometry().getUniqueId());
            data.addHeaderKeywordValue("primary", "SequencerFileName", shortFileName, true);
            data.addHeaderKeywordValue("primary", "ExposureTime", new Float(0), true);
            data.addHeaderKeywordValue("primary", "ShutterDelay", new Float(0), true);
            subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());
                        
            sliceCount.add(nSlice);
        }
        updateReadoutParameters();        
        return sliceCount;
    }

        /**
     *  Loads all sequencers from a precompiled sequencer model.
     *
     *  @param  model  The compiled sequencer
     *  @param shortFileName The filename to be used in the FITS header
     *  @return  A list of the number of slices expected to be read from each REB
     *  @throws  Exception
     *  @throws  IOException
     */
    @Command(type=Command.CommandType.ACTION, description="Load all sequencers from a file")
    public List<Integer> loadCompiledSequencer(
            @Argument(name="model", description="Compiled sequencer") FPGA2Model model,
            @Argument(name="filename", description="Sequencer file name") String shortFileName)
        throws Exception, IOException
    {
        List<Integer> sliceCount = new ArrayList<>();
        for (int j = 0; j < sequencers.size(); j++) {
            SequencerProc seq = sequencers.get(j);
            int nSlice = seq.loadSequencer(model);

            FitsHeaderKeywordData data = new FitsHeaderKeywordData(imageProcs.get(j).getRebGeometry().getUniqueId());
            data.addHeaderKeywordValue("primary", "SequencerFileName", shortFileName, true);
            // TODO: Understand why these are here -- and ideally remove them
            data.addHeaderKeywordValue("primary", "ExposureTime", new Float(0), true);
            data.addHeaderKeywordValue("primary", "ShutterDelay", new Float(0), true);
            subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());
                        
            sliceCount.add(nSlice);
        }
        updateReadoutParameters();        
        return sliceCount;
    }
    
    /**
     *  Gets the list of all main program maps.
     *
     *  @return  The list of name-to-address maps of main programs
     */
    @Command(type=Command.CommandType.QUERY, description="Get the maps of mains")
    public List<Map<String, Integer>> getSequencerMainMap()
    {
        List<Map<String, Integer>> list = new ArrayList<>();
        for (SequencerProc seq : sequencers) {
            list.add(seq.getMainMap());
        }
        return list;
    }

    @Deprecated
    @Command(type=Command.CommandType.QUERY, description="Get the maps of mains")
    public List<Map<String, Integer>> getMainMap()
    {
        return getSequencerMainMap();
    }


    /**
     *  Gets the list of all subroutine maps.
     *
     *  @return  The list of name-to-address maps of subroutines
     */
    @Command(type=Command.CommandType.QUERY, description="Get the maps of subroutines")
    public List<Map<String, Integer>> getSequencerSubroutineMap()
    {
        List<Map<String, Integer>> list = new ArrayList<>();
        for (SequencerProc seq : sequencers) {
            list.add(seq.getSubroutineMap());
        }
        return list;
    }

    @Deprecated
    @Command(type=Command.CommandType.QUERY, description="Get the maps of subroutines")
    public List<Map<String, Integer>> getSubroutineMap()
    {
        return getSequencerSubroutineMap();
    }


    /**
     *  Gets the list of all function maps.
     *
     *  @return  The name-to-number map of functions
     */
    @Command(type=Command.CommandType.QUERY, description="Get the maps of sequencer functions")
    public List<Map<String, Integer>> getSequencerFunctionMap()
    {
        List<Map<String, Integer>> list = new ArrayList<>();
        for (SequencerProc seq : sequencers) {
            list.add(seq.getFunctionMap());
        }
        return list;
    }

    @Deprecated
    @Command(type=Command.CommandType.QUERY, description="Get the maps of functions")
    public List<Map<String, Integer>> getFunctionMap()
    {
        return getSequencerFunctionMap();
    }


    /**
     *  Gets the list of all pointer information maps.
     *
     *  @return  The list of name-to-type-and-address maps of pointers
     */
    @Command(type=Command.CommandType.QUERY, description="Get the maps of sequencer pointers")
    public List<Map<String, PointerInfo>> getSequencerPointers()
    {
        List<Map<String, PointerInfo>> list = new ArrayList<>();
        for (SequencerProc seq : sequencers) {
            list.add(seq.getPointers());
        }
        return list;
    }

    @Deprecated
    @Command(type=Command.CommandType.QUERY, description="Get the maps of pointers")
    public List<Map<String, PointerInfo>> getPointers()
    {
        return getSequencerPointers();
    }


    /**
     *  Gets the list of all values for a parameter.
     *
     *  @param  name  The parameter name
     *  @return  The list of values, one per REB
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.QUERY, description="Get sequencer parameter values")
    public List<Integer> getSequencerParameter(@Argument(name="name", description="The parameter name")
                                               String name) throws Exception
    {
        List<Integer> list = new ArrayList<>();
        for (SequencerProc seq : sequencers) {
            list.add(seq.getParameter(name));
        }
        return list;
    }

    @Deprecated
    @Command(type=Command.CommandType.QUERY, description="Get parameter values")
    public List<Integer> getParameter(@Argument(name="name", description="The parameter name")
                                      String name) throws Exception
    {
        return getSequencerParameter(name);
    }


    /**
     *  Sets a sequencer parameter value in all REBs.
     *
     *  @param  name   The parameter name
     *  @param  value  The value to set
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Set a sequencer parameter value")
    public void setSequencerParameter(@Argument(name="name", description="The parameter name")
                                      String name,
                                      @Argument(name="value", description="The value to set")
                                      int value) throws Exception
    {
        for (SequencerProc seq : sequencers) {
            seq.setParameter(name, value);
        }
        if ( SequencerData.imageRegs.containsValue(name) ) {
            updateReadoutParameters();
        }        
    }

    @Deprecated
    @Command(type=Command.CommandType.ACTION, description="Set a sequencer parameter value")
    public void setParameter(@Argument(name="name", description="The parameter name")
                             String name,
                             @Argument(name="value", description="The value to set")
                             int value) throws Exception
    {
        setSequencerParameter(name, value);
    }


    // note for future implementation
    // there will be a table of 8 active mains 8 DAQ registers
    // add a lockSequencerStart / unlockSequencerStart that will lock
    // this start in one of the registers.
    // setSequencerStart :
    //  -- if loaded : start with this index (0-7)
    //  -- if not, choose the unlocked LRU, unload, reload this one instead,
    //     sync/lookatme, and start with the new index.
    /**
     *  Sets the start address for all sequencers.
     *
     *  @param  mainName  The name of the main sequencer program to start at
     *  @return  A list of the number of slices expected to be read from each REB by this main
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Set all sequencer start addresses")
    public List<Integer> setSequencerStart(@Argument(name="mainName", description="Main routine name")
                                           String mainName) throws Exception
    {
        List<Integer> sliceCount = new ArrayList<>();
        for (SequencerProc seq : sequencers) {
            sliceCount.add(seq.setStart(mainName));
        }

        SequencerMainState state = SequencerMainState.OTHER;
        String loweredMainName = mainName.toLowerCase();
        if (loweredMainName.contains("clear")) {
            state = SequencerMainState.CLEAR;
        } else if (loweredMainName.contains("expose")) {
            state = SequencerMainState.INTEGRATE;
        } else if (loweredMainName.contains("acquire") || loweredMainName.contains("bias")) {
            state = SequencerMainState.READOUT;
        } 
        if ( !subsys.isInState(state) ) {
            subsys.updateAgentState(state);
        }
        
        sequencerStartMain = mainName;
        return sliceCount;
    }

    /**
     * Get the name of the Main routine that will be invoked when the sequencer
     * starts.
     * @return The name of the main routine.
     */
    public String getSequencerStart() {
        return sequencerStartMain;
    }
    
    @Deprecated
    @Command(type=Command.CommandType.ACTION, description="Set all sequencer start addresses")
    public List<Integer> setStart(@Argument(name="mainName", description="Main routine name")
                                  String mainName) throws Exception
    {
        return setSequencerStart(mainName);
    }


    /**
     *  Starts the sequencers.
     *
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Start the sequencers")
    public void startSequencer() throws Exception
    {
        for (SequencerProc seq : sequencers) {
            seq.resetError();
            seq.startSequencer();
        }
        subsys.updateAgentState(SequencerState.RUNNING);
    }


    /**
     *  Sends a sequencer stop command.
     *
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Send a sequencer stop command")
    public void stopSequencer() throws Exception
    {
        for (SequencerProc seq : sequencers) {
            seq.sendStop();
        }
    }


    /**
     *  Sends a sequencer step command.
     *
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Send a sequencer step command")
    public void stepSequencer() throws Exception
    {
        for (SequencerProc seq : sequencers) {
            seq.sendStep();
        }
    }
    

    /**
     *  Gets the running state of all sequencers.
     *
     *  @return  A list of the sequencer running states for each REB
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.QUERY, description="Get the running states of the sequencers")
    public List<Boolean> isSequencerRunning() throws Exception
    {
        List<Boolean> running = new ArrayList<>();
        for (SequencerProc seq : sequencers) {
            running.add(seq.isRunning());
        }

        return running;
    }


    /**
     *  Waits for the sequencers to be done.
     *
     *  @param  timeout  The maximum tiime to wait (ms)
     *  @throws  Exception
     */
    @Command(type=Command.CommandType.ACTION, description="Start the sequencers", autoAck = false)
    public void waitSequencerDone(@Argument(name="timeout", description="Timeout (ms)")
                                  int timeout) throws Exception
    {
        subsys.sendAck(Duration.ofMillis(timeout));
        long tmo = timeout;
        long endTime = System.currentTimeMillis() + tmo;
        for (SequencerProc seq : sequencers) {
            seq.waitDone((int) tmo);
            tmo = endTime - System.currentTimeMillis();
        }
    }

    /**
     * Starts image acquisition, using image name obtained by ImageNameService,
     * @return The allocated image name
     * @throws REBException If image acquisition fails, or no ImageNameService has been configured.
     * @see @{link ImageNameService} 
     */
    @Command(type=Command.CommandType.ACTION, description="Start image acquisition using standard LSST image name")
    public ImageName acquireLSSTImage() throws REBException {
        if (imageNameService != null) {
            ImageName imageName = imageNameService.getImageName();
            acquireImage(imageName.toString());
            return imageName;
        } else { 
            throw new REBException("ImageNameService not configured");
        }
    }

    /**
     *  Starts image acquisition.
     *
     *  @param imageName Optional imageName. If absent subsystem makes up its own imageName,
     *  @throws REBException
     */
    @Command(type=Command.CommandType.ACTION, description="Start image acquisition")
    public void acquireImage(@Argument(name="imageName", defaultValue = "") String  imageName) throws REBException
    {
        for (SequencerProc seq : sequencers) {
            seq.resetError();
        }
        for (ImageProc img : imageProcs) {
            img.initImageWait();
        }
        if (globalProc != null) {
            /* TODO: Send integration parameters event. This requires 
             * Getting the registers addresses (from globelProc)
             * Reading the register values
             * Constructing readout parameters using createReadOutParametersFromRegistersMetadata
             * Generating the event
             * Sending the event
             */
            
            if (imageName == null || imageName.trim().length() == 0) {
                globalProc.acquireImage();
            } else {
                globalProc.acquireImage(imageName);
            }
        }
    }

    /**
     *  Waits for image to arrive
     * 
     *  @param  timeout  The timeout period (msec)
     *  @return  The number of REB images received
     */
    @Command(type=Command.CommandType.ACTION, description="Wait for images to arrive", autoAck = false)
    public int waitForImage(int timeout)
    {
        subsys.sendAck(Duration.ofMillis(timeout));
        long tmo = timeout;
        long endTime = System.currentTimeMillis() + tmo;
        int count = 0;
        for (ImageProc img : imageProcs) {
            count += img.waitForImage((int) tmo);  // Okay if tmo <= 0
            tmo = endTime - System.currentTimeMillis();
        }
        return count;
    }


    /**
     *  Saves the current image data as one or more sets of raw bytes.
     *
     *  @param  dirName  The name of the directory where the image file is to be saved.
     *  @return  A list containing the names of the saved files
     *  @throws  RaftException
     *  @throws  IOException
     */
    @Command(type=Command.CommandType.ACTION, description="Save the current image as raw bytes")
    public List<String> saveImage(@Argument(name="dirname", description="The directory to use",
                                            defaultValue="")
                                  String dirName) throws RaftException, IOException
    {
        List<String> names = new ArrayList<>();
        for (ImageProc imp : imageProcs) {
            names.add(imp.saveImage(dirName));
        }
        return names;
    }


    /**
     *  Saves the current image data as one or more FITS files.
     *
     *  @param  dirName  The name of the directory where the FITS file is to be saved.
     *  @return  A list containing the names of the saved files
     *  @throws  RaftException
     *  @throws  IOException
     */
    @Command(type=Command.CommandType.ACTION, description="Save the current image as a FITS file")
    public List<String> saveFitsImage(@Argument(name="dirname", description="The directory to use",
                                                defaultValue="")
                                      String dirName)
        throws IOException, RaftException
    {
        return saveFitsImage(dirName, null);
    }
    public List<String> saveFitsImage(String dirName, FitsHeaderMetadataProvider provider)
        throws IOException, RaftException
    {
        List<String> names = new ArrayList<>();
        for (ImageProc imp : imageProcs) {
            names.addAll(imp.saveFitsImage(dirName, provider));
        }

        return names;
    }


    /**
     *  Sets whether image data values are inverted
     * 
     *  @param  invert  Whether data values are to be inverted
     */
    @Command(type=Command.CommandType.ACTION, description="Set whether image data values are inverted")
    public void setImageDataInversion(@Argument(name="invert", description="Whether to invert")
                                      boolean invert)
    {
        for (ImageProc imp : imageProcs) {
            imp.setDataInversion(invert);
        }
    }


    /**
     *  Sets the default image directory.
     *
     *  @param  dirName  The directory name
     */
    @Command(type=Command.CommandType.ACTION, description="Set the default image directory")
    public void setDefaultImageDirectory(@Argument(name="dirname", description="The directory name")
                                         String dirName)
    {
        for (ImageProc imp : imageProcs) {
            imp.setDefaultImageDirectory(dirName);
        }
    }


    /**
     *  Sets the FITS image file name pattern.
     *
     *  @param  pattern  The file name pattern to set
     */
    @Command(type=Command.CommandType.ACTION, description="Set the FITS file name pattern")
    public void setFitsFileNamePattern(@Argument(name="pattern", description="The file name pattern")
                                       String pattern)
    {
        for (ImageProc imp : imageProcs) {
            imp.setFitsFileNamePattern(pattern);
        }
    }


    /**
     *  Sets the raw image data file name pattern.
     *
     *  @param  pattern  The file name pattern to set
     */
    @Command(type=Command.CommandType.ACTION, description="Set the raw image data file name pattern")
    public void setImageDataFileNamePattern(@Argument(name="pattern", description="The file name pattern")
                                            String pattern)
    {
        for (ImageProc imp : imageProcs) {
            imp.setImageDataFileNamePattern(pattern);
        }
    }


    /**
     *  Sets the FITS image exposure time.
     *
     *  @param  expTime  The exposure time (secs)
     */
    @Command(type=Command.CommandType.ACTION, description="Set the FITS exposure time")
    public void setExposureTime(@Argument(name="exptime", description="The exposure time")
                                double expTime)
    {
        for (ImageProc imp : imageProcs) {
            FitsHeaderKeywordData data = new FitsHeaderKeywordData(imp.getRebGeometry().getUniqueId());
            data.addHeaderKeywordValue("primary", "ExposureTime", expTime, true);
            subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());
        }
    }
    
    
    /**
     *  Set a pattern property for resolving file name and output directory.
     *
     *  @param  propertyName The name of the property
     *  @param  propertyValue The value of the property
     * 
     */
    @Command(type=Command.CommandType.ACTION, description="Set a pattern property for resolving file names and output directory")
    public void setPatternProperty(String propertyName, String propertyValue)
    {
        for (ImageProc imp : imageProcs) {
            imp.setPatternProperty(propertyName, propertyValue);
        }
    }
    
    /**
     * We check the sequencer state.
     * If any sequencer is running the state is RUNNING.
     * If any sequencer throws an exception when checked we go into ERROR.
     * Otherwise we are in IDLE state.
     * 
     * TO-DO: should we go in ENGINEERING_FAULT if we are in ERROR?
     */
    public void checkSequencerState() {
        Enum state = SequencerState.IDLE;
        for ( SequencerProc seq : sequencers ) {
            try {
                if (seq.isRunning()) {
                    state = SequencerState.RUNNING;
                    break;
                }
            } catch (REBException ex) {
                state = SequencerState.ERROR;
                break;
            }
        }
        if ( ! subsys.isInState(state) ) {
            subsys.updateAgentState(state);
        }
    }
    
    /**
     * Set an external sequence number.
     * @param sequenceNumber the external sequence number.
     */
    public void setSequenceNumber(int sequenceNumber) {
        for (ImageProc imageProc : imageProcs) {
            imageProc.setExternalSequenceNumber(sequenceNumber);
        }
    }
    
    @Command(simulation=true) 
    public void generateAlert(
            @Argument(name = "raftAlert") RaftAlert raftAlert, @Argument(name = "alertState") AlertState state) {
        subsys.getAgentService(AlertService.class).raiseAlert(raftAlert.getAlert(), state, "Simulated Raft Alert");
    }
    
    
    private void updateReadoutParameters() throws REBException {
        if (globalProc == null) return;
        Set<Integer> types = new HashSet<>();
        for (int j = 0; j < sequencers.size(); j++) {
            int type = rebDevices.get(j).getRebType();
            if (!types.contains(type)) {
                globalProc.setRegisters(type, sequencers.get(j).getImageRegisters());
                types.add(type);
            }
        }        		
    }

}
