package org.lsst.ccs.subsystem.rafts;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.config.BulkValidationException;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.reb.BoardDacs;
import org.lsst.ccs.drivers.reb.PowerAdcs;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.rafts.config.BiasDACS;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.CcdType;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.REBDevice.AdcData;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements control of CCD biases and CS gate.
 *
 *  @author Owen Saxton
 */
public class BiasControl extends Control implements ConfigurationBulkChangeHandler {

    /**
     *  Constants.
     */
    public static final String
        GD_P      = "gdP",
        OD_P      = "odP",
        OG_P      = "ogP",
        RD_P      = "rdP",
        CS_GATE_P = "csGateP",

        GD        = "gd",
        OD        = "od",
        OG        = "og",
        OG_SH     = "ogSh",
        RD        = "rd",
        CS_GATE   = "csGate";

    private static final int
        GD_CH     = 0x01,
        OD_CH     = 0x02,
        OG_CH     = 0x04,
        OG_SH_CH  = 0x08,
        RD_CH     = 0x10,
        CSGATE_CH = 0x20;

    private static final String[] checkList = {GD, OD, OG, RD};

    private static final int[]
        OD_ADCS = {REBDevice.ADC_OD_0, REBDevice.ADC_OD_1, REBDevice.ADC_OD_2},
        GD_ADCS = {REBDevice.ADC_GD_0, REBDevice.ADC_GD_1, REBDevice.ADC_GD_2},
        RD_ADCS = {REBDevice.ADC_RD_0, REBDevice.ADC_RD_1, REBDevice.ADC_RD_2},
        OG_ADCS = {REBDevice.ADC_OG_0, REBDevice.ADC_OG_1, REBDevice.ADC_OG_2};

    // Test parameters
    private static final double DAC_TOLERANCE = 0.33;
    private static final int DEFAULT_LOAD_WAIT = 160, DEFAULT_CLEAR_WAIT = 50;
    private static final double TEST_VOLTS = 0.5, ZERO_ERROR = 0.2, VALUE_ERROR = 0.2,
            OD_SHORTS_LIMIT = 0.005, CLKH_SHORTS_LIMIT = 0.005, CLKL_SHORTS_LIMIT = 0.005;

    /*
     *  Data fields.
     */
    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    // Supplied configurable fields - bias voltages
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Guard diode", units="volts")
    private double  gdP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Output drain", units="volts")
    private double  odP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Output gate", units="volts")
    private double  ogP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset drain", units="volts")
    private double  rdP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Current source", units="mA")
    private double  csGateP;

    // Configurable bias voltage limits
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Guard diode min", units="volts")
    private double  gdMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Guard diode max", units="volts")
    private double  gdMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Output drain min", units="volts")
    private double  odMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Output drain max", units="volts")
    private double  odMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Output gate min", units="volts")
    private double  ogMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Output gate max", units="volts")
    private double  ogMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Reset drain min", units="volts")
    private double  rdMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Reset drain max", units="volts")
    private double  rdMax;

    // Supplied configurable fields - bias DAC settings
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Guard diode", units="DAC counts")
    private int  gd;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Output drain", units="DAC counts")
    private int  od;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Output gate", units="DAC counts")
    private int  og;
    @ConfigurationParameter(category=REBDevice.RAFTS, description ="Output gate shifted", units="DAC counts")
    private int  ogSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset drain", units="DAC counts")
    private int  rd;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Current source", units="DAC counts")
    private int  csGate;

    // PowerCCDsOn test parameters
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Guard diode tolerance", units="volts")
    private double  gdTol;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Guard diode offset", units="volts")
    private double  gdOff;
    // this would have "path": Rxy/RebI/BiasJ/odTol (I,J) = {0,1,2}
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Output drain tolerance", units="volts")
    private double  odTol;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Output drain offset", units="volts")
    private double  odOff;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Output gate tolerance", units="volts")
    private double  ogTol;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Output gate offset", units="volts")
    private double  ogOff;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Reset drain tolerance", units="volts")
    private double  rdTol;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Reset drain offset", units="volts")
    private double  rdOff;
    // current limit for node: /Rxy/RebI/BiasJ/ODIMax (I,J) = {0,1,2} Can step value for J
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Max ODI value", units="Amps")
    private double  odIMax;

    // Shorts test parameters (N.B. OD is not tested)
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Shorts Test Voltage for GD", units="Volts")
    private double  gdTestVolts;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for GD at 0V", units="Volts")
    private double  gdZeroErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for GD at TestVolts", units="Volts")
    private double  gdValueErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Shorts Test Voltage for OG", units="Volts")
    private double  ogTestVolts;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for OG at 0V", units="Volts")
    private double  ogZeroErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for OG at TestVolts", units="Volts")
    private double  ogValueErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Shorts Test Voltage for RD", units="Volts")
    private double  rdTestVolts;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for RD at 0V", units="Volts")
    private double  rdZeroErr;
    @ConfigurationParameter(category=REBDevice.RAFTS_POWER, description="Tolerance for RD at TestVolts", units="Volts")
    private double  rdValueErr;

    // Set from Groovy file
    private boolean raw = false;

    // Other
    private static final Logger LOG = Logger.getLogger(BiasControl.class.getName());
    private REBDevice rebDevc;
    private boolean cfgValid;
    private int hwVersion;
    private BoardDacs dac;
    private int changed = -1;
    private final Map<String, Double> lowLimitMap = new HashMap<>();
    private final Map<String, Double> highLimitMap = new HashMap<>();
    private double gdConv, odConv, rdConv, ogConv, ogShiftConv, csConv, csOffset;
    private int adcType;


    /**
     *  Constructor.
     */
    public BiasControl()
    {
        for (String pName : checkList) {
            lowLimitMap.put(pName, 0.0);
            highLimitMap.put(pName, 0.0);
        }
    }


    /**
     *  Configures channel description.
     *
     *  @param  mon   The associated monitor object
     *  @param  devc  The associated device object
     */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        rebDevc = (REBDevice)devc;
        dac = rebDevc.getBoardDacs();
    }


    /**
     *  Checks configuration.
     *
     *  @return  Whether configuration is okay
     */
    public boolean checkConfig()
    {
        hwVersion = getHwVersion(dac);
        if (hwVersion == REB.VERSION_UNSUPP) {
            LOG.severe(getName() + " configuration incompatible with the firmware");
            return false;
        }

        try {
            if (hwChan < 0 || hwChan >= dac.getNumStrips()) {
                MonitorLogUtils.reportError(LOG, getName(), "HW channel", hwChan);
            }
        }
        catch (DriverException e) {
            LOG.severe("Error getting number of strips for " + getName());
            return false;
        }
        catch (Exception e) {
            /* Thrown by reportError */
            return false;
        }

        if (!raw) {
            switch (hwVersion) {
                case REB.VERSION_1:
                    odConv = REBDevice.DAC_CONV / (1.0 + 49.9 / 10.0);
                    gdConv = odConv;
                    rdConv = odConv;
                    break;
                case REB.VERSION_2:
                    odConv = REBDevice.DAC_CONV / (1.0 + 62.0 / 9.9);  // Zener diode leakage
                    gdConv = REBDevice.DAC_CONV / (1.0 + 62.0 / 10.0);
                    rdConv = REBDevice.DAC_CONV / (1.0 + 40.2 / 10.0);
                    break;
                default:
                    odConv = REBDevice.DAC_CONV / (1.0 + 62.0 / 10.0);
                    gdConv = odConv;
                    rdConv = REBDevice.DAC_CONV / (1.0 + 40.2 / 10.0);
                    break;
            }
            double ogGain = 10.0 / 10.0;
            double fudge = hwVersion == REB.VERSION_3 ? 1.0 : 1.1;
            ogShiftConv = fudge * REBDevice.DAC_CONV / ogGain;
            ogConv = fudge * REBDevice.DAC_CONV / (1.0 + ogGain);
            csConv = REBDevice.DAC_CONV * 2.0;    // 2 Kohm -> 1 mA = 2 V
            csOffset = -0.95; // current offset in mA
        }
        else {
            cfgValid = true;   // No checking of values
        }
        adcType = hwVersion == REB.VERSION_2 ? REBDevice.TYPE_BIAS_VOLT : REBDevice.TYPE_CR_VOLT;

        return true;
    }


    /**
     *  Gets the raw state.
     *
     *  @return  Whether configuration data are raw ADC values
     */
    public boolean isRaw()
    {
        return raw;
    }


    /**
     *  Gets the hardware version.
     *
     *  @param  dac  The board DAC object
     *  @return  The hardware version
     */
    public static int getHwVersion(BoardDacs dac)
    {
        int version = dac.getVersion();
        switch (version) {
            case BoardDacs.VERSION_WREB1:
            case BoardDacs.VERSION_WREB2:
            case BoardDacs.VERSION_GREB1:
                version = REB.VERSION_1;
                break;
            case BoardDacs.VERSION_REB3:
            case BoardDacs.VERSION_REB5:
                version = REB.VERSION_2;
                break;
            case BoardDacs.VERSION_GREB2:
            case BoardDacs.VERSION_WREB4:
                version = REB.VERSION_3;
                break;
            default:
                version = REB.VERSION_UNSUPP;
                break;
        }

        return version;
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version
     */
    @Deprecated
    public int getHwVersion()
    {
        return hwVersion;
    }


    /**
     *  Validates configuration parameter change.
     *
     *  @param  params  The map of configuration parameter values
     *  @throws  IllegalArgumentException
     */
    @Override
    public void validateBulkChange(Map<String, Object> params) throws IllegalArgumentException
    {
        //System.out.println(getName() + " validateBulkChange called");
        String errMsg = null;
        if (!raw) {
            for (String pName : checkList) {
                double value = (Double)params.get(pName + "P");
                final Double max = (Double)params.get(pName + "Max");
                if (value > max) {
                    errMsg = String.format("%s: %sP (%5.5g) is above high limit (%5.5g)", getName(), pName, value, max);
                    break;
                }
                final Double min = (Double)params.get(pName + "Min");
                if (value < min) {
                    errMsg = String.format("%s: %sP (%5.5g) is below low limit (%5.5g)", getName(), pName, value, min);
                    break;
                }
            }
        }
        if (errMsg != null) {
            LOG.severe("Configuration failure: " + errMsg);
            throw new IllegalArgumentException(errMsg);
        }
        else {
            cfgValid = true;
        }
    }


    /**
     *  Sets the Guard Diode voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGdP(double value)
    {
        gdP = value;
        changed |= GD_CH;
    }


    /**
     *  Gets the Guard Diode voltage.
     *
     *  @return  The value
     */
    @Command(type=CommandType.QUERY, description="Get the Guard Diode voltage")
    public double getGdP()
    {
        return gdP;
    }


    /**
     *  Sets the Guard Diode voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGdMax(double value)
    {
        gdMax = value;
        highLimitMap.put(GD, value);
    }


    /**
     *  Sets the Guard Diode voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGdMin(double value)
    {
        gdMin = value;
        lowLimitMap.put(GD, value);
    }


    /**
     *  Sets the Guard Diode DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGd(int value)
    {
        gd = value;
        changed |= GD_CH;
    }


    /**
     *  Gets the Guard Diode DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Get the Guard Diode DAC value")
    public int getGd()
    {
        return gd;
    }


    /**
     *  Sets the Output Drain voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOdP(double value)
    {
        odP = value;
        changed |= OD_CH;
    }


    /**
     *  Gets the Output Drain voltage.
     *
     *  @return  The value
     */
    @Command(type=CommandType.QUERY, description="Get the readout mode Output Drain voltage")
    public double getOdP()
    {
        return odP;
    }


    /**
     *  Sets the Output Drain voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOdMax(double value)
    {
        odMax = value;
        highLimitMap.put(OD, value);
    }


    /**
     *  Sets the Output Drain voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOdMin(double value)
    {
        odMin = value;
        lowLimitMap.put(OD, value);
    }


    /**
     *  Sets the Output Drain DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOd(int value)
    {
        od = value;
        changed |= OD_CH;
    }


    /**
     *  Gets the Output Drain DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Get the readout mode Output Drain DAC value")
    public int getOd()
    {
        return od;
    }


    /**
     *  Sets the Output Gate voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOgP(double value)
    {
        ogP = value;
        changed |= OG_CH;
    }


    /**
     *  Gets the Output Gate voltage.
     *
     *  @return  The value
     */
    @Command(type=CommandType.QUERY, description="Get the Output Gate voltage")
    public double getOgP()
    {
        return ogP;
    }


    /**
     *  Sets the Output Gate voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOgMax(double value)
    {
        ogMax = value;
        highLimitMap.put(OG, value);
    }


    /**
     *  Sets the Output Gate voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOgMin(double value)
    {
        ogMin = value;
        lowLimitMap.put(OG, value);
    }


    /**
     *  Sets the Output Gate DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOg(int value)
    {
        og = value;
        changed |= OG_CH;
    }


    /**
     *  Gets the Output Gate DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Get the Output Gate DAC value")
    public int getOg()
    {
        return og;
    }


    /**
     *  Sets the Output Gate shift DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOgSh(int value)
    {
        ogSh = value;
        changed |= OG_SH_CH;
    }


    /**
     *  Gets the Output Gate shift DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Get the Output Gate shift DAC value")
    public int getOgSh()
    {
        return ogSh;
    }


    /**
     *  Sets the Reset Drain voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRdP(double value)
    {
        rdP = value;
        changed |= RD_CH;
    }


    /**
     *  Gets the Reset Drain voltage.
     *
     *  @return  The value
     */
    @Command(type=CommandType.QUERY, description="Get the Reset Drain voltage")
    public double getRdP()
    {
        return rdP;
    }


    /**
     *  Sets the Reset Drain voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRdMax(double value)
    {
        rdMax = value;
        highLimitMap.put(RD, value);
    }


    /**
     *  Sets the Reset Drain voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRdMin(double value)
    {
        rdMin = value;
        lowLimitMap.put(RD, value);
    }


    /**
     *  Sets the Reset Drain DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRd(int value)
    {
        rd = value;
        changed |= RD_CH;
    }


    /**
     *  Gets the Reset Drain DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Get the Reset Drain DAC value")
    public int getRd()
    {
        return rd;
    }


    /**
     *  Sets the Current Source gate current.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setCsGateP(double value)
    {
        csGateP = value;
    }


    /**
     *  Gets the Current Source gate current.
     *
     *  @return  The value
     */
    @Command(type=CommandType.QUERY, description="Get the Current Source gate")
    public double getCsGateP()
    {
        return csGateP;
    }


    /**
     *  Sets the Current Source gate DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setCsGate(int value)
    {
        csGate = value;
        changed |= CSGATE_CH;
    }


    /**
     *  Gets the Current Source gate DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Get the Current Source gate")
    public int getCsGate()
    {
        return csGate;
    }


    /**
     *  Sets all the configuration data.
     * 
     *  @param  bias  The bias DAC values
     *  @throws  RaftException
     */
    void setConfig(BiasDACS bias) throws RaftException
    {
        String name = getName();
        if (!raw) {
            double[] values = bias.getPValues();
            sce.submitChange(name, GD_P, values[BiasDACS.GD]);
            sce.submitChange(name, OD_P, values[BiasDACS.OD]);
            sce.submitChange(name, OG_P, values[BiasDACS.OG]);
            sce.submitChange(name, RD_P, values[BiasDACS.RD]);
            sce.submitChange(name, CS_GATE_P, values[BiasDACS.CS_GATE]);
        }
        else {
            int[] values = bias.getValues();
            sce.submitChange(name, GD, values[BiasDACS.GD]);
            sce.submitChange(name, OD, values[BiasDACS.OD]);
            sce.submitChange(name, OG, values[BiasDACS.OG]);
            sce.submitChange(name, OG_SH, values[BiasDACS.OG_SH]);
            sce.submitChange(name, RD, values[BiasDACS.RD]);
            sce.submitChange(name, CS_GATE, values[BiasDACS.CS_GATE]);
        }
        try {
            sce.commitBulkChange();
        }
        catch (BulkValidationException e) {
            sce.dropSubmittedChangesForComponent(name);
            throw new RaftException("Bulk configuration validation failed");
        }
    }


    /**
     *  Gets all the configuration data.
     * 
     *  @return  The bias DAC values
     */
    BiasDACS getConfig()
    {
        BiasDACS bias = new BiasDACS();
        if (!raw) {
            // Note, this returns the values array by reference
            // so changing the values here also changes them inside the
            // bias object
            @SuppressWarnings("MismatchedReadAndWriteOfArray")
            double[] values = bias.getPValues();
            values[BiasDACS.GD] = gdP;
            values[BiasDACS.OD] = odP;
            values[BiasDACS.OG] = ogP;
            values[BiasDACS.RD] = rdP;
            values[BiasDACS.CS_GATE] = csGateP;
        }
        else {
            @SuppressWarnings("MismatchedReadAndWriteOfArray")
            int[] values = bias.getValues();
            values[BiasDACS.GD] = gd;
            values[BiasDACS.OD] = od;
            values[BiasDACS.OG] = og;
            values[BiasDACS.OG_SH] = ogSh;
            values[BiasDACS.RD] = rd;
            values[BiasDACS.CS_GATE] = csGate;
        }
        return bias;
    }


    /**
     *  Loads configuration data to the biases DAC.
     *
     *  @param  check     Whether to check the read-back value
     *  @param  adcDataList  List to receive read-back values of slowAdcs 
     *  @return  The number of DAC channels loaded: 6
     *  @throws  RaftException
     */
    int load(boolean check, List<AdcData> adcDataList) throws RaftException
    {
        return load(DEFAULT_LOAD_WAIT, check, false, adcDataList);
    }


    /**
     *  Loads configuration data to the biases DAC.
     *
     *  @param  check     Whether to check the read-back value
     *  @param  dataList  List to receive read-back values of slowAdcs 
     *  @return  The number of DAC channels loaded: 6
     *  @throws  RaftException
     */
    int load(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        return load(wait, check, false, dataList);
    }


    /**
     *  Loads configuration data to the biases DAC.
     *
     *  @param  wait      The time to wait after each DAC setting (ms)
     *  @param  check     Whether to check the read-back value
     *  @param  dataList  List to receive ADC data
     *  @param  odStep    pass odStep through to loadOd(), true in powerCCDsOn()
     *  @return  The number of DAC channels loaded: 6
     *  @throws  RaftException
     */
    int load(int wait, boolean check, boolean odStep, List<AdcData> dataList) throws RaftException
    {
        if (!cfgValid) {
            throw new RaftException("DAC load error: invalid configuration");
        }
        if (!rebDevc.isSerialNumValid()) {
            throw new RaftException("DAC load error: invalid REB serial number");
        }
        rebDevc.readAllAdcs(dataList);
        if (isItlCcd()) {
            loadOg(wait, check, dataList);
            loadGd(wait, check, dataList);
            loadRd(wait, check, dataList);
            loadOd(wait, check, dataList, odStep);
        }
        else {
            loadRd(wait, check, dataList);
            loadOd(wait, check, dataList, odStep);
            loadGd(wait, check, dataList);
            loadOg(wait, check, dataList);
        }
        loadCsGate(0);
        changed = 0;
        return 6;
    }


    /**
     *  Loads changed configuration data to the biases DAC.
     *
     *  @param  check     Whether to check the read-back value
     *  @param  dataList  List to receive read-back values  
     *  @return  The number of DAC channels loaded
     *  @throws  RaftException
     */
    int loadChanged(boolean check, List<AdcData> dataList) throws RaftException
    {
        return loadChanged(DEFAULT_LOAD_WAIT, check, dataList);
    }


    /**
     *  Loads changed configuration data to the biases DAC.
     *
     *  @param  check     Whether to check the read-back value
     *  @param  dataList  List to receive read-back values  
     *  @return  The number of DAC channels loaded
     *  @throws  RaftException
     */
    int loadChanged(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        return loadChanged(DEFAULT_LOAD_WAIT, check, false, dataList);
    }


    /**
     *  Loads changed configuration data to the biases DAC.
     *
     *  @param  wait      The time to wait after each DAC setting (ms)
     *  @param  check     Whether to check the read-back value
     *  @param  dataList  List to receive read-back values  
     *  @return  The number of DAC channels loaded
     *  @throws  RaftException
     */
    int loadChanged(int wait, boolean check, boolean odStep, List<AdcData> dataList) throws RaftException
    {
        if (!cfgValid) {
            throw new RaftException("DAC load error: invalid configuration");
        }
        if (!rebDevc.isSerialNumValid()) {
            throw new RaftException("DAC load error: invalid REB serial number");
        }
        int count = 0;
        rebDevc.readAllAdcs(dataList);
        if (isItlCcd()) {
            if ((changed & OG_CH) != 0) {
                loadOg(wait, check, dataList);
                count += 2;
            }
            if ((changed & GD_CH) != 0) {
                loadGd(wait, check, dataList);
                count++;
            }
            if ((changed & RD_CH) != 0) {
                loadRd(wait, check, dataList);
                count++;
            }
            if ((changed & OD_CH) != 0) {
                loadOd(wait, check, dataList, odStep);
                count++;
            }
        }
        else {
            if ((changed & RD_CH) != 0) {
                loadRd(wait, check, dataList);
                count++;
            }
            if ((changed & OD_CH) != 0) {
                loadOd(wait, check, dataList, odStep);
                count++;
            }
            if ((changed & GD_CH) != 0) {
                loadGd(wait, check, dataList);
                count++;
            }
            if ((changed & OG_CH) != 0) {
                loadOg(wait, check, dataList);
                count += 2;
            }
        }
        if ((changed & CSGATE_CH) != 0) {
            loadCsGate(0);
            count++;
        }
        changed = 0;
        return count;
    }


    /**
     *  Clears (zeroes) the biases DAC values.
     *
     *  @throws  RaftException
     */
    void clear(List<AdcData> dataList) throws RaftException
    {
        clear(DEFAULT_CLEAR_WAIT, dataList);
    }


    /**
     *  Clears (zeroes) the biases DAC values.
     *
     *  @param  wait  The time to wait after each DAC setting (ms)
     *  @throws  RaftException
     */
    void clear(int wait, List<AdcData> dataList) throws RaftException
    {
        clearCsGate(0);
        if (isItlCcd()) {
            clearOd(wait);
            clearRd(wait);
            clearGd(wait);
            clearOg(wait);
        }
        else {
            clearOg(wait);
            clearGd(wait);
            clearOd(wait);
            clearRd(wait);
        }
        changed = -1;
        rebDevc.readAllAdcs(dataList);
    }


    /**
     *  Tests for biases shorts.
     *
     *  @param  dataList  A list to receive read ADC values
     *  @param  fcause  A StringBuilder to append any failure cause to
     *  @return  Whether the test failed
     *  @throws  RaftException
     *
     *  Note OD not in test: driver doesn't turn on below ~15V
     */
    public boolean testShorts(List<AdcData> dataList, StringBuilder fcause) throws RaftException
    {
        boolean failed = false;
        failed =  testShorts(BoardDacs.CHAN_RD, rdConv, RD_ADCS[hwChan],
                             rdTestVolts, rdValueErr, rdZeroErr, dataList, fcause);
        failed |= testShorts(BoardDacs.CHAN_GD, gdConv, GD_ADCS[hwChan],
                             gdTestVolts, gdValueErr, gdZeroErr, dataList, fcause);
        failed |= testShorts(BoardDacs.CHAN_OG, ogConv, OG_ADCS[hwChan],
                             ogTestVolts, ogValueErr, ogZeroErr, dataList, fcause);
        // read one more time but not needed to evaluate test, just to get good trending points
        rebDevc.readAllAdcs(dataList);
        return failed;
    }


    /**
     *  Tests for biases shorts. 
     *  Apply voltages and take readings: 1 before, 2@testV, 2 @0V,
     *  then perform tests on accumulated data.
     *
     *  @param  dac   The DAC channel number
     *  @param  conv  The conversion factor
     *  @param  adc   The corresponding ADC number
     *  @param  adcTestVolts The test voltage for that adc
     *  @param  adcValueErr  The tolerance for the test voltage
     *  @param  adcZeroErr   The tolerance for change in other biases
     *  @param  dataList  A list to receive read slow ADC values
     *  @param  fcause  A StringBuilder to append any failure cause to
     *  @return  Whether the test failed
     *  @throws  RaftException
     */
    private boolean testShorts(int dac, double conv, int adc, double adcTestVolts, double adcValueErr,
                               double adcZeroErr, List<AdcData> dataList, StringBuilder fcause)
        throws RaftException
    {
        boolean failed = false;
        // read base
        double[] origPower = rebDevc.readPowerAdcs(dataList);
        double[] origAdcs = rebDevc.readSlowAdcs(dataList);
        // load the test voltage and record
        setDac(dac, (int)(conv * adcTestVolts));
        loadDac(DEFAULT_LOAD_WAIT);
        rebDevc.readAllAdcs(dataList);
        // give it a little extra time to settle
        try {
            Thread.sleep(DEFAULT_CLEAR_WAIT);
            }
        catch (InterruptedException e) {}
        // read for test at test voltage
        double[] curPower = rebDevc.readPowerAdcs(dataList);
        double[] currAdcs = rebDevc.readSlowAdcs(dataList);
        setDac(dac, 0);
        loadDac(DEFAULT_CLEAR_WAIT);
        rebDevc.readAllAdcs(dataList);
        // give it a little extra time to settle
        try {
            Thread.sleep(DEFAULT_CLEAR_WAIT);
            }
        catch (InterruptedException e) {}
        // final read at zero voltage
        rebDevc.readAllAdcs(dataList);
        // Perform voltage checks
        for (int ad = 0; ad < currAdcs.length; ad++) {  // cycle through the "SlowAdcs"
            double value = currAdcs[ad] - origAdcs[ad]; // change in voltage from baseline
            if (ad == adc) { // where the test voltage was applied
                if (Math.abs(value - adcTestVolts) > adcValueErr) {
                    fcause.append(String.format("%s %s value (%.2f) is out of bounds (+/-%s)\n",
                                                rebDevc.getName(), REBDevice.getAdcName(ad), value, adcValueErr));
                    failed = true;
                }
            }
            else { // all others should stay at zero
                if (Math.abs(value) > adcZeroErr) {  // note this is ad == adc!, reflects effect of "ad" change
                    fcause.append(String.format("%s %s value (%.2f) is non-zero while testing %s\n",
                                                rebDevc.getName(), REBDevice.getAdcName(ad), value, REBDevice.getAdcName(adc)));
                    failed = true;
                }
            }
        }
        // Perform current increase checks
        if (curPower[PowerAdcs.ADC_OD_CURRENT] > origPower[PowerAdcs.ADC_OD_CURRENT] + OD_SHORTS_LIMIT) {
            fcause.append(String.format("%s OD current increased excessively from %.1f mA to %.1f mA while testing %s\n",
                                        rebDevc.getName(), 1000.0 * origPower[PowerAdcs.ADC_OD_CURRENT],
                                        1000.0 * curPower[PowerAdcs.ADC_OD_CURRENT], REBDevice.getAdcName(adc)));
            failed = true;
        }
        if (curPower[PowerAdcs.ADC_CLKH_CURRENT] > origPower[PowerAdcs.ADC_CLKH_CURRENT] + CLKH_SHORTS_LIMIT) {
            fcause.append(String.format("%s CLKH current increased excessively from %.1f mA to %.1f mA while testing %s\n",
                                        rebDevc.getName(), 1000.0 * origPower[PowerAdcs.ADC_CLKH_CURRENT],
                                        1000.0 * curPower[PowerAdcs.ADC_CLKH_CURRENT], REBDevice.getAdcName(adc)));
            failed = true;
        }
        if (curPower[PowerAdcs.ADC_CLKL_CURRENT] > origPower[PowerAdcs.ADC_CLKL_CURRENT] + CLKL_SHORTS_LIMIT) {
            fcause.append(String.format("%s CLKL current increased excessively from %.1f mA to %.1f mA while testing %s\n",
                                        rebDevc.getName(), 1000.0 * origPower[PowerAdcs.ADC_CLKL_CURRENT],
                                        1000.0 * curPower[PowerAdcs.ADC_CLKL_CURRENT], REBDevice.getAdcName(adc)));
            failed = true;
        }
        return failed;
    }


    /**
     *  Loads the Guard Diode DAC.
     */
    private void loadGd(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        if (!raw) {
            setDac(BoardDacs.CHAN_GD, (int)(gdConv * gdP));
            loadDac(wait);
            RaftException ex = checkAdc(GD_ADCS[hwChan], dataList, check, gdP, gdOff, gdTol);
            if (ex != null) {
                clearGd(0);
                throw ex;
            }
        }
        else {
            setDac(BoardDacs.CHAN_GD, gd);
            loadDac(wait);
        }
    }

    /**
     *  Loads the Output Drain DAC.
     */
    private void loadOd(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        loadOd(wait, check, dataList, false);
    }

    /**
     *  Loads the Output Drain DAC.
     */
    private void loadOd(int wait, boolean check, List<AdcData> dataList, boolean odStep) throws RaftException
    {
        if (!raw) {
            if (odStep) {
                double odStepP = 15.0;  // make configurable
                if (odP > odStepP) { // load OD in 2 steps
                    setDac(BoardDacs.CHAN_OD, (int)(odConv * odStepP));
                    loadDac(wait);
                    RaftException ex =checkAdc(OD_ADCS[hwChan], dataList, check, odStepP, odOff, odTol);
                    if (ex != null) {
                        clearOd(0);
                        throw ex;
                    }
                } else { // no clear needed since nothing loaded
                    throw new RaftException(String.format("odStepP (%.2f) > odP (%.2f), odP setting too low", odStepP, odP));
                }
            }
            setDac(BoardDacs.CHAN_OD, (int)(odConv * odP));
            loadDac(wait);
            RaftException ex =checkAdc(OD_ADCS[hwChan], dataList, check, odP, odOff, odTol);
            if (ex != null) {
                clearOd(0);
                throw ex;
            }
        }
        else {
            setDac(BoardDacs.CHAN_OD, od);
            loadDac(wait);
        }
    }


    /**
     *  Loads the Output Gate DAC.
     */
    private void loadOg(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        if (!raw) {
            setDac(BoardDacs.CHAN_OG, ogP > 0.0 ? (int)(ogConv * ogP) : 0);
            setDac(BoardDacs.CHAN_OG_SH, ogP < 0.0 ? (int)(-ogShiftConv * ogP) : 0);
            loadDac(wait);
            RaftException ex = checkAdc(OG_ADCS[hwChan], dataList, check, ogP, ogOff, ogTol);
            if (ex != null) {
                clearOg(0);
                throw ex;
            }
        }
        else {
            setDac(BoardDacs.CHAN_OG, og);
            setDac(BoardDacs.CHAN_OG_SH, ogSh);
            loadDac(wait);
        }
    }


    /**
     *  Loads the Reset Drain DAC.
     */
    private void loadRd(int wait, boolean check, List<AdcData> dataList) throws RaftException
    {
        if (!raw) {
            setDac(BoardDacs.CHAN_RD, (int)(rdConv * rdP));
            loadDac(wait);
            RaftException ex = checkAdc(RD_ADCS[hwChan], dataList, check, rdP, rdOff, rdTol);
            if (ex != null) {
                clearRd(0);
                throw ex;
            }
        }
        else {
            setDac(BoardDacs.CHAN_RD, rd);
            loadDac(wait);
        }
    }


    /**
     *  Loads the Current Source gate DAC.
     */
    private void loadCsGate(int wait) throws RaftException
    {
        if (!raw) {
            setDac(BoardDacs.CHAN_CSGATE, (int)(csConv * (csGateP + csOffset)));
        }
        else {
            setDac(BoardDacs.CHAN_CSGATE, csGate);
        }
        loadDac(wait);
    }


    /**
     *  Obtains an ADC value and checks voltage and overcurrent
     *
     *  @param adc       index of Bias channel in SlowAdcs being checked
     *  @param dataList  for storing measured values
     *  @param check     record and perform check (true) otherwise just record values
     *  @param setPoint  voltage set point for channel being checked
     *  @param adcOff    added to measured voltage before comparing to tolerance
     *  @param adcTol    Tolerance for comparison
     *  @return          null or a RaftException that the caller can throw
     *  
     */
    private RaftException checkAdc(int adc, List<AdcData> dataList, boolean check,
                                   double setPoint, double adcOff, double adcTol) throws RaftException
    {
        StringBuilder exString = new StringBuilder();
        RaftException re = null;
        boolean failed = false;
        if (!check && dataList == null) return re;
        double[] adcValues = rebDevc.readSlowAdcs(dataList);
        double[] curPower = rebDevc.readPowerAdcs(dataList);
        if (check) {  // checking for (2) failure modes to allow caller to throw the returned exception
            if (Math.abs(adcValues[adc] + adcOff - setPoint) > adcTol) {
                exString.append(String.format("%s %s out of range: setpt=%.2f, reading: %.2f, offset: %.2f, tol: %.2f\n",
                                              rebDevc.getName(), REBDevice.getAdcName(adc), 
                                              setPoint, adcValues[adc], adcOff, adcTol));
                failed = true;
            }
            if (curPower[PowerAdcs.ADC_OD_CURRENT] > odIMax) {  // this is RXY/RebN/BiasM/odIMax so can step M=0,1,2
                exString.append(String.format("%s %s caused overcurrent on ODI: measured %.3f, limit: %.3f",
                                              rebDevc.getName(), REBDevice.getAdcName(adc), 
                                              curPower[PowerAdcs.ADC_OD_CURRENT], odIMax));
                failed = true;
            }
        }
        if (failed) {
            return new RaftException(exString.toString());
        }
        return re;
    }


    /**
     *  Checks whether a read value is okay
     */
    private static boolean isValueOkay(double read, double set)
    {
        return Math.abs(read - set) <= DAC_TOLERANCE;
    }


    /**
     *  Clears the Guard Diode DAC.
     */
    private void clearGd(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_GD, 0);
        loadDac(wait);
    }


    /**
     *  Clears the Output Drain DAC.
     */
    private void clearOd(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_OD, 0);
        loadDac(wait);
    }


    /**
     *  Clears the Output Gate DAC.
     */
    private void clearOg(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_OG, 0);
        setDac(BoardDacs.CHAN_OG_SH, 0);
        loadDac(wait);
    }


    /**
     *  Clears the Reset Drain DAC.
     */
    private void clearRd(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_RD, 0);
        loadDac(wait);
    }


    /**
     *  Clears the Current Source gate DAC.
     */
    private void clearCsGate(int wait) throws RaftException
    {
        setDac(BoardDacs.CHAN_CSGATE, 0);
        loadDac(wait);
    }


    /**
     *  Sets a DAC value.
     *
     *  @param  chan   The channel number
     *  @param  value  The value to set
     *  @throws  RaftException
     */
    private void setDac(int chan, int value) throws RaftException
    {
        testOnline();
        try {
            dac.set(hwChan, chan, Math.max(0, Math.min(value, REBDevice.DAC_LIMIT)));
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
    }


    /**
     *  Loads DAC values.
     *
     *  @param  wait  The time to wait before loading the DAC (ms)
     *  @throws  RaftException
     */
    private void loadDac(int wait) throws RaftException
    {
        testOnline();
        try {
            dac.loadStrip(hwChan);
        }
        catch (REBException e) {
            throw new RaftException(e.getMessage());
        }
        try {
            Thread.sleep(wait);
        }
        catch (InterruptedException e) {}
    }


    /**
     *  Determines whether ITL CCD type
     */
    private boolean isItlCcd()
    {
        return rebDevc.getCcdType() == CcdType.ITL;
    }

}
