package org.lsst.ccs.subsystem.rafts;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements a temperature controller for the rafts system.
 *
 *  @author Owen Saxton
 */
public class TempControl implements HasLifecycle {

    private static final String TIMER_NAME = "temp-loop-timer";

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupName
    private String name;

    private Double  gain;              // loop gain
    private Double  timeConst;         // integration time constant (secs)
    private Double  smoothTime;        // input smoothing time (secs)   
    private Double  maxOutput;         // maximum PID output (watts)
    private Double  awGain;            // anti-windup gain
    private Double  basePower;         // base power input
    private Double  tolerance;         // maximum on-target error (%)
    private double  minOutput = 0.0;   // minimum PID output (watts)
    private double  maxInput = 100.0;  // maximum input
    private double  minInput = -200.0; // minimum input
    private Integer updateTime;        // The update time interval (msec)
    private String[] tempChans;        // Temperature channels to use
    private String[] rebs;             // The REBs with heaters to use

    private PIController pic;          // The PI controller
    private final List<Channel> tempChansL = new ArrayList<>();  // Temperature channels to use
    private final List<REBDevice> rebsL = new ArrayList<>();     // REBs with heaters to use

    private double lastPower;          // The last power value set
    private boolean active;            // True if loop is active
    private double setTemp;            // Temperature set point
    private Logger log;                // The logger


    /**
     *  Start temperature control task.
     */
    @Override
    public void build() {
        pts.scheduleAgentPeriodicTask(new AgentPeriodicTask(TIMER_NAME, () -> {
            iterateLoop();
        }).withPeriod(Duration.ofMillis(updateTime)));
    }


    /**
     *  Initializes the parameters.
     *
     *  @param  log     The subsystem logger
     */
    public void initialize(Logger log)
    {
        ComponentLookup clu = subsys.getComponentLookup();
        if (gain == null) {
            MonitorLogUtils.reportConfigError(log, name, "gain", "is missing");
        }
        if (timeConst == null) {
            MonitorLogUtils.reportConfigError(log, name, "timeConst", "is missing");
        }
        if (smoothTime == null) {
            MonitorLogUtils.reportConfigError(log, name, "smoothTime", "is missing");
        }
        if (maxOutput == null) {
            MonitorLogUtils.reportConfigError(log, name, "maxOutput", "is missing");
        }
        if (awGain == null) {
            MonitorLogUtils.reportConfigError(log, name, "awGain", "is missing");
        }
        if (basePower == null) {
            MonitorLogUtils.reportConfigError(log, name, "basePower", "is missing");
        }
        if (tolerance == null) {
            MonitorLogUtils.reportConfigError(log, name, "tolerance", "is missing");
        }
        if (updateTime == null) {
            MonitorLogUtils.reportConfigError(log, name, "updateTime", "is missing");
        }
        if (tempChans == null) {
            MonitorLogUtils.reportConfigError(log, name, "tempChans", "is missing");
        }
        else {
            for (String cName : tempChans) {
                Object cmpt = clu.getComponentByName(cName);
                if (cmpt instanceof Channel) {
                    tempChansL.add((Channel)cmpt);
                }
                else {
                    MonitorLogUtils.reportConfigError(log, name, "tempChans", "contains non-Channel item");
                }
            }
        }
        if (rebs == null) {
            MonitorLogUtils.reportConfigError(log, name, "rebs", "is missing");
        }
        else {
            for (String cName : rebs) {
                Object cmpt = clu.getComponentByName(cName);
                if (cmpt instanceof REBDevice) {
                    rebsL.add((REBDevice)cmpt);
                }
                else {
                    MonitorLogUtils.reportConfigError(log, name, "rebs", "contains non-REBDevice item");
                }
            }
        }
        this.log = log;
        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(basePower);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
    }


    /**
     *  Sets the heater power.
     *
     *  @param  power  The power to set
     */
    void setPower(double power)
    {
        for (REBDevice reb : rebsL) {
            try {
                reb.setHeaterPower(0, power / rebs.length);
            }
            catch (Exception e) {
                log.error("Error setting " + reb.getName() + " heater: " + e.getMessage());
            }
        }
    }


    /**
     *  Sets the target temperature.
     *
     *  @param  temp  The temperature to set
     */
    public void setTemperature(double temp)
    {
        setTemp = temp;
        pic.setSetpoint(setTemp);
    }


    /**
     *  Gets the target temperature.
     *
     *  @return  The set temperature
     */
    public double getTemperature()
    {
        return setTemp;
   }


    /**
     *  Starts the control loop.
     *
     *  @param  power  The initial power value
     */
    public void startLoop(double power)
    {
        if (active) return;
        lastPower = power;
        startLoop();
    }


    /**
     *  Starts the control loop.
     */
    public void startLoop()
    {
        if (active) return;
        pic.reset();
        pic.setIntegral(lastPower - basePower);
        active = true;
    }


    /**
     *  Stops the control loop.
     */
    public void stopLoop()
    {
        if (!active) return;
        active = false;
        setPower(0.0);
    }


    /**
     *  Gets the control loop state.
     *
     *  @return  Whether the control loop is active
     */
    public boolean isLoopActive()
    {
        return active;
    }


    /**
     *  Resets the controller.
     */
    public void reset()
    {
        pic.reset();
    }


    /**
     *  Timer method for control loop iteration.
     */
    private void iterateLoop()
    {
        if (!active) return;
        int count = 0;
        double temp = 0.0;
        for (Channel tempChan : tempChansL) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                temp += value;
                count++;
            }
        }
        if (count > 0) {
            double tod = (double)System.currentTimeMillis() / 1000;
            lastPower = pic.performPI(new double[]{temp / count}, tod);
            setPower(lastPower);
        }
        else {
            log.error("Control loop iteration failed: no valid temperature values available");
        }
    }

}
