package org.lsst.ccs.subsystem.rafts;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.AlertService;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.utilities.FitsHeaderKeywordData;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.daq.utilities.FitsService;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.reb.Asic;
import org.lsst.ccs.drivers.reb.Aspic;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.BoardDacs;
import org.lsst.ccs.drivers.reb.Cabac;
import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.drivers.reb.ImageClient;
import org.lsst.ccs.drivers.reb.ImageMetadata;
import org.lsst.ccs.drivers.reb.PowerAdcs;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.SlowAdcs;
import org.lsst.ccs.drivers.reb.StatusSet;
import org.lsst.ccs.drivers.reb.TempAdcs;
import org.lsst.ccs.drivers.reb.TempRtds;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.rafts.alerts.RaftAlert;
import org.lsst.ccs.subsystem.rafts.config.ASPIC;
import org.lsst.ccs.subsystem.rafts.config.BiasDACS;
import org.lsst.ccs.subsystem.rafts.config.CABAC;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.CcdType;
import org.lsst.ccs.subsystem.rafts.data.ImageData;
import org.lsst.ccs.subsystem.rafts.data.RegisterData;
import org.lsst.ccs.subsystem.rafts.data.StatusData;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.states.RebDeviceState;
import org.lsst.ccs.subsystem.rafts.states.RebValidationState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements access to a REB device.
 *
 *  @author Owen Saxton
 */
public class REBDevice extends Device implements ClearAlertHandler, ConfigurationBulkChangeHandler {

    /*
     *  Constants.
     */
    private static final String
        SERIAL_NUM = "serialNum";

    public static final String
        RAFTS = "Rafts",
        RAFTS_LIMITS = "RaftsLimits";

    public static final int
        TYPE_BD_TEMP   = 0,
        TYPE_BD_POWER  = 1,
        TYPE_RTD       = 2,
        TYPE_ASP_TEMP  = 3,
        TYPE_BIAS_VOLT = 4,
        TYPE_CCD_CURR  = 5,
        TYPE_CR_VOLT   = 6,
        TYPE_HEATER    = 7,
        TYPE_HV_SWITCH = 8;

    public static final int
        CHAN_TOTAL_POWER = -1,
        CHAN_HTR_VOLTS = 0,
        CHAN_HTR_POWER = 1,
        NUM_HTR_CHANS  = 2,
        CHAN_RTD_INT_TEMP = TempRtds.NUM_RTD_TEMPS,
        CHAN_RTD_INT_VOLT = TempRtds.NUM_RTD_TEMPS + 1,
        NUM_RTD_CHANS     = TempRtds.NUM_RTD_TEMPS + 2;

    static final int
        REB_NUM_MASK = 0x03,
        RAFT_NUM_MASK = 0x3f,
        RAFT_NUM_SHIFT = 2;

    static final int
        POWER_ON_DELAY = 200,
        POWER_OFF_DELAY = 50;

    static final int
        DAC_LIMIT = 4095;

    static final double
        DAC_CONV = DAC_LIMIT / 5.0,
        HTR_CONV = DAC_CONV / (1.0 + 10.0 / 10.0),
        HTR_CONV_R5 = 2.0 * DAC_CONV / (1.0 + 30.0 / 10.0),
        HEATER_OHMS = 14.5,   // Estimated value at -100C
        HTR_LEAD_OHMS = 0.3;  // Lead resistance

    /*
     *  Private lookup maps.
     */
    private static final Map<String,Integer> hdwTypeMap = new HashMap<>();
    static {
        hdwTypeMap.put("DAQ0", BaseSet.HDW_TYPE_DAQ0);
        hdwTypeMap.put("DAQ1", BaseSet.HDW_TYPE_DAQ1);
        hdwTypeMap.put("DAQ2", BaseSet.HDW_TYPE_DAQ2);
        hdwTypeMap.put("PCI",  BaseSet.HDW_TYPE_PCI);  // Same as PCI0
        hdwTypeMap.put("PCI0", BaseSet.HDW_TYPE_PCI0);
        hdwTypeMap.put("PCI1", BaseSet.HDW_TYPE_PCI1);
    }
    private static final Map<String,Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("TEMP",   TYPE_BD_TEMP);
        typeMap.put("POWER",  TYPE_BD_POWER);
        typeMap.put("RTD",    TYPE_RTD);
        typeMap.put("ATEMP",  TYPE_ASP_TEMP);
        typeMap.put("BIAS",   TYPE_BIAS_VOLT);
        typeMap.put("CURR",   TYPE_CCD_CURR);
        typeMap.put("CRVOLT", TYPE_CR_VOLT);
        typeMap.put("HEATER", TYPE_HEATER);
        typeMap.put("HVSWCH", TYPE_HV_SWITCH);
    }

    /*
     *  Data fields.
     */
    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private FitsService fitsService;

  		  
    @ConfigurationParameter(name=SERIAL_NUM, category=RAFTS)
    private long serialNum;
    @ConfigurationParameter(category=RAFTS, isFinal=true)
    private int id;
    @ConfigurationParameter(category=RAFTS, isFinal=true)
    private String ifcName;
    //@ConfigurationParameter(category=RAFTS)
    protected CcdType ccdType = CcdType.NONE;

    // Set from the Groovy file
    private String hdwType = "daq2";
    private ClientFactory clientFactory = null;
    private int rebType = BaseSet.REB_TYPE_SCIENCE;
    private int ccdMask = 0;  // Dummy, but some groovy files might still use it
    private int[] dataSegmentMap = null;
    
    //private boolean isConnectedOk = true;
    //private final Object stateUpdateLock = new Object();

    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private final BaseSet bss = new BaseSet();
    private final StatusSet sts = new StatusSet(bss);
    private final TempAdcs tmp = new TempAdcs(bss);
    private final PowerAdcs pwr = new PowerAdcs(bss);
    private final SlowAdcs slow = new SlowAdcs(bss);
    private final TempRtds rtd = new TempRtds(bss);
    private final BoardDacs dac = new BoardDacs(bss);
    private final Aspic asp = new Aspic(bss);
    private final Cabac cbc = new Cabac(bss);
    private final ImageClient imc;
    private final BiasControl[] biases = new BiasControl[Asic.MAX_STRIPS];
    private final AspicControl[] aspics = new AspicControl[Asic.MAX_ASICS];
    private final CabacControl[] cabacs = new CabacControl[Asic.MAX_ASICS];
    private ImageProc img;
    private final SequencerProc seq = new SequencerProc(bss);
    private DacControl dacCtrl;
    private long hwSerialNum = 0;
    private boolean serialNumValid = false;
    protected int hdwTypeI, configCcdMask, realCcdMask, hwVersion;
    private int maxTemp, maxPower, maxAtemp, maxCcdI,
                numTemp, numPower, numAtemp, numCrVolt, numRtd;
    private boolean dacRaw, biasRaw;
    private double[] tempValues, powerValues, atempValues;
    private final double[] heaterValues = new double[NUM_HTR_CHANS];
    private double heaterVolts;
    private int errorCount;
    private int dacLoadDelay = 0, dacClearDelay = 0;

    /**
     *  Constructor.
     *
     *  @param  id       The REB ID
     *  @param  ifcName  The name of the network interface to use
     *  @param  ccdMask  The mask of active CCDs
     */
    public REBDevice(int id, String ifcName, int ccdMask)
    {
        this("DAQ0", id, ifcName, ccdMask);
    }


    /**
     *  Constructor.
     *
     *  @param  hdwType  The hardware type (DAQ0, DAQ1 or PCI)
     *  @param  id       The REB ID
     *  @param  ifcName  The name of the network interface to use
     *  @param  ccdMask  The mask of active CCDs
     *
     */
    public REBDevice(String hdwType, int id, String ifcName, int ccdMask)
    {
        this(hdwType, id, ifcName, ccdMask, null);
    }


    /**
     *  Constructor.
     *
     *  @param  hdwType  The hardware type (DAQ0, DAQ1 or PCI)
     *  @param  id       The REB ID
     *  @param  ifcName  The name of the network interface to use
     *  @param  ccdMask  The mask of active CCDs
     *  @param  clientFactory  The client factory for this RebDevice. It is
     *                         used to inject simulation.
     */
    public REBDevice(String hdwType, int id, String ifcName, int ccdMask,
                     ClientFactory clientFactory)
    {
        super();
        this.imc = new ImageClient(bss,"reb-"+id);
        this.img = new ImageProc(imc);
        // If a client factory is given, use it for both the BaseSet and ImageClient
        if (clientFactory != null) {
            bss.setClientFactory(clientFactory);
            imc.setClientFactory(clientFactory);
            this.clientFactory = clientFactory;
        }
        setHdwType(hdwType);
        this.id = id;
        this.ifcName = ifcName;
        this.ccdMask = ccdMask;
    }
    
    @Override
    public void build() {
        super.build();
        ComponentLookup lookupService = subsys.getComponentLookup();
        ComponentNode rebNode = lookupService.getComponentNodeForObject(this);

        //Add an image processing node
        img = new ImageProc(imc);
        ComponentNode imageProcNode = new ComponentNode(rebNode, "imageProc", img);
        lookupService.addComponentNodeToLookup(rebNode, imageProcNode);                
        img.setCcdMask(ccdMask);
        if (dataSegmentMap != null) {
            img.setDataSegmentMap(dataSegmentMap);
        }
       
        if ( fitsService == null ) {
            sLog.warn("No fits service is defined in groovy. For now using internal instance.");
            FitsService fitsSrvc = new FitsService();
            //Since the fits service was not defined in groovy we add it here.
            ComponentNode fitsServiceNode = new ComponentNode(rebNode, rebNode + "_fitsService", fitsSrvc);
            lookupService.addComponentNodeToLookup(rebNode, fitsServiceNode);
            fitsSrvc.setHeaderFilesList(Arrays.asList(new String[]{"primary", "extended", "Fits_primary_header.spec:primary"}));
        }
        
    }
    
    /**
     *  Sets the REB Hardware Type.
     *
     *  @param  hdwType  The REB hardware type
     */
    @ConfigurationParameterChanger
    public final void setHdwType(String hdwType)
    {
        this.hdwType = hdwType;
        /*
         *  Encode the hardware type
         */
        Integer type = hdwTypeMap.get(hdwType.toUpperCase());
        if (type == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "hdwType", "is invalid");
        }
        this.hdwTypeI = type;
    }
    
    
    /**
     *  Validates configuration parameter change.
     *
     *  @param  params  The map of configuration parameter values
     */
    @Override
    public void validateBulkChange(Map<String, Object> params)
    {
        long sNum = (Long)params.get(SERIAL_NUM);
        if (hwSerialNum != 0) {
            if (sNum != -1 && sNum != hwSerialNum) {
                subsys.getAgentService(AgentStateService.class).updateAgentComponentState(this, RebValidationState.INVALID);        
                String msg = "Configuration has wrong REB s/n: " + sNum + ". " + fullName + " s/n is " + hwSerialNum;
                throw new IllegalArgumentException(msg);
            }
            serialNumValid = true;
            subsys.getAgentService(AgentStateService.class).updateAgentComponentState(this, RebValidationState.VALID);                        
        }
    }


    /**
     *  Gets the REB ID.
     *
     *  @return  The REB ID
     */
    @Command(type=CommandType.QUERY, description="Get the REB ID")
    public int getId()
    {
        return id;
    }


    /**
     *  Gets the network interface name.
     *
     *  @return  The name of the network interface being used
     */
    @Command(type=CommandType.QUERY, description="Get the network interface name")
    public String getIfcName()
    {
        return ifcName;
    }


    /**
     *  Gets the mask of CCDs being used.
     *
     *  @return  The mask of active CCDs
     */
    @Command(type=CommandType.QUERY, description="Get the mask of CCDs being used")
    public int getCcdMask()
    {
        return realCcdMask;
    }


    /**
     *  Sets the CCD type.
     *
     *  @param  ccdType  The CCD type
     */
    //@ConfigurationParameterChanger
    public void setCcdType(CcdType ccdType)
    {
        this.ccdType = ccdType;
        dacLoadDelay = ccdType == CcdType.NONE ? 0 : POWER_ON_DELAY;
        dacClearDelay = ccdType == CcdType.NONE ? 0 : POWER_OFF_DELAY;
    }


    /**
     *  Gets the CCD type.
     *
     *  @return  The CCD type
     */
    @Command(type=CommandType.QUERY, description="Get the type of CCDs being used")
    public CcdType getCcdType()
    {
        return ccdType;
    }


    /**
     *  Sets the client factory.
     *
     *  @param  factory  The client factory
     */
    public void setClientFactory(ClientFactory factory)
    {
        if (factory != null) {
            bss.setClientFactory(factory);
            imc.setClientFactory(factory);
        }
    }


    /**
     *  Gets the REB type.
     *
     *  @return  The REB type
     */
    @Command(type=CommandType.QUERY, description="Gets the REB type")
    public int getRebType()
    {
        return rebType;
    }


    /**
     *  Gets the REB number.
     *
     *  @return  The REB number (0 - 2) within its raft
     */
    @Command(type=CommandType.QUERY, description="Gets the REB number")
    public int getRebNumber()
    {
        return id & REB_NUM_MASK;
    }


    /**
     *  Gets the raft number.
     *
     *  @return  The raft number (00 - 44) for the REB
     */
    @Command(type=CommandType.QUERY, description="Gets the raft number")
    public int getRaftNumber()
    {
        return (id >> RAFT_NUM_SHIFT) & RAFT_NUM_MASK;
    }


    /**
     *  Gets the hardware type.
     *
     *  @return  The hardware type, or UNKNOWN if the REB is not connected
     */
    @Command(type=CommandType.QUERY, description="Gets the hardware type")
    public int getHdwType()
    {
        return hdwTypeI;
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version, or 0 if the REB is not connected
     */
    @Command(type=CommandType.QUERY, description="Get the hardware version")
    public int getHwVersion()
    {
        return hwVersion;
    }


    /**
     *  Gets the serial number.
     *
     *  @return  The serial number, or 0 if the REB is not connected
     */
    @Command(type=CommandType.QUERY, description="Get the serial number")
    public long getSerialNumber()
    {
        return hwSerialNum;
    }


    /**
     *  Gets whether the serial number is valid.
     *
     *  @return  The serial number validity
     */
    public boolean isSerialNumValid()
    {
        return serialNumValid;
    }


    /**
     *  Clears an alert.
     * 
     *  @param  alert  The alert
     *  @return 
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert) {
        if ( alert.getAlertId().equals(RaftAlert.REB_NOT_CONNECTED.getAlertId()) ) {
            return ClearAlertCode.CLEAR_ALERT;
        }
        return ClearAlertCode.UNKWNOWN_ALERT;
    }
    

    /**
     *  Performs basic initialization.
     * 
     * Invoked when HasLifecycle::init is called on Device class.
     * 
     *  Checks that Groovy specifications are consistent
     */
    @Override
    protected void initDevice() {
        
        subsys.getAgentService(AgentStateService.class).registerState(RebDeviceState.class, "Reb Device ONLINE/OFFLINE State", this);
        subsys.getAgentService(AgentStateService.class).registerState(RebValidationState.class, "Reb Device Validation State", this);        
                
        subsys.getAgentService(AgentStateService.class).updateAgentComponentState(this, RebDeviceState.OFFLINE);
        subsys.getAgentService(AgentStateService.class).updateAgentComponentState(this, RebValidationState.UNKNWONW);        
        
        /*
         *  Propagate the client factory
         */
        setClientFactory(clientFactory);

        /*
         *  Configure the image processor
         */
        img.configure(log);

        /*
         *  Encode the hardware type
         */
        Integer type = hdwTypeMap.get(hdwType.toUpperCase());
        if (type == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "hdwType", "is invalid");
        }
        hdwTypeI = type;

        /*
         *  Get masks of Bias DACs, CABACs and ASPICs present
         */
        int biasMask = 0, aspMask = 0, aspicMask = 0, cabMask = 0, cabacMask = 0;

        for (Control ctrl : ctlChans.values()) {
            if (ctrl instanceof BiasControl) {
                biasMask |= 1 << ctrl.getHwChan();
            }
            else if (ctrl instanceof AspicControl) {
                aspMask |= 1 << ctrl.getHwChan();
            }
            else if (ctrl instanceof CabacControl) {
                cabMask |= 1 << ctrl.getHwChan();
            }
        }

        /*
         *  Check that CABACs and ASPICs come in pairs
         *  Generate the maximal mask of strips present
         */
        for (int j = 0; j < Asic.MAX_STRIPS; j++, aspMask >>= 2) {
            int pair = aspMask & 3;
            if (pair == 0) continue;
            if (pair == 3) {
                aspicMask |= (1 << j);
            }
            else {
                sLog.error("Unpaired configuration for ASPIC "
                            + (2 * j + pair - 1) + " ignored");
            }
        }
        for (int j = 0; j < Asic.MAX_STRIPS; j++, cabMask >>= 2) {
            int pair = cabMask & 3;
            if (pair == 0) continue;
            if (pair == 3) {
                cabacMask |= (1 << j);
            }
            else {
                sLog.error("Unpaired configuration for CABAC "
                            + (2 * j + pair - 1) + " ignored");
            }
        }
        if (cabacMask != 0 && biasMask != 0) {
            sLog.error("Both bias and CABAC configurations defined");
        }
        if (cabacMask == 0 && biasMask == 0) {
            sLog.warning("No bias or CABAC configuration defined");
        }
        if (cabacMask != 0 && cabacMask != aspicMask) {
            sLog.warning("ASPIC and CABAC configurations don't match");
        }
        if (biasMask != 0 && biasMask != aspicMask) {
            sLog.warning("ASPIC and bias configurations don't match");
        }
        configCcdMask = biasMask | aspicMask | cabacMask;

        /*
         *  Fill the record of control devices
         */
        for (Control ctrl : ctlChans.values()) {
            if (ctrl instanceof DacControl) {
                dacCtrl = (DacControl)ctrl;
            }
            else if (ctrl instanceof BiasControl) {
                if (((1 << ctrl.getHwChan()) & configCcdMask) != 0) {
                    biases[ctrl.getHwChan()] = (BiasControl)ctrl;
                }
            }
            else if (ctrl instanceof AspicControl) {
                if (((1 << ctrl.getHwChan() / 2) & configCcdMask) != 0) {
                    aspics[ctrl.getHwChan()] = (AspicControl)ctrl;
                }
            }
            else if (ctrl instanceof CabacControl) {
                if (((1 << ctrl.getHwChan() / 2) & configCcdMask) != 0) {
                    cabacs[ctrl.getHwChan()] = (CabacControl)ctrl;
                }
            }
        }

        /*
         *  Set initial value of maximum CCD mask
         */
        bss.setDefaultRebType(BaseSet.REB_TYPE_SCIENCE);
        try {
            realCcdMask = configCcdMask & ((1 << bss.getNumStrips()) - 1);
        }
        catch (REBException e) {
            sLog.error("Error getting expected CCD count: " + e); // Shouldn't happen
        }
        fullName = "REB " + id + (ifcName != null ? " (" + ifcName + ")" : "");
    }
    
    
    /**
     *  Performs full initialization.
     * 
     *  Opens connection to the REB and obtains information about it
     */
    @Override
    protected void initialize()
    {
        try {
            bss.open(hdwTypeI, id, ifcName);
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
            }
            bss.disable();
            bss.setTime();
            bss.enable();
            rebType = bss.getRebType();
            hwVersion = bss.getHwVersion();
            maxTemp = tmp.getNumRegs();
            maxPower = pwr.getNumRegs();
            maxAtemp = slow.getNumAspicTemps();
            maxCcdI = slow.getNumCcdCurrents();
            numTemp = numPower = numAtemp = numCrVolt = numRtd = 0;
            tempValues = new double[maxTemp];
            powerValues = new double[maxPower];
            atempValues = new double[maxAtemp];
            realCcdMask = configCcdMask & ((1 << bss.getNumStrips()) - 1);
            img.setCcdMask(realCcdMask);
            img.setNumRebCcds(bss.getNumStrips());
            img.setDataInversion(bss.getRebType() == BaseSet.REB_TYPE_SCIENCE);
            /*
            try {
                seq.setCcdMask(maxCcdMask);
            }
            catch (REBException e) {
                sLog.error("Error writing stripe selection");
            }
            */
            img.enableScan(seq.isScanEnabled());
            if (dacCtrl != null && !dacCtrl.checkConfig()) {
                dacCtrl = null;
            }
            dacRaw = dacCtrl != null ? dacCtrl.isRaw() : false;
            biasRaw = false;
            for (int j = 0; j < biases.length; j++) {
                BiasControl bias = biases[j];
                if (bias != null && !bias.checkConfig()) {
                    bias = null;
                }
                if (bias != null) {
                    biasRaw = bias.isRaw();
                }
                biases[j] = bias;
            }
            for (int j = 0; j < aspics.length; j++) {
                AspicControl aspic = aspics[j];
                if (aspic != null && !aspic.checkConfig()) {
                    aspic = null;
                }
                aspics[j] = aspic;
            }
            for (int j = 0; j < cabacs.length; j++) {
                CabacControl cabac = cabacs[j];
                if (cabac != null && !cabac.checkConfig()) {
                    cabac = null;
                }
                cabacs[j] = cabac;
            }
            try {
                hwSerialNum = bss.getRebSerial();
            }
            catch (REBException e) {
                sLog.error("Error reading serial number");
                hwSerialNum = -1;
            }
            serialNumValid = serialNum == -1 || serialNum == hwSerialNum;
            if (!serialNumValid) {
                subsys.getAgentService(AgentStateService.class).updateAgentComponentState(this, RebValidationState.INVALID);        
                sLog.error(name + " s/n (" + hwSerialNum + ") doesn't match configuration value (" + serialNum + ")");
            } else {
                subsys.getAgentService(AgentStateService.class).updateAgentComponentState(this, RebValidationState.VALID);                        
            }
            errorCount = 0;
            initSensors();
            if (numRtd > 0) {
                rtd.initialize();
            }
            imc.open(hdwTypeI, id, ifcName);
            
            //	LSSTCCSRAFTS-386 the first time this is invoked it's too early
            //  so we publish in the postStart method and publish any other time
            //  as long as the Agent is OPERATIONAL
            if ( stateService.isInState(PhaseState.OPERATIONAL) ) {
                publishCCDControllerSerialNumber();
            }
            

            setOnline(true);
            sLog.info("Connected to " + fullName);
            if (inited) {                    
                alertService.raiseAlert(RaftAlert.REB_NOT_CONNECTED.getAlert(), AlertState.NOMINAL,
                                        fullName + " is connected");
            }
        }
        catch (REBException e) {
            if (!inited) {
                sLog.error("Error connecting to " + fullName + ": " + e);
                close("could not connect");
            }
            else {
                close(null);
            }
        }
        inited = true;
//        if ( !isConnectedOk ) {
//            throw new RuntimeException("Could not connect with REB " + name);
//        }
    }

    @Override
    public void postStart() {
        publishCCDControllerSerialNumber();
    }

    
    
    private void publishCCDControllerSerialNumber() {
        FitsHeaderKeywordData data = new FitsHeaderKeywordData(img.getRebGeometry().getUniqueId());
        data.addHeaderKeywordValue("primary", "CCDControllerSerial", String.format("%012x", hwSerialNum), true);
        subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());        
    }
    

    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        close("was disconnected");
    }


    private void close(String alertText)
    {
        try {
            rebType = BaseSet.REB_TYPE_UNKNOWN;
            hwVersion = 0;
            hwSerialNum = 0;
            bss.close();
        }
        catch (REBException e) {
        }
        try {
            imc.close();
        }
        catch (REBException e) {
        }
        if (alertText != null) {
            alertService.raiseAlert(RaftAlert.REB_NOT_CONNECTED.getAlert(), AlertState.ALARM,
                                    fullName + " " + alertText);
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  Two-element array containing the encoded type [0] and
     *           subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
                                 String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(sLog, name, "channel type", type);
        }
        
        return new int[]{iType, 0};
    }


    /**
     *  Initializes a channel (in this case only checks HW channel number).
     *
     *  @param  name     The channel name
     *  @param  id       The channel ID
     *  @param  hwChan   The hardware channel
     *  @param  type     The channel type
     *  @param  subtype  The channel subtype
     */
    @Override
    protected void initChannel(String name, int id, int hwChan, int type, int subtype)
    {
        try {
            boolean chanOk;
            if (type == TYPE_BIAS_VOLT || type == TYPE_CR_VOLT) {
                chanOk = slow.testChannel(hwChan);
            }
            else {
                int maxChan = type == TYPE_BD_TEMP ? maxTemp :
                              type == TYPE_BD_POWER ? maxPower :
                              type == TYPE_ASP_TEMP ? maxAtemp :
                              type == TYPE_CCD_CURR ? maxCcdI :
                              type == TYPE_RTD ? NUM_RTD_CHANS :
                              type == TYPE_HEATER ? NUM_HTR_CHANS : 
                              type == TYPE_HV_SWITCH ? 1 : 0;
                chanOk = (hwChan >= 0 || type == TYPE_BD_POWER) && hwChan < maxChan;
            }
            if (!chanOk) {
                MonitorLogUtils.reportError(sLog, name, "hw channel number", hwChan);
            }
            switch (type) {
            case TYPE_BD_TEMP:
                numTemp++;
                break;
            case TYPE_BD_POWER:
                numPower++;
                break;
            case TYPE_ASP_TEMP:
                numAtemp++;
                break;
            case TYPE_CR_VOLT:
                numCrVolt++;
                break;
            case TYPE_RTD:
                numRtd++;
                break;
            }
        }
        catch (Exception e) {
            dropChannel(id);
        }
    }


    /**
     *  Reads grouped channels.
     */
    @Override
    protected void readChannelGroup()
    {
        if (!online) return;
        String item = "";
        try {
            if (numTemp > 0) {
                try {
                    tempValues = tmp.readAdcs();
                }
                catch (REBException e) {
                    if (e.getMessage().contains("Completion wait")) {
                        for (int j = 0; j < tempValues.length; j++) {
                            tempValues[j] = Double.NaN;
                        }
                    }
                    else {
                        item = "board temperatures";
                        throw(e);
                    }
                }
            }
            if (numPower > 0) {
                try {
                    powerValues = pwr.readAdcs();
                }
                catch (REBException e) {
                    if (e.getMessage().contains("Completion wait")) {
                        for (int j = 0; j < powerValues.length; j++) {
                            powerValues[j] = Double.NaN;
                        }
                    }
                    else {
                        item = "board power";
                        throw(e);
                    }
                }
            }
            if (numAtemp > 0) {
                item = "aspic temperatures";
                atempValues = slow.readAspicTemps(0, maxAtemp);
            }
            if (numCrVolt > 0) {
                item = "CCD voltages";
                slow.fetchVoltages();
            }
            errorCount = 0;
        }
        catch (REBException e) {
            sLog.error("Error reading " + item + ": " + e.getMessage());
            checkErrorCount();
        }
        double volts = heaterVolts;
        double amps = volts / (HEATER_OHMS + HTR_LEAD_OHMS);
        heaterValues[CHAN_HTR_VOLTS] = volts;
        heaterValues[CHAN_HTR_POWER] = amps * amps * HEATER_OHMS;
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        String item = "";
        try {
            switch (type) {
            case TYPE_CCD_CURR:
                item = "CCD current";
                value = slow.readCurrent(hwChan);
                errorCount = 0;
                break;

            case TYPE_BIAS_VOLT:
            case TYPE_CR_VOLT:
                item = "CCD voltage";
                value = slow.readVoltage(hwChan);
                errorCount = 0;
                break;

            case TYPE_RTD:
                item = "RTD";
                if (hwChan < TempRtds.NUM_RTD_TEMPS) {
                    value = rtd.readTemperature(hwChan);
                }
                else if (hwChan == CHAN_RTD_INT_TEMP) {
                    value = rtd.readIntTemperature();
                }
                else {
                    value = rtd.readIntVoltage();
                }
                errorCount = 0;
                break;

            case TYPE_HV_SWITCH:
                item = "HV bias switch";
                value = bss.isBackBiasOn() ? 1.0 : 0.0;
                errorCount = 0;
                break;

            case TYPE_BD_POWER:
                if (hwChan >= 0) {
                    value = powerValues[hwChan];
                }
                else {
                    value = 0.0;
                    for (int j = 0; j < powerValues.length; j += 2) {
                        value += powerValues[j] * powerValues[j + 1];
                    }
                    value += heaterValues[CHAN_HTR_POWER];
                }
                break;

            case TYPE_BD_TEMP:
                value = tempValues[hwChan];
                break;

            case TYPE_ASP_TEMP:
                value = atempValues[hwChan];
                break;

            case TYPE_HEATER:
                value = heaterValues[hwChan];
                break;
            }
        }
        catch (DriverException e) {
            sLog.error("Error reading " + item + ": " + e.getMessage());
            checkErrorCount();
        }
        return value;
    }


    /**
     *  Reads a channel immediately.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    public double readChannelNow(int hwChan, int type)
    {
        double value = Double.NaN;
        String item = "";
        try {
            switch (type) {
            case TYPE_BD_TEMP:
                item = "board temperature";
                value = tmp.readAdc(hwChan);
                errorCount = 0;
                break;

            case TYPE_BD_POWER:
                item = "board power";
                value = pwr.readAdc(hwChan);
                errorCount = 0;
                break;

            case TYPE_ASP_TEMP:
                item = "aspic temperature";
                value = slow.readAspicTemp(hwChan / Asic.NUM_SIDES, hwChan % Asic.NUM_SIDES);
                errorCount = 0;
                break;

            case TYPE_CR_VOLT:
                item = "CCD voltage";
                value = slow.readVoltageNow(hwChan);
                errorCount = 0;
                break;

            default:
                value = super.readChannelNow(hwChan, type);
            }
        }
        catch (DriverException e) {
            sLog.error("Error reading " + item + ": " + e.getMessage());
            checkErrorCount();
        }

        return value;
    }


    /**
     *  Sets the REB configuration data.
     *
     *  @param  reb  The REB configuration (object)
     */
    @Command(type=CommandType.ACTION, description="Sets the REB configuration data")
    public void setREBConfig(@Argument(name="reb", description="REB configuration data")
                             REB reb)
    {
        dacCtrl.setConfig(reb.getDacs());
        BiasDACS[] biasConfig = reb.getBiases();
        for (int j = 0; j < biasConfig.length; j++) {
            BiasControl bias = biases[j];
            if (bias != null) {
                bias.setConfig(biasConfig[j]);
            }
        }
        ASPIC[] aspicConfig = reb.getAspics();
        for (int j = 0; j < aspicConfig.length; j++) {
            AspicControl aspic = aspics[j];
            if (aspic != null) {
                aspic.setConfig(aspicConfig[j]);
            }
        }
        CABAC[] cabacConfig = reb.getCabacs();
        for (int j = 0; j < cabacConfig.length; j++) {
            CabacControl cabac = cabacs[j];
            if (cabac != null) {
                cabac.setConfig(cabacConfig[j]);
            }
        }
    }


    /**
     *  Gets the REB configuration data.
     *
     *  @return  The REB configuration (object)
     */
    @Command(type=CommandType.QUERY, description="Gets the REB configuration data")
    public REB getREBConfig() 
    {
        REB reb = new REB();
        reb.setId(id);
        reb.setIfcName(ifcName);
        reb.setCcdMask(realCcdMask);
        reb.setMaxCcdMask(realCcdMask);

        reb.setDacVersion(DacControl.getHwVersion(dac));
        reb.setDacRaw(dacRaw);
        reb.setBiasVersion(BiasControl.getHwVersion(dac));
        reb.setBiasRaw(biasRaw);
        reb.setAspicVersion(AspicControl.getHwVersion(asp));
        reb.setCabacVersion(CabacControl.getHwVersion(cbc));

        dacCtrl.getConfig(reb.getDacs());
        BiasDACS[] biasConfig = reb.getBiases();
        for (int j = 0; j < biases.length; j++) {
            BiasControl bias = biases[j];
            if (bias != null) {
                biasConfig[j] = bias.getConfig();
            }
        }
        ASPIC[] aspicConfig = reb.getAspics();
        for (int j = 0; j < aspics.length; j++) {
            AspicControl aspic = aspics[j];
            if (aspic != null) {
                aspicConfig[j] = aspic.getConfig();
            }
        }
        CABAC[] cabacConfig = reb.getCabacs();
        for (int j = 0; j < cabacs.length; j++) {
            CabacControl cabac = cabacs[j];
            if (cabac != null) {
                cabacConfig[j] = cabac.getConfig();
            }
        }

        return reb;
    }


    /**
     * Transversal setting of all TM for the Aspics of this reb.
     * It internally uses the configuration service.
     * @param tm true for TM, false for normal
     */
    @Command(type=CommandType.CONFIGURATION, description="Set TM for all ASPICs on this REB")
    public void setAllAspicTM(boolean tm) {
        setAllAspic(AspicControl.TM, tm ? 1 : 0);
    }

    @Deprecated
    @Command(type=CommandType.CONFIGURATION, description="[Deprecated] transversal TM setting for all aspic controls for this reb")
    public void setAllTM(boolean tm) {
        setAllAspicTM(tm);
    }


    /**
     * Transversal setting of all Gains for the Aspics of this reb.
     * It internally uses the configuration service.
     * @param gain the new gain value
     */
    @Command(type=CommandType.CONFIGURATION, description="Set gain for all ASPICs on this REB")
    public void setAllAspicGain(int gain) {
        setAllAspic(AspicControl.GAIN, gain);
    }

    @Deprecated
    @Command(type=CommandType.CONFIGURATION, description="[Deprecated] transversal Gain setting for all aspic controls for this reb")
    public void setAllGain(int gain) {
        setAllAspicGain(gain);
    }


    /**
     * Transversal setting of all Rcs for the Aspics of this reb.
     * It internally uses the configuration service.
     * @param rc the new rc value
     */
    @Command(type=CommandType.CONFIGURATION, description="Set RC for all ASPICs on this REB")
    public void setAllAspicRc(int rc) {
        setAllAspic(AspicControl.RC, rc);
    }

    @Deprecated
    @Command(type=CommandType.CONFIGURATION, description="[Deprecated] transversal Rc setting for all aspic controls for this reb")
    public void setAllRc(int rc) {
        setAllAspicRc(rc);
    }
    
    private void setAllAspic(String parmName, int val) {
        for (AspicControl aspic : aspics) {
            subsys.getComponentConfigurationEnvironment(aspic).submitChange(parmName, val);
        }
        sce.commitBulkChange();
    }


    /**
     *  Performs CCD power on/off sequences.
     *
     *  @param  on  Whether to power on (true) or off (false)
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Perform CCD power on/off sequence")
    public void powerCCDs(@Argument(description="Whether to power on") boolean on) throws Exception
    {
        if (on) {
            if (ccdType == CcdType.ITL) {
                loadDacs(true);
                loadBiasDacs(true);
            }
            else {
                loadBiasDacs(true);
                loadDacs(true);
            }
        }
        else {
            setBackBias(false);
            if (ccdType == CcdType.ITL) {
                clearBiasDacs();
                clearDacs();
            }
            else {
                clearDacs();
                clearBiasDacs();
            }
        }
    }


    /**
     *  Loads configured DAC values.
     *
     *  @param  all  If true, all data are loaded; otherwise only changes
     *  @return  The number of DACs loaded
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Loads configured DAC values")
    public int loadDacs(@Argument(description="Whether all DACs are loaded, or only changed ones")
                        boolean all) throws Exception
    {
        int count = all ? dacCtrl.loadConfig(dacLoadDelay) : dacCtrl.loadChanged(dacLoadDelay);
        sLog.info("Loaded " + count + " " + name + " DACs");
        return count;
    }


    /**
     *  Clears DAC values.
     *
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Clear DAC values")
    public void clearDacs() throws Exception
    {
        dacCtrl.clear(dacClearDelay);
    }


    /**
     *  Loads configured bias DAC values.
     *
     *  @param  all  If true, all data are loaded; otherwise only changes
     *  @return  The number of bias DACs loaded
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Loads configured bias DAC values")
    public int loadBiasDacs(@Argument(description="Whether all DACs are loaded, or only changed ones")
                            boolean all) throws Exception
    {
        int count = 0;
        for (BiasControl bias : biases) {
            if (bias == null) continue;
            count += all ? bias.load(dacLoadDelay) : bias.loadChanged(dacLoadDelay);
        }
        sLog.info("Loaded " + count + " " + name + " bias DACs");
        return count;
    }


    /**
     *  Clears bias DAC values.
     *
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Clear bias DAC values")
    public void clearBiasDacs() throws Exception
    {
        for (BiasControl bias : biases) {
            if (bias == null) continue;
            bias.clear(dacClearDelay);
        }
    }


    /**
     *  Loads configuration data to the CABACs in a safe way.
     *
     *  @return  The number of CABACs loaded
     *  @throws  Exception
     */
    @Deprecated
    @Command(type=CommandType.ACTION, description="[Deprecated] Loads configuration data to the CABACs")
    public int loadCabacs() throws Exception
    {
        int count = 0;
        try {
            if (cbc.getVersion() == Cabac.VERSION_0) {
                for (int j = 0; j < cabacs.length; j += 2) {
                    if (cabacs[j] == null) continue;
                    int[] dataT = new int[Cabac.NUM_CABAC_FLDS];
                    int[] dataB = new int[Cabac.NUM_CABAC_FLDS];
                    for (int k = 0;; k++) {
                        if (!cabacs[j].getData(k, dataT)) break;
                        if (!cabacs[j + 1].getData(k, dataB)) break;
                        cbc.set(Cabac.SIDE_TOP, dataT);
                        cbc.set(Cabac.SIDE_BOTTOM, dataB);
                        cbc.load(j / 2);
                    }
                    count += 2;
                }
            }
            else {
                for (CabacControl cabac : cabacs) {
                    if (cabac != null) {
                        count += cabac.load();
                    }
                }
            }
        }
        catch (REBException e) {
            checkTimeout(e);
        }
        sLog.info("Loaded " + count + " " + name + " CABACs");
        return count;
    }


    /**
     *  Checks loaded CABAC configuration data.
     *
     *  @return  A list containing, for each configured CABAC, a mask showing
     *           which fields don't match
     *  @throws  Exception
     */
    @Deprecated
    @Command(type=CommandType.QUERY, description="[Deprecated] Checks loaded CABAC configuration data")
    public List<Integer> checkCabacs() throws Exception
    {
        try {
            List<Integer> diff = new ArrayList<>();
            for (int j = 0; j < cabacs.length; j += 2) {
                if (cabacs[j] == null) continue;
                if (cbc.getVersion() == Cabac.VERSION_0) {
                    cbc.fetch(j / 2);
                    diff.add(cabacs[j].compare(cbc.get(Cabac.SIDE_TOP)));
                    diff.add(cabacs[j + 1].compare(cbc.get(Cabac.SIDE_BOTTOM)));
                }
                else {
                    diff.add(cabacs[j].check());
                    diff.add(cabacs[j + 1].check());
                }
            }
            return diff;
        }
        catch (REBException e) {
            checkTimeout(e);
            return null;
        }
    }


    /**
     *  Sets the CABAC power enable bits.
     *
     *  @param  enables  The enable bits
     *  @throws  DriverException
     */
    @Deprecated
    @Command(type=CommandType.ACTION, description="[Deprecated] Set the CABAC power enable bits")
    public void setCabacPower(@Argument(description="Enable bits")
                              int enables) throws DriverException
    {
        cbc.setRegulator(enables);
    }


    /**
     *  Gets the CABAC power enable bits.
     *
     *  @return  The enable bits
     */
    @Deprecated
    @Command(type=CommandType.QUERY, description="[Deprecated] Get the CABAC power enable bits")
    public int getCabacPower()
    {
        try {
            return cbc.getRegulator();
        }
        catch (DriverException e) {
            return -1;
        }
    }


    /**
     *  Loads configuration data to the ASPICs.
     *
     *  @param  all  If true, all data are loaded; otherwise only changes
     *  @return  The number of ASPICs loaded
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Load configuration data to the ASPICs")
    public int loadAspics(boolean all) throws Exception
    {
        int count = 0;
        for (AspicControl aspic : aspics) {
            if (aspic != null) {
                count += all ? aspic.load() : aspic.loadChanged();
            }
        }
        sLog.info("Loaded " + count + " " + name + " ASPICs");
        return count;
    }


    /**
     *  Resets the ASPICs.
     *
     *  @return  The number of ASPICs reset
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Reset the ASPICs")
    public int resetAspics() throws Exception
    {
        int count = 0, mask = 0;
        for (AspicControl aspic : aspics) {
            if (aspic != null) {
                int strip = aspic.getHwChan() / 2;
                if ((mask & (1 << strip)) == 0) {
                    count += aspic.reset();
                    mask |= (1 << strip);
                }
            }
        }
        sLog.info("Reset " + count + " " + name + " ASPICs");
        return count;
    }


    /**
     *  Checks loaded ASPIC configuration data.
     *
     *  @return  The mask of mismatched fields
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Check loaded ASPIC configuration")
    public int checkAspics() throws Exception
    {
        int mask = 0, index = 0;
        for (AspicControl aspic : aspics) {
            if (aspic != null) {
                mask |= (aspic.check() << index);
                index += 4;
            }
        }
        return mask;
    }


    /**
     *  Gets the back bias on state.
     *
     *  @return  Whether back bias is turned on
     *  @throws  Exception
     */
    @Command(type=CommandType.QUERY, description="Get the back bias on state")
    public boolean isBackBiasOn() throws Exception
    {
        try {
            return bss.isBackBiasOn();
        }
        catch (REBException e) {
            checkTimeout(e);
            return false;
        }
    }


    /**
     *  Turns back bias on or off.
     *
     *  @param  value  The back bias on state to set (true or false)
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Turn back bias on or off")
    public void setBackBias(@Argument(description="The on state")
                            boolean value) throws Exception
    {
        try {
            bss.setBackBias(value);
        }
        catch (REBException e) {
            checkTimeout(e);
        }
    }


    /**
     *  Sets a heater voltage.
     *
     *  @param  heater  The heater number: 0 or 1
     *  @param  volts   The voltage to set
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Set heater voltage")
    public void setHeater(@Argument(description="Heater number")
                          int heater,
                          @Argument(description="The voltage")
                          double volts) throws Exception
    {
        heaterVolts = volts;
        int value = (int)((dac.getVersion() == BoardDacs.VERSION_REB5 ? HTR_CONV_R5 : HTR_CONV) * volts);
        try {
            dac.set(heater == 0 ? BoardDacs.CHAN_HEATER_1 : BoardDacs.CHAN_HEATER_2,
                    value < 0 ? 0 : value > DAC_LIMIT ? DAC_LIMIT : value, true);
        }
        catch (REBException e) {
            checkTimeout(e);
        }
    }


    /**
     *  Sets heater power.
     *
     *  @param  heater  The heater number: 0 or 1
     *  @param  power   The power value to set
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Set heater power")
    public void setHeaterPower(@Argument(description="Heater number")
                               int heater,
                               @Argument(description="The power value")
                               double power) throws Exception
    {
        setHeater(heater, (HEATER_OHMS + HTR_LEAD_OHMS) * Math.sqrt(power / HEATER_OHMS));
    }


    /**
     *  Sets the time base to the current system time.
     *
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Sets the time base to the current system time")
    public void setTime() throws Exception
    {
        try {
            bss.disable();
            bss.setTime();
            bss.enable();
        }
        catch (REBException e) {
            checkTimeout(e);
        }
    }


    /**
     *  Gets the time base as Unix time.
     *
     *  @return  The time base converted to Unix time
     *  @throws  Exception
     */
    @Command(type=CommandType.QUERY, description="Gets the time base as Unix time")
    public long getTime() throws Exception
    {
        try {
            return bss.getTime();
        }
        catch (REBException e) {
            checkTimeout(e);
            return 0;
        }
    }


    /**
     *  Gets a trigger time as Unix time.
     *
     *  @param  rSet  The name of the trigger time's register set: "stat",
     *                "time" or "seq:.
     *  @return  The trigger time converted to Unix time
     *  @throws  Exception
     */
    private static final Map<String,Integer> rsetNames = new HashMap();
    static {
        rsetNames.put("stat", BaseSet.RSET_STATUS);
        rsetNames.put("time", BaseSet.RSET_TIME_BASE);
        rsetNames.put("seq",  BaseSet.RSET_SEQUENCER);
        rsetNames.put("tadc", BaseSet.RSET_TEMP_ADCS);
        rsetNames.put("padc", BaseSet.RSET_POWER_ADCS);
    }

    @Command(type=CommandType.QUERY, description="Gets a trigger time as Unix time")
    public long getTime(@Argument(name="regset", description="Register set name")
                        String rSet) throws Exception
    {
        Integer regSet = rsetNames.get(rSet.toLowerCase());
        if (regSet == null) {
            throw new RaftException("Invalid register set name");
        }
        try {
            return bss.getTriggerTime(regSet);
        }
        catch (REBException e) {
            checkTimeout(e);
            return 0;
        }
    }


    /**
     *  Sets the FITS file test conditions.
     *
     *  @param  filter       The filter name
     *  @param  temperature  The temperature set point
     *  @param  wavelength   The monochromator wavelength
     */
    @Command(type=CommandType.ACTION, description="Sets the FITS file test conditions")
    public void setFitsConditions(@Argument(description="Filter name")
                                  String filter, 
                                  @Argument(description="The temperature")
                                  double temperature,
                                  @Argument(description="The wavelength")
                                  double wavelength)
    {        
        FitsHeaderKeywordData data = new FitsHeaderKeywordData(img.getRebGeometry().getUniqueId());
        data.addHeaderKeywordValue("primary", "FilterName", filter, true);
        data.addHeaderKeywordValue("primary", "TemperatureSetPoint", temperature, true);
        data.addHeaderKeywordValue("primary", "MonochromatorWavelength", wavelength, true);                        
        subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());
    }


    /**
     *  Gets the DAQ metadata for the current image.
     *
     *  @return  The image metadata
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Get the DAQ metadata for the current image")
    public ImageMetadata getImageMetadata() throws RaftException
    {
        return img.getImageMetadata();
    }


    /**
     *  Gets a portion of the current image.
     *
     *  @param  offset  The offset (in pixels) to the first pixel data to obtain.
     *  @param  count   The number of data pixels to obtain.  If zero, all
     *                  the data, starting at offset, is obtained.
     *  @return  The returned pixel data
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Gets a portion of the current image")
    public ImageData getImage(@Argument(description="Offset to the pixel data")
                              int offset,
                              @Argument(description="Number of pixels")
                              int count) throws RaftException
    {
        return img.getImage(0, offset, count);
    }


    /**
     *  Gets a portion of the current image.
     *
     *  @param  ccd     The CCD number
     *  @param  offset  The offset (in pixels) to the first pixel data to obtain.
     *  @param  count   The number of data pixels to obtain.  If zero, all
     *                  the data, starting at offset, is obtained.
     *  @return  The returned pixel data
     *  @throws  RaftException
     */
    @Command(type=CommandType.QUERY, description="Gets a portion of the current image")
    public ImageData getImage(@Argument(description="CCD number")
                              int ccd,
                              @Argument(description="Offset to the pixel data")
                              int offset,
                              @Argument(description="Number of pixels")
                              int count) throws RaftException
    {
        return img.getImage(ccd, offset, count);
    }


    /**
     *  Resets the front end.
     *
     *  This is the only way to stop a running sequencer program
     *
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Resets the front end")
    public void resetFrontEnd() throws Exception
    {
        try {
            imc.resetFrontEnd();
            bss.enable();
            bss.setTime();
        }
        catch (REBException e) {
            checkTimeout(e);
        }
    }


    /**
     *  Gets the contents of REB registers.
     *
     *  @param  address  The address of the first register to read
     *  @param  count    The number of registers to read
     *  @return  The block of register values
     *  @throws  Exception
     */
    @Command(type=CommandType.QUERY, description="Gets the contents of REB registers")
    public RegisterData getRegister(@Argument(description="Address of the first register")
                                    int address,
                                    @Argument(description="Number of registers to read")
                                    int count) throws Exception
    {
        int[] values = new int[count];
        try {
            bss.read(address, values);
        }
        catch (REBException e) {
            checkTimeout(e);
        }

        return new RegisterData(address, count, values);
    }


    /**
     *  Sets the contents of REB registers.
     *
     *  @param  address  The address of the first register to write
     *  @param  values   The array of values to write to consecutive registers
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Sets the contents of REB registers")
    public void setRegister(@Argument(description="Address of the first register")
                            int address,
                            @Argument(description="Values to write")
                            int[] values) throws Exception
    {
        try {
            bss.write(address, values);
        }
        catch (REBException e) {
            checkTimeout(e);
        }
    }


    /**
     *  Gets the REB status block.
     *
     *  @return  The REB status block
     *  @throws  Exception
     */
    @Command(type=CommandType.QUERY, description="Gets the REB status block")
    public StatusData getRebStatus() throws Exception
    {
        try {
            return new StatusData(sts.readStatus());
        }
        catch (REBException e) {
            checkTimeout(e);
            return null;
        }
    }


    /**
     *  Gets the associated board DACs object.
     *
     *  @return  The DAC object
     */
    BoardDacs getBoardDacs()
    {
        return dac;
    }


    /**
     *  Gets the associated ASPIC object.
     *
     *  @return  The ASPIC object
     */
    Aspic getAspic()
    {
        return asp;
    }


    /**
     *  Gets the associated CABAC object.
     *
     *  @return  The CABAC object
     */
    Cabac getCabac()
    {
        return cbc;
    }


    /**
     *  Gets the associated image processor.
     *
     *  @return  The image
     */
    public ImageProc getImageProc()
    {
        return img;
    }


    /**
     *  Gets the associated sequencer.
     *
     *  @return  The sequencer
     */
    public SequencerProc getSequencer()
    {
        return seq;
    }


    /**
     *  Checks whether exception is a timeout.
     */
    void checkTimeout(DriverException e) throws Exception
    {
        checkTimeout(e, RaftException.class);
    }


    /**
     *  Gets a temperature sensor value.
     *
     *  @return  The value
     */
    double getTempValue(int hwChan)
    {
        return online ? tempValues[hwChan] : 0;
    }


    /**
     *  Gets a power sensor value.
     *
     *  @return  The value
     */
    double getPowerValue(int hwChan)
    {
        return online ? powerValues[hwChan] : 0;
    }


    /**
     * Simulate a disconnect
     */
    @Command(simulation = true)
    public void simulateDisconnection() {
        setOnline(false);
    }


    /**
     * Checks error count.
     * 
     * Increments consecutive error count and sets offline if excessive
     */
    private void checkErrorCount() {
        if (++errorCount > 4) {
            setOnline(false);
        }
    }
    
    //This could be a temporary method
    /**
     * Override superclass method to set RebDevice state
     * 
     */
    @Override
    protected void setOnline(boolean online) {
        super.setOnline(online);
        subsys.getAgentService(AgentStateService.class).updateAgentComponentState(this, isOnline() ? RebDeviceState.ONLINE : RebDeviceState.OFFLINE);
    }
    
}
