package org.lsst.ccs.subsystem.rafts;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.drivers.reb.GlobalClient;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Routines for accessing the whole raft
 *
 * @author  Owen Saxton
 */
public class GlobalProc implements HasLifecycle {

    private GlobalClient gbl;
    private final Logger log = Logger.getLogger(getClass().getPackage().getName());
    //This is a temporary variable to make sure we don't initialize this class twice
    //as we are migrating from direct instatiation to the HasLifecycle::init phase 
    private boolean globalProcAlreadyInitialized = false;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final List<REBDevice> rebDevices = new ArrayList();
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ClientFactory clientFactory;
    
    @Override
    public void init() {  
        globalProcAlreadyInitialized = true;
        internalInit(rebDevices, clientFactory, log);
        if ( clientFactory != null ) {
            for (REBDevice rebDevice : rebDevices) {
                rebDevice.setClientFactory(clientFactory);
            }
        }        
    }

    /**
     *  Initializes the global raft system.
     *
     *  @param  rebDevices     The list of REB devices
     *  @param  clientFactory  The client factory
     *  @param  sLog           The subsystem logger
     */
    public void initialize(List<REBDevice> rebDevices, ClientFactory clientFactory, Logger sLog) {
        if ( globalProcAlreadyInitialized ) {
            throw new RuntimeException("This method should never be invoked when GlobalProc is added to the lookup tree.");
        }
        internalInit(rebDevices, clientFactory, sLog);
    }
    
    private void internalInit(List<REBDevice> rebDevices, ClientFactory clientFactory, Logger sLog)
    {
        int[] ids = new int[rebDevices.size()];
        String[] ifcNames = new String[rebDevices.size()];
        int hdwType = GlobalClient.HDW_TYPE_DAQ1;
        for (int j = 0; j < ids.length; j++) {
            REBDevice reb = rebDevices.get(j);
            ids[j] = reb.getId();
            ifcNames[j] = reb.getIfcName();
            hdwType = reb.getHdwType();
        }
        try {
            if (gbl == null) {
                gbl = new GlobalClient();
                if (clientFactory != null) {
                    gbl.setClientFactory(clientFactory);
                }
            }
            else {
                gbl.close();
            }
            gbl.open(hdwType, ids, ifcNames);
        }
        catch (REBException e) {
            sLog.error("Error during global initialization" , e);
        }
    }


    /**
     *  Sets the registers to be read during image acquisition.
     *
     *  @param  rebType    The encoded REB type
     *  @param  registers  The array of addresses of registers to be read
     *  @throws REBException
     */
    public void setRegisters(int rebType, int[] registers) throws REBException
    {
        gbl.setRegisterlist(rebType, registers);
    }


    /**
     *  Starts image acquisition.
     *
     *  @throws REBException
     */
    public void acquireImage() throws REBException
    {
        gbl.acquireImage(String.format("Image_%016x", System.currentTimeMillis()));
    }


    /**
     *  Starts image acquisition.
     *
     *  @param  name  The image name
     *  @throws REBException
     */
    public void acquireImage(String name) throws REBException
    {
        gbl.acquireImage(name);
    }

}
