package org.lsst.ccs.subsystem.rafts;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.rafts.data.CcdType;
import org.lsst.ccs.subsystem.rafts.data.ImageData;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.data.RaftFullState;
import org.lsst.ccs.subsystem.rafts.data.RaftState;
import org.lsst.ccs.subsystem.rafts.data.RaftsAgentProperties;
import org.lsst.ccs.subsystem.rafts.states.SequencerMainState;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.E2VCCDType;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the rafts test subsystem.
 *
 *  @author Owen Saxton
 */
public class RaftsMain implements HasLifecycle {

    private ClientFactory clientFactory = null;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final List<REBDevice> rebDevices = new ArrayList();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    protected final Map<String, REBDevice> rebDevicesMap = new HashMap();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private TempControl tempCtrl;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Monitor mon;

    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private ImageNameService imageNameService;
        
    @LookupName
    private String name;
    
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private CcdType ccdType = CcdType.NONE;

    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private final GlobalProc globalProc = new GlobalProc();
    private RaftsCommands raftsCommands;
    // TODO: tonyj: I don't get what this is for???
    private final CCDType ccdTypeX = new E2VCCDType();
    private final Raft raftGeometry = Raft.createRaft(ccdTypeX);
    private boolean running = false;

    @Override
    public void postBuild() {
        //Add a periodic task to check if Sequencers are running
        //This will be replaced once the DAQ v2 will be able to call us back
        //when the sequencer is done running
        periodicTaskService.scheduleAgentPeriodicTask( new AgentPeriodicTask("sequencerMonitor",
                () -> { raftsCommands.checkSequencerState(); }).withIsFixedRate(false).withPeriod(Duration.ofSeconds(5)));        
    }

    
    /**
     *  Initializes the rafts subsystem.
     */
    @Override
    public void postInit() {

        raftsCommands = new RaftsCommands(subsys, rebDevices, globalProc, raftGeometry, tempCtrl, imageNameService);
        
        // Add AgentInfo properties
        subsys.setAgentProperty(RaftsAgentProperties.RAFT_TYPE_AGENT_PROPERTY,
                                RaftsMain.class.getCanonicalName());

        // Initialize temperature control object
        if (tempCtrl != null) {
            tempCtrl.initialize(sLog);
        }
        else {
            sLog.info("No temperature control available");
        }

        // Add the RaftsCommands to the CommandSet
        subsys.addCommandsFromObject(raftsCommands, "");
        
        // Initialize for triggering
        globalProc.initialize(rebDevices, clientFactory, sLog);
                
    }


    /**
     *  Starts the subsystem.
     *
     */
    @Override
    public void postStart() 
    {
        // Propagete the CCD type everywhere
        raftsCommands.setCcdType(ccdType.name().toLowerCase());

        //  The following should be moved to the common code
        for (REBDevice devc : rebDevices) {
            BitSet s = new BitSet(3);
            for (int i = 0; i < 3; i++) {
                s.set(i, (devc.getCcdMask() & (1 << i)) != 0);
            }
            //TO-DO TS8 and Rafts should have a uniform way of dealing
            //with Geometry.
            Reb rebGeometry = Reb.createReb(ccdTypeX, s);
            raftGeometry.addReb(rebGeometry, devc.getRebNumber());
        }

        running = true;
        sLog.info("Rafts subsystem started");
        //At start we don't know what is the main state of the Sequencer
        //so we set it to UNKNOWN
        subsys.updateAgentState(SequencerMainState.UNKNOWN);
        publishState();
    }


    /**
     *  Sets the CCD type
     */
    @ConfigurationParameterChanger
    public void setCcdType(CcdType type)
    {
        ccdType = type;
        raftsCommands.setCcdType(ccdType.name().toLowerCase());
    }


    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(name="value", description="The tick period (ms)")
                                int value)
    {
        setTickPeriod(value);
        publishState();
    }


    /**
     *  Saves a named rafts configuration.
     *
     *  @param  name  The configuration name
     *  @throws  IOException
     */
    @Command(type=CommandType.ACTION, description="Save the current configuration")
    public void saveNamedConfig(String name) throws IOException
    {
        if (name.isEmpty()) {
            sce.saveChangesForCategories(REBDevice.RAFTS);
        }
        else {
            sce.saveChangesForCategoriesAs(REBDevice.RAFTS + ":" + name);
        }
    }


    /**
     *  Loads a named rafts configuration.
     *
     *  @param  name  The configuration name
     *  @throws  IOException
     */
    @Command(type=CommandType.ACTION, description="Load a named rafts configuration")
    public void loadNamedConfig(String name) throws IOException
    {
        sce.loadCategories(REBDevice.RAFTS_LIMITS + ":" + name, REBDevice.RAFTS + ":" + name);
    }


    /**
     *  Gets the full state of the rafts module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The full Raft state
     */
    @Command(type=CommandType.QUERY, description="Get the full Raft module state")
    public RaftFullState getFullState()
    {
        return new RaftFullState(new RaftState(getTickPeriod()), mon.getFullState());
    }    


    /**
     *  Gets a portion of a saved raw image.
     *
     *  @param  fileName  The name of the file containing the image
     *  @param  offset    The offset (in pixels) to the first pixel data to obtain.
     *  @param  count     The number of data pixels to obtain.
     *  @return  The returned pixel data
     *  @throws  RaftException
     *  @throws  IOException
     */
    @Command(type=CommandType.QUERY, description="Get a portion of a saved raw image")
    public static ImageData getImage(@Argument(name="filename", description="The image file name")
                                     String fileName,
                                     @Argument(name="offset", description="The offset to the first pixel")
                                     int offset,
                                     @Argument(name="count", description="The number of pixels to get")
                                     int count)
        throws RaftException, IOException
    {
        return ImageProc.getImage(fileName, offset, count);
    }


    /**
     *  Publishes the state of the raft module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    void publishState()
    {
        if (!running) return;
        KeyValueData data = new KeyValueData(RaftState.KEY, new RaftState(getTickPeriod()));
        subsys.publishSubsystemDataOnStatusBus(data);
    }


    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }


    /**
     *  Protected methods. Used only by tests.
     */
    RaftsCommands getRaftsCommands() {
        return raftsCommands;
    }
    
}
