package org.lsst.ccs.subsystem.rafts;

import hep.aida.IAnalysisFactory;
import hep.aida.IDataPoint;
import hep.aida.IDataPointSet;
import hep.aida.IDataPointSetFactory;
import hep.aida.IManagedObject;
import hep.aida.util.XMLUtils;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.IntBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TimeZone;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import nom.tam.fits.FitsException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.utilities.FitsService;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.Image;
import org.lsst.ccs.drivers.reb.ImageClient;
import org.lsst.ccs.drivers.reb.ImageMetadata;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.rafts.data.ImageData;
import org.lsst.ccs.subsystem.rafts.data.ImageState;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.data.SequencerData;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.image.DefaultImageSet;
import org.lsst.ccs.utilities.image.FitsFileWriter;
import org.lsst.ccs.utilities.image.FitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.image.ImageSet;
import org.lsst.ccs.utilities.image.MetaDataSet;
import org.lsst.ccs.utilities.image.ReadOutParameters;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.ccd.Segment;
import org.lsst.ccs.utilities.pattern.FileNamePatternUtils;

/**
 * Routines for handling image data
 *
 * @author Owen Saxton
 */
public class ImageProc {

    /**
     * The order of the Segments in the fits files is: S10->...
     * ->S17->S07->...->S00
     *
     * while the order of the segments coming from the DAQ is
     * S00->...->S07->S10->...->S17
     *
     * So we introduce the array below to describe the needed mapping. Note, the
     * ATS appears to have a different mapping, so this needs to be made
     * settable.
     */
    private int[] dataSegmentMap = {15, 14, 13, 12, 11, 10, 9, 8, 0, 1, 2, 3, 4, 5, 6, 7};
    private int[] invertedDataSegmentMap;

    /*
     *  Constants.
     */
    private static final int NUM_ADCS = 16,
            DATA_MASK_STRAIGHT = 0x20000,
            DATA_MASK_INVERTED = 0x1ffff;

    /*
     *  Data fields.
     */
    private final IDataPointSetFactory dpsf = IAnalysisFactory.create().createDataPointSetFactory(null);

    @LookupField(strategy = LookupField.Strategy.ANCESTORS)
    private REBDevice reb;
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy = LookupField.Strategy.SIBLINGS)
    private FitsService fitsService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private GlobalVisualizationClient globalVisualizationClient;

    private Logger log;
    private int ccdMask, numCcds, numRebCcds;
    private Image currImage = new Image();
    private ByteBuffer[][] currImageBuffer;
    private RaftException currImageExcp = new RaftException("No image data available");
    private int fitsSeqnum;
    private boolean externalSequenceNumber = false;
    private long imageTime;
    private boolean scanMode;

    private String defaultDirectory = ".";
    private String fitsFileNamePattern = "Image_${rebName}_${sensorId}_${imageName}.fits";
    private String imageDataFileNamePattern = "Image_${rebName}_${imageName}.dat";
    private final Properties fileNamePatternProperties = new Properties();
    private Reb rebGeometry;
    private volatile ReadOutParameters readOutParameters;
    private int dataInversionMask = DATA_MASK_INVERTED;
    private final BlockingQueue<Integer> imageQueue = new ArrayBlockingQueue<>(1);

    private final ImageProcFitsHeaderMetadataProvider metadataProvider = new ImageProcFitsHeaderMetadataProvider();

    /**
     * Inner class for receiving an image.
     */
    private class GetImage implements ImageClient.Listener {

        /**
         * Processes received images.
         *
         * Broadcasts a message on the status bus containing the image metadata.
         *
         * If scan mode is enabled, creates an AIDA data set from the data from
         * each amplifier and sends them on the status bus.
         */
        @Override
        public void processImage(Image image) {
            currImage = image;
            log.info("Received image: Reb = " + reb.getName() + ", Name = " + currImage.getName()
                    + ", Length = " + currImage.getLength() + ", Status = " + currImage.getStatus());
            try {
                currImageBuffer = splitImage();
                currImageExcp = null;
            } catch (RaftException e) {
                log.error(e.getMessage());
                currImageBuffer = null;
                currImageExcp = e;
                return;
            }
            imageTime = currImage.getTimestamp() / 1_000_000;
            final String timestampString = timestampString(currImage.getTimestamp());
            fileNamePatternProperties.setProperty("timestamp", timestampString);
            fileNamePatternProperties.setProperty("imageName", currImage.getName() == null ? timestampString : currImage.getName());
            readOutParameters = createReadOutParametersFromRegistersMetadata(image.getRegisters());
            imageQueue.offer(0);

            ImageState iState = new ImageState(currImage.getTimestamp(), currImage.getLength());
            KeyValueData kvd = new KeyValueData(ImageState.KEY, iState);
            subsys.publishSubsystemDataOnStatusBus(kvd);

            try {
                if (globalVisualizationClient != null && globalVisualizationClient.isConnected()) {
                    for (int ccd = 0; ccd < numCcds; ccd++) {
                        CCD ccdGeometry = rebGeometry.getChild(0, ccd);

                        //For now let's take all the data including pre/over scan regions
                        ByteBuffer ccdData = ByteBuffer.allocate(readOutParameters.getTotalParallelSize() * readOutParameters.getTotalSerialSize() * NUM_ADCS * 4);

                        for (int adc = 0; adc < NUM_ADCS; adc++) {
                            ByteBuffer buff = currImageBuffer[ccd][adc];
                            ccdData.put(buff);
                            buff.rewind();
                        }
                        ccdData.rewind();
                        log.debug("Sending image data " + image.getName());
                        globalVisualizationClient.sendImageData(image.getMetadata(), ccdData, ccdGeometry, readOutParameters);
                    }
                }
            } catch (Exception e) {
                //Do nothing and continue
                e.printStackTrace();
            }

            /*
            double [][] stats;
            try {
                stats = getImageStats();
            }
            catch (RaftException e) {
                log.error(e.getMessage());
                return;
            }
            for (int j = 0; j < stats.length / 3; j++) {
                for (int k = 0; k < 3; k++) {
                    int ix = k * stats.length / 3 + j;
                    System.out.format("    %2s-%s: %7.1f %5.1f", j, k,
                                      stats[ix][0], stats[ix][1]);
                }
                System.out.println();
            }
             */
            if (scanMode) {
                if (currImageBuffer[0][0].limit() % (4 * 256) != 0) {
                    log.warning("Scan mode image data size not a multiple of 256");
                }
                IDataPointSet[] sets = new IDataPointSet[numCcds * NUM_ADCS];
                KeyValueDataList plots = new KeyValueDataList(reb.getName() + " scan plots");
                for (int strip = 0; strip < numCcds; strip++) {
                    for (int amp = 0; amp < NUM_ADCS; amp++) {
                        int j = NUM_ADCS * strip + amp;
                        IntBuffer data = currImageBuffer[strip][amp].asIntBuffer();
                        String title = reb.getName() + ": Strip " + strip + ", Ampl " + amp;
                        IDataPointSet points = dpsf.create("", title, 2);
                        sets[j] = points;
                        for (int k = 0; k < Math.min(1024, data.limit()); k++) {
                            IDataPoint point = points.addPoint();
                            point.coordinate(0).setValue(k);
                            point.coordinate(1).setValue(data.get());
                        }
                        try {
                            plots.addData(String.valueOf(j),
                                    XMLUtils.createXMLString((IManagedObject) sets[j]),
                                    KeyValueData.KeyValueDataType.KeyValuePlotData);
                        } catch (IOException ex) {
                            log.warning("Error writing dataPointSet to xml", ex);
                        }
                    }
                }
                subsys.publishSubsystemDataOnStatusBus(plots);
            }
        }
    }

    /**
     * Constructor.
     *
     * @param imc The image client object
     */
    public ImageProc(ImageClient imc) {
        imc.setListener(new GetImage(), currImage);
        this.invertedDataSegmentMap = invertDataSegmentMap(dataSegmentMap);
        fileNamePatternProperties.setProperty("raftLoc", "99");
    }

    private int[] invertDataSegmentMap(int[] dataSegmentMap) {
        int[] result = new int[dataSegmentMap.length];
        for (int j = 0; j < result.length; j++) {
            result[dataSegmentMap[j]] = j;
        }
        return result;
    }

    void setDataSegmentMap(int[] map) {
        if (map.length != dataSegmentMap.length) {
            throw new IllegalArgumentException("Invalid map length");
        }
        int[] inverted = invertDataSegmentMap(map);
        // Sanity check
        if (!Arrays.equals(invertDataSegmentMap(inverted), map)) {
            throw new IllegalArgumentException("Invalue dataSegmentMap " + Arrays.toString(dataSegmentMap));
        }
        this.dataSegmentMap = map;
        this.invertedDataSegmentMap = inverted;
    }

    /**
     * Performs configuration.
     *
     * @param log The associated logger
     */
    public void configure(Logger log) {
        this.log = log;
        fileNamePatternProperties.setProperty("rebName", reb.getName());
    }

    /**
     * Performs configuration.
     *
     * @param rebName The REB device name
     * @param rebId The REB ID (address)
     * @param subsys The associated subsystem
     * @param log The associated logger
     */
    @Deprecated
    public void configure(String rebName, int rebId, Subsystem subsys, Logger log) {
//        this.rebName = rebName;
//        this.rebId = rebId;
        this.log = log;
        this.subsys = subsys;
        fileNamePatternProperties.setProperty("rebName", rebName);
    }

    /**
     * Set a property for resolving file name and output directory
     */
    void setPatternProperty(String propertyName, String propertyValue) {
        fileNamePatternProperties.setProperty(propertyName, propertyValue);
    }

    /**
     * Enables/disables scan mode.
     *
     * @param enable True to enable, false to disable
     */
    public void enableScan(boolean enable) {
        scanMode = enable;
    }

    /**
     * Sets the mask of CCDs being used.
     *
     * @param ccdMask The mask of active CCDs
     */
    public void setCcdMask(int ccdMask) {
        this.ccdMask = ccdMask;
        numCcds = Integer.bitCount(this.ccdMask);
    }

    /**
     * Sets the number of CCDs on the REB.
     *
     * @param numRebCcds The number of CCDs
     */
    public void setNumRebCcds(int numRebCcds) {
        this.numRebCcds = numRebCcds;
    }

    /**
     * Set the Reb Geometry corresponding to this ImageProc instance.
     *
     * @param reb The Reb Geometry.
     */
    public void setRebGeometry(Reb reb) {
        this.rebGeometry = reb;
        fitsService.setGeometry(rebGeometry);
    }

    public Reb getRebGeometry() {
        return rebGeometry;
    }

    /**
     * Sets whether image data values are inverted
     *
     * @param invert Whether data values are to be inverted
     */
    public void setDataInversion(boolean invert) {
        dataInversionMask = invert ? DATA_MASK_INVERTED : DATA_MASK_STRAIGHT;
    }

    /**
     * Initializes for a new set of images
     */
    public void initImageWait() {
        imageQueue.clear();
    }

    /**
     * Waits for an image to arrive
     *
     * @param timeout The timeout (msec)
     * @return The number of images received: 0 (if timed out) or 1
     */
    public int waitForImage(int timeout) {
        Integer value = null;
        try {
            value = imageQueue.poll(timeout, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
        }
        return value != null ? 1 : 0;
    }

    /**
     * Saves the current image as raw bytes.
     *
     * The default name of the file is "Image_[reb]_[imagename].dat", where
     * [imagename] is the name provided in the DAQ meta-data or if not specified
     * 16-digit hexadecimal representation of the image timestamp, and [reb] is
     * the name of the REB that produced the image.
     *
     * @param dName The name of the directory where the image file is to saved.
     * @return The name of the saved file
     * @throws RaftException
     * @throws IOException
     */
    public String saveImage(String dName) throws RaftException, IOException {
        if ((currImage.isInterleaved() ? currImage.getData() : currImage.getData(0)) == null) {
            throw new RaftException("Image contains no valid data");
        }

        String dirName = ((dName == null || dName.isEmpty()) ? defaultDirectory : dName) + "/";
        String filName = FileNamePatternUtils.resolveFileName(imageDataFileNamePattern, fileNamePatternProperties);
        String shortFileName = Paths.get(filName).getFileName().toString();
        setFitsFileName(shortFileName);

        try (FileOutputStream ofs = new FileOutputStream(dirName + filName)) {
            if (currImage.isInterleaved()) {
                ofs.getChannel().write(currImage.getData());
            } else {
                for (int ccd = 0; ccd < numCcds; ccd++) {
                    ofs.getChannel().write(currImage.getData(ccd));
                }
            }
        }

        return filName;
    }

    void setExternalSequenceNumber(int sequenceNumber) {
        fitsSeqnum = sequenceNumber;
        externalSequenceNumber = true;
    }

    /**
     * Saves the current image data as one or more FITS files.
     *
     * Each active CCD on the REB produces a separate file. The default name of
     * each file is "Image_[reb]_[ccd]_[imagename].fits", where [imagename] is
     * the name provided in the DAQ meta-data or if not specified the 16-digit
     * hexadecimal representation of the image timestamp, [reb] is the name of
     * the REB, and [ccd] is the CCD number.
     *
     * @param dName The name of the directory where the FITS file is to be
     * saved.
     * @param provider An external FitsHeaderMetadataProvider.
     * @return A list containing the names of the saved files
     * @throws RaftException
     * @throws IOException
     */
    public List<String> saveFitsImage(String dName, FitsHeaderMetadataProvider provider)
            throws IOException, RaftException {
        if (currImageExcp != null) {
            throw currImageExcp;
        }
        int numberOfColumns = readOutParameters.getSerialReadPixels();
        int numberOfRows = readOutParameters.getParallelReadPixels();

        if (currImageBuffer[0][0].limit() != 4 * numberOfColumns * numberOfRows) {
            throw new RaftException("Image size is inconsistent with slice count. Size: "
                    + (currImageBuffer[0][0].limit() / 4) + ", expected: "
                    + (numberOfColumns * numberOfRows)
                    + " (" + numberOfColumns + "x" + numberOfRows + ")");
        }
        List<String> names = new ArrayList<>();
        String dirPatt = ((dName == null || dName.isEmpty()) ? defaultDirectory : dName) + "/";
        for (int ccd = 0; ccd < numCcds; ccd++) {
            CCD ccdGeometry = rebGeometry.getChild(0, ccd);

            //TO-DO: can this be stored/built more efficiently? When the image is changed?
            DefaultImageSet imageSet = new DefaultImageSet(ccdGeometry, readOutParameters);

            fitsService.setHeaderKeywordValue("SequenceNumber", externalSequenceNumber ? fitsSeqnum : ++fitsSeqnum);
            fitsService.setHeaderKeywordValue("FileCreationTime", new Date(System.currentTimeMillis()));
            fitsService.setHeaderKeywordValue("ObservationDate", new Date(imageTime));
            fitsService.setHeaderKeywordValue("Tag", String.valueOf(imageTime));

            try {
                ImageName in = new ImageName(currImage.getName());
                fitsService.setHeaderKeywordValue("ImageName", in.toString());
                fitsService.setHeaderKeywordValue("ImageDate", in.getDateString());
                fitsService.setHeaderKeywordValue("ImageNumber", in.getNumber());
                fitsService.setHeaderKeywordValue("ImageController", in.getController().getCode());
                fitsService.setHeaderKeywordValue("ImageSource", in.getSource().getCode());
            } catch (IllegalArgumentException x) {
                log.debug("Error parsing "+currImage.getName(), x);
            }

            FitsFileWriter writer;
            List<FitsHeaderMetadataProvider> providers = new ArrayList<>();
            providers.add(metadataProvider);
            providers.add(fitsService);
            if (provider != null) {
                providers.add(provider);
            }

            //Convert the metadata providers to properties
            Properties resolverProperties = getPropertiesFromMetadataProviders(providers, imageSet);

            //Resolve the fileName and output directory pattern.
            fileNamePatternProperties.setProperty("sensorId", String.valueOf(ccd));
            fileNamePatternProperties.setProperty("sensorLoc", String.valueOf(reb.getRebNumber()) + String.valueOf(ccd));
            try {
                fileNamePatternProperties.setProperty("testType", resolverProperties.getProperty("TestType").toLowerCase());
            } catch (RuntimeException ex) {
            }
            try {
                fileNamePatternProperties.setProperty("imageType", resolverProperties.getProperty("ImageType").toLowerCase());
            } catch (RuntimeException ex) {
            }

            resolverProperties.putAll(fileNamePatternProperties);

            String fileName = FileNamePatternUtils.resolveFileName(fitsFileNamePattern, resolverProperties);
            String shortFileName = Paths.get(fileName).getFileName().toString();
            setFitsFileName(shortFileName);

            String dirName = FileNamePatternUtils.resolveFileName(dirPatt, resolverProperties);
            File file = new File(dirName, fileName);
            file.getParentFile().mkdirs();
            try {
                writer = new FitsFileWriter(file, imageSet, fitsService.getHeaderSpecificationMap(), providers);
            } catch (FitsException e) {
                throw new RaftException("FITS error: " + e);
            }
            for (int adc = 0; adc < NUM_ADCS; adc++) {
                ByteBuffer buff = currImageBuffer[ccd][adc];
                buff.rewind();
                writer.write(dataSegmentMap[adc], buff);
            }
            writer.close();
            names.add(file.getName());
        }
        fitsService.clearNonStickyHeaderKeywordValues();
        return names;
    }

    public List<String> saveFitsImage(String dName)
            throws IOException, RaftException {
        return saveFitsImage(dName, null);
    }

    private Properties getPropertiesFromMetadataProviders(List<FitsHeaderMetadataProvider> providers, ImageSet imageSet) {
        Properties p = new Properties();
        for (FitsHeaderMetadataProvider provider : providers) {
            MetaDataSet md = provider.getPrimaryHeaderMetadata(imageSet);
            if (md != null) {
                p.putAll(md.convertToProperties());
            }
        }
        return p;
    }

    /**
     * Gets the DAQ metadata for the current image.
     *
     * @return The image metadata
     * @throws RaftException
     */
    public ImageMetadata getImageMetadata() throws RaftException {
        return currImage.getMetadata();
    }

    /**
     * Gets a portion of the current image.
     *
     * @param ccd The CCD number
     * @param offset The offset (in pixels) to the first pixel data to obtain.
     * @param count The number of data pixels to obtain. If zero, all the data,
     * starting at offset, is obtained.
     * @return The returned pixel data
     * @throws RaftException
     */
    public ImageData getImage(int ccd, int offset, int count) throws RaftException {

        if ((currImage.isInterleaved() ? currImage.getData() : currImage.getData(0)) == null) {
            throw new RaftException("Image contains no valid data");
        }

        int length = count;
        int iLength = currImage.getLength();
        if (length < 0 || offset < 0 || offset >= iLength) {
            throw new RaftException("Invalid length or offset");
        }
        if (length == 0 || length + offset > iLength) {
            length = iLength - offset;
        }
        int[] data = new int[length];
        ByteBuffer buff = currImage.isInterleaved() ? currImage.getData() : currImage.getData(ccd);
        buff.order(ByteOrder.LITTLE_ENDIAN);
        buff.position(4 * offset);
        buff.asIntBuffer().get(data);

        return new ImageData(currImage.getTimestamp(), offset, data);
    }

    /**
     * Gets a portion of a saved raw image.
     *
     * @param fileName The name of the file containing the image
     * @param offset The offset (in pixels) to the first pixel data to obtain.
     * @param count The number of data pixels to obtain.
     * @return The returned pixel data
     * @throws RaftException
     * @throws IOException
     */
    static ImageData getImage(String fileName, int offset, int count)
            throws RaftException, IOException {

        offset *= 4;
        int length = 4 * count;
        FileInputStream ifs = new FileInputStream(fileName);
        ByteBuffer buff;
        try {
            FileChannel chan = ifs.getChannel();
            long iLength = chan.size();
            if (length < 0 || offset < 0 || offset >= iLength) {
                throw new RaftException("Invalid length or offset");
            }
            if (length == 0 || length + offset > iLength) {
                length = (int) (iLength - offset);
            }
            buff = chan.map(FileChannel.MapMode.READ_ONLY, offset, length);
        } finally {
            ifs.close();
        }
        int[] data = new int[length / 4];
        buff.order(ByteOrder.LITTLE_ENDIAN);
        buff.asIntBuffer().get(data);

        return new ImageData(0, offset / 4, data);
    }

    /**
     * Gets pixel value statistics for the current image.
     *
     * @return The array of (average, stddev) pairs, one per image segment
     * @throws RaftException
     */
    public double[][] getImageStats() throws RaftException {
        if (currImageExcp != null) {
            throw currImageExcp;
        }
        double[][] stats = new double[numCcds * NUM_ADCS][2];
        for (int ccd = 0; ccd < numCcds; ccd++) {
            for (int adc = 0; adc < NUM_ADCS; adc++) {
                double sum = 0.0, sumSq = 0.0;
                ByteBuffer buff = currImageBuffer[ccd][adc];
                int count = buff.limit() / 4;
                for (int k = 0; k < count; k++) {
                    double value = buff.getInt();
                    sum += value;
                    sumSq += value * value;
                }
                double avg = sum / count;
                int j = ccd * NUM_ADCS + adc;
                stats[j][0] = avg;
                stats[j][1] = Math.sqrt(sumSq / count - avg * avg);
            }
        }

        return stats;
    }

    /**
     * Splits the current image into its segments.
     *
     * Each active CCD on the REB produces 16 segments.
     *
     * @return The array of image segment buffers
     * @throws RaftException
     */
    public ByteBuffer[][] splitImage() throws RaftException {
        ByteBuffer data = currImage.isInterleaved() ? currImage.getData() : currImage.getData(0);
        if (data == null || currImage.getStatus() != Image.STATUS_GOOD) {
            throw new RaftException("Image contains no valid data");
        }
        int numSeg = numCcds * NUM_ADCS;
        int buffSegs = currImage.isInterleaved() ? numSeg : NUM_ADCS;
        if (data.limit() % (4 * buffSegs) != 0) {
            throw new RaftException(("Invalid image size: " + data.limit()));
        }
        int segSize = data.limit() / buffSegs;
        ByteBuffer[][] adata = new ByteBuffer[numCcds][NUM_ADCS];
        for (int ccd = 0; ccd < numCcds; ccd++) {
            for (int adc = 0; adc < NUM_ADCS; adc++) {
                adata[ccd][adc] = ByteBuffer.allocate(segSize);
            }
        }
        if (currImage.isInterleaved()) {
            data.order(ByteOrder.nativeOrder());
            for (int j = 0; j < segSize / 4; j++) {
                for (int ccd = 0; ccd < numCcds; ccd++) {
                    int actCcd = reb.getRebType() != BaseSet.REB_TYPE_SCIENCE || numCcds < 3 ? ccd : numRebCcds - 1 - ccd;
                    for (int adc = 0; adc < NUM_ADCS; adc++) {
                        adata[actCcd][adc].putInt(data.getInt() ^ dataInversionMask);  // Raw values are usually inverted
                    }
                }
            }
        } else {
            ByteBuffer[] cdata = new ByteBuffer[numCcds];
            for (int ccd = 0; ccd < numCcds; ccd++) {
                cdata[ccd] = currImage.getData(ccd);
                cdata[ccd].order(ByteOrder.nativeOrder());
            }
            for (int j = 0; j < segSize / 4; j++) {
                for (int ccd = 0; ccd < numCcds; ccd++) {
                    for (int adc = 0; adc < NUM_ADCS; adc++) {
                        adata[ccd][adc].putInt(cdata[ccd].getInt() ^ dataInversionMask);  // Raw values are usually inverted
                    }
                }
            }
        }
        for (int ccd = 0; ccd < numCcds; ccd++) {
            for (int adc = 0; adc < NUM_ADCS; adc++) {
                adata[ccd][adc].flip();
            }
        }
        return adata;
    }

    /**
     * Sets constant FITS metadata.
     *
     * @param serial The REB serial number
     */
    private void setFitsFileName(String fitsFileName) {
        fitsService.setHeaderKeywordValue("OriginalFileName", fitsFileName);
    }

    /**
     * Sets the default image directory.
     *
     * @param dirName The directory name
     */
    public void setDefaultImageDirectory(String dirName) {
        defaultDirectory = dirName == null || dirName.isEmpty() ? "." : dirName;
    }

    /**
     * Sets the FITS image file name pattern.
     *
     * @param pattern The file name pattern to set
     */
    public void setFitsFileNamePattern(String pattern) {
        fitsFileNamePattern = pattern;
    }

    /**
     * Sets the raw image data file name pattern.
     *
     * @param pattern The file name pattern to set
     */
    public void setImageDataFileNamePattern(String pattern) {
        imageDataFileNamePattern = pattern;
    }

    /**
     * Get the corresponding FitsService
     */
    public FitsService getFitsService() {
        return fitsService;
    }

    /**
     * Converts image timestamp (ns) to canonical string (YYYYMMDDHHMMSS).
     */
    private static String timestampString(long tstamp) {
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(tstamp / 1_000_000);
        cal.setTimeZone(TimeZone.getTimeZone("GMT"));
        return String.format("%tY%<tm%<td%<tH%<tM%<tS", cal);
    }

    private class ImageProcFitsHeaderMetadataProvider implements FitsHeaderMetadataProvider {

        @Override
        public MetaDataSet getPrimaryHeaderMetadata(ImageSet imageSet) {
            if (rebGeometry.getChildrenList().contains(imageSet.getCCD())) {
                MetaDataSet m = new MetaDataSet();
                m.addMetaData("primary", "CCDManufacturer", imageSet.getCCD().getType().getName().toUpperCase());
                m.addMetaData("primary", "CCDModel", imageSet.getCCD().getType().getType());
                return m;
            }
            return null;
        }

        @Override
        public MetaDataSet getDataExtendedHeaderMetadata(ImageSet imageSet, int extendedIndex) {
            if (rebGeometry.getChildrenList().contains(imageSet.getCCD())) {
                MetaDataSet segmentMetaDataSet = new MetaDataSet();
                Map<String, Object> metadata = new HashMap<>();
                try {
                    analyzeImage(imageSet.getCCD().getSegments().get(extendedIndex),
                            imageSet.getCCD().getSerialPosition(), extendedIndex, metadata);
                } catch (RaftException ex) {
                    log.error(ex);
                }
                segmentMetaDataSet.addMetaDataMap("seg", metadata);
                return segmentMetaDataSet;
            }
            return null;
        }

        @Override
        public MetaDataSet getAdditionalExtendedHeaderMetadata(ImageSet imageSet, String extendedKeyword) {
            return null;
        }
    }

    /*
     * This map determines the DAQ CCD index (0, 1 or 2) within the image data
     * for a specified CCD number.  Since it's possible that not all CCDs were
     * read out, the value of the mask of read CCDs (0 - 7) is needed.
     *
     * The three rows of the map correspond to the three CCDs.  Each row contains
     * eight entries, one for each possible value of the CCD mask.  Negative
     * values correspond to invalid combinations.  Non-negative values provide
     * the desired mapping.
     *
     * So a given column in the table below formed by the initialization values
     * contains the index within the data for each CCD, numbering from the top.
     * E.g. the "5" column (0, -1, 1) shows that CCD 0 is at index 0, CCD 1 is
     * not present, and CCD 2 is at index 1.
     */
    private static final int[][] CCD_MAP = {{-1, 0, -1, 0, -1, 0, -1, 0},
    {-1, -1, 0, 1, -1, -1, 0, 1},
    {-1, -1, -1, -1, 0, 1, 1, 2}};

    /**
     * Calculates image statistics for a given segment for the active and
     * overscan regions separately. The results are added to the metadata for
     * the segment header.
     */
    private void analyzeImage(Segment segment, int ccdNum, int segNum, Map<String, Object> imageMetaData)
            throws RaftException {

        if (currImageExcp != null) {
            throw currImageExcp;
        }

        int daqCcdNum = CCD_MAP[ccdNum][ccdMask];
        if (daqCcdNum < 0) {
            throw new RaftException("Invalid CCD number: " + ccdNum + "; CCD mask = " + ccdMask);
        }
        ByteBuffer buff = currImageBuffer[daqCcdNum][invertedDataSegmentMap[segNum]];
        buff.rewind();

        double pix_sum_active = 0.;
        double pix_sum_sq_active = 0.;
        int npix_active = 0;
        double pix_sum_bias = 0.;
        double pix_sum_sq_bias = 0.;
        int npix_bias = 0;

        //Total serial read pixels: readCols + readCols2 + overCols
        int numberOfColumns = readOutParameters.getSerialReadPixels();
        //Total parallel read pixels: readRows + overRows
        int numberOfRows = readOutParameters.getParallelReadPixels();

        //The total serial active size. These are the active pixels and don't
        //include the serial prescan pixels.
        int segmentSerialActiveSize = segment.getSegmentSerialActiveSize();

        //The serial prescan. This depends on the CCD type
        int serialPrescan = readOutParameters.getSerialPrescan();

        for (int row = 1; row <= numberOfRows; row++) {
            for (int col = 1; col <= numberOfColumns; col++) {
                double pixval = buff.getInt();
//                int pixval = segment.getImagePixelData().getPixelData(row, col);
                if (col > serialPrescan && col <= (serialPrescan + segmentSerialActiveSize)) {
                    pix_sum_active += pixval;
                    pix_sum_sq_active += (double) pixval * pixval;
                    npix_active++;
                } else if (col > (serialPrescan + segmentSerialActiveSize)) {
                    pix_sum_bias += pixval;
                    pix_sum_sq_bias += (double) pixval * pixval;
                    npix_bias++;
                }
                //if (row == 100) {
                //    System.out.println(preCols + " " + segmentSerialActiveSize + " : " + col + " " + pixval);
                //}
            }
        }

        log.debug("Active " + npix_active + " Bias " + npix_bias + " PreCols " + serialPrescan + " ActiveSize " + segmentSerialActiveSize + " NCols " + numberOfColumns + " NRows " + numberOfRows);

        if (npix_active > 0) {
            double average = (double) pix_sum_active / npix_active;
            double stdev = Math.sqrt(pix_sum_sq_active / npix_active - (average * average));
            log.debug("Adding AVERAGE " + average + " STDEV " + stdev);
            imageMetaData.put("AVERAGE", average);
            imageMetaData.put("STDEV", stdev);
        }
        if (npix_bias > 0) {
            double average = (double) pix_sum_bias / npix_bias;
            double stdev = Math.sqrt(pix_sum_sq_bias / npix_bias - (average * average));
            log.debug("Adding AVGBIAS " + average + " STDVBIAS " + stdev);
            imageMetaData.put("AVGBIAS", average);
            imageMetaData.put("STDVBIAS", stdev);
        }

    }

    private ReadOutParameters createReadOutParametersFromRegistersMetadata(int[] registers) {
        CCDType ccdType = rebGeometry.getChildrenList().get(0).getType();
        if (ccdType == null) {
            throw new RuntimeException("The CCD Type must have been set to evaluate Read Out Parameters");
        }
        int segRows = ccdType.getCCDGeometryConstants().getSegmentParallelActiveSize();
        int serCols = ccdType.getCCDGeometryConstants().getSegmentSerialActiveSize() + ccdType.getCCDGeometryConstants().getSegmentSerialPrescanSize();

        try {
            int preRows = registers[SequencerData.REG_PRE_ROWS];
            int readRows = registers[SequencerData.REG_READ_ROWS];
            int postRows = registers[SequencerData.REG_POST_ROWS];
            int overRows = registers[SequencerData.REG_OVER_ROWS];
            int preCols = registers[SequencerData.REG_PRE_COLS];
            int readCols = registers[SequencerData.REG_READ_COLS];
            int postCols = registers[SequencerData.REG_POST_COLS];
            int readCols2 = registers[SequencerData.REG_READ_COLS2];
            int overCols = registers[SequencerData.REG_OVER_COLS];
            //log.info("Got readout parameters from registers : "+preRows+" "+readRows+" "+postRows+" "+overRows+" "+preCols+" "+readCols+" "+postCols+" "+readCols2+" "+overCols);
            if (preRows == 0 && postRows == 0 && preCols == 0 && postCols == 0 && readCols2 == 0) {
                if (overRows == 0 && readRows > segRows) {
                    overRows = readRows - segRows;
                    readRows = segRows;
                }
                if (overCols == 0 && readCols > serCols) {
                    overCols = readCols - serCols;
                    readCols = serCols;
                }
            }
            if (readCols != 0 && readRows != 0) {
                //log.info("Creating readout parameters object : "+preRows+" "+readRows+" "+overRows+" "+overRows+" "+preCols+" "+readCols+" "+postCols+" "+readCols2+" "+overCols);
                return new ReadOutParameters(ccdType, preRows, readRows, postRows, overRows, preCols, readCols, postCols, readCols2, overCols);
            } else {
                log.info("Reverting back to default readout parameters " + readRows + " " + readCols);
                return new ReadOutParameters(ccdType);
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.info("Got Exception: Reverting back to default readout parameters " + e.getMessage());
            return new ReadOutParameters(ccdType);
        }
    }

}
