package org.lsst.ccs.subsystem.rafts;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.drivers.reb.Asic;
import org.lsst.ccs.drivers.reb.BoardDacs;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.subsystem.rafts.config.DACS;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements control of the DACs on a REB.
 *
 *  @author Owen Saxton
 */
public class DacControl extends Control implements ConfigurationBulkChangeHandler {

    /*
     *  Constants.
     */
    public static final String
        SCLK_LOW_P   = "sclkLowP",
        SCLK_HIGH_P  = "sclkHighP",
        PCLK_LOW_P   = "pclkLowP",
        PCLK_HIGH_P  = "pclkHighP",
        RG_LOW_P     = "rgLowP",
        RG_HIGH_P    = "rgHighP",

        SCLK_LOW     = "sclkLow",
        SCLK_LOW_SH  = "sclkLowSh",
        SCLK_HIGH    = "sclkHigh",
        SCLK_HIGH_SH = "sclkHighSh",
        PCLK_LOW     = "pclkLow",
        PCLK_LOW_SH  = "pclkLowSh",
        PCLK_HIGH    = "pclkHigh",
        PCLK_HIGH_SH = "pclkHighSh",
        RG_LOW       = "rgLow",
        RG_LOW_SH    = "rgLowSh",
        RG_HIGH      = "rgHigh",
        RG_HIGH_SH   = "rgHighSh",
        CS_GATE      = "csGate";

    private static final int
        SCLK_LOW_CH     = 0x00001,
        SCLK_LOW_SH_CH  = 0x00002,
        SCLK_HIGH_CH    = 0x00004,
        SCLK_HIGH_SH_CH = 0x00008,
        PCLK_LOW_CH     = 0x00010,
        PCLK_LOW_SH_CH  = 0x00020,
        PCLK_HIGH_CH    = 0x00040,
        PCLK_HIGH_SH_CH = 0x00080,
        RG_LOW_CH       = 0x00100,
        RG_LOW_SH_CH    = 0x00200,
        RG_HIGH_CH      = 0x00400,
        RG_HIGH_SH_CH   = 0x00800,
        CS_GATE_CH      = 0x01000;

    private static final String[]
        checkList = {SCLK_LOW, SCLK_HIGH, PCLK_LOW, PCLK_HIGH, RG_LOW, RG_HIGH};

    private static final int DEFAULT_LOAD_WAIT = 200;

    /*
    *  Data fields.
    */
    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    // Supplied configurable fields - rail voltages
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock low", units="volts")
    private double sclkLowP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock high", units="volts")
    private double sclkHighP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock low", units="volts")
    private double pclkLowP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock high", units="volts")
    private double pclkHighP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate low", units="volts")
    private double rgLowP;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate high", units="volts")
    private double rgHighP;

    // Configurable rail voltage limits
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Serial clock low min", units="volts")
    private double sclkLowMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Serial clock low max", units="volts")
    private double sclkLowMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Serial clock high min", units="volts")
    private double sclkHighMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Serial clock high max", units="volts")
    private double sclkHighMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Parallel clock low min", units="volts")
    private double pclkLowMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Parallel clock low max", units="volts")
    private double pclkLowMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Parallel clock high min", units="volts")
    private double pclkHighMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Parallel clock high max", units="volts")
    private double pclkHighMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Reset gate low min", units="volts")
    private double rgLowMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Reset gate low max", units="volts")
    private double rgLowMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Reset gate high min", units="volts")
    private double rgHighMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS, description="Reset gate high max", units="volts")
    private double rgHighMax;

    // Supplied configurable fields - raw DAC settings
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock low", units= "DAC counts")
    private int    sclkLow;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock low shifted", units= "DAC counts")
    private int    sclkLowSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock high", units= "DAC counts")
    private int    sclkHigh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Serial clock high shifted", units= "DAC counts")
    private int    sclkHighSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock low", units= "DAC counts")
    private int    pclkLow;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock low shifted", units= "DAC counts")
    private int    pclkLowSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock high", units= "DAC counts")
    private int    pclkHigh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Parallel clock high shifted", units= "DAC counts")
    private int    pclkHighSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate low", units= "DAC counts")
    private int    rgLow;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate low shifted", units= "DAC counts")
    private int    rgLowSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate high", units= "DAC counts")
    private int    rgHigh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Reset gate high shifted", units= "DAC counts")
    private int    rgHighSh;
    @ConfigurationParameter(category=REBDevice.RAFTS, description="Current source", units= "DAC counts")
    private final int[]  csGate = new int[Asic.MAX_STRIPS];

    // Set from Groovy file
    private boolean raw = false;

    // Other fields
    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private final int version;
    private boolean cfgValid, isCurrent, isCorner;
    private int hwVersion;
    private REBDevice rebDevc;
    private BoardDacs dac;
    private int changed = -1;
    private final Map<String, Double> lowLimitMap = new HashMap<>();
    private final Map<String, Double> highLimitMap = new HashMap<>();
    private String configError;
    private double railShiftConvH, railConvH, railShiftConvL, railConvL;


    /**
     *  No-argument constructor
     */
    public DacControl()
    {
        super();
        for (String pName : checkList) {
            lowLimitMap.put(pName, 0.0);
            highLimitMap.put(pName, 0.0);
        }
        version = -1;
    }


    /**
     *  Constructor, physical values.
     *
     *  @param  desc
     *  @param  hwChan
     *  @param  sclkLowP
     *  @param  sclkHighP
     *  @param  pclkLowP
     *  @param  pclkHighP
     *  @param  rgLowP
     *  @param  rgHighP
     */
    @Deprecated
    public DacControl(String desc, int hwChan, double sclkLowP, double sclkHighP,
                      double pclkLowP, double pclkHighP, double rgLowP, double rgHighP)
    {
        this();
        this.description = desc;
        this.hwChan = hwChan;
        this.sclkLowP = sclkLowP;
        this.sclkHighP = sclkHighP;
        this.pclkLowP = pclkLowP;
        this.pclkHighP = pclkHighP;
        this.rgLowP = rgLowP;
        this.rgHighP = rgHighP;
        raw = false;
    }


    /**
     *  Constructor, raw DAC values, version 0 (ancient science raft).
     *
     *  @param  desc
     *  @param  hwChan
     *  @param  sclkLow
     *  @param  sclkHigh
     *  @param  pclkLow
     *  @param  pclkHigh
     *  @param  rgLow
     *  @param  rgHigh
     *  @param  csGate
     */
    @Deprecated
    public DacControl(String desc, int hwChan,
                      int sclkLow, int sclkHigh, int pclkLow,
                      int pclkHigh, int rgLow, int rgHigh,
                      int[] csGate)
    {
        super();
        this.description = desc;
        this.hwChan = hwChan;
        this.sclkLow = sclkLow;
        this.sclkHigh = sclkHigh;
        this.pclkLow = pclkLow;
        this.pclkHigh = pclkHigh;
        this.rgLow = rgLow;
        this.rgHigh = rgHigh;
        System.arraycopy(csGate, 0, this.csGate, 0, this.csGate.length);
        version = REB.VERSION_0;
        raw = true;
    }


    /**
     *  Constructor, RAW DAC values, version 1 (corner raft).
     *
     *  @param  desc
     *  @param  hwChan
     *  @param  sclkLow
     *  @param  sclkLowSh
     *  @param  sclkHigh
     *  @param  sclkHighSh
     *  @param  pclkLow
     *  @param  pclkLowSh
     *  @param  pclkHigh
     *  @param  pclkHighSh
     *  @param  rgLow
     *  @param  rgLowSh
     *  @param  rgHigh
     *  @param  rgHighSh
     */
    @Deprecated
    public DacControl(String desc, int hwChan,
                      int sclkLow, int sclkLowSh, int sclkHigh,
                      int sclkHighSh, int pclkLow, int pclkLowSh,
                      int pclkHigh, int pclkHighSh, int rgLow,
                      int rgLowSh, int rgHigh, int rgHighSh)
    {
        super();
        this.description = desc;
        this.hwChan = hwChan;
        this.sclkLow = sclkLow;
        this.sclkLowSh = sclkLowSh;
        this.sclkHigh = sclkHigh;
        this.sclkHighSh = sclkHighSh;
        this.pclkLow = pclkLow;
        this.pclkLowSh = pclkLowSh;
        this.pclkHigh = pclkHigh;
        this.pclkHighSh = pclkHighSh;
        this.rgLow = rgLow;
        this.rgLowSh = rgLowSh;
        this.rgHigh = rgHigh;
        this.rgHighSh = rgHighSh;
        version = REB.VERSION_1;
        raw = true;
    }


    /**
     *  Constructor, raw DAC values, version 2 (science raft).
     *
     *  @param  desc
     *  @param  hwChan
     *  @param  sclkLow
     *  @param  sclkLowSh
     *  @param  sclkHigh
     *  @param  pclkLow
     *  @param  pclkLowSh
     *  @param  pclkHigh
     *  @param  rgLow
     *  @param  rgLowSh
     *  @param  rgHigh
     */
    @Deprecated
    public DacControl(String desc, int hwChan,
                      int sclkLow, int sclkLowSh, int sclkHigh,
                      int pclkLow, int pclkLowSh, int pclkHigh,
                      int rgLow, int rgLowSh, int rgHigh)
    {
        super();
        this.description = desc;
        this.hwChan = hwChan;
        this.sclkLow = sclkLow;
        this.sclkLowSh = sclkLowSh;
        this.sclkHigh = sclkHigh;
        this.pclkLow = pclkLow;
        this.pclkLowSh = pclkLowSh;
        this.pclkHigh = pclkHigh;
        this.rgLow = rgLow;
        this.rgLowSh = rgLowSh;
        this.rgHigh = rgHigh;
        version = REB.VERSION_2;
        raw = true;
    }


    /**
     *  Configures DAC description.
     *
     *  @param  mon   The associated monitor object
     *  @param  devc  The associated device object
     */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        if (configError != null) {
            sLog.error("Config error (startup): " + getName() + ": " + configError);
        }
        rebDevc = (REBDevice)devc;
        dac = rebDevc.getBoardDacs();
    }


    /**
     *  Checks configuration.
     *
     *  @return  Whether configuration is okay
     */
    public boolean checkConfig()
    {
        hwVersion = getHwVersion(dac);
        isCurrent = hwVersion >= REB.VERSION_1;
        isCorner = hwVersion == REB.VERSION_1 || hwVersion == REB.VERSION_3;
        if (raw && version >= 0 && version != hwVersion || !raw && !isCurrent) {
            sLog.error("DAC configuration incompatible with the firmware");
            return false;
        }

        if (!raw) {
            double shiftGainH = (hwVersion == REB.VERSION_1) ? 20.0 / 10.0 : 24.9 / 10.0;
            double shiftGainL = (hwVersion == REB.VERSION_2) ? 24.9 / 10.09 : shiftGainH;
            double fudge = (hwVersion == REB.VERSION_1) ? 1.0 : 1.04;
            railShiftConvH = fudge * REBDevice.DAC_CONV / shiftGainH;
            railConvH = fudge * REBDevice.DAC_CONV / (1 + shiftGainH);
            railShiftConvL = fudge * REBDevice.DAC_CONV / shiftGainL;
            railConvL = fudge * REBDevice.DAC_CONV / (1 + shiftGainL);
        }
        else {
            cfgValid = true;
        }

        return true;
    }


    /**
     *  Gets the raw state.
     *
     *  @return  Whether data are raw ADC values
     */
    public boolean isRaw()
    {
        return raw;
    }


    /**
     *  Gets the hardware version.
     *
     *  @param  dac  The board DACs object
     *  @return  The hardware version
     */
    public static int getHwVersion(BoardDacs dac)
    {
        int version = dac.getVersion();
        if (version == BoardDacs.VERSION_REB0_0
              || version == BoardDacs.VERSION_REB0_1) {
            version = REB.VERSION_0;
        }
        else if (version == BoardDacs.VERSION_WREB1
                  || version == BoardDacs.VERSION_WREB2
                  || version == BoardDacs.VERSION_GREB1) {
            version = REB.VERSION_1;
        }
        else if (version == BoardDacs.VERSION_REB3
                  || version == BoardDacs.VERSION_REB5) {
            version = REB.VERSION_2;
        }
        else if (version == BoardDacs.VERSION_GREB2
                  || version == BoardDacs.VERSION_WREB4) {
            version = REB.VERSION_3;
        }
        else {
            version = REB.VERSION_UNSUPP;
        }
        return version;
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version
     */
    @Deprecated
    public int getHwVersion()
    {
        return hwVersion;
    }


    /**
     *  Validates configuration changes
     *
     *  @param  params  Map of parameter values
     *  @throws IllegalArgumentException 
     */
    @Override
    public void validateBulkChange(Map<String, Object> params) throws IllegalArgumentException
    {
        //System.out.println(getName() + " validateBulkChange called");
        String errMsg = null;
        if (!raw) {
            for (String pName : checkList) {
                double value = (Double)params.get(pName + "P");
                if (value > (Double)params.get(pName + "Max")) {
                    errMsg = pName + "P is above high limit";
                    break;
                }
                if (value < (Double)params.get(pName + "Min")) {
                    errMsg = pName + "P is below low limit";
                    break;
                }
            }
            if (errMsg != null) {
                errMsg = getName() + ": " + errMsg;
            }
        }
        if (errMsg != null) {
            sLog.error("Configuration failure: " + errMsg);
            throw new IllegalArgumentException(errMsg);
        }
        else {
            cfgValid = true;
        }
    }


    /**
     *  Sets the low parallel clock rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkLowP(double value)
    {
        pclkLowP = value;
    }


    /**
     *  Gets the low parallel clock rail voltage.
     *
     *  @return  The value
     */
    public double getPclkLowP()
    {
        return pclkLowP;
    }


    /**
     *  Sets the low parallel clock rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkLowMax(double value)
    {
        pclkLowMax = value;
        highLimitMap.put(PCLK_LOW, value);
    }


    /**
     *  Sets the low parallel clock rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkLowMin(double value)
    {
        pclkLowMin = value;
        lowLimitMap.put(PCLK_LOW, value);
    }


    /**
     *  Sets the high parallel clock rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkHighP(double value)
    {
        pclkHighP = value;
    }


    /**
     *  Gets the high parallel clock rail voltage.
     *
     *  @return  The value
     */
    public double getPclkHighP()
    {
        return pclkHighP;
    }


    /**
     *  Sets the high parallel clock rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkHighMax(double value)
    {
        pclkHighMax = value;
        highLimitMap.put(PCLK_HIGH, value);
    }


    /**
     *  Sets the high parallel clock rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkHighMin(double value)
    {
        pclkHighMin = value;
        lowLimitMap.put(PCLK_HIGH, value);
    }


    /**
     *  Sets the low serial clock rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkLowP(double value)
    {
        sclkLowP = value;
    }


    /**
     *  Gets the low serial clock rail voltage.
     *
     *  @return  The value
     */
    public double getSclkLowP()
    {
        return sclkLowP;
    }


    /**
     *  Sets the low serial clock rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkLowMax(double value)
    {
        sclkLowMax = value;
        highLimitMap.put(SCLK_LOW, value);
    }


    /**
     *  Sets the low serial clock rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkLowMin(double value)
    {
        sclkLowMin = value;
        lowLimitMap.put(SCLK_LOW, value);
    }


    /**
     *  Sets/Gets the high serial clock rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkHighP(double value)
    {
        sclkHighP = value;
    }

    public double getSclkHighP()
    {
        return sclkHighP;
    }


    /**
     *  Sets the high serial clock rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkHighMax(double value)
    {
        sclkHighMax = value;
        highLimitMap.put(SCLK_HIGH, value);
    }


    /**
     *  Sets the high serial clock rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkHighMin(double value)
    {
        sclkHighMin = value;
        lowLimitMap.put(SCLK_HIGH, value);
    }


    /**
     *  Sets the low reset gate rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgLowP(double value)
    {
        rgLowP = value;
    }


    /**
     *  Gets the low reset gate rail voltage.
     *
     *  @return  The value
     */
    public double getRgLowP()
    {
        return rgLowP;
    }


    /**
     *  Sets the low reset gate rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgLowMax(double value)
    {
        rgLowMax = value;
        highLimitMap.put(RG_LOW, value);
    }


    /**
     *  Sets the low reset gate rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgLowMin(double value)
    {
        rgLowMin = value;
        lowLimitMap.put(RG_LOW, value);
    }


    /**
     *  Sets the high reset gate rail voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgHighP(double value)
    {
        rgHighP = value;
    }


    /**
     *  Gets the high reset gate rail voltage.
     *
     *  @return  The value
     */
    public double getRgHighP()
    {
        return rgHighP;
    }


    /**
     *  Sets the high reset gate rail voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgHighMax(double value)
    {
        rgHighMax = value;
        highLimitMap.put(RG_HIGH, value);
    }


    /**
     *  Sets the high reset gate rail voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgHighMin(double value)
    {
        rgHighMin = value;
        lowLimitMap.put(RG_HIGH, value);
    }


    /**
     *  Sets/Gets the low parallel clock rail.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkLow(int value)
    {
        pclkLow = value;
        changed |= PCLK_LOW_CH;
    }

    public int getPclkLow()
    {
        return pclkLow;
    }


    /**
     *  Sets/Gets the high parallel clock rail.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkHigh(int value)
    {
        pclkHigh = value;
        changed |= PCLK_HIGH_CH;
    }

    public int getPclkHigh()
    {
        return pclkHigh;
    }


    /**
     *  Sets/Gets the low serial clock rail.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkLow(int value)
    {
        sclkLow = value;
        changed |= SCLK_LOW_CH;
    }

    public int getSclkLow()
    {
        return sclkLow;
    }


    /**
     *  Sets/Gets the high serial clock rail.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkHigh(int value)
    {
        sclkHigh = value;
        changed |= SCLK_HIGH_CH;
    }

    public int getSclkHigh()
    {
        return sclkHigh;
    }


    /**
     *  Sets/Gets the low reset gate rail.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgLow(int value)
    {
        rgLow = value;
        changed |= RG_LOW_CH;
    }

    public int getRgLow()
    {
        return rgLow;
    }


    /**
     *  Sets/Gets the high reset gate rail.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgHigh(int value)
    {
        rgHigh = value;
        changed |= RG_HIGH_CH;
    }

    public int getRgHigh()
    {
        return rgHigh;
    }


    /**
     *  Sets/Gets the current source gates.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setCsGate(int[] value)
    {
        System.arraycopy(value, 0, csGate, 0, csGate.length);
        changed |= CS_GATE_CH;
    }

    public int[] getCsGate()
    {
        return csGate;
    }


    /**
     *  Sets/Gets the low parallel clock rail shift.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkLowSh(int value)
    {
        pclkLowSh = value;
        changed |= PCLK_LOW_SH_CH;
    }

    public int getPclkLowSh()
    {
        return pclkLowSh;
    }


    /**
     *  Sets/Gets the high parallel clock rail shift.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkHighSh(int value)
    {
        pclkHighSh = value;
        changed |= PCLK_HIGH_SH_CH;
    }

    public int getPclkHighSh()
    {
        return pclkHighSh;
    }


    /**
     *  Sets/Gets the low serial clock rail shift.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkLowSh(int value)
    {
        sclkLowSh = value;
        changed |= SCLK_LOW_SH_CH;
    }

    public int getSclkLowSh()
    {
        return sclkLowSh;
    }


    /**
     *  Sets/Gets the high serial clock rail shift.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkHighSh(int value)
    {
        sclkHighSh = value;
        changed |= SCLK_HIGH_SH_CH;
    }

    public int getSclkHighSh()
    {
        return sclkHighSh;
    }


    /**
     *  Sets/Gets the low reset gate rail shift.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgLowSh(int value)
    {
        rgLowSh = value;
        changed |= RG_LOW_SH_CH;
    }

    public int getRgLowSh()
    {
        return rgLowSh;
    }


    /**
     *  Sets/Gets the high reset gate rail shift.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgHighSh(int value)
    {
        rgHighSh = value;
        changed |= RG_HIGH_SH_CH;
    }

    public int getRgHighSh()
    {
        return rgHighSh;
    }


    /**
     *  Sets all the configuration data.
     * 
     *  @param  dacs  The DAC values
     */
    void setConfig(DACS dacs) //throws Exception
    {
        String name = getName();
        if (!raw) {
            double[] values = dacs.getPValues();
            sce.submitChange(name, RG_HIGH_P, values[DACS.RG_HIGH]);
            sce.submitChange(name, RG_LOW_P, values[DACS.RG_LOW]);
            sce.submitChange(name, PCLK_HIGH_P, values[DACS.PCLK_HIGH]);
            sce.submitChange(name, PCLK_LOW_P, values[DACS.PCLK_LOW]);
            sce.submitChange(name, SCLK_HIGH_P, values[DACS.SCLK_HIGH]);
            sce.submitChange(name, SCLK_LOW_P, values[DACS.SCLK_LOW]);
        }
        else {
            int[] values = dacs.getValues();
            sce.submitChange(name, RG_HIGH, values[DACS.RG_HIGH]);
            sce.submitChange(name, RG_LOW, values[DACS.RG_LOW]);
            sce.submitChange(name, PCLK_HIGH, values[DACS.PCLK_HIGH]);
            sce.submitChange(name, PCLK_LOW, values[DACS.PCLK_LOW]);
            sce.submitChange(name, SCLK_HIGH, values[DACS.SCLK_HIGH]);
            sce.submitChange(name, SCLK_LOW, values[DACS.SCLK_LOW]);
            if (isCurrent) {
                sce.submitChange(name, RG_LOW_SH, values[DACS.RG_LOW_SH]);
                sce.submitChange(name, PCLK_LOW_SH, values[DACS.PCLK_LOW_SH]);
                sce.submitChange(name, SCLK_LOW_SH, values[DACS.SCLK_LOW_SH]);
                if (isCorner) {
                    sce.submitChange(name, RG_HIGH_SH, values[DACS.RG_HIGH_SH]);
                    sce.submitChange(name, PCLK_HIGH_SH, values[DACS.PCLK_HIGH_SH]);
                    sce.submitChange(name, SCLK_HIGH_SH, values[DACS.SCLK_HIGH_SH]);
                }
            }
            else {
                sce.submitChange(name, CS_GATE, new int[]{values[DACS.CSGATE_1],
                                 values[DACS.CSGATE_2], values[DACS.CSGATE_3]});
            }
        }
        try {
            sce.commitBulkChange();
        }
        catch (IllegalArgumentException e) {
            sce.dropSubmittedChangesForComponent(name);
            throw e;
        }
    }


    /**
     *  Gets all the configuration data.
     *
     *  @param  dacs  The DAC values
     */
    void getConfig(DACS dacs)
    {
        if (!raw) {
            double[] values = dacs.getPValues();
            values[DACS.RG_HIGH] = rgHighP;
            values[DACS.RG_LOW] = rgLowP;
            values[DACS.PCLK_HIGH] = pclkHighP;
            values[DACS.PCLK_LOW] = pclkLowP;
            values[DACS.SCLK_HIGH] = sclkHighP;
            values[DACS.SCLK_LOW] = sclkLowP;
        }
        else {
            int[] values = dacs.getValues();
            values[DACS.RG_HIGH] = rgHigh;
            values[DACS.RG_LOW] = rgLow;
            values[DACS.PCLK_HIGH] = pclkHigh;
            values[DACS.PCLK_LOW] = pclkLow;
            values[DACS.SCLK_HIGH] = sclkHigh;
            values[DACS.SCLK_LOW] = sclkLow;
            if (isCurrent) {
                values[DACS.RG_LOW_SH] = rgLowSh;
                values[DACS.PCLK_LOW_SH] = pclkLowSh;
                values[DACS.SCLK_LOW_SH] = sclkLowSh;
                if (isCorner) {
                    values[DACS.RG_HIGH_SH] = rgHighSh;
                    values[DACS.PCLK_HIGH_SH] = pclkHighSh;
                    values[DACS.SCLK_HIGH_SH] = sclkHighSh;
                }
            }
            else {
                values[DACS.CSGATE_1] = csGate[0];
                values[DACS.CSGATE_2] = csGate[1];
                values[DACS.CSGATE_3] = csGate[2];
            }
        }
    }


    /**
     *  Loads configured DAC values onto the REB.
     *
     *  @return  The number of DACs loaded
     *  @throws  Exception
     */
    public int loadConfig() throws Exception
    {
        return loadConfig(DEFAULT_LOAD_WAIT);
    }


    /**
     *  Loads configured DAC values onto the REB.
     *
     *  @param  wait  The time to wait after loading a DAC (ms)
     *  @return  The number of DACs loaded
     *  @throws  Exception
     */
    public int loadConfig(int wait) throws Exception
    {
        if (!cfgValid) {
            throw new RaftException(getName() + " load error: invalid configuration");
        }
        if (!rebDevc.isSerialNumValid()) {
            throw new RaftException(getName() + " load error: invalid REB serial number");
        }
        int count = 0;
        count += loadSclkHigh(wait);
        count += loadSclkLow(wait);
        count += loadPclkHigh(wait);
        count += loadPclkLow(wait);
        count += loadRgHigh(wait);
        count += loadRgLow(wait);
        if (!isCurrent) {
            setDac(BoardDacs.CHAN_CSGATE_1, csGate[0]);
            setDac(BoardDacs.CHAN_CSGATE_2, csGate[1]);
            setDac(BoardDacs.CHAN_CSGATE_3, csGate[2]);
            loadDac(wait);
            count += 3;
        }
        changed = 0;
        return count;
    }


    /**
     *  Loads changed configured DAC values onto the REB.
     *
     *  @return  The number of DACs loaded
     *  @throws  Exception
     */
    public int loadChanged() throws Exception
    {
        return loadChanged(DEFAULT_LOAD_WAIT);
    }


    /**
     *  Loads changed configured DAC values onto the REB.
     *
     *  @param  wait  The time to wait after loading a DAC (ms)
     *  @return  The number of DACs loaded
     *  @throws  Exception
     */
    public int loadChanged(int wait) throws Exception
    {
        if (!cfgValid) {
            throw new RaftException(getName() + " load error: invalid configuration");
        }
        if (!rebDevc.isSerialNumValid()) {
            throw new RaftException(getName() + " load error: invalid REB serial number");
        }
        int count = 0;
        if ((changed & SCLK_HIGH_CH) != 0) {
            count += loadSclkHigh(wait);
        }
        if ((changed & SCLK_LOW_CH) != 0) {
            count += loadSclkLow(wait);
        }
        if ((changed & PCLK_HIGH_CH) != 0) {
            count += loadPclkHigh(wait);
        }
        if ((changed & PCLK_LOW_CH) != 0) {
            count += loadPclkLow(wait);
        }
        if ((changed & RG_HIGH_CH) != 0) {
            count += loadRgHigh(wait);
        }
        if ((changed & RG_LOW_CH) != 0) {
            count += loadRgLow(wait);
        }
        if (!isCurrent) {
            if ((changed & CS_GATE_CH) != 0) {
                setDac(BoardDacs.CHAN_CSGATE_1, csGate[0]);
                setDac(BoardDacs.CHAN_CSGATE_2, csGate[1]);
                setDac(BoardDacs.CHAN_CSGATE_3, csGate[2]);
                loadDac(wait);
                count += 3;
            }
        }
        changed = 0;
        return count;
    }


    /**
     *  Clears (zeroes) all DAC values on the REB.
     *
     *  @throws  Exception
     */
    public void clear() throws Exception
    {
        clear(0);
    }


    /**
     *  Clears (zeroes) all DAC values on the REB.
     *
     *  @param  wait  The time to wait after loading a DAC (ms)
     *  @throws  Exception
     */
    public void clear(int wait) throws Exception
    {
        clearRgLow(wait);
        clearRgHigh(wait);
        clearPclkLow(wait);
        clearPclkHigh(wait);
        clearSclkLow(wait);
        clearSclkHigh(wait);
        if (!isCurrent) {
            setDac(BoardDacs.CHAN_CSGATE_1, 0);
            setDac(BoardDacs.CHAN_CSGATE_2, 0);
            setDac(BoardDacs.CHAN_CSGATE_3, 0);
            loadDac(wait);
        }
        changed = -1;
    }


    /**
     *  Loads the low parallel clock rail DACs.
     */
    private int loadPclkLow(int wait) throws Exception
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_PCLK_L, pclkLowP >= 0.0 ? (int)(railConvL * pclkLowP) : 0);
            setDac(BoardDacs.CHAN_PCLK_L_SH, pclkLowP <= 0.0 ? (int)(-railShiftConvL * pclkLowP) : 0);
            count++;
        }
        else {
            setDac(BoardDacs.CHAN_PCLK_L, pclkLow);
            if (isCurrent) {
                setDac(BoardDacs.CHAN_PCLK_L_SH, pclkLowSh);
                count++;
            }
        }
        loadDac(wait);
        return count;
    }


    /**
     *  Clears the low parallel clock rail DACs.
     */
    private void clearPclkLow(int wait) throws Exception
    {
        setDac(BoardDacs.CHAN_PCLK_L, 0);
        if (isCurrent) {
            setDac(BoardDacs.CHAN_PCLK_L_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Loads the high parallel clock rail DACs.
     */
    private int loadPclkHigh(int wait) throws Exception
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_PCLK_H, pclkHighP >= 0.0 ? (int)(railConvH * pclkHighP) : 0);
            if (isCorner) {
                setDac(BoardDacs.CHAN_PCLK_H_SH, pclkHighP <= 0.0 ? (int)(-railShiftConvH * pclkHighP) : 0);
                count++;
            }
        }
        else {
            setDac(BoardDacs.CHAN_PCLK_H, pclkHigh);
            if (isCorner) {
                setDac(BoardDacs.CHAN_PCLK_H_SH, pclkHighSh);
                count++;
            }
        }
        loadDac(wait);
        return count;
    }


    /**
     *  Clears the high parallel clock rail DACs.
     */
    private void clearPclkHigh(int wait) throws Exception
    {
        setDac(BoardDacs.CHAN_PCLK_H, 0);
        if (isCorner) {
            setDac(BoardDacs.CHAN_PCLK_H_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Loads the low serial clock rail DACs.
     */
    private int loadSclkLow(int wait) throws Exception
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_SCLK_L, sclkLowP >= 0.0 ? (int)(railConvL * sclkLowP) : 0);
            setDac(BoardDacs.CHAN_SCLK_L_SH, sclkLowP <= 0.0 ? (int)(-railShiftConvL * sclkLowP) : 0);
            count++;
        }
        else {
            setDac(BoardDacs.CHAN_SCLK_L, sclkLow);
            if (isCurrent) {
                setDac(BoardDacs.CHAN_SCLK_L_SH, sclkLowSh);
                count++;
            }
        }
        loadDac(wait);
        return count;
    }


    /**
     *  Clears the low serial clock rail DACs.
     */
    private void clearSclkLow(int wait) throws Exception
    {
        setDac(BoardDacs.CHAN_SCLK_L, 0);
        if (isCurrent) {
            setDac(BoardDacs.CHAN_SCLK_L_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Loads the high serial clock rail DACs.
     */
    private int loadSclkHigh(int wait) throws Exception
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_SCLK_H, sclkHighP >= 0.0 ? (int)(railConvH * sclkHighP) : 0);
            if (isCorner) {
                setDac(BoardDacs.CHAN_SCLK_H_SH, sclkHighP <= 0.0 ? (int)(-railShiftConvH * sclkHighP) : 0);
                count++;
            }
        }
        else {
            setDac(BoardDacs.CHAN_SCLK_H, sclkHigh);
            if (isCorner) {
                setDac(BoardDacs.CHAN_SCLK_H_SH, sclkHighSh);
                count++;
            }
        }
        loadDac(wait);
        return count;
    }


    /**
     *  Clears the high serial clock rail DACs.
     */
    private void clearSclkHigh(int wait) throws Exception
    {
        setDac(BoardDacs.CHAN_SCLK_H, 0);
        if (isCorner) {
            setDac(BoardDacs.CHAN_SCLK_H_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Loads the low reset gate rail DACs.
     */
    private int loadRgLow(int wait) throws Exception
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_RG_L, rgLowP >= 0.0 ? (int)(railConvL * rgLowP) : 0);
            setDac(BoardDacs.CHAN_RG_L_SH, rgLowP <= 0.0 ? (int)(-railShiftConvL * rgLowP) : 0);
            count++;
        }
        else {
            setDac(BoardDacs.CHAN_RG_L, rgLow);
            if (isCurrent) {
                setDac(BoardDacs.CHAN_RG_L_SH, rgLowSh);
                count++;
            }
        }
        loadDac(wait);
        return count;
    }


    /**
     *  Clears the low reset gate rail DACs.
     */
    private void clearRgLow(int wait) throws Exception
    {
        setDac(BoardDacs.CHAN_RG_L, 0);
        if (isCurrent) {
            setDac(BoardDacs.CHAN_RG_L_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Loads the high reset gate rail DACs.
     */
    private int loadRgHigh(int wait) throws Exception
    {
        int count = 1;
        if (!raw) {
            setDac(BoardDacs.CHAN_RG_H, rgHighP >= 0.0 ? (int)(railConvH * rgHighP) : 0);
            if (isCorner) {
                setDac(BoardDacs.CHAN_RG_H_SH, rgHighP <= 0.0 ? (int)(-railShiftConvH * rgHighP) : 0);
                count++;
            }
        }
        else {
            setDac(BoardDacs.CHAN_RG_H, rgHigh);
            if (isCorner) {
                setDac(BoardDacs.CHAN_RG_H_SH, rgHighSh);
                count++;
            }
        }
        loadDac(wait);
        return count;
    }


    /**
     *  Clears the high reset gate rail DACs.
     */
    private void clearRgHigh(int wait) throws Exception
    {
        setDac(BoardDacs.CHAN_RG_H, 0);
        if (isCorner) {
            setDac(BoardDacs.CHAN_RG_H_SH, 0);
        }
        loadDac(wait);
    }


    /**
     *  Sets a DAC value.
     *
     *  @param  chan   The channel number
     *  @param  value  The value to set
     *  @throws  Exception
     */
    private void setDac(int chan, int value) throws Exception
    {
        testOnline();
        try {
            dac.set(chan, value < 0 ? 0 : value > REBDevice.DAC_LIMIT ? REBDevice.DAC_LIMIT : value);
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
        }
    }


    /**
     *  Loads DAC values.
     *
     *  @param  wait  The time to wait before loading the DAC (ms)
     *  @throws  Exception
     */
    private void loadDac(int wait) throws Exception
    {
        testOnline();
        try {
            Thread.sleep(wait);
        }
        catch (InterruptedException e) {}
        try {
            dac.loadGlobal();
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
        }
    }

}
