package org.lsst.ccs.subsystem.rafts;

import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.reb.Aspic;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.rafts.config.ASPIC;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements control of an ASPIC.
 *
 *  @author Owen Saxton
 */
public class AspicControl extends Control {

    /*
     *  Constants.
     */
    static final String
        RC     = "rc",
        GAIN   = "gain",
        CLAMP  = "clamp",
        AF1    = "af1",
        TM     = "tm";

    private static final int
        GAIN_RC_CH = 0x01,
        CLAMP_CH   = 0x02,
        MODE_CH    = 0x04;

    /*
     *  Data fields.
     */
    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    // Supplied configurable fields
    @ConfigurationParameter(name=GAIN, category=REBDevice.RAFTS, description="Gain Selector", units="DAC counts")
    private int gain;
    @ConfigurationParameter(name=RC, category=REBDevice.RAFTS, description="RC Selector", units="DAC counts")
    private int rc;
    @ConfigurationParameter(name=CLAMP, category=REBDevice.RAFTS, description="Clamp bits", units= "DAC counts")
    private int clamp;
    @ConfigurationParameter(name=AF1, category=REBDevice.RAFTS, description="First Stage Gain", units= "DAC counts")
    private int af1;
    @ConfigurationParameter(name=TM, category=REBDevice.RAFTS, description="Transparent mode", units= "DAC counts")
    private int tm;

    // Other fields
    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private Aspic asp;
    private int hwVersion;
    private int change = -1;


    /**
     *  No-argument constructor.
     */
    public AspicControl()
    {
    }


    /**
     *  Constructor.
     *
     *  @param  desc    The ASPIC description
     *  @param  hwChan  The ASPIC number (0 - 5)
     *  @param  gain    The gain
     *  @param  rc      The rc
     *  @param  clamp   The mask of ADCs to clamp
     *  @param  af1     If non-zero, set first-stage gain to 1
     *  @param  tm      If non-zero, set transparent mode
     */
    @Deprecated
    public AspicControl(String desc, int hwChan, int gain, int rc, int clamp, int af1, int tm)
    {
        super();
        super.description = desc;
        super.hwChan = hwChan;
        this.gain = gain;
        this.rc = rc;
        this.clamp = clamp;
        this.af1 = af1;
        this.tm = tm;
    }


    /**
     *  Configures channel description.
     *
     *  @param  mon   The associated monitor object
     *  @param  devc  The associated device object
     */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        asp = ((REBDevice)devc).getAspic();
    }


    /**
     *  Checks configuration.
     *
     *  @return  Whether configuration is okay
     */
    public boolean checkConfig()
    {
        hwVersion = getHwVersion(asp);
        if (hwVersion == REB.VERSION_UNSUPP) {
            sLog.error("ASPIC " + hwChan + " configuration not supported by the firmware");
            return false;
        }

        try {
            if (hwChan >= 0 && hwChan < Aspic.NUM_SIDES * asp.getNumStrips()) return true;
        }
        catch (DriverException e) {
            sLog.error("Error getting number of strips");
            return false;
        }
        try {
            MonitorLogUtils.reportError(sLog, getName(), "HW channel", hwChan);
        }
        catch (Exception e) {
            /* Can be safely ignored */
        }
        return false;
    }


    /**
     *  Gets the hardware version.
     *
     *  @param  asp  The ASPIC object
     *  @return  The hardware version
     */
    public static int getHwVersion(Aspic asp)
    {
        return asp.getVersion() == Aspic.VERSION_UNSUPP
                 ? REB.VERSION_UNSUPP : REB.VERSION_0;
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version
     */
    @Deprecated
    public int getHwVersion()
    {
        return hwVersion;
    }


    /**
     *  Sets/gets the gain.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGain(int value)
    {
        gain = value;
        change |= GAIN_RC_CH;
    }

    @Command(type=CommandType.QUERY, description="Gets the gain")
    public int getGain()
    {
        return gain;
    }


    /**
     *  Sets/gets the rc value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRc(int value)
    {
        rc = value;
        change |= GAIN_RC_CH;
    }

    @Command(type=CommandType.QUERY, description="Gets the rc value")
    public int getRc()
    {
        return rc;
    }


    /**
     *  Sets/gets the clamp mask.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setClamp(int value)
    {
        clamp = value;
        change |= CLAMP_CH;
    }

    @Command(type=CommandType.QUERY, description="Gets the clamp")
    public int getClamp()
    {
        return clamp;
    }


    /**
     *  Sets/gets the af1 (force first-stage gain to 1) value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setAf1(int value)
    {
        af1 = value;
        change |= MODE_CH;
    }

    @Command(type=CommandType.QUERY, description="Gets the af1 value")
    public int getAf1()
    {
        return af1;
    }


    /**
     *  Sets/gets the tm (transparent mode) value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setTm(int value)
    {
        tm = value;
        change |= MODE_CH;
    }

    @Command(type=CommandType.QUERY, description="Gets the tm")
    public int getTm()
    {
        return tm;
    }


    /**
     *  Sets all the configuration data.
     */
    void setConfig(ASPIC aspic)
    {
        String name = getName();
        sce.change(name, GAIN, aspic.getGain());
        sce.change(name, RC, aspic.getRc());
        sce.change(name, CLAMP, aspic.getClamp());
        sce.change(name, AF1, aspic.isAf1() ? 1 : 0);
        sce.change(name, TM, aspic.isTm() ? 1 : 0);
    }


    /**
     *  Gets all the configuration data.
     */
    ASPIC getConfig()
    {
        ASPIC aspic = new ASPIC();
        aspic.setGain(gain);
        aspic.setRc(rc);
        aspic.setClamp(clamp);
        aspic.setAf1(af1 != 0);
        aspic.setTm(tm != 0);

        return aspic;
    }


    /**
     *  Loads configuration data to an ASPIC.
     *
     *  @return  The number of ASPICs loaded: 0 or 1
     *  @throws  Exception
     */
    int load() throws Exception
    {
        if (asp.getVersion() == Aspic.VERSION_UNSUPP) return 0;
        try {
            int strips = 1 << hwChan / 2;
            int side = (hwChan & 1) == 0 ? Aspic.SIDE_TOP : Aspic.SIDE_BOTTOM;
            asp.writeGainRc(strips, 1 << side, gain, rc);
            asp.writeClamp(strips, 1 << side, clamp);
            int mode = (af1 != 0 ? Aspic.MODE_AF1 : 0) | (tm != 0 ? Aspic.MODE_TM : 0);
            asp.writeModes(strips, 1 << side, mode);
            change = 0;
            return 1;
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
            return 0;
        }
    }


    /**
     *  Loads changed configuration data to an ASPIC.
     *
     *  @return  The number of ASPICs loaded
     *  @throws  Exception
     */
    int loadChanged() throws Exception
    {
        if (asp.getVersion() == Aspic.VERSION_UNSUPP) return 0;
        int count = 0;
        if (change != 0) {
            try {
                int strips = 1 << hwChan / 2;
                int side = (hwChan & 1) == 0 ? Aspic.SIDE_TOP : Aspic.SIDE_BOTTOM;
                if ((change & GAIN_RC_CH) != 0) {
                    asp.writeGainRc(strips, 1 << side, gain, rc);
                }
                if ((change & CLAMP_CH) != 0) {
                    asp.writeClamp(strips, 1 << side, clamp);
                }
                if ((change &= MODE_CH) != 0){
                    int mode = (af1 != 0 ? Aspic.MODE_AF1 : 0) | (tm != 0 ? Aspic.MODE_TM : 0);
                    asp.writeModes(strips, 1 << side, mode);
                }
                count = 1;
                change = 0;
            }
            catch (REBException e) {
                checkTimeout(e, RaftException.class);
            }
        }
        return count;
    }


    /**
     *  Checks loaded ASPIC data against its configuration.
     *
     *  @return  A mask of field numbers that didn't match
     *  @throws  Exception
     */
    int check() throws Exception
    {
        if (asp.getVersion() == Aspic.VERSION_UNSUPP) return 0;
        try {
            int mask = 0;
            int strips = 1 << hwChan / 2;
            int side = (hwChan & 1) == 0 ? Aspic.SIDE_TOP : Aspic.SIDE_BOTTOM;
            int[][] values = asp.readGainRc(strips, side);
            if (values[0][0] != gain) {
                mask |= 1;
            }
            if (values[0][1] != rc) {
                mask |= 2;
            }
            values[0] = asp.readClamp(strips, side);
            if (values[0][0] != clamp) {
                mask |= 4;
            }
            values[0] = asp.readModes(strips, side);
            int value = values[0][0] & (Aspic.MODE_AF1 | Aspic.MODE_TM);
            if (value != ((af1 != 0 ? Aspic.MODE_AF1 : 0) | (tm != 0 ? Aspic.MODE_TM : 0))) {
                mask |= 8;
            }
            return mask;
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
            return 0;
        }
    }


    /**
     *  Resets an ASPIC and its strip companion.
     *
     *  @return  The number of ASPICs reset: 0 or 2
     *  @throws  Exception
     */
    int reset() throws Exception
    {
        if (asp.getVersion() == Aspic.VERSION_UNSUPP) return 0;
        try {
            asp.reset(hwChan / 2);
            return 2;
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
            return 0;
        }
    }

}
