package org.lsst.ccs.subsystem.rafts;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.Sequencer;
import org.lsst.ccs.drivers.reb.SequencerUtils;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.data.SequencerData;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo.PointerKind;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;

/**
 *  Routines for accessing a REB sequencer.
 *
 *  @author Owen Saxton
 */
public class SequencerProc {

    /*
     *  Constants.
     */
    private static final Map<PointerKind, Integer> pointerTypeMap = new HashMap<>();
    static {
        pointerTypeMap.put(PointerKind.FUNCTION, SequencerData.PTR_TYPE_FUNC);
        pointerTypeMap.put(PointerKind.SUBROUTINE, SequencerData.PTR_TYPE_SUBR);
        pointerTypeMap.put(PointerKind.REPEAT_FUNCTION, SequencerData.PTR_TYPE_FUNC_REP);
        pointerTypeMap.put(PointerKind.REPEAT_SUBROUTINE, SequencerData.PTR_TYPE_SUBR_REP);
    }

    /*
     *  Data fields.
     */
    private final BaseSet bss;
    private final SequencerUtils seq;
    private Map<String, Integer> mainsMap;
    private Map<Integer, String> invMainsMap;
    private Map<String, Integer> subroutineMap;
    private Map<Integer, String> invSubroutineMap;
    private Map<String, Integer> functionMap;
    private Map<Integer, String> invFunctionMap;
    private final Map<String, Integer> pointerMap = new HashMap<>();
    private int zeroReg;


    /**
     *  Constructor.
     *
     *  @param  bss
     */
    public SequencerProc(BaseSet bss)
    {
        this.bss = bss;
        seq = new SequencerUtils(bss);
    }


    /**
     *  Loads the sequencer from a file.
     *
     *  The file format may be XML, LPNHE assembler, or simple instructions
     *  (SIF)
     *
     *  @param  fileName  The name of the file containing sequencer instructions
     *  @return  The number of slices expected to be read
     *  @throws  Exception
     */
    public int loadSequencer(String fileName) throws Exception
    {
        for (int j = 0; j < Sequencer.SEQ_MAX_PROGRAM; j += 16) {
            bss.write(Sequencer.REG_SEQ_PROGRAM + j + 15, 0);
        }
        int nSlice;
        if (fileName.endsWith(".sif")) {
            nSlice = seq.loadFile(fileName);
            mainsMap = null;
            pointerMap.clear();
        }
        else {
            nSlice = loadSequencerXML(fileName);
        }
        zeroReg = 0;
        for (int j = 0; j < Sequencer.SEQ_MAX_PROGRAM; j += 16) {
            if (bss.read(Sequencer.REG_SEQ_PROGRAM + j + 15) == 0) {
                zeroReg = Sequencer.REG_SEQ_PROGRAM + j + 15;
            }
        }
        
        return nSlice;
    }


    /**
     *  Loads the sequencer from an XML or assembler file.
     *
     *  @param  fileName  The name of the file
     *  @return  The number of slices expected to be read
     *  @throws  Exception
     */
    private int loadSequencerXML(String fileName) throws Exception
    {
        //If the fileName starts with a forward slash try to load it as a File,
        //otherwise use the Bootstrap.
        //This strategy will have to be changed in the future when we find a way
        //to pass an object representing the sequencer as loaded by a client
        //application
        FPGA2Model model = null;
        try {
            if (fileName.startsWith("/")) {
                model = new FPGA2ModelBuilder().compileFile(new File(fileName));
            } else {
                InputStream sequencerFileInputStream = BootstrapResourceUtils.getBootstrapResource(fileName);
                if (sequencerFileInputStream == null) {
                    throw new RuntimeException("Could not load sequencer file " + fileName);
                }
                model = new FPGA2ModelBuilder().compileFile(sequencerFileInputStream, "xml");
            }
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception e) {
            Exception ec = (Exception) e.getCause();
            ec = ec == null ? e : ec;
            throw ec;
        }
        mainsMap = model.getMainAddresses();
        invMainsMap = invertMap(mainsMap);
        subroutineMap = model.getSubroutineAddresses();
        invSubroutineMap = invertMap(subroutineMap);
        functionMap = model.getFunctionAddresses();
        invFunctionMap = invertMap(functionMap);
        List<PointerInfo> pointerList = model.getPointers();
        pointerMap.clear();
        for (PointerInfo pi : pointerList) {
            int type = pointerTypeMap.get(pi.getKind());
            pointerMap.put(pi.getName(), pi.getOffset() | (type << 8));
        }
        seq.clearCache();

        for (int[] command : model.getCommands()) {
            /*
            System.out.print("Command:");
            for (int j = 0; j < command.length; j++) {
                System.out.format(" %d", command[j]);
            }
            System.out.println();
            */
            switch (command[0]) {

            case FPGA2Model.CMD_LINES:
                int lines = 0;
                for (int j = 3; j < command.length; j++) {
                    lines |= (1 << command[j]);
                }
                seq.writeLines(command[1], command[2], lines);
                break;

            case FPGA2Model.CMD_TIME:
                seq.writeTimes(command[1], command[2], command[3]);
                break;

            case FPGA2Model.CMD_PROGFUNC:
                seq.writeProgExec(command[1], command[2], command[3]);
                break;

            case FPGA2Model.CMD_PROGFUNC_FP:
                seq.writeProgExec(command[1], Sequencer.SEQ_ARG_IND_FUNC,
                                  command[2], command[3]);
                seq.writeExecFunc(command[2], command[4]);
                break;

            case FPGA2Model.CMD_PROGFUNC_RP:
                seq.writeProgExec(command[1], Sequencer.SEQ_ARG_IND_COUNT,
                                  command[2], command[3]);
                seq.writeExecCount(command[3], command[4]);
                break;

            case FPGA2Model.CMD_PROGFUNC_FRP:
                seq.writeProgExec(command[1],
                                  Sequencer.SEQ_ARG_IND_FUNC
                                    | Sequencer.SEQ_ARG_IND_COUNT,
                                  command[2], command[3]);
                seq.writeExecFunc(command[2], command[4]);
                seq.writeExecCount(command[3], command[5]);
                break;

            case FPGA2Model.CMD_PROGJUMP:
                seq.writeProgJump(command[1], command[2], command[3]);
                break;

            case FPGA2Model.CMD_PROGJUMP_AP:
                seq.writeProgJump(command[1], Sequencer.SEQ_ARG_IND_JUMP,
                                  command[2], command[3]);
                seq.writeJumpSubr(command[2], command[4]);
                break;

            case FPGA2Model.CMD_PROGJUMP_RP:
                seq.writeProgJump(command[1], Sequencer.SEQ_ARG_IND_COUNT,
                                  command[2], command[3]);
                seq.writeJumpCount(command[3], command[4]);
                break;

            case FPGA2Model.CMD_PROGJUMP_ARP:
                seq.writeProgJump(command[1],
                                  Sequencer.SEQ_ARG_IND_JUMP
                                    | Sequencer.SEQ_ARG_IND_COUNT,
                                  command[2], command[3]);
                seq.writeJumpSubr(command[2], command[4]);
                seq.writeJumpCount(command[3], command[5]);
                break;

            case FPGA2Model.CMD_PROGSUBE:
                seq.writeProgEndSubr(command[1]);
                break;

            case FPGA2Model.CMD_PROGEND:
                seq.writeProgEnd(command[1]);
                break;

            default:
            }
        }

        int nSlice = seq.getCacheSliceCount();
        int version = seq.getVersion();
        if (version == BaseSet.VERSION_3) {
            seq.writeStartAddr(0);
        }
        else if (version == BaseSet.VERSION_0) {
            seq.writeSliceCount(nSlice);
        }

        return nSlice;
    }


    /**
     *  Gets the list of main programs.
     *
     *  @return  The list of known names of main programs, or null is there
     *           are no such names.
     */
    public List<String> getMains()
    {
        if (mainsMap == null) return null;
        return new ArrayList<>(mainsMap.keySet());
    }


    /**
     *  Gets the map of main programs.
     *
     *  @return  The name-to-address map of main programs
     */
    public Map<String, Integer> getMainMap()
    {
        return mainsMap;
    }


    /**
     *  Gets the map of subroutines.
     *
     *  @return  The name-to-address map of subroutines
     */
    public Map<String, Integer> getSubroutineMap()
    {
        return subroutineMap;
    }


    /**
     *  Gets the map of functions.
     *
     *  @return  The name-to-number map of functions
     */
    public Map<String, Integer> getFunctionMap()
    {
        return functionMap;
    }


    /**
     *  Gets the map of pointer information.
     *
     *  @return  The name-to-type-and-address map of pointers
     */
    public Map<String, Integer> getPointers()
    {
        return pointerMap;
    }


    /**
     *  Gets the array of image parameter registers to be read
     *
     *  @return  The array of register addresses
     */
    public int[] getImageRegisters()
    {
        int[] registers = new int[SequencerData.NUM_REGISTERS];
        for (int reg : SequencerData.imageRegs.keySet()) {
            int addr = zeroReg;
            Integer desc = pointerMap.get(SequencerData.imageRegs.get(reg));
            if (desc != null) {
                int type = desc >> 8, index = desc & 0xff;
                if (type == SequencerData.PTR_TYPE_FUNC_REP) {
                    addr = Sequencer.REG_SEQ_EXEC_REPS + index;
                }
                if (type == SequencerData.PTR_TYPE_SUBR_REP) {
                    addr = Sequencer.REG_SEQ_JUMP_REPS + index;
                }
            }
            registers[reg] = addr;
        }
        return registers;
    }


    /**
     *  Sets the sequencer start address.
     *
     *  @param  mainName  The name of the main to start at
     *  @return  The number of slices produced by this main
     *  @throws  REBException
     *  @throws  RaftException
     */
    public int setStart(String mainName) throws REBException, RaftException
    {
        if (mainsMap != null) {
            Integer addr = mainsMap.get(mainName);
            if (addr != null) {
                seq.writeStartAddr(addr);
                return seq.getCacheSliceCount();
            }
        }
        throw new RaftException("Unknown main program name: " +mainName);
    }


    /**
     *  Sets the sequencer start address.
     *
     *  @param  address  The address of the main to start at
     *  @return  The number of slices produced by this main
     *  @throws  REBException
     */
    public int setStart(int address) throws REBException
    {
        seq.writeStartAddr(address);
        return seq.getCacheSliceCount();
    }


    /**
     *  Sets a parameter value.
     *
     *  @param  parmName  The name of the parameter to set
     *  @param  value     The value to set
     *  @throws  REBException
     *  @throws  RaftException
     */
    public void setParameter(String parmName, int value)
        throws REBException, RaftException
    {
        if (pointerMap != null) {
            Integer typeAddr = pointerMap.get(parmName);
            if (typeAddr != null) {
                int type = typeAddr >> 8;
                int addr = typeAddr & 0xff;
                switch (type) {
                case SequencerData.PTR_TYPE_FUNC:
                    seq.writeExecFunc(addr, value);
                    break;
                case SequencerData.PTR_TYPE_FUNC_REP:
                    seq.writeExecCount(addr, value);
                    break;
                case SequencerData.PTR_TYPE_SUBR:
                    seq.writeJumpSubr(addr, value);
                    break;
                case SequencerData.PTR_TYPE_SUBR_REP:
                    seq.writeJumpCount(addr, value);
                    break;
                }
                return;
            }
        }
        throw new RaftException("Unknown parameter name: " + parmName);
    }


    /**
     *  Gets a parameter value.
     *
     *  @param  parmName  The name of the parameter to set
     *  @return  The value
     *  @throws  REBException
     *  @throws  RaftException
     */
    public int getParameter(String parmName) throws REBException, RaftException
    {
        if (pointerMap != null) {
            Integer typeAddr = pointerMap.get(parmName);
            if (typeAddr != null) {
                int type = typeAddr >> 8;
                int addr = typeAddr & 0xff;
                int value = 0;
                switch (type) {
                case SequencerData.PTR_TYPE_FUNC:
                    value = seq.readExecFunc(addr);
                    break;
                case SequencerData.PTR_TYPE_FUNC_REP:
                    value = seq.readExecCount(addr);
                    break;
                case SequencerData.PTR_TYPE_SUBR:
                    value = seq.readJumpSubr(addr);
                    break;
                case SequencerData.PTR_TYPE_SUBR_REP:
                    value = seq.readJumpCount(addr);
                    break;
                }
                return value;
            }
        }
        throw new RaftException("Unknown parameter name: " + parmName);
    }


    /**
     *  Sets the mask of CCDs being used.
     *
     *  @param  ccdMask  The mask of active CCDs
     *  @throws  REBException
     */
    public void setCcdMask(int ccdMask) throws REBException
    {
        seq.writeStripeSelect(ccdMask);
    }


    /**
     *  Gets the data source.
     *
     *  @return  The encoded data source: CCD or pattern
     *  @throws  REBException
     */
    public int getDataSource() throws REBException
    {
        return seq.readDataSource();
    }


    /**
     *  Sets the data source.
     *
     *  @param  value  The encoded data source: CCD or pattern
     *  @throws  REBException
     */
    public void setDataSource(int value) throws REBException
    {
        seq.writeDataSource(value);
    }


    /**
     *  Gets the scan mode enabled state.
     *
     *  @return  Whether scan mode is enabled
     *  @throws  REBException
     */
    public boolean isScanEnabled() throws REBException
    {
        return seq.isScanEnabled();
    }


    /**
     *  Enables or disables scan mode.
     *
     *  @param  value  The scan mode enabled state to set (true or false)
     *  @throws  REBException
     */
    public void enableScan(boolean value) throws REBException
    {
        seq.enableScan(value);
    }


    /**
     *  Resets scan mode.
     *
     *  @throws  REBException
     */
    public void resetScan() throws REBException
    {
        seq.resetScan();
    }


    /**
     *  Gets error address.
     *
     *  @return  The sequencer error address
     *  @throws  REBException
     */
    public int getErrorAddr() throws REBException
    {
        return seq.getErrorAddr();
    }


    /**
     *  Resets error state.
     *
     *  @throws  REBException
     */
    public void resetError() throws REBException
    {
        seq.resetError();
    }


    /**
     *  Starts the sequencer.
     *
     *  @throws  REBException
     */
    public void startSequencer() throws REBException
    {
        resetError();
        seq.enable();
    }


    /**
     *  Sends a sequencer stop command.
     *
     *  @throws  REBException
     */
    public void sendStop() throws REBException
    {
        seq.sendStop();
    }


    /**
     *  Sends a sequencer step command.
     *
     *  @throws  REBException
     */
    public void sendStep() throws REBException
    {
        seq.sendStep();
    }


    /**
     *  Checks whether the sequencer is running.
     *
     *  @return  Whether the sequencer is running
     *  @throws  REBException 
     */
    public boolean isRunning() throws REBException
    {
        return seq.isEnabled();
    }


    /**
     *  Waits for the sequencer to be done (idle)
     *
     *  @param  timeout  The maximum time to wait (ms)
     *  @throws  REBException
     */
    public void waitDone(int timeout) throws REBException
    {
        seq.waitDone(timeout);
    }


    /**
     *  Inverts a string-to-integer map.
     */
    private Map<Integer, String> invertMap(Map<String, Integer> map)
    {
        Map<Integer, String> invMap = new HashMap<>();
        for (Map.Entry e : map.entrySet()) {
            invMap.put((Integer)e.getValue(), (String)e.getKey());
        }
        return invMap;
    }

}
