package org.lsst.ccs.subsystem.rafts;

import java.util.List;
import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.drivers.reb.GlobalClient;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Routines for accessing the whole raft
 *
 * @author  Owen Saxton
 */
public class GlobalProc {

    private GlobalClient gbl;


    /**
     *  Initializes the global raft system.
     *
     *  @param  rebDevices     The list of REB devices
     *  @param  clientFactory  The client factory
     *  @param  sLog           The subsystem logger
     */
    public void initialize(List<REBDevice> rebDevices, ClientFactory clientFactory, Logger sLog)
    {
        int[] ids = new int[rebDevices.size()];
        String[] ifcNames = new String[rebDevices.size()];
        int hdwType = GlobalClient.HDW_TYPE_DAQ1;
        for (int j = 0; j < ids.length; j++) {
            REBDevice reb = rebDevices.get(j);
            ids[j] = reb.getId();
            ifcNames[j] = reb.getIfcName();
            hdwType = reb.getHdwType();
        }
        try {
            if (gbl == null) {
                gbl = new GlobalClient();
                if (clientFactory != null) {
                    gbl.setClientFactory(clientFactory);
                }
            }
            else {
                gbl.close();
            }
            gbl.open(hdwType, ids, ifcNames);
        }
        catch (REBException e) {
            sLog.error("Error during global initialization" , e);
        }
    }


    /**
     *  Sets the registers to be read during image acquisition.
     *
     *  @param  rebType    The encoded REB type
     *  @param  registers  The array of addresses of registers to be read
     *  @throws REBException
     */
    public void setRegisters(int rebType, int[] registers) throws REBException
    {
        gbl.setRegisterlist(rebType, registers);
    }


    /**
     *  Starts image acquisition.
     *
     *  @throws REBException
     */
    public void acquireImage() throws REBException
    {
        gbl.acquireImage(String.format("Image_%016x", System.currentTimeMillis()));
    }


    /**
     *  Starts image acquisition.
     *
     *  @param  name  The image name
     *  @throws REBException
     */
    public void acquireImage(String name) throws REBException
    {
        gbl.acquireImage(name);
    }

}
