package org.lsst.ccs.subsystem.rafts;

import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.reb.Cabac;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.rafts.config.CABAC;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements control of a CABAC.
 *
 *  @author Owen Saxton
 */
public class CabacControl extends Control {

    /*
     *  Constants.
     */
    static final String
        RD        = "rd",
        OG        = "og",
        GD        = "gd",
        RG_RISE   = "rgRise",
        RG_FALL   = "rgFall",
        SCLK_RISE = "sclkRise",
        SCLK_FALL = "sclkFall",
        PCLK_RISE = "pclkRise",
        PCLK_FALL = "pclkFall",
        OD_RDO    = "odRdo",
        OD_EXP    = "odExp",
        PULSE     = "pulse";

    static final double
        LOW_SCALE = 0.019,
        HIGH_SCALE = 0.150;

    private static final int[][] cfgFields =
      {{Cabac.CABAC_FLD_OD_EXP0, Cabac.CABAC_FLD_OD_EXP1,
        Cabac.CABAC_FLD_OD_RDO0, Cabac.CABAC_FLD_OD_RDO1,
        Cabac.CABAC_FLD_PCLK0, Cabac.CABAC_FLD_PCLK1,
        Cabac.CABAC_FLD_PCLK2, Cabac.CABAC_FLD_PCLK3,
        Cabac.CABAC_FLD_SCLK0, Cabac.CABAC_FLD_SCLK1,
        Cabac.CABAC_FLD_SCLK2, Cabac.CABAC_FLD_RG,
        Cabac.CABAC_FLD_GD, Cabac.CABAC_FLD_OG,
        Cabac.CABAC_FLD_RD}};

    /*
     *  Data fields.
     */
    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    // Supplied configurable fields
    @ConfigurationParameter(name=RD, category=REBDevice.RAFTS)
    private int          rd;   
    @ConfigurationParameter(name=OG, category=REBDevice.RAFTS)
    private int          og;   
    @ConfigurationParameter(name=GD, category=REBDevice.RAFTS)
    private int          gd;
    @ConfigurationParameter(name=RG_RISE, category=REBDevice.RAFTS)
    private int          rgRise;
    @ConfigurationParameter(name=RG_FALL, category=REBDevice.RAFTS)
    private int          rgFall;
    @ConfigurationParameter(name=SCLK_RISE, category=REBDevice.RAFTS)
    final private int[]  sclkRise = new int[3];
    @ConfigurationParameter(name=SCLK_FALL, category=REBDevice.RAFTS)
    final private int[]  sclkFall = new int[3];
    @ConfigurationParameter(name=PCLK_RISE, category=REBDevice.RAFTS)
    final private int[]  pclkRise = new int[4];
    @ConfigurationParameter(name=PCLK_FALL, category=REBDevice.RAFTS)
    final private int[]  pclkFall = new int[4];
    @ConfigurationParameter(name=OD_RDO, category=REBDevice.RAFTS)
    final private int[]  odRdo = new int[2];
    @ConfigurationParameter(name=OD_EXP, category=REBDevice.RAFTS)
    final private int[]  odExp = new int[2];

    // Other
    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private final int version;
    private int hwVersion;
    private Cabac cbc;


    /**
     *  Constructor, version 0.
     *
     *  @param  desc
     *  @param  hwChan
     *  @param  rd
     *  @param  og
     *  @param  gd
     *  @param  rgRise
     *  @param  sclkRise
     *  @param  pclkRise
     *  @param  odRdo
     *  @param  odExp
     */
    public CabacControl(String desc, int hwChan,
                        int rd, int og, int gd, int rgRise, int[] sclkRise,
                        int[] pclkRise, int[] odRdo, int[] odExp)
    {
        super();
        super.description = desc;
        super.hwChan = hwChan;
        this.rd = rd;
        this.og = og;
        this.gd = gd;
        this.rgRise = rgRise;
        System.arraycopy(sclkRise, 0, this.sclkRise, 0, this.sclkRise.length);
        System.arraycopy(pclkRise, 0, this.pclkRise, 0, this.pclkRise.length);
        System.arraycopy(odRdo, 0, this.odRdo, 0, this.odRdo.length);
        System.arraycopy(odExp, 0, this.odExp, 0, this.odExp.length);
        version = REB.VERSION_0;
    }


    /**
     *  Constructor, version 1.
     *
     *  @param  desc
     *  @param  hwChan
     *  @param  rd
     *  @param  og
     *  @param  gd
     *  @param  rgRise
     *  @param  rgFall
     *  @param  sclkRise
     *  @param  sclkFall
     *  @param  pclkRise
     *  @param  pclkFall
     *  @param  odRdo
     *  @param  odExp
     */
    public CabacControl(String desc, int hwChan,
                        int rd, int og, int gd, int rgRise, int rgFall,
                        int[] sclkRise, int[] sclkFall, int[] pclkRise,
                        int[] pclkFall, int[] odRdo, int[] odExp)
    {
        super();
        super.description = desc;
        super.hwChan = hwChan;
        this.rd = rd;
        this.og = og;
        this.gd = gd;
        this.rgRise = rgRise;
        this.rgFall = rgFall;
        System.arraycopy(sclkRise, 0, this.sclkRise, 0, this.sclkRise.length);
        System.arraycopy(sclkFall, 0, this.sclkFall, 0, this.sclkFall.length);
        System.arraycopy(pclkRise, 0, this.pclkRise, 0, this.pclkRise.length);
        System.arraycopy(pclkFall, 0, this.pclkFall, 0, this.pclkFall.length);
        System.arraycopy(odRdo, 0, this.odRdo, 0, this.odRdo.length);
        System.arraycopy(odExp, 0, this.odExp, 0, this.odExp.length);
        version = REB.VERSION_1;
    }


    /**
     *  Configures channel description.
     *
     *  @param  mon   The associated monitor object
     *  @param  devc  The associated device object
     */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        cbc = ((REBDevice)devc).getCabac();
    }


    /**
     *  Checks configuration.
     *
     *  @return  Whether configuration is okay
     */
    public boolean checkConfig()
    {
        hwVersion = getHwVersion(cbc);
        if (version != hwVersion) {
            sLog.error("CABAC " + hwChan
                        + " configuration incompatible with the firmware");
            return false;
        }

        try {
            if (hwChan >= 0 && hwChan < Cabac.NUM_SIDES * cbc.getNumStrips()) return true;
        }
        catch (DriverException e) {
            sLog.error("Error getting number of strips");
            return false;
        }
        try {
            MonitorLogUtils.reportError(sLog, getName(), "HW channel", hwChan);
        }
        catch (Exception e) {
            /* Can be safely ignored */
        }
        return false;
    }


    /**
     *  Gets the hardware version.
     *
     *  @param  cbc  The CABAC object
     *  @return  The hardware version
     */
    public static int getHwVersion(Cabac cbc)
    {
        int version = cbc.getVersion();
        if (version == Cabac.VERSION_0) {
            version = REB.VERSION_0;
        }
        else if (version == Cabac.VERSION_1) {
            version = REB.VERSION_1;
        }
        else {
            version = REB.VERSION_UNSUPP;
        }
        return version;
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version
     */
    @Deprecated
    public int getHwVersion()
    {
        return hwVersion;
    }


    /**
     *  Sets/gets the Reset Drain voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRd(int value)
    {
        rd = value;
    }

    @Command(type=CommandType.QUERY, description="Gets the Reset Drain voltage")
    public int getRd()
    {
        return rd;
    }


    /**
     *  Sets/gets the Output Gate voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOg(int value)
    {
        og = value;
    }

    @Command(type=CommandType.QUERY, description="Gets the Output Gate voltage")
    public int getOg()
    {
        return og;
    }


    /**
     *  Sets/gets the Guard Diode voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGd(int value)
    {
        gd = value;
    }

    @Command(type=CommandType.QUERY, description="Gets the Guard Diode voltage")
    public int getGd()
    {
        return gd;
    }


    /**
     *  Sets/gets the Reset Gate rise current.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgRise(int value)
    {
        rgRise = value;
    }

    @Command(type=CommandType.QUERY,
             description="Gets the Reset Gate rise current")
    public int getRgRise()
    {
        return rgRise;
    }


    /**
     *  Sets/gets the Reset Gate fall current.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRgFall(int value)
    {
        rgFall = value;
    }

    @Command(type=CommandType.QUERY,
             description="Gets the Reset Gate fall current")
    public int getRgFall()
    {
        return rgFall;
    }


    /**
     *  Sets/gets the Serial Clock rise currents.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkRise(int[] value)
    {
        System.arraycopy(value, 0, sclkRise, 0, sclkRise.length);
    }

    @Command(type=CommandType.QUERY,
             description="Gets the Serial Clock rise currents")
    public int[] getSclkRise()
    {
        return sclkRise;
    }


    /**
     *  Sets/gets the Serial Clock fall currents.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setSclkFall(int[] value)
    {
        System.arraycopy(value, 0, sclkFall, 0, sclkFall.length);
    }

    @Command(type=CommandType.QUERY,
             description="Gets the Serial Clock fall currents")
    public int[] getSclkFall()
    {
        return sclkFall;
    }


    /**
     *  Sets/gets the Parallel Clock rise currents.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkRise(int[] value)
    {
        System.arraycopy(value, 0, pclkRise, 0, pclkRise.length);
    }

    @Command(type=CommandType.QUERY,
             description="Gets the Parallel Clock rise currents")
    public int[] getPclkRise()
    {
        return pclkRise;
    }


    /**
     *  Sets/gets the Parallel Clock fall currents.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setPclkFall(int[] value)
    {
        System.arraycopy(value, 0, pclkFall, 0, pclkFall.length);
    }

    @Command(type=CommandType.QUERY,
             description="Gets the Parallel Clock fall currents")
    public int[] getPclkFall()
    {
        return pclkFall;
    }


    /**
     *  Sets/gets the readout mode Output Drain voltages.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOdRdo(int[] value)
    {
        System.arraycopy(value, 0, odRdo, 0, odRdo.length);
    }

    @Command(type=CommandType.QUERY,
             description="Gets the readout mode Output Drain voltages")
    public int[] getOdRdo()
    {
        return odRdo;
    }


    /**
     *  Sets/gets the exposure mode Output Drain voltages.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOdExp(int[] value)
    {
        System.arraycopy(value, 0, odExp, 0, odExp.length);
    }

    @Command(type=CommandType.QUERY,
             description="Gets the exposure mode Output Drain voltages")
    public int[] getOdExp()
    {
        return odExp;
    }


    /**
     *  Sets all the configuration data.
     */
    void setConfig(CABAC cabac) //throws Exception
    {
        int[] values = cabac.getValues();
        String name = getName();
        sce.change(name, RD, values[CABAC.RD]);
        sce.change(name, OG, values[CABAC.OG]);
        sce.change(name, GD, values[CABAC.GD]);
        sce.change(name, RG_RISE, values[CABAC.RG_RISE]);
        sce.change(name, SCLK_RISE, new int[]{values[CABAC.SCLK0_RISE],
                                    values[CABAC.SCLK1_RISE],
                                    values[CABAC.SCLK2_RISE]});
        sce.change(name, PCLK_RISE, new int[]{values[CABAC.PCLK0_RISE],
                                    values[CABAC.PCLK1_RISE],
                                    values[CABAC.PCLK2_RISE],
                                    values[CABAC.PCLK3_RISE]});
        if (version != REB.VERSION_0) {
            sce.change(name, RG_FALL, values[CABAC.RG_FALL]);
            sce.change(name, SCLK_FALL, new int[]{values[CABAC.SCLK0_FALL],
                                   values[CABAC.SCLK1_FALL],
                                   values[CABAC.SCLK2_FALL]});
            sce.change(name, PCLK_FALL, new int[]{values[CABAC.PCLK0_FALL],
                                   values[CABAC.PCLK1_FALL],
                                   values[CABAC.PCLK2_FALL],
                                   values[CABAC.PCLK3_FALL]});
        }
        sce.change(name, OD_RDO, new int[]{values[CABAC.OD0_RDO],
                                 values[CABAC.OD1_RDO]});
        sce.change(name, OD_EXP, new int[]{values[CABAC.OD0_EXP],
                                 values[CABAC.OD1_EXP]});
    }


    /**
     *  Gets all the configuration data.
     */
    CABAC getConfig()
    {
        CABAC cabac = new CABAC();
        int[] values = cabac.getValues();
        values[CABAC.RD] = rd;
        values[CABAC.OG] = og;
        values[CABAC.GD] = gd;
        values[CABAC.RG_RISE] = rgRise;
        values[CABAC.SCLK0_RISE] = sclkRise[0];
        values[CABAC.SCLK1_RISE] = sclkRise[1];
        values[CABAC.SCLK2_RISE] = sclkRise[2];
        values[CABAC.PCLK0_RISE] = pclkRise[0];
        values[CABAC.PCLK1_RISE] = pclkRise[1];
        values[CABAC.PCLK2_RISE] = pclkRise[2];
        values[CABAC.PCLK3_RISE] = pclkRise[3];
        if (version != REB.VERSION_0) {
            values[CABAC.RG_FALL] = rgFall;
            values[CABAC.SCLK0_FALL] = sclkFall[0];
            values[CABAC.SCLK1_FALL] = sclkFall[1];
            values[CABAC.SCLK2_FALL] = sclkFall[2];
            values[CABAC.PCLK0_FALL] = pclkFall[0];
            values[CABAC.PCLK1_FALL] = pclkFall[1];
            values[CABAC.PCLK2_FALL] = pclkFall[2];
            values[CABAC.PCLK3_FALL] = pclkFall[3];
        }
        values[CABAC.OD0_RDO] = odRdo[0];
        values[CABAC.OD1_RDO] = odRdo[1];
        values[CABAC.OD0_EXP] = odExp[0];
        values[CABAC.OD1_EXP] = odExp[1];

        return cabac;
    }


    /**
     *  Loads configuration data to a CABAC.
     *
     *  @return  The number of CABACs loaded: 0 or 1
     *  @throws  Exception
     */
    int load() throws Exception
    {
        if (cbc.getVersion() == Cabac.VERSION_0) return 0;
        try {
            int strips = 1 << hwChan / 2;
            int side = (hwChan & 1) == 0 ? Cabac.SIDE_TOP : Cabac.SIDE_BOTTOM;
            int[] data = getData();
            for (int[] fields : cfgFields) {
                for (int j : fields) {
                    cbc.writeField(strips, 1 << side, j, data[j]);
                }
            }
            return 1;
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
            return 0;
        }
    }


    /**
     *  Gets selected configuration data.
     *
     *  @param  index  The index of the set of data to obtain
     *  @param  data   An array into which to copy the selected data
     *  @return  Whether the specified set of data exists
     */
    boolean getData(int index, int[] data)
    {
        if (index >= cfgFields.length) return false;
        int[] fields = cfgFields[index], cData = getData();
        for (int j : fields) {
            data[j] = cData[j];
        }
        return true;
    }


    /**
     *  Checks loaded CABAC data against its configuration.
     *
     *  @return  A mask of field numbers that didn't match
     *  @throws  Exception
     */
    int check() throws Exception
    {
        if (cbc.getVersion() == Cabac.VERSION_0) return -1;
        try {
            int mask = 0;
            int strips = 1 << hwChan / 2;
            int side = (hwChan & 1) == 0 ? Cabac.SIDE_TOP : Cabac.SIDE_BOTTOM;
            int[] data = getData();
            for (int[] fields : cfgFields) {
                for (int j : fields) {
                    if (cbc.readField(strips, side, j)[0] != data[j]) {
                        mask |= 1 << j;
                    }
                }
            }
            return mask;
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
            return 0;
        }
    }


    /**
     *  Compares read CABAC values with configuration.
     *
     *  @param  read  The read values
     *  @return  The mask of fields which don't match
     */
    int compare(int[] read)
    {
        int mask = 0;
        int[] config = getData();
        for (int[] fields : cfgFields) {
            for (int j : fields) {
                if (read[j] != config[j]) {
                    mask |= 1 << j;
                }
            }
        }
        return mask;
    }


    /**
     *  Creates an array of configured CABAC data.
     */
    private int[] getData()
    {
        int[] data = new int[Cabac.NUM_CABAC_FLDS];
        data[Cabac.CABAC_FLD_OD_EXP0] = odExp[0];
        data[Cabac.CABAC_FLD_OD_EXP1] = odExp[1];
        data[Cabac.CABAC_FLD_OD_RDO0] = odRdo[0];
        data[Cabac.CABAC_FLD_OD_RDO1] = odRdo[1];
        if (version == REB.VERSION_0) {
            data[Cabac.CABAC_FLD_PCLK0] = pclkRise[0];
            data[Cabac.CABAC_FLD_PCLK1] = pclkRise[1];
            data[Cabac.CABAC_FLD_PCLK2] = pclkRise[2];
            data[Cabac.CABAC_FLD_PCLK3] = pclkRise[3];
            data[Cabac.CABAC_FLD_SCLK0] = sclkRise[0];
            data[Cabac.CABAC_FLD_SCLK1] = sclkRise[1];
            data[Cabac.CABAC_FLD_SCLK2] = sclkRise[2];
            data[Cabac.CABAC_FLD_RG] = rgRise;
        }
        else {
            data[Cabac.CABAC_FLD_PCLK0] = (pclkRise[0] << 8) | (pclkFall[0] & 0xff);
            data[Cabac.CABAC_FLD_PCLK1] = (pclkRise[1] << 8) | (pclkFall[1] & 0xff);
            data[Cabac.CABAC_FLD_PCLK2] = (pclkRise[2] << 8) | (pclkFall[2] & 0xff);
            data[Cabac.CABAC_FLD_PCLK3] = (pclkRise[3] << 8) | (pclkFall[3] & 0xff);
            data[Cabac.CABAC_FLD_SCLK0] = (sclkRise[0] << 8) | (sclkFall[0] & 0xff);
            data[Cabac.CABAC_FLD_SCLK1] = (sclkRise[1] << 8) | (sclkFall[1] & 0xff);
            data[Cabac.CABAC_FLD_SCLK2] = (sclkRise[2] << 8) | (sclkFall[2] & 0xff);
            data[Cabac.CABAC_FLD_RG] = (rgRise << 8) | (rgFall & 0xff);
        }
        data[Cabac.CABAC_FLD_GD] = gd;
        data[Cabac.CABAC_FLD_OG] = og;
        data[Cabac.CABAC_FLD_RD] = rd;

        return data;
    }

}
