package org.lsst.ccs.subsystem.rafts;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.reb.BoardDacs;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.rafts.config.BiasDACS;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements control of CCD biases and CS gate.
 *
 *  @author Owen Saxton
 */
public class BiasControl extends Control implements ConfigurationBulkChangeHandler {

    /**
     *  Constants.
     */
    public static final String
        GD_P      = "gdP",
        OD_P      = "odP",
        OG_P      = "ogP",
        RD_P      = "rdP",
        CS_GATE_P = "csGateP",

        GD        = "gd",
        OD        = "od",
        OG        = "og",
        OG_SH     = "ogSh",
        RD        = "rd",
        CS_GATE   = "csGate";

    private static final int
        GD_CH     = 0x01,
        OD_CH     = 0x02,
        OG_CH     = 0x04,
        OG_SH_CH  = 0x08,
        RD_CH     = 0x10,
        CSGATE_CH = 0x20;

    private static final String[] checkList = {GD, OD, OG, RD};

    /*
     *  Data fields.
     */
    @LookupField (strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    // Supplied configurable fields - bias voltages
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private double  gdP;
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private double  odP;
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private double  ogP;
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private double  rdP;
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private double  csGateP;

    // Configurable bias voltage limits
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS)
    private double  gdMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS)
    private double  gdMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS)
    private double  odMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS)
    private double  odMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS)
    private double  ogMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS)
    private double  ogMax;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS)
    private double  rdMin;
    @ConfigurationParameter(category=REBDevice.RAFTS_LIMITS)
    private double  rdMax;

    // Supplied configurable fields - bias DAC settings
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private int  gd;
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private int  od;
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private int  og;
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private int  ogSh;
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private int  rd;
    @ConfigurationParameter(category=REBDevice.RAFTS)
    private int  csGate;

    // Set from Groovy file
    private boolean raw = false;

    // Other
    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private boolean valid;
    private int hwVersion;
    private BoardDacs dac;
    private int changed = -1;
    private final Map<String, Double> lowLimitMap = new HashMap<>();
    private final Map<String, Double> highLimitMap = new HashMap<>();
    private String configError;
    private double gdConv, odConv, rdConv, ogConv, ogShiftConv, csConv, csOffset;


    /**
     *  Empty constructor - implies physical quantities.
     */
    public BiasControl()
    {
        for (String pName : checkList) {
            lowLimitMap.put(pName, 0.0);
            highLimitMap.put(pName, 0.0);
        }
    }


    /**
     *  Constructor using raw DAC values.
     *
     *  @param  desc
     *  @param  hwChan
     *  @param  gd
     *  @param  od
     *  @param  og
     *  @param  ogSh
     *  @param  rd
     *  @param  csGate
     */
    @Deprecated
    public BiasControl(String desc, int hwChan, int gd, int od, int og, int ogSh, int rd, int csGate)
    {
        super();
        super.description = desc;
        super.hwChan = hwChan;
        this.gd = gd;
        this.od = od;
        this.og = og;
        this.ogSh = ogSh;
        this.rd = rd;
        this.csGate = csGate;
        raw = true;
    }


    /**
     *  Configures channel description.
     *
     *  @param  mon   The associated monitor object
     *  @param  devc  The associated device object
     */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        if (configError != null) {
            sLog.error("Config error (startup): " + getName() + ": " + configError);
        }
        dac = ((REBDevice)devc).getBoardDacs();
    }


    /**
     *  Checks configuration.
     *
     *  @return  Whether configuration is okay
     */
    public boolean checkConfig()
    {
        hwVersion = getHwVersion(dac);
        if (hwVersion == REB.VERSION_UNSUPP) {
            sLog.error("BIAS " + hwChan + " configuration incompatible with the firmware");
            return false;
        }

        try {
            if (hwChan < 0 || hwChan >= dac.getNumStrips()) {
                MonitorLogUtils.reportError(sLog, getName(), "HW channel", hwChan);
            }
        }
        catch (DriverException e) {
            sLog.error("Error getting number of strips");
            return false;
        }
        catch (Exception e) {
            /* Thrown by reportError */
            return false;
        }

        if (!raw) {
            if (hwVersion == REB.VERSION_1) {
                odConv = REBDevice.DAC_CONV / (1.0 + 49.9 / 10.0);
                gdConv = odConv;
                rdConv = odConv;
            }
            else if (hwVersion == REB.VERSION_2) {
                odConv = REBDevice.DAC_CONV / (1.0 + 62.0 / 9.9);  // Zener diode leakage
                gdConv = REBDevice.DAC_CONV / (1.0 + 62.0 / 10.0);
                rdConv = REBDevice.DAC_CONV / (1.0 + 40.2 / 10.0);
            }
            else {
                odConv = REBDevice.DAC_CONV / (1.0 + 62.0 / 10.0);
                gdConv = odConv;
                rdConv = REBDevice.DAC_CONV / (1.0 + 40.2 / 10.0);
            }
            double ogGain = 10.0 / 10.0;
            double fudge = hwVersion == REB.VERSION_3 ? 1.0 : 1.1;
            ogShiftConv = fudge * REBDevice.DAC_CONV / ogGain;
            ogConv = fudge * REBDevice.DAC_CONV / (1.0 + ogGain);
            csConv = REBDevice.DAC_CONV * 2.0;    // 2 Kohm -> 1 mA = 2 V
            csOffset = -0.95; // current offset in mA
        }
        else {
            valid = true;   // No checking of values
        }

        return true;
    }


    /**
     *  Gets the raw state.
     *
     *  @return  Whether configuration data are raw ADC values
     */
    public boolean isRaw()
    {
        return raw;
    }


    /**
     *  Gets the hardware version.
     *
     *  @param  dac  The board DAC object
     *  @return  The hardware version
     */
    public static int getHwVersion(BoardDacs dac)
    {
        int version = dac.getVersion();
        if (version == BoardDacs.VERSION_WREB1
              || version == BoardDacs.VERSION_WREB2
              || version == BoardDacs.VERSION_GREB1) {
            version = REB.VERSION_1;
        }
        else if (version == BoardDacs.VERSION_REB3
                   || version == BoardDacs.VERSION_REB5) {
            version = REB.VERSION_2;
        }
        else if (version == BoardDacs.VERSION_GREB2
                   || version == BoardDacs.VERSION_WREB4) {
            version = REB.VERSION_3;
        }
        else {
            version = REB.VERSION_UNSUPP;
        }

        return version;
    }


    /**
     *  Gets the hardware version.
     *
     *  @return  The hardware version
     */
    @Deprecated
    public int getHwVersion()
    {
        return hwVersion;
    }


    /**
     *  Validates configuration parameter change.
     *
     *  @param  params  The map of configuration parameter values
     *  @throws  IllegalArgumentException
     */
    @Override
    public void validateBulkChange(Map<String, Object> params) throws IllegalArgumentException
    {
        if (raw) return;
        String errMsg = null;
        boolean newLimits = false;
        for (String pName : checkList) {
            double value = (Double)params.get(pName + "P");
            if (value > (Double)params.get(pName + "Max")) {
                errMsg = pName + "P is above high limit";
                break;
            }
            if (value < (Double)params.get(pName + "Min")) {
                errMsg = pName + "P is below low limit";
                break;
            }
        }
        //if (errMsg == null && (Double)params.get(OD_P) < (Double)params.get(RD_P)) {
        //    errMsg = "odP is below rdP";
        //}
        if (errMsg != null && getName() != null) {
            for (String pName : checkList) {
                if ((Double)params.get(pName + "Max") != highLimitMap.get(pName)
                      || (Double)params.get(pName + "Min") != lowLimitMap.get(pName)) {
                    newLimits = true;
                    break;
                }
            }
        }
        if (errMsg != null) {
            if (getName() == null) {
                valid = false;
                configError = errMsg;
            }
            else if (newLimits) {
                valid = false;
                sLog.error("Config error (new limits): " + getName() + ": " + errMsg);
            }
            else {
                errMsg = getName() + ": " + errMsg;
                sLog.error("Config error: " + errMsg);
                throw new IllegalArgumentException(errMsg);
            }
        }
        else {
            valid = true;
        }
    }


    /**
     *  Sets the Guard Diode voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGdP(double value)
    {
        gdP = value;
    }


    /**
     *  Gets the Guard Diode voltage.
     *
     *  @return  The value
     */
    @Command(type=CommandType.QUERY, description="Gets the Guard Diode voltage")
    public double getGdP()
    {
        return gdP;
    }


    /**
     *  Sets the Guard Diode voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGdMax(double value)
    {
        gdMax = value;
        highLimitMap.put(GD, value);
    }


    /**
     *  Sets the Guard Diode voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGdMin(double value)
    {
        gdMin = value;
        lowLimitMap.put(GD, value);
    }


    /**
     *  Sets the Output Drain voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOdP(double value)
    {
        odP = value;
    }


    /**
     *  Gets the Output Drain voltage.
     *
     *  @return  The value
     */
    @Command(type=CommandType.QUERY, description="Gets the readout mode Output Drain voltage")
    public double getOdP()
    {
        return odP;
    }


    /**
     *  Sets the Output Drain voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOdMax(double value)
    {
        odMax = value;
        highLimitMap.put(OD, value);
    }


    /**
     *  Sets the Output Drain voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOdMin(double value)
    {
        odMin = value;
        lowLimitMap.put(OD, value);
    }


    /**
     *  Sets the Output Gate voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOgP(double value)
    {
        ogP = value;
    }


    /**
     *  Gets the Output Gate voltage.
     *
     *  @return  The value
     */
    @Command(type=CommandType.QUERY, description="Gets the Output Gate voltage")
    public double getOgP()
    {
        return ogP;
    }


    /**
     *  Sets the Output Gate voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOgMax(double value)
    {
        ogMax = value;
        highLimitMap.put(OG, value);
    }


    /**
     *  Sets the Output Gate voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOgMin(double value)
    {
        ogMin = value;
        lowLimitMap.put(OG, value);
    }


    /**
     *  Sets the Reset Drain voltage.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRdP(double value)
    {
        rdP = value;
    }


    /**
     *  Gets the Reset Drain voltage.
     *
     *  @return  The value
     */
    @Command(type=CommandType.QUERY, description="Gets the Reset Drain voltage")
    public double getRdP()
    {
        return rdP;
    }


    /**
     *  Sets the Reset Drain voltage maximum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRdMax(double value)
    {
        rdMax = value;
        highLimitMap.put(RD, value);
    }


    /**
     *  Sets the Reset Drain voltage minimum.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRdMin(double value)
    {
        rdMin = value;
        lowLimitMap.put(RD, value);
    }


    /**
     *  Sets the Current Source gate current.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setCsGateP(double value)
    {
        csGateP = value;
    }


    /**
     *  Gets the Current Source gate current.
     *
     *  @return  The value
     */
    @Command(type=CommandType.QUERY, description="Gets the Current Source gate")
    public double getCsGateP()
    {
        return csGateP;
    }


    /**
     *  Sets the Guard Diode DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setGd(int value)
    {
        gd = value;
        changed |= GD_CH;
    }


    /**
     *  Gets the Guard Diode DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Gets the Guard Diode DAC value")
    public int getGd()
    {
        return gd;
    }


    /**
     *  Sets the Output Drain DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOd(int value)
    {
        od = value;
        changed |= OD_CH;
    }


    /**
     *  Gets the Output Drain DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Gets the readout mode Output Drain DAC value")
    public int getOd()
    {
        return od;
    }


    /**
     *  Sets the Output Gate DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOg(int value)
    {
        og = value;
        changed |= OG_CH;
    }


    /**
     *  Gets the Output Gate DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Gets the Output Gate DAC value")
    public int getOg()
    {
        return og;
    }


    /**
     *  Sets the Output Gate shift DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setOgSh(int value)
    {
        ogSh = value;
        changed |= OG_SH_CH;
    }


    /**
     *  Gets the Output Gate shift DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Gets the Output Gate shift DAC value")
    public int getOgSh()
    {
        return ogSh;
    }


    /**
     *  Sets the Reset Drain DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setRd(int value)
    {
        rd = value;
        changed |= RD_CH;
    }


    /**
     *  Gets the Reset Drain DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Gets the Reset Drain DAC value")
    public int getRd()
    {
        return rd;
    }


    /**
     *  Sets the Current Source gate DAC value.
     *
     *  @param  value
     */
    @ConfigurationParameterChanger
    public void setCsGate(int value)
    {
        csGate = value;
        changed |= CSGATE_CH;
    }


    /**
     *  Gets the Current Source gate DAC value.
     *
     *  @return  value
     */
    @Command(type=CommandType.QUERY, description="Gets the Current Source gate")
    public int getCsGate()
    {
        return csGate;
    }


    /**
     *  Sets all the configuration data.
     * 
     *  @param  bias  The bias DAC values
     */
    void setConfig(BiasDACS bias)
    {
        String name = getName();
        if (!raw) {
            double[] values = bias.getPValues();
            sce.submitChange(name, GD_P, values[BiasDACS.GD]);
            sce.submitChange(name, OD_P, values[BiasDACS.OD]);
            sce.submitChange(name, OG_P, values[BiasDACS.OG]);
            sce.submitChange(name, RD_P, values[BiasDACS.RD]);
            sce.submitChange(name, CS_GATE_P, values[BiasDACS.CS_GATE]);
        }
        else {
            int[] values = bias.getValues();
            sce.submitChange(name, GD, values[BiasDACS.GD]);
            sce.submitChange(name, OD, values[BiasDACS.OD]);
            sce.submitChange(name, OG, values[BiasDACS.OG]);
            sce.submitChange(name, OG_SH, values[BiasDACS.OG_SH]);
            sce.submitChange(name, RD, values[BiasDACS.RD]);
            sce.submitChange(name, CS_GATE, values[BiasDACS.CS_GATE]);
        }
        try {
            sce.commitBulkChange();
        }
        catch (IllegalArgumentException e) {
            sce.dropSubmittedChangesForComponent(name);
            throw e;
        }
    }


    /**
     *  Gets all the configuration data.
     * 
     *  @return  The bias DAC values
     */
    BiasDACS getConfig()
    {
        BiasDACS bias = new BiasDACS();
        if (!raw) {
            double[] values = bias.getPValues();
            values[BiasDACS.GD] = gdP;
            values[BiasDACS.OD] = odP;
            values[BiasDACS.OG] = ogP;
            values[BiasDACS.RD] = rdP;
            values[BiasDACS.CS_GATE] = csGateP;
        }
        else {
            int[] values = bias.getValues();
            values[BiasDACS.GD] = gd;
            values[BiasDACS.OD] = od;
            values[BiasDACS.OG] = og;
            values[BiasDACS.OG_SH] = ogSh;
            values[BiasDACS.RD] = rd;
            values[BiasDACS.CS_GATE] = csGate;
        }
        return bias;
    }


    /**
     *  Loads configuration data to the biases DAC.
     *
     *  @return  The number of DAC channels loaded: 6
     *  @throws  Exception
     */
    int load() throws Exception
    {
        if (!valid) {
            throw new RaftException(getName() + " configuration not valid");
        }
        if (!raw) {
            setDac(BoardDacs.CHAN_GD, (int)(gdConv * gdP));
            setDac(BoardDacs.CHAN_OD, (int)(odConv * odP));
            setDac(BoardDacs.CHAN_OG, ogP > 0.0 ? (int)(ogConv * ogP) : 0);
            setDac(BoardDacs.CHAN_OG_SH, ogP < 0.0 ? (int)(-ogShiftConv * ogP) : 0);
            setDac(BoardDacs.CHAN_RD, (int)(rdConv * rdP));
            setDac(BoardDacs.CHAN_CSGATE, (int)(csConv * (csGateP + csOffset)));
        }
        else {
            setDac(BoardDacs.CHAN_GD, gd);
            setDac(BoardDacs.CHAN_OD, od);
            setDac(BoardDacs.CHAN_OG, og);
            setDac(BoardDacs.CHAN_OG_SH, ogSh);
            setDac(BoardDacs.CHAN_RD, rd);
            setDac(BoardDacs.CHAN_CSGATE, csGate);
        }
        loadDac();
        changed = 0;
        return 6;
    }


    /**
     *  Loads changed configuration data to the biases DAC.
     *
     *  @return  The number of DAC channels loaded
     *  @throws  Exception
     */
    int loadChanged() throws Exception
    {
        if (!valid) {
            throw new RaftException(getName() + " configuration not valid");
        }
        int count = 0;
        if (changed != 0) {
            if (!raw) {
                if ((changed & GD_CH) != 0) {
                    setDac(BoardDacs.CHAN_GD, (int)(gdConv * gdP));
                    count++;
                }
                if ((changed & OD_CH) != 0) {
                    setDac(BoardDacs.CHAN_OD, (int)(odConv * odP));
                    count++;
                }
                if ((changed & OG_CH) != 0) {
                    setDac(BoardDacs.CHAN_OG, ogP > 0.0 ? (int)(ogConv * ogP) : 0);
                    setDac(BoardDacs.CHAN_OG_SH, ogP < 0.0 ? (int)(-ogShiftConv * ogP) : 0);
                    count += 2;
                }
                if ((changed & RD_CH) != 0) {
                    setDac(BoardDacs.CHAN_RD, (int)(rdConv * rdP));
                    count++;
                }
                if ((changed & CSGATE_CH) != 0) {
                    setDac(BoardDacs.CHAN_CSGATE, (int)(csConv * (csGateP + csOffset)));
                    count++;
                }
            }
            else {
                if ((changed & GD_CH) != 0) {
                    setDac(BoardDacs.CHAN_GD, gd);
                    count++;
                }
                if ((changed & OD_CH) != 0) {
                    setDac(BoardDacs.CHAN_OD, od);
                    count++;
                }
                if ((changed & OG_CH) != 0) {
                    setDac(BoardDacs.CHAN_OG, og);
                    count++;
                }
                if ((changed & OG_SH_CH) != 0) {
                    setDac(BoardDacs.CHAN_OG_SH, ogSh);
                    count++;
                }
                if ((changed & RD_CH) != 0) {
                    setDac(BoardDacs.CHAN_RD, rd);
                    count++;
                }
                if ((changed & CSGATE_CH) != 0) {
                    setDac(BoardDacs.CHAN_CSGATE, csGate);
                    count++;
                }
            }
            loadDac();
            changed = 0;
        }
        return count;
    }


    /**
     *  Clears (zeroes) the biases DAC values.
     *
     *  @throws  Exception
     */
    void clear() throws Exception
    {
        setDac(BoardDacs.CHAN_GD, 0);
        setDac(BoardDacs.CHAN_OD, 0);
        setDac(BoardDacs.CHAN_OG, 0);
        setDac(BoardDacs.CHAN_OG_SH, 0);
        setDac(BoardDacs.CHAN_RD, 0);
        setDac(BoardDacs.CHAN_CSGATE, 0);
        loadDac();
        changed = -1;
    }


    /**
     *  Sets a DAC value.
     *
     *  @param  chan   The channel number
     *  @param  value  The value to set
     *  @throws  Exception
     */
    void setDac(int chan, int value) throws Exception
    {
        testOnline();
        try {
            dac.set(hwChan, chan, value < 0 ? 0 : value > REBDevice.DAC_LIMIT ? REBDevice.DAC_LIMIT : value);
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
        }
    }


    /**
     *  Loads DAC values.
     *
     *  @throws  Exception
     */
    void loadDac() throws Exception
    {
        testOnline();
        try {
            dac.loadStrip(hwChan);
        }
        catch (REBException e) {
            checkTimeout(e, RaftException.class);
        }
    }

}
