package org.lsst.ccs.subsystem.rafts.ui;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.text.NumberFormatter;
import org.lsst.ccs.subsystem.rafts.config.BiasDACS;

/**
 *  Biases configuration editor.
 *
 *  @author  Owen Saxton
 */
class BiasesConfigPanel extends JPanel {

    private static final Insets INSETS_STD = new Insets(4, 4, 4, 4);
    private static final int TF_WIDTH = 50, TF_WIDTH_L = 80;
    private final NumberFormatter fmt = new NumberFormatter();
    private final JLabel[] lbBiases = new JLabel[BiasDACS.NUM_VALUES];
    private final JFormattedTextField[] tfBiases = new JFormattedTextField[BiasDACS.NUM_VALUES];
    private final int iBias;
    private final BiasDACS bias;
    private final boolean raw;
    private final Map biasNames;
    private final List<Integer> biasList;

    private static final Map<Integer, String> biasNamesP = new LinkedHashMap<>();
    static {
        biasNamesP.put(BiasDACS.GD, "GD");
        biasNamesP.put(BiasDACS.RD, "RD");
        biasNamesP.put(BiasDACS.OG, "OG");
        biasNamesP.put(BiasDACS.OD, "OD");
        biasNamesP.put(BiasDACS.CS_GATE, "CS Gate");
    }
    private static final List<Integer> biasListP = new ArrayList<>(biasNamesP.keySet());

    private static final Map<Integer, String> biasNamesR = new LinkedHashMap<>();
    static {
        biasNamesR.put(BiasDACS.GD, "GD");
        biasNamesR.put(BiasDACS.RD, "RD");
        biasNamesR.put(BiasDACS.OG, "OG");
        biasNamesR.put(BiasDACS.OG_SH, "OG Shift");
        biasNamesR.put(BiasDACS.OD, "OD");
        biasNamesR.put(BiasDACS.CS_GATE, "CS Gate");
    }
    private static final List<Integer> biasListR = new ArrayList<>(biasNamesR.keySet());


   /**
    *  Constructor.
    */
    BiasesConfigPanel(int ia, BiasDACS a, boolean r) {

        iBias = ia;
        bias = a;
        raw = r;

        fmt.setFormat(new DecimalFormat(raw ? "####0" : "##0.0#"));
        fmt.setValueClass(raw ? Integer.class : Double.class);
        JFormattedTextField tf = new JFormattedTextField(fmt);
        Dimension d5 = tf.getPreferredSize();
        d5.width = TF_WIDTH;
        biasList = raw ? biasListR : biasListP;
        biasNames = raw ? biasNamesR : biasNamesP;
        for (int bnum : biasList) {
            lbBiases[bnum] = new JLabel(biasNames.get(bnum) + ":");
            tfBiases[bnum] = new JFormattedTextField(fmt);
            tfBiases[bnum].setPreferredSize(d5);
            tfBiases[bnum].setMinimumSize(d5);
        }

        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = INSETS_STD;

        String title = "Bias DAC " + iBias;
        setBorder(BorderFactory.createTitledBorder(title));

        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = INSETS_STD;

        int fld = 0;
        for (int bnum : biasList) {
            gbc.gridx = 2 * (fld & 1);
            gbc.gridy += 1 - (fld & 1);
            add(lbBiases[bnum], gbc);
            gbc.gridx++;
            add(tfBiases[bnum], gbc);
            fld++;
        }
    }


   /**
    *  Saves new values to the bias data.
    */
    void save() {

        if (raw) {
            int[] values = bias.getValues();
            for (int bnum : biasList) {
                values[bnum] = (Integer)tfBiases[bnum].getValue();
            }
        }
        else {
            double[] values = bias.getPValues();
            for (int bnum : biasList) {
                values[bnum] = (Double)tfBiases[bnum].getValue();
            }
        }
    }


   /**
    *  Reads values from the bias data and populates fields.
    */
    void read() {

        if (raw) {
            int[] values = bias.getValues();
            for (int bnum : biasList) {
                tfBiases[bnum].setValue(values[bnum]);
            }
        }
        else {
            double[] values = bias.getPValues();
            for (int bnum : biasList) {
                tfBiases[bnum].setValue(values[bnum]);
            }
        }
    }

    private static final long serialVersionUID = 1L;
}
