package org.lsst.ccs.subsystem.rafts.ui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JWindow;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.ImageState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the rafts control panel.
 *
 *  @author Owen Saxton
 */
public class RaftsControlPanel extends JPanel implements CommandSender.ReplyHandler, UiUtilities.ActionHandler {

    private static final String LF = "\n";

    private final JPanel mainPanel = UiUtilities.newBorderedPanel(null);
    private final JPanel subMainPanel = new JPanel();

    private final JPanel raftsStPanel = new JPanel();
    private JLabel lblStValue;
    private final JPanel rebStPanel = new JPanel();
    private JLabel lblRebs;

    private final JPanel cfgPanel = UiUtilities.newBorderedPanel("Config");
    private JButton[] btnCfgREB = new JButton[0];
    private JButton btnCfgSave;

    private final JPanel setPanel = UiUtilities.newBorderedPanel("Setup");
    private JButton btnLoadDacs, btnLoadAspics;
    private JCheckBox cbxBackBias;

    private final JPanel seqPanel = UiUtilities.newBorderedPanel("Sequencer");
    private JButton btnSeqLoad, btnSeqStart, btnSeqStop, btnSeqStep, btnResetScan, btnResetFE;
    private JCheckBox cbxScanMode, cbxPattern;

    private final JPanel imgPanel = UiUtilities.newBorderedPanel("Images");
    private JButton btnAcquire, btnSetDir, btnSaveRaw, btnSaveFits;

    private final JPanel msgPanel = UiUtilities.newBorderedPanel("Messages");
    private final JTextArea taMessage = new JTextArea(8, 40);
    private final JScrollPane spMessage = new JScrollPane(taMessage);

    private static final Logger LOG = Logger.getLogger(RaftsControlPanel.class.getName());
    private final UiUtilities uiUtils;
    private final CommandSender sender;
    private List<String> rebNames = new ArrayList<>();
    private static final Map<String, REBConfigPanel> rebConfigMap = new HashMap<>();
    private String imageDir;
    private boolean active;
    private int aspicCount;


   /**
    *  Constructor.
    *
    *  @param  agent   The name of the subsystem agent.
    */
    public RaftsControlPanel(String agent) {
        sender = new CommandSender(agent, this);
        uiUtils = new UiUtilities(this);
        initStatusPanels();
        initConfigPanel();
        initSetupPanel();
        initSequencerPanel();
        initImagePanel();
        initSubMainPanel();
        initMessagePanel();
        initMainPanel();
        initTopPanel();
    }


    public void initPanel() {
        sender.sendCommand(true, null, "getFullState");
    }


   /**
    *  Enables the rafts subsystem.
    */
    public void enablePanel() {
        active = true;
        SwingUtilities.invokeLater(new UpdateStatus());
    }


   /**
    *  Disables the rafts subsystem.
    */
    public void disablePanel() {
        active = false;
        SwingUtilities.invokeLater(new UpdateStatus());
    }


   /**
    *  Reports image received messages from the rafts subsystem.
    *
    *  @param  state  The image state data
    */
    public void reportImage(ImageState state) {
        String message = String.format("Received image: timestamp = %016x, length = %s",
                                       state.getTimestamp(), state.getLength());
        updateMessage(message);
    }


   /**
    *  Sends a command to the rafts subsystem.
    *
    *  @param  reply    True if a reply is expected
    *  @param  name     The sub-component (REB) name
    *  @param  command  The command to send
    *  @param  params   The command parameters
    */
    void sendCommand(boolean reply, String name, String command, Object... params) {
        if (active) {
            sender.sendCommand(reply, name, command, params);
        }
        else {
            displayError("Rafts system not running");
        }
    }


    /**
     *  Handles command replies.
     * 
     *  @param  reply    The reply object
     *  @param  path     The sub-path to the cammand receiver
     *  @param  command  The command
     *  @param  args     The command arguments
     */
    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        switch (command) {
        case "getFullState":
            enablePanel();
            break;

        case "setREBConfig":
            display(path + " configuration applied");
            sendCommand(true, path, "getREBConfig");
            break;

        case "getREBConfig":
            REBConfigPanel panel = rebConfigMap.get(path);
            if (panel != null) {
                panel.setConfig((REB)reply);
            }
            else {
                panel = new REBConfigPanel((REB)reply, path, this);
                rebConfigMap.put(path, panel);
                Window pWindow = SwingUtilities.getWindowAncestor(this);
                JDialog dlg = new JDialog(pWindow, path + " Configuration", Dialog.ModalityType.MODELESS);
                dlg.add(panel, BorderLayout.CENTER);
                dlg.pack();
                dlg.setLocationRelativeTo(pWindow);
                dlg.addWindowListener(new WindowAdapter() {
                    @Override
                    public void windowClosing(WindowEvent e) {
                        rebConfigMap.remove(path);
                        dlg.setVisible(false);
                        dlg.dispose();
                    }
                });
                dlg.setVisible(true);
            }
            break;

        case "loadDacs":
            display((Integer)reply + (path == null ? "" : " " + path) + " DACs loaded");
            sendCommand(true, path, "loadBiasDacs", true);
            break;

        case "loadBiasDacs":
            display((Integer)reply + (path == null ? "" : " " + path) + " bias DACs loaded");
            break;

        case "loadAspics":
            aspicCount = (Integer)reply;
            sendCommand(true, path, "checkAspics");
            break;

        case "checkAspics":
            List<Integer> masks;
            if (path == null) {
                masks = (List)reply;
            }
            else {
                masks = new ArrayList();
                masks.add((Integer)reply);
            }
            display(aspicCount + (path == null ? "" : " " + path) + " ASPICs loaded: " + getResult(masks));
            break;

        case "loadSequencer":
            List<Integer> nSlice = (List)reply;
            StringBuilder msg = new StringBuilder("Slice counts = ");
            boolean first = true;
            for (int count : nSlice) {
                msg.append(first ? "" : ", ").append(count);
                first = false;
            }
            display(msg.toString());
            break;

        case "setBackBias":
            setBackBiasCB();
            break;

        case "isBackBiasOn":
            List<Boolean> bValues = (List)reply;
            boolean on = true;
            for (boolean val : bValues) {
                on = !val ? false : on;
            }
            cbxBackBias.setSelected(on);
            break;

        case "enableScan":
            setScanModeCB();
            break;

        case "isScanEnabled":
            bValues = (List)reply;
            boolean scanMode = true;
            for (boolean val : bValues) {
                scanMode = !val ? false : scanMode;
            }
            cbxScanMode.setSelected(scanMode);
            break;

        case "setDataSource":
            setPatternCB();
            break;

        case "getDataSource":
            List<Integer> iValues = (List)reply;
            boolean pattern = true;
            for (int val : iValues) {
                pattern = (val & 1) == 0 ? false : pattern;
            }
            cbxPattern.setSelected(pattern);
            break;

        case "resetFirmware":
            setScanModeCB();
            setPatternCB();
            break;

        case "saveImage":
        case "saveFitsImage":
            List<String> names = (List)reply;
            display("The following files were created:");
            for (String name : names) {
                display("  " + name);
            }
            break;

        case "getREBDeviceNames":
            rebNames = (List)reply;
            StringBuilder text = new StringBuilder();
            for (String name : rebNames) {
                text.append(" ").append(name);
            }
            lblRebs.setText(text.toString());
            updateConfig();
            updateSetup();
            setPatternCB();
            setScanModeCB();
            enableButtons(true);
            break;

        default:
            displayError("Reply received from unrecognized command: " + command);
        }
    }


   /**
    *  Displays a message in the message area.
    *
    *  @param  message  The message to be displayed
    */
    void display(String message) {
        taMessage.append(message + LF);
    }


    /**
     *  Handles button pushes
     *
     *  @param  name  The button name
     */
    @Override
    public void handleButton(String name) {
        switch (name.substring(0, 2)) {
        case "CR":
            cfgREB(name.substring(2));
            break;

        case "CS":
            sendCommand(false, null, "saveNamedConfig", "");
            break;

        case "LD":
            sendCommand(true, null, "loadDacs", true);
            break;

        case "LA":
            sendCommand(true, null, "loadAspics", true);
            break;

        case "LS":
            seqLoad();
            break;

        case "SS":
            sendCommand(false, null, "startSequencer");
            break;

        case "OS":
            sendCommand(false, null, "stopSequencer");
            break;

        case "ES":
            sendCommand(false, null, "stepSequencer");
            break;

        case "RS":
            sendCommand(false, null, "resetScan");
            break;

        case "RF":
            sendCommand(true, null, "resetFirmware");
            break;

        case "AI":
            sendCommand(false, null, "acquireImage");
            break;

        case "SD":
            setImageDir();
            break;

        case "SR":
            saveImage("saveImage");
            break;

        case "SF":
            saveImage("saveFitsImage");
            break;
        }
    }


    /**
     *  Handles check boxes
     *
     *  @param  name  The check box name
     *  @param  chkd  Whether checked
     */
    @Override
    public void handleCheckBox(String name, boolean chkd) {
        switch (name.charAt(0)) {
        case 'B':
            sendCommand(true, null, "setBackBias", chkd);
            break;

        case 'S':
            sendCommand(true, null, "enableScan", chkd);
            break;

        case 'D':
            sendCommand(true, null, "setDataSource", chkd ? 1 : 0);
            break;
        }
    }


    /**
     *  Closes a configuration dialog.
     *
     *  @param  name  The REB name
     */
    void closeConfig(String name) {
        JPanel pnl = rebConfigMap.get(name);
        if (pnl != null) {
            rebConfigMap.remove(name);
            JDialog dlg = (JDialog)SwingUtilities.getWindowAncestor(pnl);
            dlg.setVisible(false);
            dlg.dispose();
        }
    }


    /**
    *  Initializes the status sub-panels.
    */
    private void initStatusPanels() {
        lblStValue = UiUtilities.newLabel("X", UiUtilities.maxLabelWidth(new String[]{"RUNNING", "STOPPED"}, ""));
        lblStValue.setForeground(UiConstants.RED);
        lblStValue.setText("STOPPED");
        raftsStPanel.add(UiUtilities.newLabel("Rafts Status: ", 0));
        raftsStPanel.add(lblStValue);

        lblRebs = UiUtilities.newLabel("none", -1);
        rebStPanel.add(UiUtilities.newLabel("Connected REBs:", 0));
        rebStPanel.add(lblRebs);
    }


   /**
    *  Initializes the configuration sub-panel.
    */
    private void initConfigPanel() {
        btnCfgREB = new JButton[rebNames.size()];
        for (int j = 0; j < btnCfgREB.length; j++) {
            String name = rebNames.get(j);
            btnCfgREB[j] = uiUtils.newButton(name + "...", "CR" + name);
        }
        btnCfgSave = uiUtils.newButton("Save", "CS");

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        for (JButton btn : btnCfgREB) {
            cfgPanel.add(btn, gbc);
            gbc.gridy++;
        }
        cfgPanel.add(btnCfgSave, gbc);
    }


   /**
    *  Initializes the setup sub-panel.
    */
    private void initSetupPanel() {
        btnLoadDacs = uiUtils.newButton("Load DACs", "LD");
        btnLoadAspics = uiUtils.newButton("Load ASPICs", "LA");
        cbxBackBias = uiUtils.newCheckBox("Back Bias", "B");

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        setPanel.add(btnLoadDacs, gbc);
        gbc.gridy++;
        setPanel.add(btnLoadAspics, gbc);
        gbc.gridy++;
        setPanel.add(cbxBackBias, gbc);
    }


   /**
    *  Initializes the sequencer sub-panel.
    */
    private void initSequencerPanel() {
        btnSeqLoad = uiUtils.newButton("Load...", "LS");
        btnSeqStart = uiUtils.newButton("Start", "SS");
        btnSeqStop = uiUtils.newButton("Stop", "OS");
        btnSeqStep = uiUtils.newButton("Step", "ES");
        cbxScanMode = uiUtils.newCheckBox("Scan Mode", "S");
        btnResetScan = uiUtils.newButton("Reset Scan", "RS");
        cbxPattern = uiUtils.newCheckBox("Pattern", "D");
        btnResetFE = uiUtils.newButton("Reset FE", "RF");
        
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        seqPanel.add(btnSeqLoad, gbc);
        gbc.gridy++;
        seqPanel.add(btnSeqStart, gbc);
        gbc.gridy++;
        seqPanel.add(btnSeqStop, gbc);
        gbc.gridy++;
        seqPanel.add(btnSeqStep, gbc);
        gbc.gridx = 1;
        gbc.gridy = 0;
        seqPanel.add(cbxScanMode, gbc);
        gbc.gridy++;
        seqPanel.add(btnResetScan, gbc);
        gbc.gridy++;
        seqPanel.add(cbxPattern, gbc);
        gbc.gridy++;
        seqPanel.add(btnResetFE, gbc);
    }


   /**
    *  Initializes the images sub-panel.
    */
    private void initImagePanel() {
        btnAcquire = uiUtils.newButton("Acquire", "AI");
        btnSetDir = uiUtils.newButton("Directory...", "SD");
        btnSaveRaw = uiUtils.newButton("Save Raw", "SR");
        btnSaveFits = uiUtils.newButton("Save FITS", "SF");

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        imgPanel.add(btnAcquire, gbc);
        gbc.gridy++;
        imgPanel.add(btnSetDir, gbc);
        gbc.gridy++;
        imgPanel.add(btnSaveRaw, gbc);
        gbc.gridy++;
        imgPanel.add(btnSaveFits, gbc);
    }


   /**
    *  Initializes the sub-main sub-panel.
    */
    private void initSubMainPanel() {
        subMainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.insets = new Insets(0, 0, 0, 10);
        subMainPanel.add(cfgPanel, gbc);
        gbc.gridx++;
        gbc.insets.left = 10;
        subMainPanel.add(setPanel, gbc);
        gbc.gridx++;
        subMainPanel.add(seqPanel, gbc);
        gbc.gridx++;
        gbc.insets.right = 0;
        subMainPanel.add(imgPanel, gbc);
    }


   /**
    *  Initializes the messages sub-panel.
    */
    private void initMessagePanel() {
        msgPanel.add(spMessage);
        spMessage.setMinimumSize(spMessage.getPreferredSize());
        taMessage.setEditable(false);
    }


   /**
    *  Initializes the main sub-panel.
    */
    private void initMainPanel() {
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.insets = new Insets(10, 10, 5, 10);

        gbc.gridx = 0;
        gbc.gridy = 0;
        mainPanel.add(raftsStPanel, gbc);

        gbc.insets.top = 0;
        gbc.insets.bottom = 10;
        gbc.gridy++;
        mainPanel.add(rebStPanel, gbc);

        gbc.insets.top = 10;
        gbc.gridy++;
        mainPanel.add(subMainPanel, gbc);

        gbc.gridy++;
        mainPanel.add(msgPanel, gbc);
    }


   /**
    *  Initializes the top-level panel.
    */
    private void initTopPanel() {
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTH;
        add(mainPanel, gbc);
    }


   /**
    *  Configures a REB.
    */
    private void cfgREB(String name) {
        JPanel pnl = rebConfigMap.get(name);
        if (pnl != null) {
            SwingUtilities.getWindowAncestor(pnl).toFront();
        }
        else {
            sendCommand(true, name, "getREBConfig");
        }
    }


   /**
    *  Forms a result string from a list of mismatch masks.
    */
    private String getResult(List<Integer> masks) {
        if (masks == null) return "check not done";
        int sum = 0;
        for (int mask : masks) {
            sum |= mask;
        }
        if (sum == 0) return "no errors";
        StringBuilder result = new StringBuilder("mismatches =");
        for (int mask : masks) {
            result.append(String.format(" %08x", mask));
        }
        return result.toString();
    }


   /**
    *  Loads the sequencers.
    */
    private void seqLoad() {
        JFileChooser fc = new JFileChooser("./");
        fc.setFileFilter(new FileNameExtensionFilter(null, "xml", "seq", "sif"));
        if (fc.showOpenDialog(getTopFrame()) != JFileChooser.APPROVE_OPTION) return;
        String fn;
        try {
            fn = fc.getSelectedFile().getCanonicalPath();
        }
        catch (IOException e) {
            displayError(e.toString());
            return;
        }
        sendCommand(true, null, "loadSequencer", fn);
    }


   /**
    *  Sets the back bias checkbox.
    */
    private void setBackBiasCB() {
        sendCommand(true, null, "isBackBiasOn");
    }


   /**
    *  Sets the scan mode checkbox.
    */
    private void setScanModeCB() {
        sendCommand(true, null, "isScanEnabled");
    }


   /**
    *  Sets the image pattern checkbox.
    */
    private void setPatternCB() {
        sendCommand(true, null, "getDataSource");
    }


   /**
    *  Sets the directory for saving images.
    */
    private void setImageDir() {
        JFileChooser fc = new JFileChooser(imageDir == null ? "./" : imageDir);
        fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        if (fc.showOpenDialog(getTopFrame()) == JFileChooser.APPROVE_OPTION) {
            imageDir = fc.getSelectedFile().getPath();
        }
    }


   /**
    *  Saves the current images.
    */
    private void saveImage(String cmnd) {
        if (imageDir != null) {
            sendCommand(true, null, cmnd, imageDir);
        }
        else {
            displayError("Image directory hasn't been set");
        }
    }


   /**
    *  Updates the configuration sub-panel.
    */
    private void updateConfig() {
        if (rebNames.size() != btnCfgREB.length) {
            cfgPanel.removeAll();
            initConfigPanel();
        }
        else {
            for (int j = 0; j < btnCfgREB.length; j++) {
                btnCfgREB[j].setName(rebNames.get(j) + "...");
            }
        }
    }


   /**
    *  Updates the setup sub-panel.
    */
    private void updateSetup() {
        setBackBiasCB();
    }


   /**
    *  Gets the top-most container of this panel.
    */
    private Component getTopFrame() {
        Component c = this;
        while (!(c instanceof JFrame) && !(c instanceof JWindow)
                && !(c instanceof JDialog) && c.getParent() != null) {
            c = c.getParent();
        }
        return c;
    }


   /**
    *  Updates the message area.
    */
    private void updateMessage(String msg) {
        SwingUtilities.invokeLater(new UpdateMessage(msg));
    }


   /**
    *  Displays an error message in the message area.
    */
    private void displayError(String message) {
        //taMessage.setForeground(Color.RED);
        taMessage.append(message + LF);
    }


   /**
    *  Class for processing external status updates.
    */
    class UpdateStatus implements Runnable {

        @Override
        public void run() {
            if (active) {
                lblStValue.setText("RUNNING");
                lblStValue.setForeground(UiConstants.GREEN);
                sendCommand(true, null, "getREBDeviceNames");
            }
            else {
                lblStValue.setText("STOPPED");
                lblStValue.setForeground(UiConstants.RED);
                enableButtons(false);
            }
        }
    }


   /**
    *  Class for processing externally-generated messages.
    */
    class UpdateMessage implements Runnable {

        String message;

        UpdateMessage(String message) {
            this.message = message;
        }

        @Override
        public void run() {
            display(message);
        }

    }
        
    private void enableButtons(boolean enable) {
        for ( JButton b : btnCfgREB ) {
            b.setEnabled(enable);
        }
        btnCfgSave.setEnabled(enable);
        btnLoadDacs.setEnabled(enable);
        btnLoadAspics.setEnabled(enable);
        cbxBackBias.setEnabled(enable);
        btnSeqLoad.setEnabled(enable);
        btnSeqStart.setEnabled(enable);
        btnSeqStop.setEnabled(enable);
        btnSeqStep.setEnabled(enable);
        cbxScanMode.setEnabled(enable);
        btnResetScan.setEnabled(enable);
        cbxPattern.setEnabled(enable);
        btnResetFE.setEnabled(enable);
        btnAcquire.setEnabled(enable);
        btnSaveRaw.setEnabled(enable);
        btnSaveFits.setEnabled(enable);
        repaint();
    }

    private static final long serialVersionUID = 1L;
}
