package org.lsst.ccs.subsystem.rafts.ui;

import java.awt.Color;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JWindow;
import javax.swing.SwingUtilities;
import javax.swing.border.TitledBorder;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.rafts.data.ImageState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements the rafts control panel.
 *
 *  @author Owen Saxton
 */
public class RaftsControlPanel extends JPanel implements REBConfigPanel.Parent {

    private static final Color RED = new Color(150, 0, 0), GREEN = new Color(0, 150, 0);
    private static final String LF = "\n";

    private final JPanel mainPanel = new JPanel();
    private final JPanel subMainPanel = new JPanel();
    private final JPanel raftsStPanel = new JPanel();
    private final JLabel lblStName = new JLabel("Rafts Status: ");
    private final JLabel lblStValue = new JLabel();
    private final JPanel rebStPanel = new JPanel();
    private final JLabel lblRbName = new JLabel("Connected REBs:");
    private final JLabel lblRebs = new JLabel();

    private final JPanel cfgPanel = new JPanel();
    private JButton[] btnCfgREB = new JButton[0];
    private final JButton btnCfgSave = new JButton("Save");

    private final JPanel setPanel = new JPanel();
    private final JButton btnLoadDacs = new JButton("Load DACs");
    private final JButton btnLoadAspics = new JButton("Load ASPICs");
    private final JCheckBox cbxBackBias = new JCheckBox("Back Bias");

    private final JPanel seqPanel = new JPanel();
    private final JButton btnSeqLoad = new JButton("Load...");
    private final JButton btnSeqStart = new JButton("Start");
    private final JButton btnSeqStop = new JButton("Stop");
    private final JButton btnSeqStep = new JButton("Step");
    private final JCheckBox cbxScanMode = new JCheckBox("Scan Mode");
    private final JButton btnResetScan = new JButton("Reset Scan");
    private final JCheckBox cbxPattern = new JCheckBox("Pattern");
    private final JButton btnResetFE = new JButton("Reset FE");

    private final JPanel imgPanel = new JPanel();
    private final JButton btnAcquire = new JButton("Acquire");
    private final JButton btnSetDir = new JButton("Directory...");
    private final JButton btnSaveRaw = new JButton("Save Raw");
    private final JButton btnSaveFits = new JButton("Save FITS");

    private final JPanel msgPanel = new JPanel();
    private final JTextArea taMessage = new JTextArea(8, 40);
    private final JScrollPane spMessage = new JScrollPane(taMessage);

    private static final Logger LOG = Logger.getLogger(RaftsControlPanel.class.getName());
    private final CommandSender gui;
    private List<String> rebNames = new ArrayList<>();
    private String imageDir;
    private boolean active;
    private static final long serialVersionUID = 1L;


   /**
    *  Constructor.
    *
    *  @param  gui   The controlling GUI.
    */
    public RaftsControlPanel(CommandSender gui) {
        this.gui = gui;
        initStatusPanels();
        initConfigPanel();
        initSetupPanel();
        initSequencerPanel();
        initImagePanel();
        initSubMainPanel();
        initMessagePanel();
        initMainPanel();
        initTopPanel();
    }


   /**
    *  Sends a command to the rafts subsystem.
    *
    *  @param  name     The sub-component (REB) name
    *  @param  command  The command to send
    *  @param  params   The command parameters
    *  @return  The command response
    */
    @Override
    public Object sendCommand(String name, String command, Object... params) {
        if (active) {
            return sendSubsysCommand(name, command, params);
        }
        else {
            displayError("Rafts system not running");
            return null;
        }
    }


   /**
    *  Displays a message in the message area.
    *
    *  @param  message  The message to be displayed
    */
    @Override
    public void display(String message) {
        taMessage.append(message + LF);
    }


   /**
    *  Initializes the status sub-panels.
    */
    private void initStatusPanels() {
        raftsStPanel.add(lblStName);
        lblStValue.setForeground(RED);
        lblStValue.setText("STOPPED");
        raftsStPanel.add(lblStValue);

        rebStPanel.add(lblRbName);
        lblRebs.setText("none");
        rebStPanel.add(lblRebs);
    }


   /**
    *  Initializes the configuration sub-panel.
    */
    private void initConfigPanel() {
        TitledBorder border = new TitledBorder("Config");
        border.setTitleJustification(TitledBorder.CENTER);
        cfgPanel.setBorder(border);
        cfgPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;

        btnCfgREB = new JButton[rebNames.size()];
        for (int j = 0; j < btnCfgREB.length; j++) {
            JButton btn = new JButton(rebNames.get(j) + "...");
            btnCfgREB[j] = btn;
            cfgPanel.add(btn, gbc);
            btn.setFocusable(false);
            btn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    cfgREB(e);
                }
            });
            gbc.gridy++;
        }

        cfgPanel.add(btnCfgSave, gbc);
        btnCfgSave.setFocusable(false);
        btnCfgSave.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand(null, "saveNamedConfig", "");
            }
        });
    }


   /**
    *  Initializes the setup sub-panel.
    */
    private void initSetupPanel() {
        TitledBorder border = new TitledBorder("Setup");
        border.setTitleJustification(TitledBorder.CENTER);
        setPanel.setBorder(border);
        setPanel.setLayout(new GridBagLayout());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;

        setPanel.add(btnLoadDacs, gbc);
        btnLoadDacs.setFocusable(false);
        btnLoadDacs.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                Object reply = sendCommand(null, "loadDacs", true);
                if (reply != null) {
                    display((Integer)reply + " DACs loaded");
                }
                reply = sendCommand(null, "loadBiasDacs", true);
                if (reply != null) {
                    display((Integer)reply + " bias DACs loaded");
                }
            }
        });

        gbc.gridy++;
        setPanel.add(btnLoadAspics, gbc);
        btnLoadAspics.setFocusable(false);
        btnLoadAspics.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                Integer count = (Integer)sendCommand(null, "loadAspics", true);
                if (count == null) return;
                List<Integer> masks = (List)sendCommand(null, "checkAspics");
                display(count + " ASPICs loaded: " + getResult(masks));
            }
        });

        gbc.gridy++;
        setPanel.add(cbxBackBias, gbc);
        cbxBackBias.setFocusable(false);
        cbxBackBias.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand(null, "setBackBias", cbxBackBias.isSelected());
                setBackBiasCB();
            }
        });
    }


   /**
    *  Initializes the sequencer sub-panel.
    */
    private void initSequencerPanel() {

        TitledBorder border = new TitledBorder("Sequencer");
        border.setTitleJustification(TitledBorder.CENTER);
        seqPanel.setBorder(border);
        seqPanel.setLayout(new GridBagLayout());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridx = 0;
        gbc.gridy = 0;

        seqPanel.add(btnSeqLoad, gbc);
        btnSeqLoad.setFocusable(false);
        btnSeqLoad.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                seqLoad();
            }
        });

        gbc.gridy++;
        seqPanel.add(btnSeqStart, gbc);
        btnSeqStart.setFocusable(false);
        btnSeqStart.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand(null, "startSequencer");
            }
        });

        gbc.gridy++;
        seqPanel.add(btnSeqStop, gbc);
        btnSeqStop.setFocusable(false);
        btnSeqStop.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand(null, "stopSequencer");
            }
        });

        gbc.gridy++;
        seqPanel.add(btnSeqStep, gbc);
        btnSeqStep.setFocusable(false);
        btnSeqStep.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand(null, "stepSequencer");
            }
        });

        gbc.gridx = 1;
        gbc.gridy = 0;
        seqPanel.add(cbxScanMode, gbc);
        cbxScanMode.setFocusable(false);
        cbxScanMode.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand(null, "enableScan", cbxScanMode.isSelected());
                setScanModeCB();
            }
        });

        gbc.gridy++;
        seqPanel.add(btnResetScan, gbc);
        btnResetScan.setFocusable(false);
        btnResetScan.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand(null, "resetScan");
            }
        });

        gbc.gridy++;
        seqPanel.add(cbxPattern, gbc);
        cbxPattern.setFocusable(false);
        cbxPattern.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand(null, "setDataSource",
                            cbxPattern.isSelected() ? 1 : 0);
                setPatternCB();
            }
        });

        gbc.gridy++;
        seqPanel.add(btnResetFE, gbc);
        btnResetFE.setFocusable(false);
        btnResetFE.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand(null, "resetFirmware");
                setScanModeCB();
                setPatternCB();
            }
        });
    }


   /**
    *  Initializes the images sub-panel.
    */
    private void initImagePanel() {
        TitledBorder border = new TitledBorder("Images");
        border.setTitleJustification(TitledBorder.CENTER);
        imgPanel.setBorder(border);
        imgPanel.setLayout(new GridBagLayout());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;

        imgPanel.add(btnAcquire, gbc);
        btnAcquire.setFocusable(false);
        btnAcquire.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendCommand(null, "acquireImage");
            }
        });

        gbc.gridy++;
        imgPanel.add(btnSetDir, gbc);
        btnSetDir.setFocusable(false);
        btnSetDir.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                setImageDir();
            }
        });

        gbc.gridy++;
        imgPanel.add(btnSaveRaw, gbc);
        btnSaveRaw.setFocusable(false);
        btnSaveRaw.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                saveImage("saveImage");
            }
        });

        gbc.gridy++;
        imgPanel.add(btnSaveFits, gbc);
        btnSaveFits.setFocusable(false);
        btnSaveFits.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                saveImage("saveFitsImage");
            }
        });
    }


   /**
    *  Initializes the sub-main sub-panel.
    */
    private void initSubMainPanel() {
        subMainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;

        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.insets = new Insets(0, 0, 0, 10);
        subMainPanel.add(cfgPanel, gbc);
        gbc.gridx++;
        gbc.insets.left = 10;
        subMainPanel.add(setPanel, gbc);
        gbc.gridx++;
        subMainPanel.add(seqPanel, gbc);
        gbc.gridx++;
        gbc.insets.right = 0;
        subMainPanel.add(imgPanel, gbc);
    }


   /**
    *  Initializes the messages sub-panel.
    */
    private void initMessagePanel() {
        TitledBorder border = new TitledBorder("Messages");
        border.setTitleJustification(TitledBorder.CENTER);
        msgPanel.setBorder(border);
        msgPanel.add(spMessage);
        spMessage.setMinimumSize(spMessage.getPreferredSize());
        taMessage.setEditable(false);
    }


   /**
    *  Initializes the main sub-panel.
    */
    private void initMainPanel() {
        mainPanel.setBorder(new TitledBorder(""));
        mainPanel.setLayout(new GridBagLayout());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.insets = new Insets(10, 10, 5, 10);

        gbc.gridx = 0;
        gbc.gridy = 0;
        mainPanel.add(raftsStPanel, gbc);

        gbc.insets.top = 0;
        gbc.insets.bottom = 10;
        gbc.gridy++;
        mainPanel.add(rebStPanel, gbc);

        gbc.insets.top = 10;
        gbc.gridy++;
        mainPanel.add(subMainPanel, gbc);

        gbc.gridy++;
        mainPanel.add(msgPanel, gbc);
    }


   /**
    *  Initializes the top-level panel.
    */
    private void initTopPanel() {
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTH;
        add(mainPanel, gbc);
    }


   /**
    *  Configures a REB.
    */
    private void cfgREB(ActionEvent e) {
        for (int id = 0; id < rebNames.size(); id++) {
            if (!e.getSource().equals(btnCfgREB[id])) continue;
            String name = rebNames.get(id);
            REBConfigPanel.edit(SwingUtilities.getWindowAncestor(this),
                                name, this);
            return;
        }
    }


   /**
    *  Forms a result string from a list of mismatch masks.
    */
    private String getResult(List<Integer> masks) {
        if (masks == null) return "check not done";
        int sum = 0;
        for (int mask : masks) {
            sum |= mask;
        }
        if (sum == 0) return "no errors";
        StringBuilder result = new StringBuilder("mismatches =");
        for (int mask : masks) {
            result.append(String.format(" %08x", mask));
        }
        return result.toString();
    }


   /**
    *  Loads the sequencers.
    */
    private void seqLoad() {
        JFileChooser fc = new JFileChooser("./");
        fc.setFileFilter(new FileNameExtensionFilter(null, "xml", "seq", "sif"));
        if (fc.showOpenDialog(getTopFrame()) != JFileChooser.APPROVE_OPTION) return;
        String fn;
        try {
            fn = fc.getSelectedFile().getCanonicalPath();
        }
        catch (IOException e) {
            displayError(e.toString());
            return;
        }
        List<Integer> nSlice = (List)sendCommand(null, "loadSequencer", fn);
        if (nSlice != null) {
            StringBuilder msg = new StringBuilder("Slice counts = ");
            boolean first = true;
            for (int count : nSlice) {
                msg.append(first ? "" : ", ").append(count);
                first = false;
            }
            display(msg.toString());
        }
    }


   /**
    *  Sets the back bias checkbox.
    */
    private void setBackBiasCB() {
        List<Boolean> values = (List)sendCommand(null, "isBackBiasOn");
        boolean on = (values != null);
        if (on) {
            for (boolean val : values) {
                if (!val) {
                    on = false;
                }
            }
        }
        cbxBackBias.setSelected(on);
    }


   /**
    *  Sets the scan mode checkbox.
    */
    private void setScanModeCB() {
        List<Boolean> values = (List)sendCommand(null, "isScanEnabled");
        boolean scanMode = (values != null);
        if (scanMode) {
            for (boolean val : values) {
                if (!val) {
                    scanMode = false;
                }
            }
        }
        cbxScanMode.setSelected(scanMode);
    }


   /**
    *  Sets the image pattern checkbox.
    */
    private void setPatternCB() {
        List<Integer> values = (List)sendCommand(null, "getDataSource");
        boolean pattern = (values != null);
        if (pattern) {
            for (int val : values) {
                if ((val & 1) == 0) {
                    pattern = false;
                }
            }
        }
        cbxPattern.setSelected(pattern);
    }


   /**
    *  Sets the directory for saving images.
    */
    private void setImageDir() {
        JFileChooser fc = new JFileChooser(imageDir == null ? "./" : imageDir);
        fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        if (fc.showOpenDialog(getTopFrame()) == JFileChooser.APPROVE_OPTION) {
            imageDir = fc.getSelectedFile().getPath();
        }
    }


   /**
    *  Saves the current images.
    */
    private void saveImage(String cmnd) {
        if (imageDir != null) {
            List<String> names = (List)sendCommand(null, cmnd, imageDir);
            if (names != null) {
                display("The following files were created:");
                for (String name : names) {
                    display("  " + name);
                }
            }
        }
        else {
            displayError("Image directory hasn't been set");
        }
    }


   /**
    *  Updates the configuration sub-panel.
    */
    private void updateConfig() {
        if (rebNames.size() != btnCfgREB.length) {
            cfgPanel.removeAll();
            initConfigPanel();
        }
        else {
            for (int j = 0; j < btnCfgREB.length; j++) {
                btnCfgREB[j].setName(rebNames.get(j) + "...");
            }
        }
    }


   /**
    *  Updates the setup sub-panel.
    */
    private void updateSetup() {
        setBackBiasCB();
    }


   /**
    *  Sends a command to a subsystem.
    */
    private Object sendSubsysCommand(String child, String name, Object... params)
    {
        try {
            return gui.sendCommandRaw(child, name, params);
        }
        catch (Exception e) {
            LOG.error("Command error: ", e);
            String text = e.getMessage();
            displayError(text == null || text.isEmpty() ? e.toString() : text);
            return null;
        }
    }


   /**
    *  Gets the top-most container of this panel.
    */
    private Component getTopFrame() {
        Component c = this;
        while (!(c instanceof JFrame) && !(c instanceof JWindow)
                && !(c instanceof JDialog) && c.getParent() != null) {
            c = c.getParent();
        }
        return c;
    }


   /**
    *  Updates the message area.
    */
    private void updateMessage(String msg) {
        SwingUtilities.invokeLater(new UpdateMessage(msg));
    }


   /**
    *  Displays an error message in the message area.
    */
    private void displayError(String message) {
        //taMessage.setForeground(Color.RED);
        taMessage.append(message + LF);
    }


   /**
    *  Enables the rafts subsystem.
    */
    public void enableSystem() {
        active = true;
        SwingUtilities.invokeLater(new UpdateStatus());
        SwingUtilities.invokeLater( () -> enableButtons(true) );
    }


   /**
    *  Disables the rafts subsystem.
    */
    public void disableSystem() {
        active = false;
        SwingUtilities.invokeLater(new UpdateStatus());
        SwingUtilities.invokeLater( () -> enableButtons(false) );
    }


   /**
    *  Reports image received messages from the rafts subsystem.
    *
    *  @param  state  The image state data
    */
    public void reportImage(ImageState state) {
        String message = String.format("Received image: timestamp = %016x, length = %s",
                                       state.getTimestamp(), state.getLength());
        updateMessage(message);
    }


   /**
    *  Class for processing external status updates.
    */
    class UpdateStatus implements Runnable {

        @Override
        public void run() {
            if (active) {
                lblStValue.setText("RUNNING");
                lblStValue.setForeground(GREEN);
                rebNames = (List)sendCommand(null, "getREBDeviceNames");
                StringBuilder text = new StringBuilder();
                for (String name : rebNames) {
                    text.append(" ").append(name);
                }
                lblRebs.setText(text.toString());
                updateConfig();
                updateSetup();
                setPatternCB();
                setScanModeCB();
            }
            else {
                lblStValue.setText("STOPPED");
                lblStValue.setForeground(RED);
            }
        }
    }


   /**
    *  Class for processing externally-generated messages.
    */
    class UpdateMessage implements Runnable {

        String message;

        UpdateMessage(String message) {
            this.message = message;
        }

        @Override
        public void run() {
            display(message);
        }

    }
        
    private void enableButtons(boolean enable) {
        for ( JButton b : btnCfgREB ) {
            b.setEnabled(enable);
        }
        btnCfgSave.setEnabled(enable);
        btnLoadDacs.setEnabled(enable);
        btnLoadAspics.setEnabled(enable);
        cbxBackBias.setEnabled(enable);
        btnSeqLoad.setEnabled(enable);
        btnSeqStart.setEnabled(enable);
        btnSeqStop.setEnabled(enable);
        btnSeqStep.setEnabled(enable);
        cbxScanMode.setEnabled(enable);
        btnResetScan.setEnabled(enable);
        cbxPattern.setEnabled(enable);
        btnResetFE.setEnabled(enable);
        btnAcquire.setEnabled(enable);
        btnSaveRaw.setEnabled(enable);
        btnSaveFits.setEnabled(enable);
    }

}
