package org.lsst.ccs.subsystem.rafts.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.rafts.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.rafts.data.RaftState;

/**
 *  Implements the Rafts monitoring panel.
 *
 *  @author Owen Saxton
 */
public class RaftsMonitorPanel extends javax.swing.JPanel {

    static final Font f = new java.awt.Font("Tahoma", 1, 12);
    static final Color RED = new Color(150, 0, 0), GREEN = new Color(0, 150, 0);
    final private CommandSender gui;
    private final JPanel topPanel = new JPanel();
    private final JPanel statePanel = new JPanel();
    private final JLabel stateLabel = new JLabel("System State: ");
    private final JLabel stateStatus = new JLabel("....");
    private final JPanel periodPanel = new JPanel();
    private final JLabel periodLabel = new JLabel("Update Period: ");
    private final JTextField periodTextField = new JTextField("...");
    private final JLabel periodUnitsLabel = new JLabel("sec");
    private String mainName;
    private double mainPeriod = 5;

    public RaftsMonitorPanel(CommandSender gui) {
        this.gui = gui;
        initComponents();
        (new UpdateSystemStatus(new RaftState(0))).run();
        (new DisableSystem()).run();
    }

    public void setSubsystem(String mainName) {
        this.mainName = mainName;
    }

    private void initComponents()
    {
        //** Create all low-level components

        // State indicator
        Dimension d = stateStatus.getPreferredSize();
        d.width = 70;
        stateStatus.setPreferredSize(d);
        stateStatus.setMinimumSize(d);

        // Update period
        d = periodTextField.getPreferredSize();
        d.width = 50;
        periodTextField.setPreferredSize(d);
        periodTextField.setMinimumSize(d);
        periodTextField.setHorizontalAlignment(JTextField.CENTER);
        periodTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setUpdatePeriod();
            }
        });

        // Power state
        statePanel.add(stateLabel);
        statePanel.add(stateStatus);

        // Update period
        periodPanel.add(periodLabel);
        periodPanel.add(periodTextField);
        periodPanel.add(periodUnitsLabel);

        // Top panel
        GridBagConstraints gbt = new GridBagConstraints();
        topPanel.setLayout(new GridBagLayout());
        gbt.gridx = 0;
        gbt.gridy = 0;
        topPanel.add(statePanel, gbt);
        gbt.insets.left = 80;
        gbt.gridx++;
        topPanel.add(periodPanel, gbt);
        
        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(topPanel, gbc);
    }

    private void setUpdatePeriod() {
        try {
            double value = Double.valueOf(periodTextField.getText());
            gui.sendCommand(mainName, null, "setUpdatePeriod", (int)(1000 * value));
            periodTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            periodTextField.setText(String.valueOf(mainPeriod));
        }
    }

    public void updateSystemStatus(RaftState rs) {
        SwingUtilities.invokeLater(new UpdateSystemStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdateSystemStatus implements Runnable {

        private final RaftState rs;

        UpdateSystemStatus(RaftState rs) {
            this.rs = rs;
        }

        @Override
        public void run() {
            stateStatus.setText("RUNNING");
            stateStatus.setForeground(GREEN);
            periodTextField.setEnabled(true);
            mainPeriod = rs.getTickMillis() / 1000.0;
            periodTextField.setText(String.valueOf(mainPeriod));
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            stateStatus.setText("STOPPED");
            stateStatus.setForeground(RED);
            periodTextField.setEnabled(false);
        } 
    }

    private static final long serialVersionUID = 1L;
}
