package org.lsst.ccs.subsystem.rafts.ui;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import javax.swing.BorderFactory;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import org.lsst.ccs.subsystem.rafts.config.ASPIC;

/**
 ********************************
 *
 *  ASPIC configuration editor.
 *
 *  @author  Owen Saxton
 *
 ********************************
 */
class ASPICConfigPanel extends JPanel {

    private static final Insets INSETS_STD = new Insets(4, 4, 4, 4);
    private static final int TF_WIDTH = 50, TF_WIDTH_L = 80;
    final int iAspic;
    final ASPIC aspic;

    private final JLabel lbGain = new JLabel("Gain:");
    private final JLabel lbRC = new JLabel("RC:");
    private final JLabel lbClamp = new JLabel("Clamp:");
    private final BinaryTextField tfGain;
    private final BinaryTextField tfRC;
    private final BinaryTextField tfClamp;
    private final JCheckBox cbAf1 = new JCheckBox("FS Gain 1");
    private final JCheckBox cbTm = new JCheckBox("Trans Mode");


   /**
    *  Inner class for implementing a binary text field.
    */
    static class BinaryTextField extends JTextField {

        int value, width;

        BinaryTextField() {
            this(0);
        }

        BinaryTextField(int width) {
            this.width = width;

            addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent ev) {
                    valueChanged();
                }
            });

            addFocusListener(new FocusListener() {
                @Override
                public void focusLost(FocusEvent ev) {
                    valueChanged();
                }
                @Override
                public void focusGained(FocusEvent ev) {
                }
            });
        }

        void setValue(int value) {
            this.value = value;
            showValue();
        }

        int getValue() {
            return value;
        }

        private void valueChanged() {
            String text = getText();
            try {
                value = Integer.parseInt(text, 2);
            }
            catch (NumberFormatException ex) {
            }
            showValue();
        }

        private void showValue() {
            String text = Integer.toBinaryString(value);
            int leng = text.length();
            if (leng < width) {
                StringBuilder txt = new StringBuilder();
                for (int j = 0; j < width - leng; j++) {
                    txt.append('0');
                }
                txt.append(text);
                text = txt.toString();
            }
            setText(text);
        }

        private static final long serialVersionUID = 1L;
    }


   /**
    *  Constructor.
    */
    ASPICConfigPanel(int ia, ASPIC a) {

        iAspic = ia;
        aspic = a;

        tfGain = new BinaryTextField(4);
        Dimension d4 = tfGain.getPreferredSize();
        d4.width = TF_WIDTH;
        tfGain.setPreferredSize(d4);
        tfGain.setMinimumSize(d4);
        tfRC = new BinaryTextField(4);
        tfRC.setPreferredSize(d4);
        tfRC.setMinimumSize(d4);
        tfClamp = new BinaryTextField(8);
        Dimension d8 = tfClamp.getPreferredSize();
        d8.width = TF_WIDTH_L;
        tfClamp.setPreferredSize(d8);
        tfClamp.setMinimumSize(d8);

        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = INSETS_STD;

        String title = "ASPIC " + iAspic
                         + (iAspic % 2 == 0 ? " (upper)" : " (lower)");
        setBorder(BorderFactory.createTitledBorder(title));

        gbc.anchor = GridBagConstraints.NORTHWEST;

        gbc.gridy = 0;
        gbc.gridx = 0;
        add(lbGain, gbc);
        gbc.gridx++;
        add(tfGain, gbc);

        gbc.gridx++;
        add(lbRC, gbc);
        gbc.gridx++;
        add(tfRC, gbc);

        gbc.gridy++;
        gbc.gridx = 0;
        add(lbClamp, gbc);
        gbc.gridx++;
        gbc.gridwidth = 3;
        add(tfClamp, gbc);

        gbc.gridy++;
        gbc.gridx = 0;
        gbc.gridwidth = 2;
        add(cbAf1, gbc);
        cbAf1.setFocusable(false);

        gbc.gridx += 2;
        add(cbTm, gbc);
        cbTm.setFocusable(false);
    }


   /**
    *  Saves new values to the ASPIC data.
    */
    void save() {

        aspic.setGain(tfGain.getValue());
        aspic.setRc(tfRC.getValue());
        aspic.setClamp(tfClamp.getValue());
        aspic.setAf1(cbAf1.isSelected());
        aspic.setTm(cbTm.isSelected());
    }


   /**
    *  Reads values from the ASPIC data and populates fields.
    */
    void read() {

        tfGain.setValue(aspic.getGain());
        tfRC.setValue(aspic.getRc());
        tfClamp.setValue(aspic.getClamp());
        cbAf1.setSelected(aspic.isAf1());
        cbTm.setSelected(aspic.isTm());
    }

    private static final long serialVersionUID = 1L;
}
