package org.lsst.ccs.subsystem.rafts.fpga.xml;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * This parses a assembler-like text format sequencer program,
 * and converts it to an object tree mapping the XML format,
 * that can be fed to the compiler.
 * 
 */
public class SequencerTextParser {


    public static Sequencer parse(File f) throws IOException {
        try (FileReader fr = new FileReader(f)) {
            return parse(fr);
        }
    }

    // we could use ANTLR. But this is simple...

    static int lnum = 0;

    public static Sequencer parse(Reader r) throws IOException {
        Sequencer s = new Sequencer();
        FileSection sec = FileSection.BEFORE;
        BufferedReader rdr = new BufferedReader(r);
        lnum = 0;

        while (true) {
            String l = rdr.readLine();
            if (l == null) break;
            l = l.trim();
            lnum++;
            if (l.isEmpty() || l.substring(0, 1).equals("#")) continue;
            sec = sec.parseLine(s, l);
        }


        s.complete();

        //new DumpVisitor().visit(s);

        return s;
    }

    static enum FileSection {

        BEFORE {
            @Override
            FileSection parseLine(Sequencer s, String line) {
                if (line.startsWith("[constants]")) {
                    return CONSTANTS;
                }
                syntaxError(line);
                return this;
            }
        },

        CONSTANTS {
            @Override
            FileSection parseLine(Sequencer s, String line) {
                if (line.startsWith("[clocks]")) {
                    return CLOCKS;
                }
                // ReadLines: 2020 # Number of rows of the sensor
                Matcher m = definePattern.matcher(line);
                if (!m.matches()) {
                    syntaxError(line);
                }
                String key = m.group(1);
                String value = m.group(2).trim();
                String comm = trim(m.group(4));
                s.getSequencerConfig().getParameters().add(new Parameter(key, comm, value));
                return this;
            }
        },

        CLOCKS {
            @Override
            FileSection parseLine(Sequencer s, String line) {
                if (line.startsWith("[pointers]")) {
                    return POINTERS;
                }
                // P0: 8 # Parallel clock 0
                Matcher m = defineNumPattern.matcher(line);
                if (!m.matches()) {
                    syntaxError(line);
                }
                String key = m.group(1);
                int value = Integer.parseInt(m.group(2));
                String comm = trim(m.group(4));
                s.getSequencerConfig().getChannels().add(new Channel(key, comm, value));
                return this;
            }
        },

        POINTERS {
            @Override
            FileSection parseLine(Sequencer s, String line) {
                if (line.startsWith("[functions]")) {
                    return FUNCTIONS;
                }
                // PTR_FUNC AFuncPtr TransferLine
                // REP_SUBR PReadLines 2020 # blabl
                Matcher m = ptrPattern.matcher(line);
                if (!m.matches()) {
                    syntaxError(line);
                }
                switch (m.group(1)) {
                case "PTR_FUNC":
                    s.getSequencerConfig().getFuncPtrs()
                        .add(new FunctionPointer(m.group(2), trim(m.group(5)), m.group(3)));
                    break;
                case "PTR_SUBR":
                    s.getSequencerConfig().getSubPtrs()
                        .add(new SubroutinePointer(m.group(2), trim(m.group(5)), m.group(3)));
                    break;
                case "REP_FUNC":
                    int count = 0;
                    if (m.group(3).equals("infinity")) {
                        count = 0x800000;
                    }
                    else if (m.group(3).matches("[0-9]+")) {
                        count = Integer.valueOf(m.group(3));
                    }
                    else {
                        syntaxError(line);
                    }
                    s.getSequencerConfig().getFuncRepPtrs()
                        .add(new RepeatFunctionPointer(m.group(2), trim(m.group(5)), count));
                    break;
                case "REP_SUBR":
                    count = 0;
                    if (m.group(3).matches("[0-9]+")) {
                        count = Integer.valueOf(m.group(3));
                    }
                    else {
                        syntaxError(line);
                    }
                    s.getSequencerConfig().getSubRepPtrs()
                        .add(new RepeatSubroutinePointer(m.group(2), trim(m.group(5)), count));
                    break;
                default:
                    syntaxError(line);
                }

                return this;
            }
        },

        FUNCTIONS {
            @Override
            FileSection parseLine(Sequencer s, String line) {
                if (line.startsWith("[subroutines]")) {
                    inSlices = false;
                    return SUBROUTINES;
                }
                if (line.startsWith("[mains]")) {
                    inSlices = false;
                    return MAINS;
                }

                // ReadPixel: # Single pixel read
                // clocks: RG, S0, S1, S2, CL, RST, RD, RU, TRG
                // slices:
                // 80 ns = 0, 0, 0, 1, 0, 0, 0, 0, 0
                // RampTime = 0, 0, 0, 1, 0, 0, 1, 0, 0
                // constants: P1=1, P2=1

                Matcher m = funcPattern.matcher(line);
                if (!m.matches()) {
                    syntaxError(line);
                }


                // move to after inSlices is updated?
                // if (inSlices != (m.group(2).equals("="))) {
                // throw new RuntimeException(
                // "sequencer file syntax error line " + lnum + ": "
                // + line);
                // }

                if (m.group(1).equals("slices")) {
                    if (!m.group(2).equals(":") || !m.group(3).isEmpty()) {
                        syntaxError(line);
                    }
                    inSlices = true;
                } else if (m.group(1).equals("clocks")) {
                    if (!m.group(2).equals(":")) {
                        syntaxError(line);
                    }
                    inSlices = false;
                    if (!m.group(3).isEmpty()) {
                        String[] clocks = m.group(3).split("\\s*,\\s*");
                        for (String c : clocks) {
                            curFunc.getClocks().add(new Clock(findChannel(s, c)));
                        }
                    }
                } else if (m.group(1).equals("constants")) {
                    if (!m.group(2).equals(":")) {
                        syntaxError(line);
                    }
                    inSlices = false;
                    if (!m.group(3).isEmpty()) {
                        String[] cst = m.group(3).split("\\s*,\\s*");
                        for (String c : cst) {
                            String[] cc = c.trim().split("=");
                            curFunc.getConstants().add(new Constant(findChannel(s, cc[0]), cc[1]));
                        }
                    }
                } else if (inSlices && m.group(2).equals("=")) {
                    curFunc.getTimeslices().add(
                            new Timeslice(m.group(1).trim(), m.group(3).trim()
                                    .replaceAll("\\s*,\\s*", "")));
                } else {
                    if (!m.group(2).equals(":") || !m.group(3).isEmpty()) {
                        syntaxError(line);
                    }
                    inSlices = false;
                    curFunc = new Function(m.group(1).trim(), trim(m.group(5)));
                    s.getSequencerConfig().getFunctions().add(curFunc);
                }

                return this;
            }

            boolean inSlices = false;
            Function curFunc = null;
        },

        SUBROUTINES {
            @Override
            FileSection parseLine(Sequencer s, String line) {
                if (line.startsWith("[mains]")) {
                    return MAINS;
                }
                Matcher m = subLabelPattern.matcher(line);
                if (m.matches()) {
                    curSub = new Subroutine(m.group(1), trim(m.group(3)));
                    s.getSequencerRoutines().getSubroutines().add(curSub);
                    return this;
                }
                if (parseSubroutineLine(s, curSub, line)) {
                    curSub = null;
                }
                return this;
            }

            Subroutine curSub = null;
        },

        MAINS {
            @Override
            FileSection parseLine(Sequencer s, String line) {
                if (line.startsWith("[subroutines]")) {
                    return SUBROUTINES;
                }
                Matcher m = subLabelPattern.matcher(line);
                if (m.matches()) {
                    curSub = new Main(m.group(1), trim(m.group(3)));
                    s.getSequencerRoutines().getMains().add(curSub);
                    return this;
                }
                if (parseSubroutineLine(s, curSub, line)) {
                    curSub = null;
                }
                return this;
            }

            Main curSub = null;
        };

        Channel findChannel(Sequencer s, String name) {
            return s.getSequencerConfig().getChannels().stream()
                    .filter(x -> x.getId().equals(name)).findFirst()
                    .orElseThrow(() -> new RuntimeException("channel " + name + " not found line " + lnum));
        }

        boolean parseSubroutineLine(Sequencer s, Subroutine curSub, String line) {
            if (curSub == null) {
                syntaxError(line);
                // will not return
                return false;
            }
            Matcher m = subRTSPattern.matcher(line);
            if (m.matches()) {
                // nothing to do, end of routine.
                return true;
            }
            m = subENDPattern.matcher(line);
            if (m.matches()) {
                // nothing to do, end of routine.
                return true;
            }
            m = subCALLPattern.matcher(line);
            if (m.matches()) {
                Call c = new Call();
                c.setFcnName(m.group(1));
                c.setRepeat(m.group(3));
                curSub.getCalls().add(c);
                return false;
            }
            m = subJSRPattern.matcher(line);
            if (m.matches()) {
                Call c = new Call();
                c.setSubName(m.group(1));
                c.setRepeat(m.group(3));
                curSub.getCalls().add(c);
                return false;
            }
            syntaxError(line);
            return false;
        }

        static void syntaxError(String line) {
            throw new RuntimeException("sequencer file syntax error line "
                                         + lnum + ": " + line);
        }

        static String trim(String str) {
            return str == null ? null : str.trim();
        }

        abstract FileSection parseLine(Sequencer s, String line);

        Pattern definePattern = Pattern
                .compile("(\\w+)\\s*:\\s*([\\w ]+)\\s*(#\\s*(.*))?");
        Pattern defineNumPattern = Pattern
                .compile("(\\w+)\\s*:\\s*(\\d+)\\s*(#\\s*(.*))?");
        Pattern ptrPattern = Pattern
                .compile("(\\w+)\\s+(\\w+)\\s+(\\w+)\\s*(#\\s*(.*))?");
        Pattern funcPattern = Pattern
                .compile("([\\w ]+)\\s*([:=])\\s*([\\w ,=]*)\\s*(#\\s*(.*))?");

        Pattern subLabelPattern = Pattern
                .compile("(\\w+)\\s*:\\s*(#\\s*(.*))?");
        Pattern subRTSPattern = Pattern.compile("RTS\\s*(#\\s*(.*))?");
        Pattern subENDPattern = Pattern.compile("END\\s*(#\\s*(.*))?");
        Pattern subCALLPattern = Pattern
                .compile("CALL\\s+(@?\\w+)\\s*(repeat\\((@?\\w+)\\))?\\s*(#\\s*(.*))?");
        Pattern subJSRPattern = Pattern
                .compile("JSR\\s+(@?\\w+)\\s*(repeat\\((@?\\w+)\\))?\\s*(#\\s*(.*))?");

        // CONSTANTS {}, CLOCKS, POINTERS, FUNCTIONS, PROGRAM;

    }

}
