package org.lsst.ccs.subsystem.rafts.fpga.main;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo.PointerKind;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelUtils;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelUtils.LongWithInfinity;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelUtils.PixelCount;

/**
 * A main routine to dump information about a sequencer
 *
 * @author tonyj
 */
public class SequencerInfo {

    private static final Pattern PARAMETER_PATTERN = Pattern.compile("(\\w+)\\s*=\\s*(.*)");

    @Argument(required = true, usage = "Input (.seq) File")
    private File file;

    @Option(name = "-P", usage = "Set a parameter, use -P parameter=value [-P anotherParameter=anotherValue]")
    @SuppressWarnings("FieldMayBeFinal")
    private List<String> parameters = new ArrayList<>();

    public static void main(String[] args) throws FileNotFoundException, IOException {
        new SequencerInfo().doMain(args);
    }

    private void doMain(String[] args) throws FileNotFoundException, IOException {
        CmdLineParser parser = new CmdLineParser(this);
        try {
            // parse the arguments.
            parser.parseArgument(args);
            compile();

        } catch (CmdLineException | IllegalArgumentException e) {

            System.err.println(e.getMessage());
            System.err.println("java Compiler [options...] arguments...");
            // print the list of available options
            parser.printUsage(System.err);
            System.err.println();
        }
    }

    private void compile() throws FileNotFoundException, IOException, CmdLineException {

        FPGA2Model compiled;
        try (InputStream input = new FileInputStream(file)) {
            FPGA2ModelBuilder builder = new FPGA2ModelBuilder();
            compiled = builder.compileFile(input);
        }
        System.out.printf("Compilation of %s at %s\n", file, new Date());
        System.out.printf("Checksum: %08x\n", compiled.computeCheckSum());

        Map<String, Integer> addressMap = new HashMap<>();

        for (Map.Entry<String, Integer> entry : compiled.getFunctionAddresses().entrySet()) {
            addressMap.put(entry.getKey(), entry.getValue());
        }
        for (Map.Entry<String, Integer> entry : compiled.getSubroutineAddresses().entrySet()) {
            addressMap.put(entry.getKey(), entry.getValue());
        }
        Map<Integer, String> inverseAddressMap = addressMap.entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));

        // Parse the parameters
        Map<String, Object> parameterMap = new HashMap<>();
        for (String p : parameters) {
            Matcher matcher = PARAMETER_PATTERN.matcher(p);
            if (!matcher.matches()) {
                throw new IllegalArgumentException("Illegal parameter: " + p);
            } else if (!compiled.getPointerMap().containsKey(matcher.group(1))) {
                throw new IllegalArgumentException("Unknown parameter: " + matcher.group(1));
            } else {
                final String key = matcher.group(1);
                final String value = matcher.group(2);
                PointerKind kind = compiled.getPointerMap().get(key).getKind();
                if (kind == PointerKind.FUNCTION || kind == PointerKind.SUBROUTINE) {
                    final Integer address = addressMap.get(value);
                    if (address == null) {
                        throw new IllegalArgumentException("Invalid parameter value: " + p);
                    }
                    parameterMap.put(key, value);
                } else {
                    Integer result = Integer.parseInt(value);
                    parameterMap.put(key, result);
                }
            }
        }

        System.out.printf("\nParameters\n    %-20s %8s %-20s %-20s %-20s\n", "Name", "Address", "Type", "Value", "Default Value");
        compiled.getPointers().stream().forEach((i) -> {
            PointerKind kind = i.getKind();
            String value;
            if (kind == PointerKind.FUNCTION || kind == PointerKind.SUBROUTINE) {
                value = inverseAddressMap.get(i.getValue());
            } else {
                value = String.format("%d", i.getValue());
            }
            if (parameterMap.containsKey(i.getName())) {
                System.out.printf("    %-20s %8x %-20s %-20s %-20s\n", i.getName(), i.getOffset(), i.getKind(), parameterMap.get(i.getName()), value);
            } else {
                System.out.printf("    %-20s %8x %-20s %-20s\n", i.getName(), i.getOffset(), i.getKind(), value);
            }
        });

        FPGA2ModelUtils utils = new FPGA2ModelUtils(compiled, parameterMap);

        System.out.printf("\nMains\n    %-20s %-8s %-20s %-20s\n", "Name", "Address", "Time (ns)", "Pixels");
        compiled.getMainAddresses().entrySet().stream().forEach((i) -> {
            LongWithInfinity nanos = utils.getNanosForRoutine(i.getKey());
            LongWithInfinity pixels = utils.getPixelsForRoutine(i.getKey());
            System.out.printf("    %-20s %08x %-20s %-20s\n", i.getKey(), i.getValue(), nanos, pixels.isZero() ? "" : pixels);
        });

        System.out.printf("\nSubroutines\n    %-20s %-8s %-20s %-20s\n", "Name", "Address", "Time (ns)", "Pixels");
        compiled.getSubroutineAddresses().entrySet().stream().forEach((f) -> {
            LongWithInfinity nanos = utils.getNanosForRoutine(f.getKey());
            LongWithInfinity pixels = utils.getPixelsForRoutine(f.getKey());
            System.out.printf("    %-20s %08x %-20s %-20s\n", f.getKey(), f.getValue(), nanos, pixels.isZero() ? "" : pixels);
        });

        System.out.printf("\nFunctions\n    %-20s %-8s %-20s %-20s\n", "Name", "Address", "Time (ns)", "Pixels");
        compiled.getFunctionAddresses().entrySet().stream().forEach((f) -> {
            long nanos = utils.getNanosForFunction(f.getKey());
            PixelCount pixels = utils.getPixelsForFunction(f.getKey());
            System.out.printf("    %-20s %08x %-,20d %-20s\n", f.getKey(), f.getValue(), nanos, pixels);
        });
    }
}
