package org.lsst.ccs.subsystem.rafts.fpga.xml;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;
import javax.xml.bind.util.JAXBSource;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.Visitor;

@XmlRootElement(name = "sequencer")
@XmlType(propOrder = { "sequencerConfig", "sequencerRoutines" })
public class Sequencer implements Visitable {

	@XmlElement(name = "sequencer-config")
	public SequencerConfig getSequencerConfig() {
		return sequencerConfig;
	}

	public void setSequencerConfig(SequencerConfig sequencerConfig) {
		this.sequencerConfig = sequencerConfig;
	}

	@XmlElement(name = "sequencer-routines")
	public SequencerRoutines getSequencerRoutines() {
		return sequencerRoutines;
	}

	public void setSequencerRoutines(SequencerRoutines sequencerRoutines) {
		this.sequencerRoutines = sequencerRoutines;
	}

	private SequencerConfig sequencerConfig = new SequencerConfig();
	private SequencerRoutines sequencerRoutines = new SequencerRoutines();

	public void complete() {

		for (FunctionPointer p : sequencerConfig.getFuncPtrs()) {
			if (p.getFunction() == null) {
				String n = p.getFcnName();
				if (n == null) {
					throw new RuntimeException("no name for function pointer " + p.getId());
				}
				Function f = sequencerConfig.getFunctions().stream().filter(m -> m.getId().equals(n)).findFirst()
						.orElseThrow(
								() -> new RuntimeException("cannot find function " + n + " for pointer " + p.getId()));
				p.setFunction(f);
			}
		}
		for (SubroutinePointer p : sequencerConfig.getSubPtrs()) {
			if (p.getSubroutine() == null) {
				String n = p.getSubName();
				if (n == null) {
					throw new RuntimeException("no name for function pointer " + p.getId());
				}
				Subroutine s = sequencerRoutines.getSubroutines().stream().filter(m -> m.getId().equals(n)).findFirst()
						.orElseThrow(() -> new RuntimeException(
								"cannot find subroutine " + n + " for pointer " + p.getId()));
				p.setSubroutine(s);
			}
		}
		for (Subroutine s : sequencerRoutines.getSubroutines()) {
			s.complete(this);
		}
		for (Main m : sequencerRoutines.getMains()) {
			m.complete(this);
		}
	}

	public void validate() {
		sequencerConfig.validate();
		sequencerRoutines.validate();
	}

	@Override
	public void accept(Visitor v) {
		v.visit(this);
	}

	public void saveAsXML(File xmlFile) throws IOException, JAXBException {
		JAXBContext context = JAXBContext.newInstance(Sequencer.class);
		Marshaller m = context.createMarshaller();
		m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
		try (FileOutputStream out = new FileOutputStream(xmlFile)) {
			m.marshal(this, out);
		}
	}

	public void saveAsText(File textFile) throws IOException, JAXBException, TransformerException {
		JAXBContext context = JAXBContext.newInstance(Sequencer.class);
		try (FileOutputStream out = new FileOutputStream(textFile)) {
			JAXBSource source = new JAXBSource(context, this);

			// set up XSLT transformation
			TransformerFactory tf = TransformerFactory.newInstance("net.sf.saxon.TransformerFactoryImpl",
					this.getClass().getClassLoader());
			try (InputStream s = this.getClass().getResourceAsStream("seq.xsl")) {
				Transformer t = tf.newTransformer(new StreamSource(s));

				// run transformation
				t.transform(source, new StreamResult(out));
			}
		}
	}

}
