package org.lsst.ccs.subsystem.rafts.fpga.compiler;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.lsst.ccs.subsystem.rafts.fpga.xml.Call;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Callable;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Channel;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Clock;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Constant;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Function;
import org.lsst.ccs.subsystem.rafts.fpga.xml.FunctionPointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Main;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Parameter;
import org.lsst.ccs.subsystem.rafts.fpga.xml.RepeatFunctionPointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.RepeatSubroutinePointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Sequencer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Subroutine;
import org.lsst.ccs.subsystem.rafts.fpga.xml.SubroutinePointer;
import org.lsst.ccs.subsystem.rafts.fpga.xml.Timeslice;

// Note : non thread-safe. Use an instance per visit.
/**
 * This visitor builds an intermediate model. Mostly gathers all parameters,
 * channels, and functions and subroutines into maps for future use.
 *
 * @author aubourg
 *
 *
 */
class ModelBuilderVisitor extends AbstractVisitor {

    private final Map<String, String> parms = new HashMap<>();
    private final Map<String, Channel> channels = new HashMap<>();
    private final Map<String, Function> functions = new HashMap<>();
    private final Map<String, Subroutine> subs = new HashMap<>();
    private final Map<String, Main> mains = new HashMap<>();
    private final Set<String> used = new HashSet<>();
    Callable current = null;
    private int nanos;
    int sliceIndex;

    @Override
    public void visit(Sequencer s) {
        super.visit(s);
    }

    @Override
    public void visit(Parameter s) {
        parms.put(s.getId(), s.getValue());
    }

    @Override
    public void visit(Channel c) {
        channels.put(c.getId(), c);
    }

    @Override
    public void visit(Function f) {
        current = f;
        used.clear();
        nanos = 0;
        sliceIndex = 0;
        super.visit(f);
        if (channels.size() != used.size()) {
            // default is to be constant with value 0
            for (Channel c : channels.values()) {
                if (!used.contains(c.getId())) {
                    Constant cc = new Constant(c, "0");
                    f.getConstants().add(cc);
                }
            }
        }
        // Special handling for the first/last slice.
        // The first slice will be 10ns longer than the value written to the register
        // The last slice will be 20ns .longer than the value written to the register
        // In addition all functions except the default function must have at least two slices
        // The duration of the firstslice was corrected in visit(Timeslice) but the last slice
        // must be dealt with here.
        if (!"Default".equals(f.getId())) {
            List<Timeslice> slices = f.getTimeslices();
            if (slices.size() < 2) {
                throw new RuntimeException("All functions except the default function must have at least two time slices");
            }

            Timeslice firstSlice = slices.get(0);
            if (firstSlice.getDurationNanos() < 10) {
                throw new RuntimeException("First slice in function must be >= 10nS");
            }
            firstSlice.setDurationNanos(firstSlice.getDurationNanos() - 10);

            Timeslice lastSlice = slices.get(slices.size() - 1);
            if (lastSlice.getDurationNanos() < 20) {
                throw new RuntimeException("Last timeslice of a function must be >= 20nS");
            }
            lastSlice.setDurationNanos(lastSlice.getDurationNanos() - 20);
        }
        functions.put(f.getId(), f);
        current = null;
    }

    /* inside function visit */
    @Override
    public void visit(Clock c) {
        if (c.getChannel() == null) {
            throw new RuntimeException("Unknown clock reference in function "
                    + current.getId());
        }
        String chanId = c.getChannel().getId();
        if (used.contains(chanId)) {
            throw new RuntimeException("Channel " + chanId
                    + " reused for clock in function " + current.getId());
        }
        used.add(chanId);
    }

    @Override
    public void visit(Constant c) {
        if (c.getChannel() == null) {
            throw new RuntimeException(
                    "Unknown constant reference in function " + current.getId());
        }
        String chanId = c.getChannel().getId();
        if (used.contains(chanId)) {
            throw new RuntimeException("Channel " + chanId
                    + " reused for constant in function " + current.getId());
        }
        String value = c.getValue();
        if (!value.equals("0") && !value.equals("1")) {
            throw new RuntimeException("Constant " + chanId + " value ("
                    + value + ") is not 0 or 1 in function " + current.getId());
        }
        used.add(chanId);
    }

    private static int parseNanos(String s) {
        Pattern p = Pattern.compile("([0-9\\.]+) *([a-z]+)");
        Matcher m = p.matcher(s);
        if (m.matches()) {
            float value = Float.valueOf(m.group(1));
            String unit = m.group(2);
            switch (unit) {
                case "ns":
                    return ((int) value);
                case "us":
                case "µs":
                    return ((int) (value * 1000));
                default:
                    throw new RuntimeException("Unknown unit " + unit);
            }
        } else {
            return -1;
        }
    }

    @Override
    public void visit(Timeslice s) {
        if (sliceIndex > 15) {
            throw new RuntimeException("Too many time slices, max=16");
        }

        String d = s.getDuration();
        int n = parseNanos(d);
        if (n > 0) {
            s.setDurationNanos(n);
        } else {
            String x = parms.get(d);
            if (x != null) {
                n = parseNanos(x);
                if (n >= 0) {
                    s.setDurationNanos(n);
                }
            } else {
                throw new RuntimeException("Cannot interpret duration " + d);
            }
        }
        s.setStartNanos(nanos);
        s.setIndex(sliceIndex);
        nanos += s.getDurationNanos();
        sliceIndex++;

        // which channels are up?
        String pat = s.getValue();
        Function f = (Function) current;
        if (pat.length() != f.getClocks().size()) {
            throw new RuntimeException("Bad length for pattern " + pat
                    + " in function " + f.getId());
        }
        for (int i = 0; i < pat.length(); i++) {
            String digit = pat.substring(i, i + 1);
            if (!digit.equals("0") && !digit.equals("1")) {
                throw new RuntimeException("Pattern digit must be 0 or 1"
                        + " in function " + f.getId());
            }
            if (digit.equals("1")) {
                s.addUpChannel(f.getClocks().get(i).getChannel());
            }
        }
    }

    /* end inside function visit */
    @Override
    public void visit(Subroutine s) {
        current = s;
        super.visit(s);
        subs.put(s.getId(), s);
        current = null;
    }

    @Override
    public void visit(Main m) {
        current = m;
        super.visit(m);
        mains.put(m.getId(), m);
        current = null;
    }

    /* inside subroutine visit */
    @Override
    public void visit(Call c) {
        super.visit(c);
        String rep = c.getRepeat();

        if (rep == null && c.getRepeatFcnPtr() == null
                && c.getRepeatSubPtr() == null) {
            c.setInfinity(false);
            c.setRepeatValue(1);
        } else if (rep != null && rep.equals("infinity")) {
            c.setInfinity(true);
            // Per Stefano, repeat count must be > 0 or subroutine will not run
            c.setRepeatValue(0x800001);
        } else if (rep != null && rep.matches("[0-9]+")) {
            int n = Integer.parseInt(rep);
            c.setInfinity(false);
            c.setRepeatValue(n);
        } else if (rep != null) {
            String p = parms.get(rep);
            if (p == null) {
                throw new RuntimeException("Undefined property " + rep);
            }
            if (!p.matches("[0-9]+")) {
                throw new RuntimeException("Property " + rep
                        + " is not a number: " + p);
            }
            int n = Integer.parseInt(p);
            c.setInfinity(false);
            c.setRepeatValue(n);
        }
    }

    /* end inside subroutine visit */
    @Override
    public void visit(FunctionPointer fp) {
        // TODO Auto-generated method stub

    }

    @Override
    public void visit(RepeatFunctionPointer rfp) {
        // TODO Auto-generated method stub

    }

    @Override
    public void visit(SubroutinePointer sp) {
        // TODO Auto-generated method stub

    }

    @Override
    public void visit(RepeatSubroutinePointer rsp) {
        // TODO Auto-generated method stub

    }

}
