package org.lsst.ccs.subsystems.powermanage;

import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.drivers.pmcard.PMCardException;
import org.lsst.ccs.drivers.pmcard.TIUCD90xxx;

/**
 ***************************************************************************
 **
 **  Implements the powermanage test modular subsystem
 **
 **  @author Xiaowen Lei
 **
 ***************************************************************************
 */
public class PMTestHdw extends PMTest {

   /**
    ***************************************************************************
    **
    **  Container class for channel hardware data.
    **
    ***************************************************************************
    */
    private static class ChanHdw {

        // xiaowen: I guess this can be used to refer to different rails
        // int  chan;       // Hardware channel for MCC channel
        // xiaowen: only one rail is used for now. need to decide later how to do this
        int  rail;       // Rail

        public ChanHdw(int rail)
        {
            this.rail = rail;
        }
    }

   /**
    ***************************************************************************
    **
    **  Inner class to implement timer task to update the channel state.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            if (!running) return;   // Do nothing if not yet running

            /*
            **  Read channels and set chanState by hand
            */
            chanState = 0;
            for (int j = 0; j < nChan; j++) {
                double value = 0.0;
                Channel ch = chanData[j];
                ChanHdw hw = hdwData[j];
                try {
                    value=ucd90xxx.getParmValue(hw.rail,ch.type);
                }
                catch (PMCardException e) {
                    if (ucd90xxxInited)
                        log.error("Error getting value for channel " + String.valueOf(j), e);
                }

                boolean ok = (value>=ch.uFaultLimit) && (value<=ch.oFaultLimit);
                if (ok) chanState |= (1 << j);
            }
                    
            // xiaowen: does not make use of the various status information yet
            //          command: 0x78, 0x79 (STATUS_WORD), 0x7A (STATUS_VOUT), 0x7B (STATUS_IOUT), 0x7D (STATUS_TEMPERATURE
            //                   0x7E, 0x81, 0x82
            //          need to decide:
            //            1. which ones to send to buses
            //            2. how to organize these information.
            //               probably want to decide first whether to keep separate "channels" for VOLT/CURR/TEMP
            //            3. when to reset etc.

            /*
            **  Report state change
            */
            if (chanState != lastChanState) {
                lastChanState = chanState;
                tick();
                publishState();
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int
        DEV_PORT = 0,
        DEV_ADDR = 101,
        N_DEV_RAILS = 16,
        UPDATE_PERIOD = 1000;

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private TIUCD90xxx ucd90xxx;
    private int chanState, lastChanState;
    private double power;
    private boolean ucd90xxxOpened, ucd90xxxInited;
    private ChanHdw[] hdwData;


   /**
    ***************************************************************************
    **
    **  Performs additional structural configuration.
    **
    ***************************************************************************
    */
    @Override
    void addStructure(Properties struct)
    {
        String[] names = {"rails"};
        hdwData = new ChanHdw[nChan];
        String[][] pValues = new String[names.length][];
        for (int j = 0; j < names.length; j++) {
            String pValue = struct.getProperty(PREFIX + names[j]);
            if (pValue != null) {
                pValues[j] = pValue.split(",");
            }
            else {
                log.error("Hardware structure is missing " + names[j]
                          + " property");
                pValues[j] = new String[nChan];
                for (int k = 0; k < nChan; k++)
                    pValues[j][k] = "-1";
            }
        }

        int[] values = new int[names.length];
        for (int k = 0; k < nChan; k++) {
            Channel ch = chanData[k];
            for (int j = 0; j < names.length; j++) {
                try {
                    int value = Integer.decode(pValues[j][k]);
                    if (j == 0) {
                            if (value < 0 || value >= N_DEV_RAILS)
                                throw new NumberFormatException();
                    }
                    values[j] = value;
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    log.error("Hardware structure is missing " + names[j]
                              + " value for " + chanData[k].name);
                    values[j] = -1;
                }
                catch (NumberFormatException e) {
                    log.error("Hardware structure contains invalid "
                              + names[j] + " value for " + chanData[k].name);
                    values[j] = -1;
                }
            }
            hdwData[k] = new ChanHdw(values[0]);
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs additional default configuration setup.
    **
    ***************************************************************************
    */
    @Override
    void addDefaultConfiguration(Properties dfltConfig)
    {
    }


   /**
    ***************************************************************************
    **
    **  Performs additional configuration.
    **
    ***************************************************************************
    */
    @Override
    void addConfiguration()
    {
    }


   /**
    ***************************************************************************
    **
    **  Opens the pmmanage hardware
    **
    ***************************************************************************
    */
    @Override
    void openHdw()
    {
        /*
        **  Connect to UCD90160 (TI UCD90xxx)
        */
        try {
            ucd90xxx = new TIUCD90xxx();
            ucd90xxx.open(DEV_PORT, DEV_ADDR);
            ucd90xxxOpened = true;
        }
        catch (Exception e) {
            log.error("Error opening TI UCD90xxx device", e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Initializes the pmmanage hardware
    **
    ***************************************************************************
    */
    @Override
    void initHdw()
    {
        /*
        **  Configure UCD90160 (TI UCD90xxx)
        */
        try {
            // xiaowen: not much is available yet. mostly set by hand
            for (int j = 0; j < nChan; j++) {
                // xiaowen: method not working yet
                //ucd90xxx.setParmLimits(j);
            }
            ucd90xxxInited=true;
        }
        catch (Exception e) {
            log.error("Error initializing TI UCD90xxx device", e);
        }

        /*
        **  Start the thread which updates the channel state periodically
        */
        (new Timer()).schedule(new UpdateState(), 0L, UPDATE_PERIOD);
    }


   /**
    ***************************************************************************
    **
    **  Gets the parameter values
    **
    ***************************************************************************
    */
    @Override
    void getParmValues()
    {
        for (int j = 0; j < nChan; j++) {
            Channel ch = chanData[j];
            ChanHdw hw = hdwData[j];
            try {
                ch.value=ucd90xxx.getParmValue(hw.rail,ch.type);
            }
            catch (PMCardException e) {
                if (ucd90xxxInited)
                    log.error("Error getting value for channel " + String.valueOf(j), e);
                ch.value=0.0;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the parameter limits for one channel
    **
    **  @param  id  The channel ID (0 - 2)
    **
    ***************************************************************************
    */
    @Override
    void setParmLimits(int id)
    {
        // xiaowen: method not available in PMCard.c yet
        log.error("setParmLimits not working yet. No corresponding method in PMCard.c");
    }


   /**
    ***************************************************************************
    **
    **  Gets the parameter limits for one channel
    **
    **  @param  id  The channel ID (0 - 2)
    **
    **  @param  limits  The array containing the limits
    **
    ***************************************************************************
    */
    @Override
    void getParmLimits(int id, double[] limits)
    {
        Channel chan = chanData[id];
        ChanHdw hw = hdwData[id];

        int rc=0;
        try {
            ucd90xxx.getParmLimits(hw.rail, chan.type, limits);
        }
        catch (PMCardException e) {
            if ( ucd90xxxOpened ) // ucd90xxxInited
                log.error("Error getting limits for channel " + String.valueOf(id), e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the value of a state bit
    **
    **  @param  id  The channel ID (0 - 2)
    **
    **  @return  The current value of the specified bit of the state word
    **
    ***************************************************************************
    */
    @Override
    int getChanStateBit(int id)
    {
        return (chanState >> id) & 1;
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the value of the entire state word
    **
    **  @return  The current value of the state word
    **
    ***************************************************************************
    */
    @Override
    int getChanState()
    {
        return chanState;
    }

}
