package org.lsst.ccs.subsystems.powermanage;

import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;
import java.util.Random;

/**
 ***************************************************************************
 **
 **  Implements the powermanage test modular subsystem
 **
 **  @author Xiaowen Lei
 **
 ***************************************************************************
 */
public class PMTestSim extends PMTest {

   /**
    ***************************************************************************
    **
    **  Container class for channel simulation data
    **
    ***************************************************************************
    */
    private static class ChanSim {

/*        double onValue;       // On (operating) value
        double onTime;        // Time (secs) to reach on value
        double offValue;      // Off (ambient) value
        double offTime;       // Time (secs) to reach off value
        double startVal;      // Starting value
        double endVal;        // Ending value
        double rate;          // Rate of change  */
        double baseVal;       

        public ChanSim()
/*        public ChanSim(double onValue, double onTime, double offValue,
                       double offTime)*/
        {
/*            this.onValue   = onValue;
            this.onTime    = onTime;
            this.offValue  = offValue;
            this.offTime   = offTime;
            this.startVal  = 0f;
            this.endVal    = 0f;
            this.rate      = 0f;*/
        }
    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement a timer task to update the channel state
    **  in a timely manner.
    **
    ***************************************************************************
    */
    private class UpdateState extends TimerTask {

        @Override
        public void run()
        {
            if (!running) return;   // Do nothing if not yet running
            
            /*
            **  Simulate changing values and set chanState
            */
            simParmValues();
            
            /*
            **  Report state change
            */
            if (chanState != lastChanState) {
                lastChanState = chanState;
                tick();
                publishState();
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static long
        UPDATE_PERIOD = 1000;

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private int chanState = 0, lastChanState;
    private ChanSim[] simData;
    private long baseTime = System.currentTimeMillis();
    Random RN = new Random();
    private double simAngle, simAngleStartVal = 0.0, simAngleStep = 0.035;


   /**
    ***************************************************************************
    **
    **  Performs additional structural configuration.
    **
    ***************************************************************************
    */
    @Override
    void addStructure(Properties struct)
    {
        simData = new ChanSim[nChan];
        for (int k = 0; k < nChan; k++) {
            simData[k] = new ChanSim();
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs additional default configuration setup.
    **
    ***************************************************************************
    */
    @Override
    void addDefaultConfiguration(Properties dfltConfig)
    {
    }


   /**
    ***************************************************************************
    **
    **  Performs additional configuration.
    **
    ***************************************************************************
    */
    @Override
    void addConfiguration()
    {
    }


  /**
    ***************************************************************************
    **
    **  Opens the simulated hardware
    **
    ***************************************************************************
    */
    @Override
    void openHdw()
    {
        // xiaowen: put here temporarily
        baseTime = System.currentTimeMillis();
        simAngle = simAngleStartVal;
    }


   /**
    ***************************************************************************
    **
    **  Initializes the simulated hardware
    **
    ***************************************************************************
    */
    @Override
    void initHdw()
    {
        /*
        **  Set initial simulator parameter values
        */
        for (int j = 0; j < nChan; j++) {
            Channel chan = chanData[j];
            ChanSim sim = simData[j];
            if ( chan.type == CHAN_TYPE_TEMP ) {
                sim.baseVal = 1.2 * chan.oWarnLimit - 0.2 * chan.oFaultLimit;
            }
            else {
                sim.baseVal = 0.5 * ( chan.uFaultLimit + chan.oFaultLimit );
            }
            // chan.value = 
        }

        /*
        **  Start the timer task which updates the channel state periodically
        */
        (new Timer()).schedule(new UpdateState(), 0L, UPDATE_PERIOD);
    }

   /**
    ***************************************************************************
    **
    **  Simulates the parameter values
    **
    ***************************************************************************
    */
    void simParmValues()
    {
        double rn, s, c, d;

        for (int j = 0; j < nChan; j++) {
            rn = RN.nextDouble();
            Channel chan = chanData[j];
            ChanSim sim = simData[j];
            switch (chan.type) {
                case CHAN_TYPE_VOLT:
                    s = Math.sin(simAngle);
                    d = s * (chan.uFaultLimit - sim.baseVal) * 1.05;
                    chan.value = d + sim.baseVal + 0.05 * sim.baseVal * (0.5 - rn);
                    break;
                case CHAN_TYPE_CURR:
                    c = Math.cos(simAngle);
                    d = c * (chan.uFaultLimit - sim.baseVal) * 1.05;
                    chan.value = d + sim.baseVal + 0.05 * sim.baseVal * (0.5 - rn);
                    break;
                case CHAN_TYPE_TEMP:
                    chan.value = sim.baseVal + 0.5 * (0.5 - rn);
                    break;
                default:
                    ;
                    break;
            }
            setChanStateBit(j);

            double intvl = (System.currentTimeMillis() - baseTime) / 1000.0;
            simAngle = simAngleStartVal + simAngleStep * intvl;
        }

        // xiaowen: reset if reach upper limit of double?
    }

   /**
    ***************************************************************************
    **
    **  Gets the parameter values
    **
    ***************************************************************************
    */
    @Override
    void getParmValues()
    {
    }


   /**
    ***************************************************************************
    **
    **  Sets the parameter limits for one channel
    **
    **  @param  id  The channel ID (0 - 2)
    **
    ***************************************************************************
    */
    @Override
    void setParmLimits(int id)
    {
    }


  /**
    ***************************************************************************
    **
    **  Gets the parameter limits for one channel
    **
    **  @param  id  The channel ID (0 - 2)
    **
    **  @param  limits  The array containing the limits
    **
    ***************************************************************************
    */
    @Override
    void getParmLimits(int id, double[] limits)
    {
        Channel chan = chanData[id];

        int rc=0;
        switch (chan.type) {
            case CHAN_TYPE_VOLT:
                limits[0] = 0.8;
                limits[1] = 0.9;
                limits[2] = 1.1;
                limits[3] = 1.2;
                break;
            case CHAN_TYPE_CURR:
                limits[0] = 8.0;
                limits[1] = 0.0;
                limits[2] = 11.0;
                limits[3] = 12.0;
                break;
            case CHAN_TYPE_TEMP:
                limits[0] = 0.0;
                limits[1] = 0.0;
                limits[2] = 40.0;
                limits[3] = 50.0;
                break;
            default:
                ;
                break;
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the value of a channel state bit
    **
    **  @param  id  The channel ID (0 - 2)
    **
    **  @return  The current value of the specified bit of the state word
    **
    ***************************************************************************
    */
    @Override
    int getChanStateBit(int id)
    {
        return (chanState >> id) & 1;
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the value of the entire state word
    **
    **  @return  The current state word value
    **
    ***************************************************************************
    */
    @Override
    int getChanState()
    {
        return chanState;
    }


   /**
    ***************************************************************************
    **
    **  Sets the value of a channel state bit
    **
    **  @param  id  The channel ID (0 - 2)
    **
    ***************************************************************************
    */
    private void setChanStateBit(int id)
    {
        Channel chan = chanData[id];
        int value = 1;
        if (chan.value < chan.uFaultLimit || chan.value > chan.oFaultLimit)
            value = 0;
        if (value != 0)
            chanState |= (1 << id);
        else
            chanState &= ~(1 << id);
    }

}
