package org.lsst.ccs.subsystems.powermanage;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Properties;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.powermanage.data.PMChannel;
import org.lsst.ccs.subsystems.powermanage.data.PMFullState;
import org.lsst.ccs.subsystems.powermanage.data.PMState;
import org.lsst.ccs.subsystems.powermanage.status.PMStateStatus;

/**
 ***************************************************************************
 **
 **  Implements the pmmanage test modular subsystem
 **
 **  <p>
 **  This is an abstract class implementing the interactions with the CCS
 **  buses, but deferring the hardware interaction, real or simulated, to
 **  sub-classes.
 **
 **  @author Xiaowen Lei
 **
 ***************************************************************************
 */
public abstract class PMTest extends Module {

   /**
    ***************************************************************************
    **
    **  Container class for channel parameters
    **
    ***************************************************************************
    */
   // xiaowen: temperarily using Channel modified from RefrigTest
   // -- saves time. whatever written now will be changed in the future
   // need to talk to Charles (Dan?) about the hardware, then change it to something more sensible
   // -- crate, rail, channel, volt/curr/temp..
   // -- if all channels outputs volt/curr/temp, get rid of type and include all three variables?
    static class Channel {

        String  name;              // Channel name
        String  description;       // Channel description
        String  units;             // Units name (derived)
        int     type;              // Channel type (Voltage, current or temperature)
        double   uFaultLimit;       // Under fault limit value. NOT USED for temperature
        double   oFaultLimit;       // Over fault limit value
        double   uWarnLimit;        // Under warning limit value. NOT USED for current and temperature
        double   oWarnLimit;        // Over warning limit value
        String  uFaultLimitName;   // Under fault limit name (derived)
        String  oFaultLimitName;   // Over fault limit name (derived)
        String  uWarnLimitName;    // Under warn limit name (derived)
        String  oWarnLimitName;    // Over warn limit name (derived)
        double   value;             // Current value read from channel

        public Channel(String name, String desc, int type)
        {
            this.name = name;
            this.description = desc;
            this.type = type;
            this.uFaultLimitName = name + "UFaultLim";
            this.oFaultLimitName = name + "OFaultLim";
            this.uWarnLimitName = name + "UWarnLim";
            this.oWarnLimitName = name + "OWarnLim";
            this.units = (this.type == CHAN_TYPE_VOLT) ? "V"
                            : this.type == CHAN_TYPE_CURR ? "A" : "C";
        }
    }

   /**
    ***************************************************************************
    **
    **  Package constants
    **
    ***************************************************************************
    */
    final static int
        MAX_CHANS      = 3, // xiaowen: will change in the future
        CHAN_TYPE_VOLT = 0,
        CHAN_TYPE_CURR = 1,
        CHAN_TYPE_TEMP = 2;

    final static String
        PREFIX        = "org.lsst.ccs.powermanage.";

   /**
    ***************************************************************************
    **
    **  Package fields
    **
    ***************************************************************************
    */
    String structFile = "structure.properties",
           configFile = "config.properties";
    Properties config;
    Channel[] chanData;
    int nChan, uFLimitChange, oFLimitChange, uWLimitChange = 0, oWLimitChange = 0;
    boolean running = false; 


   /**
    ***************************************************************************
    **
    **  Initializes the pmmanage subsystem
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize configuration data
        */
        initStructure();
        openHdw(); // needed for the default limits
        initConfiguration();

        /*
        **  Initialize the hardware
        */
        initHdw();
    }


    /**
    ***************************************************************************
    **
    **  Performs periodic readout of the values of the parameters
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        /*
        **  Broadcast the state if first time
        */
        if (!running) {
            System.out.println("PowerManage test system started");
            publishState();    // For any GUIs
            publishLimits();   // For GUIs and the trending database
            running = true;
        }

        /*
        **  Get the values of the parameters
        */
        getParmValues();
        //// xiaowen: just to convince me this is doing something
        //for (int j = 0; j < nChan; j++) {
        //    Channel chan = chanData[j];
        //    //System.out.println("[PMTestSimModule] ( Chan "+chan.name+", value = "+chan.value+" )");
        //    log.info("[PMTestSim] ( Chan "+chan.name+", value = "+chan.value+" )");
        //}

        /*
        **  Broadcast the trending data
        */
        ArrayList<ValueNotification> trendingValues
          = new ArrayList<ValueNotification>();
        for (int j = 0; j < nChan; j++) {
            Channel chan = chanData[j];
            long ts = System.currentTimeMillis(); 
            trendingValues.add(new ValueNotification(chan.name, chan.value, ts));
        }
        trendingValues.add(new ValueNotification("state", getState()));
        publishData(trendingValues);
    }


   /**
    ***************************************************************************
    **
    **  Sets the structural configuration file name
    **
    **  @param  name  The name of the structural configuration file
    **
    ***************************************************************************
    */
    public void setStructFile(String name)
    {
        structFile = name;
    }


   /**
    ***************************************************************************
    **
    **  Sets the dynamic configuration file name
    **
    **  @param  name  The name of the dynamic configuration file
    **
    ***************************************************************************
    */
    public void setConfigFile(String name)
    {
        configFile = name;
    }


   /**
    ***************************************************************************
    **
    **  Sets the tick period
    **
    **  <p>
    **  Overrides the method in Module in order to publish a status update.
    **
    **  @param  value  The tick period (milliseconds) to set.
    **
    ***************************************************************************
    */
    @Override
    public void setTickMillis(int value)
    {
        super.setTickMillis(value);
        if (running) publishState();
    }


//    /**
//     ***************************************************************************
//     **
//     **  Sets the power enabled state on or off
//     **
//     **  @param  value  The enabled state value to set: 0 = off, ~0 = on.
//     **
//     ***************************************************************************
//     */
//     public void setPowerEnable(int value)
//     {
//         boolean enab = (value != 0);
//         if (enab == mainPower) return;
//         mainPower = enab;
//         setMainPower(enab);
//         publishState();
//     }


   /**
    ***************************************************************************
    **
    **  Sets the under fault limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The under fault limit value.
    **
    ***************************************************************************
    */
    public void setUFaultLimit(int id, double limit)
    {
        Channel chan = chanData[id];
        if (chan.type==CHAN_TYPE_TEMP) ; // xiaowen: later (soon)

        if (limit == chan.uFaultLimit) return;
        chan.uFaultLimit = limit;
        String sLimit = String.valueOf(limit);
        config.setProperty(chan.uFaultLimitName, sLimit);
        setParmLimits(id);
        uFLimitChange |= 1 << id;
        publishState();
        getSubsystem().publishMetaData(chan.name, "UnderFault", sLimit);
    }


   /**
    ***************************************************************************
    **
    **  Sets the over fault limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The over fault limit value.
    **
    ***************************************************************************
    */
    public void setOFaultLimit(int id, double limit)
    {
        Channel chan = chanData[id];
        if (limit == chan.oFaultLimit) return;
        chan.oFaultLimit = limit;
        String sLimit = String.valueOf(limit);
        config.setProperty(chan.oFaultLimitName, sLimit);
        setParmLimits(id);
        oFLimitChange |= 1 << id;
        publishState();
        getSubsystem().publishMetaData(chan.name, "OverFault", sLimit);
    }


   /**
    ***************************************************************************
    **
    **  Sets the under warn limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The under warn limit value.
    **
    ***************************************************************************
    */
    public void setUWarnLimit(int id, double limit)
    {
        Channel chan = chanData[id];
        if (chan.type!=CHAN_TYPE_VOLT) return; // xiaowen: add warning messages later (soon)

        if (limit == chan.uWarnLimit) return;
        chan.uWarnLimit = limit;
        String sLimit = String.valueOf(limit);
        config.setProperty(chan.uWarnLimitName, sLimit);
        setParmLimits(id);
        uWLimitChange |= 1 << id;
        publishState();
        getSubsystem().publishMetaData(chan.name, "UnderWarn", sLimit);
    }


   /**
    ***************************************************************************
    **
    **  Sets the over warn limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The over warn limit value.
    **
    ***************************************************************************
    */
    public void setOWarnLimit(int id, double limit)
    {
        Channel chan = chanData[id];
        if (limit == chan.oWarnLimit) return;
        chan.oWarnLimit = limit;
        String sLimit = String.valueOf(limit);
        config.setProperty(chan.oWarnLimitName, sLimit);
        setParmLimits(id);
        oFLimitChange |= 1 << id;
        publishState();
        getSubsystem().publishMetaData(chan.name, "OverWarn", sLimit);
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data
    **
    ***************************************************************************
    */
    public void saveConfiguration()
    {
        FileWriter writer = null;
        try {
            writer = new FileWriter(configFile);
        }
        catch (IOException e) {
            log.error("Error opening configuration file: " + e);
        }
        if (writer != null) {
            try {
                config.store(writer,
                             "PM test configuration");
                uFLimitChange = oFLimitChange = uWLimitChange = oWLimitChange = 0;
                publishState();
            }
            catch (IOException e) {
                log.error("Error writing configuration: " + e);
            }
            try {
                writer.close();
            }
            catch (IOException e) {
                log.error("Error closing configuration file: " + e);
            }
        }
    }


    /**
     ***************************************************************************
     **
     **  Gets the operating state word
     **
     **  @return  The current value of the operating state
     **
     ***************************************************************************
     */
     public int getState()
     {
         int value = getChanState();
//          if (mainPower) value |= (1 << RefrigState.MAIN_POWER_STATE_BIT);
 
         return value;
     }


   /**
    ***************************************************************************
    **
    **  Gets the full state of the pmmanage module
    **
    **  <p>
    **  This is intended to be called by GUIs during initialization
    **
    ***************************************************************************
    */
    public PMFullState getFullState()
    {
        PMFullState status = new PMFullState();
        for ( int i = 0; i < nChan; i++) {
            Channel c = chanData[i];
            status.addChannel(new PMChannel(c.name, c.description, c.units, c.uFaultLimit, c.oFaultLimit,
                                                c.uWarnLimit, c.oWarnLimit, c.value));
        }
        status.setPMState(new PMState(getState(), uFLimitChange, oFLimitChange,
                                              uWLimitChange, oWLimitChange, getTickMillis()));
        return status;
    }    


   /**
    ***************************************************************************
    **
    **  Publishes the state of the pmmanage module
    **
    **  <p>
    **  This is intended to be called whenever any element of the state is
    **  changed.
    **
    ***************************************************************************
    */
    public void publishState()
    {
        PMState pmState = new PMState(getState(), uFLimitChange, oFLimitChange,
                                                  uWLimitChange, oWLimitChange, getTickMillis());
        sendToStatus(new PMStateStatus(pmState));
    }    


   /**
    ***************************************************************************
    **
    **  Publishes all the limit values of the pmmanage module
    **
    **  <p>
    **  This is intended to be called at startup time.
    **
    ***************************************************************************
    */
    public void publishLimits()
    {
        for ( int i = 0; i < nChan; i++) {
            Channel c = chanData[i];
            getSubsystem().publishMetaData(c.name, "UnderFault",
                                           String.valueOf(c.uFaultLimit));
            getSubsystem().publishMetaData(c.name, "OverFault",
                                           String.valueOf(c.oFaultLimit));
            getSubsystem().publishMetaData(c.name, "UnderWarn",
                                           String.valueOf(c.uWarnLimit));
            getSubsystem().publishMetaData(c.name, "OverWarn",
                                           String.valueOf(c.oWarnLimit));
        }
    }    


   /**
    ***************************************************************************
    **
    **  Initializes the structural configuration data
    **
    ***************************************************************************
    */
    private void initStructure()
    {
        Properties struct = loadProperties(structFile, null);
        try {
            nChan = Integer.decode(struct.getProperty(PREFIX + "nchan"));
            if (nChan <= 0 || nChan > MAX_CHANS)
                throw new NumberFormatException();
        }
        catch (NullPointerException e) {
            log.error("General structure is missing nchan property");
            nChan = 0;
            return;
        }
        catch (NumberFormatException e) {
            log.error("Invalid number of channels specified");
            nChan = 0;
            return;
        }

        String[] names = {"names", "descs", "types"};
        chanData = new Channel[nChan];
        String[][] pValues = new String[names.length][];
        for (int j = 0; j < names.length; j++) {
            String pValue = struct.getProperty(PREFIX + names[j]);
            if (pValue != null) {
                pValues[j] = pValue.split(",");
            }
            else {
                log.error("General structure is missing " + names[j]
                          + " property");
                pValues[j] = new String[nChan];
                for (int k = 0; k < nChan; k++)
                    pValues[j][k] = (j == 2) ? "0" : "";
            }
        }

        String[] values = new String[names.length];
        for (int k = 0; k < nChan; k++) {
            int type = 0;
            for (int j = 0; j < names.length; j++) {
                if (k < pValues[j].length) {
                    values[j] = pValues[j][k];
                    if (j == 2) {
                        String cType = values[j];
                        type = cType.equals("VOLT") ? CHAN_TYPE_VOLT
                               : cType.equals("CURR") ? CHAN_TYPE_CURR
                               : cType.equals("TEMP") ? CHAN_TYPE_TEMP : -1;
                        if (type < 0) {
                            log.error("General structure contains invalid "
                                      + names[j] + " value for channel " + k);
                            type = CHAN_TYPE_VOLT;
                        }
                    }
                }
                else {
                    log.error("General structure is missing " + names[j]
                              + " value for channel " + k);
                    values[j] = "";
                }
            }
            chanData[k] = new Channel(values[0], values[1], type);
        }

        addStructure(struct);
    }


   /**
    ***************************************************************************
    **
    **  Initializes the configuration data
    **
    ***************************************************************************
    */
    private void initConfiguration()
    {
        /*
        **  Generate a properties list containing default configuration values
        */
        Properties dfltConfig = new Properties();
        for (int j = 0; j < nChan; j++) {
            Channel chan = chanData[j];
            double limits[]={0.0,0.0,0.0,0.0};
            getParmLimits(j,limits);
            dfltConfig.setProperty(chan.uFaultLimitName, String.valueOf(limits[0]));
            dfltConfig.setProperty(chan.oFaultLimitName, String.valueOf(limits[3]));
            dfltConfig.setProperty(chan.uWarnLimitName,  String.valueOf(limits[1]));
            dfltConfig.setProperty(chan.oWarnLimitName,  String.valueOf(limits[2]));
        }

        /*
        **  Create the configuration properties list and load it
        */
        config = loadProperties(configFile, dfltConfig);

        /*
        **  Extract values from the properties list
        */
        for (int j = 0; j < nChan; j++) {
            Channel chan = chanData[j];
            chan.uFaultLimit = Double.valueOf(config.getProperty(chan.uFaultLimitName));
            chan.oFaultLimit = Double.valueOf(config.getProperty(chan.oFaultLimitName));
            chan.uWarnLimit = Double.valueOf(config.getProperty(chan.uWarnLimitName));
            chan.oWarnLimit = Double.valueOf(config.getProperty(chan.oWarnLimitName));
        }
        
        addConfiguration();
    }


   /**
    ***************************************************************************
    **
    **  Loads a properties file
    **
    ***************************************************************************
    */
    private Properties loadProperties(String file, Properties dflt)
    {
        /*
        **  Create the properties list
        */
        Properties propList;
        if (dflt == null)
            propList = new Properties();
        else
            propList = new Properties(dflt);

        /*
        **  Open the properties file
        */
        FileReader reader = null;
        try {
            reader = new FileReader(file);
        }
        catch (FileNotFoundException e) {
            log.error("Error opening properties file: ", e);
        }

        /*
        **  Load the properties list
        */
        try {
            propList.load(reader);
        }
        catch (IOException e) {
            log.error("Error reading properties file: ", e);
        }
        try {
            reader.close();
        }
        catch (IOException e) {
            log.error("Error closing properties file: ", e);
        }

        return propList;
    }


   /**
    ***************************************************************************
    **
    **  Performs additional structural configuration.
    **
    ***************************************************************************
    */
    abstract void addStructure(Properties struct);


   /**
    ***************************************************************************
    **
    **  Performs additional default configuration setup.
    **
    ***************************************************************************
    */
    abstract void addDefaultConfiguration(Properties dfltConfig);


   /**
    ***************************************************************************
    **
    **  Performs additional configuration.
    **
    ***************************************************************************
    */
    abstract void addConfiguration();


   /**
    ***************************************************************************
    **
    **  Opens the pmmanage hardware
    **
    **  <p>
    **  This is intended to be called before initConfiguration(),
    **  because the default limits are read from the hardware
    **
    ***************************************************************************
    */
    abstract void openHdw();


   /**
    ***************************************************************************
    **
    **  Initializes the pmmanage hardware
    **
    ***************************************************************************
    */
    abstract void initHdw();


   /**
    ***************************************************************************
    **
    **  Gets the parameter values
    **
    ***************************************************************************
    */
    abstract void getParmValues();


   /**
    ***************************************************************************
    **
    **  Sets the parameter limits for one channel
    **
    **  @param  id  The channel ID (0 - 2)
    **
    ***************************************************************************
    */
    abstract void setParmLimits(int id);


   /**
    ***************************************************************************
    **
    **  Gets the parameter limits for one channel
    **
    **  @param  id  The channel ID (0 - 2)
    ** 
    **  @param  limits  array to store the limits
    **
    ***************************************************************************
    */
    abstract void getParmLimits(int id, double[] limits);


//    /**
//     ***************************************************************************
//     **
//     **  Turns the main power on or off, according to the set value
//     **
//     ***************************************************************************
//     */
//     abstract void setMainPower();
           

   /**
    ***************************************************************************
    **
    **  Gets the value of a state bit
    **
    **  @param  bit   The number (0 - 2) of the state bit
    **
    **  @return  The value, 0 or 1, of the bit.
    **
    ***************************************************************************
    */
    abstract int getChanStateBit(int bit);
            

   /**
    ***************************************************************************
    **
    **  Gets the value of the entire state word
    **
    **  @return  The current state word value
    **
    ***************************************************************************
    */
    abstract int getChanState();
    
}
