package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.ascii.Session;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.eaton.EpduG3;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.power.constants.PowerConfig;

/**
 *  Interface to an Eaton PDU.
 *
 *  @author Owen Saxton
 */
public class EpduG3Device extends Device {

    /**
     *  Constants.
     */
    private static final int
        CHAN_TYPE_VOLTAGE = 0,
        CHAN_TYPE_CURRENT = 1,
        CHAN_TYPE_POWER = 2,
        CHAN_TYPE_TEMP = 3;

    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("VOLTAGE", CHAN_TYPE_VOLTAGE);
        typeMap.put("CURRENT", CHAN_TYPE_CURRENT);
        typeMap.put("POWER", CHAN_TYPE_POWER);
        typeMap.put("TEMP", CHAN_TYPE_TEMP);
    }

    /**
     *  Data fields.
     */
    @ConfigurationParameter(category=PowerConfig.CATEGORY_EATON_PDU, isFinal=true)
    private volatile String host;
    @ConfigurationParameter(category=PowerConfig.CATEGORY_EATON_PDU, isFinal=true)
    private volatile String userName = "";
    @ConfigurationParameter(category=PowerConfig.CATEGORY_EATON_PDU, isFinal=true)
    private volatile String password = "";

    private static final Logger LOG = Logger.getLogger(EpduG3Device.class.getName());
    private final EpduG3 pdu = new EpduG3();
    private boolean initError = false;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        fullName = "Eaton G3 PDU (" + host + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            pdu.open(Session.ConnType.TELNET, host, userName, password);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (Exception e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            pdu.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}:{1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  ch  The channel to check
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        String type = ch.getTypeStr();
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            ErrorUtils.reportChannelError(LOG, ch.getPath(), "Type", type);
        }
        return new int[]{iType, 0};
    }


    /**
     *  Initializes a monitoring channel (checks validity)
     *
     *  @param  ch  The channel to initialize
     */
    @Override
    protected void initChannel(Channel ch)
    {
        try {
            int hwChan = ch.getHwChan();
            int type = ch.getType();
            if (type == CHAN_TYPE_CURRENT || type == CHAN_TYPE_POWER) {
                if (hwChan < 0 || hwChan > pdu.getOutletCount()) {
                    ErrorUtils.reportChannelError(LOG, name, "hardware channel", hwChan);
                }
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error getting PDU outlet count: {0}", e);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  ch  The channel to read
     *  @return  The read value
     */
    @Override
    protected double readChannel(Channel ch)
    {
        double value = super.readChannel(ch);
        if (isOnline()) {
            int type = ch.getType();
            int hwChan = ch.getHwChan();
            try {
                switch (type) {
                case CHAN_TYPE_CURRENT:
                    value = hwChan == 0 ? pdu.readPhaseCurrent(1) : pdu.readOutletCurrent(hwChan);
                    break;
                case CHAN_TYPE_POWER:
                    value = hwChan == 0 ? pdu.readPhasePower(1) : pdu.readOutletPower(hwChan);
                    break;
                case CHAN_TYPE_VOLTAGE:
                    value = pdu.readPhaseVoltage(1);
                    break;
                case CHAN_TYPE_TEMP:
                    value = pdu.readTemperature();
                    break;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading PDU data: {0}", e);
                setOnline(false);
            }
        }

        return value;
    }

}
