package org.lsst.ccs.subsystem.power;

import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 *  Implements power supply control functions.
 *
 *  @author Owen Saxton
 */
public class PowerControl implements HasLifecycle {

    /**
     *  Constants.
     */
    private static final Logger LOG = Logger.getLogger(PowerControl.class.getName());

    /**
     *  Data fields.
     */
    @LookupName
    protected String name;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    protected PowerDevice pDevc;
    
    // Supplied configurable fields
    @ConfigurationParameter(category=Power.POWER, isFinal=true)
    private volatile double      voltage;       // Voltage
    @ConfigurationParameter(category=Power.POWER, isFinal=true)
    private volatile double      current;       // Current
    @ConfigurationParameter(category=Power.POWER, isFinal=true)
    private volatile double      onDelay;       // Turn-on delay
    @ConfigurationParameter(category=Power.POWER, isFinal=true)
    private volatile double      offDelay;      // Turn-off delay

    // From Groovy file
    private int hwChan;
    private String description;

    private double opVoltage, opCurrent, opOnDelay, opOffDelay;  // Operating values


    /**
     *  Life-cycle initialization.
     *
     *  Configures channel description.
     */
    @Override
    public void init()
    {
        if (hwChan < pDevc.getMinChannel() || hwChan > pDevc.getMaxChannel()) {
            ErrorUtils.reportConfigError(LOG, name, "hw channel number (" + hwChan + ")", "is invalid");
        }
        opVoltage = voltage;
        opCurrent = current;
        opOnDelay = onDelay;
        opOffDelay = offDelay;
    }


    /**
     *  Sets the voltage.
     *
     *  @param  value  The value to set
     */
    public void setVoltage(double value)
    {
        opVoltage = value;
    }


    /**
     *  Gets the voltage.
     *
     *  @return  The set voltage
     */
    public double getVoltage()
    {
        return opVoltage;
    }


    /**
     *  Sets the current.
     *
     *  @param  value  The value to set
     */
    public void setCurrent(double value)
    {
        opCurrent = value;
    }


    /**
     *  Gets the current.
     *
     *  @return  The set current
     */
    public double getCurrent()
    {
        return opCurrent;
    }


    /**
     *  Sets the power-on delay.
     *
     *  @param  value  The value to set (secs)
     */
    public void setOnDelay(double value)
    {
        opOnDelay = value;
    }


    /**
     *  Gets the power-on delay.
     *
     *  @return  The set delay (secs)
     */
    public double getOnDelay()
    {
        return opOnDelay;
    }


    /**
     *  Sets the power-off delay.
     *
     *  @param  value  The value to set (secs)
     */
    public void setOffDelay(double value)
    {
        opOffDelay = value;
    }


    /**
     *  Gets the power-off delay.
     *
     *  @return  The set delay (secs)
     */
    public double getOffDelay()
    {
        return opOffDelay;
    }


    /**
     *  Changes the voltage.
     *
     *  @param  value  The value to set
     *  @throws  PowerException
     */
    public void changeVoltage(double value) throws PowerException
    {
        pDevc.writeVoltage(value, hwChan);
    }


    /**
     *  Sets all the configuration data.
     *
     *  Actually just sets the operating values.
     */
    void setConfig(Power.Channel chan)
    {
        opVoltage = chan.getVoltage();
        opCurrent = chan.getCurrent();
        opOnDelay = chan.getOnDelay();
        opOffDelay = chan.getOffDelay();
    }


    /**
     *  Gets all configuration data.
     *
     *  Actually just gets the operating values.
     */
    Power.Channel getConfig()
    {
        Power.Channel chan = new Power.Channel();
        chan.setName(name);
        chan.setVoltage(opVoltage);
        chan.setCurrent(opCurrent);
        chan.setOnDelay(opOnDelay);
        chan.setOffDelay(opOffDelay);

        return chan;
    }


    /**
     *  Gets state data from the hardware.
     */
    PowerChanState getState()
    {
        PowerChanState pwr = new PowerChanState();
        pwr.setName(name);
        try {
            pwr.setVoltage(readVoltage());
            pwr.setCurrent(readCurrent());
            Boolean outState = readOutput();
            pwr.setState(outState == null ? PowerChanState.PWR_STATE_OFFLINE
                                          : outState ? PowerChanState.PWR_STATE_ON : PowerChanState.PWR_STATE_OFF);
        }
        catch (PowerException e) {
            pwr.setVoltage(0);
            pwr.setCurrent(0);
            pwr.setState(PowerChanState.PWR_STATE_OFFLINE);
        }

        return pwr;
    }


    /**
     *  Writes all set quantities to the hardware.
     *
     *  @throws  PowerException
     */
    public void writeAll() throws PowerException
    {
        writeVoltage();
        writeCurrent();
        writeOnDelay();
        writeOffDelay();
    }


    /**
     *  Writes the voltage to the hardware.
     *
     *  @throws  PowerException
     */
    public void writeVoltage() throws PowerException
    {
        pDevc.writeVoltage(opVoltage, hwChan);
    }


    /**
     *  Reads the voltage from the hardware.
     *
     *  @return  The read voltage
     *  @throws  PowerException
     */
    public double readVoltage() throws PowerException
    {
        return pDevc.readVoltage(hwChan);
    }


    /**
     *  Writes the current to the hardware.
     *
     *  @throws  PowerException
     */
    public void writeCurrent() throws PowerException
    {
        pDevc.writeCurrent(opCurrent, hwChan);
    }


    /**
     *  Reads the current from the hardware.
     *
     *  @return  The current
     *  @throws  PowerException
     */
    public double readCurrent() throws PowerException
    {
        return pDevc.readCurrent(hwChan);
    }


    /**
     *  Writes the power-on delay to the hardware.
     *
     *  @throws  PowerException
     */
    public void writeOnDelay() throws PowerException
    {
        pDevc.writeOnDelay(opOnDelay, hwChan);
    }


    /**
     *  Writes the power-off delay to the hardware.
     *
     *  @throws  PowerException
     */
    public void writeOffDelay() throws PowerException
    {
        pDevc.writeOffDelay(opOffDelay, hwChan);
    }


    /**
     *  Writes the power output state to the hardware.
     *
     *  @param  on  Whether to set output on
     *  @throws  PowerException
     */
    public void writeOutput(boolean on) throws PowerException
    {
        pDevc.writeOutput(on, hwChan);
    }


    /**
     *  Reads the power output state from the hardware.
     *
     *  @return  Whether output is on, or null if offline
     */
    public Boolean readOutput()
    {
        return pDevc.readOutput(hwChan);
    }


    /**
     *  Gets the associated device
     *
     *  @return  The power device
     */
    public PowerDevice getDevice()
    {
        return pDevc;
    }


    /**
     *  Gets the hardware channel.
     *
     *  @return  The hardware channel
     */
    public int getHwChan()
    {
        return hwChan;
    }

}
