package org.lsst.ccs.subsystem.power;

import java.util.logging.Level;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.drivers.auxelex.RebPS;
import org.lsst.ccs.subsystem.power.constants.RebPsEnum;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.states.RebPowerState;

/**
 * A class representing an individual Reb for a Corner Rat.
 * It contains the information regarding the power supply powering this reb.
 * It provides commands to control the power to this reb.
 * 
 * @author The LSST CCS Team
 */
public class CornerRaftRebPowerSupplyNode extends RebPowerSupplyNode {
    
    @ConfigurationParameter( range = "1300..4095")
    volatile int dphi = 4095;
    
    public CornerRaftRebPowerSupplyNode(PowerSupplyConfiguration powerSupplyConfiguration, int channel) {
        super(powerSupplyConfiguration, channel);
    }

    @Override
    public void build() {
        super.build(); 
        if ( RunMode.isSimulation() ) {
            powerSupplyDevice.setSimulatedPsType(RebPS.TYPE_CORNER);
        }
    }
    
    
    
    @ConfigurationParameterChanger(propertyName = "dphi")
    public void setDphi(int dphi) {
        this.dphi = dphi;
        //Write configuration value to DAC only if RebPower is ON
        //Synchornize on StateLock to avoid race conditions with thread that
        //updates states
        synchronized (agentStateService.getStateLock()) {
            if (agentStateService.isComponentInState(rebPath, RebPowerState.ON)) {
                writeDphiConfigurationValueToDac();
            }
        }
    }

    void writeDphiConfigurationValueToDac() {
        try {
            powerSupplyDevice.setDphiDac(channel, dphi);        
            LOG.log(Level.INFO,"{0} set dphi DAC value of {1}",new Object[]{rebPath,dphi});
        } catch (PowerException ex) {
            LOG.log(Level.WARNING,"Could not set the HVBias DAC value",ex);
            throw new RuntimeException(ex);
        }        
    }
        
    @Command(type=Command.CommandType.ACTION, description="Turns on the dphi bias to the configuration value", autoAck = false)
    public void dphiOn() throws PowerException {
        agent.helper().precondition(agentStateService.isComponentInState(rebPath, RebPowerState.ON),"The Reb main power must be ON.").enterFaultOnException(true).action(
                () -> {
                    powerSupplyDevice.setPowerOn(channel, RebPsEnum.DPHI.getNumber(), true);
                });
    }
    
    @Command(type=Command.CommandType.ACTION, description="Turns off the dphi bias")
    public void dphiOff() throws PowerException {
        powerSupplyDevice.setPowerOn(channel, RebPsEnum.DPHI.getNumber(), false);
    }
    
    @Command(type = Command.CommandType.ACTION, description = "Read the Dphi DAC value")
    public double readDphiDac() throws PowerException {
        return powerSupplyDevice.getDacs(false)[channel];
    }
    
}
