package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import org.lsst.ccs.drivers.auxelex.PduHV;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import static org.lsst.ccs.subsystem.power.PduCommonDevice.MON_TYPE_VOLTAGE;

/**
 *  Interface to a SLAC HV PDU.
 *
 *  @author Owen Saxton
 */
public class PduHvDevice extends PduCommonDevice {

    /**
     *  Constants.
     */
    private static final int
        MON_TYPE_MAIN_VOLTAGE = 2,
        MON_TYPE_MAIN_CURRENT = 3,
        MON_TYPE_MAIN_TEMP = 4,
        MON_TYPE_BOARD_TEMP = 5,
        MON_TYPE_FPGA_TEMP = 6;

    private static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("VOLTAGE", MON_TYPE_VOLTAGE);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
        mTypeMap.put("MAINVOLT", MON_TYPE_MAIN_VOLTAGE);
        mTypeMap.put("MAINCURR", MON_TYPE_MAIN_CURRENT);
        mTypeMap.put("MAINTEMP", MON_TYPE_MAIN_TEMP);
        mTypeMap.put("BOARDTEMP", MON_TYPE_BOARD_TEMP);
        mTypeMap.put("FPGATEMP", MON_TYPE_FPGA_TEMP);
    }

    /**
     *  Data fields
     */
    private final PduHV pduHV;


    /**
     *  Constructor.
     *
     *  @param  pdu      The PDU driver object
     *  @param  pduType  The PDU type description
     *  @param  swDevc   The switch device ID
     */
    public PduHvDevice(PduHV pdu, String pduType, int swDevc)
    {
        super(pdu, pduType, swDevc);
        pduHV = pdu;
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  Only the type can be checked at this time.
     *
     *  @param  ch  The channel to check
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        String type = ch.getTypeStr();
        Integer mType = mTypeMap.get(type.toUpperCase());
        if (mType == null) {
            ErrorUtils.reportChannelError(LOG, ch.getPath(), "type", type);
        }
        return new int[]{mType, 0};
    }


    /**
     *  Reads a monitoring channel.
     *
     *  @param  ch  The channel to check
     *  @return  The read value
     */
    @Override
    protected double readChannel(Channel ch)
    {
        double value = Double.NaN;
        String item = null;
        if (isOnline()) {
            int type = ch.getType();
            int hwChan = ch.getHwChan();
            try {
                switch (type) {
                case MON_TYPE_VOLTAGE:
                    item = "voltage";
                    value = pduHV.readVoltage(hwChan);
                    break;

                case MON_TYPE_CURRENT:
                    item = "current";
                    value = pduHV.readCurrent(hwChan);
                    break;

                case MON_TYPE_MAIN_VOLTAGE:
                    item = "main voltage";
                    value = pduHV.readMainVoltage();
                    break;

                case MON_TYPE_MAIN_CURRENT:
                    item = "main current";
                    value = pduHV.readMainCurrent();
                    break;

                case MON_TYPE_MAIN_TEMP:
                    item = "main temperature";
                    value = pduHV.readMainTemperature();
                    break;

                case MON_TYPE_BOARD_TEMP:
                    item = "board temperature";
                    value = pduHV.readBoardTemperature();
                    break;

                case MON_TYPE_FPGA_TEMP:
                    item = "FPGA temperature";
                    value = pduHV.readFpgaTemperature();
                    break;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0} {1}: {2}", new Object[]{fullName, item, e});
                setOnline(false);
            }
        }
        return value;
    }

}
