package org.lsst.ccs.subsystem.power;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.actions.RebPowerAction;

/**
 * This class manages the emergency response actions based on Alert payloads.
 *
 * It uses the AgentPresenceManager to detect which Agents on the buses might be
 * sending Alert payloads to trigger emergency responses. This is done by adding
 * the payload fully qualified class name as part of the Agent properties.
 *
 *
 * @author The LSST CCS Team
 */
public class EmergencyResponseManager implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(EmergencyResponseManager.class.getName());

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    //The list of all the Rebs
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String, RebPowerSupplyNode> rebs = new HashMap<>();

    @ConfigurationParameter(description = "Only cleared alerts are processed.")
    private volatile boolean processOnlyClearedAlerts = true;
    
    private OriginManager originManager;
    private AlertPayloadListener alertPayloadListener = new AlertPayloadListener();

    private final Map<String, RebEmergencyResponseDelegate> rebEmergencyResponseDelegates = new ConcurrentHashMap<>();

    @Override
    public void init() {
        alertService.startStatusAlertListening((info) -> {
            return info.isAgentWorkerOrService();
        });

        //Create the originManager
        originManager = new OriginManager(agent.getName());

        //Fill the map of RebEmergencyResponse delegates
        for (Entry<String, RebPowerSupplyNode> e : rebs.entrySet()) {
            rebEmergencyResponseDelegates.put(e.getKey(), new RebEmergencyResponseDelegate(e.getValue()));
        }
        
        //Add the oringManager to the AgentPresenceManager
        agent.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(originManager);
        
    }

    @Override
    public void start() {

        //Alert listener responsible to listen for Alert emergency payloads
        //published by the origins maintained by the OriginManager
        alertService.addListener(alertPayloadListener);

    }

    @Override
    public void shutdown() {
        //Remove the oringManager from the AgentPresenceManager
        agent.getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(originManager);
        
        alertService.removeListener(alertPayloadListener);
    }

    private class RebEmergencyResponseDelegate {

        private final RebPowerSupplyNode rebNode;

        //Map of outstanding alerts that caused actions
        private final Map<RebPowerAction.Type, List<String>> outstandingAlerts = new ConcurrentHashMap<>();

        public RebEmergencyResponseDelegate(RebPowerSupplyNode rebNode) {
            this.rebNode = rebNode;
        }

        synchronized void processEmergencyResponse(String origin, Alert alert, RebPowerAction rebPowerAction) {
            addAlertCause(origin, alert, rebPowerAction);
            rebNode.processEmergencyResponse(origin, alert, rebPowerAction);
        }

        private void addAlertCause(String origin, Alert alert, RebPowerAction action) {
            List<String> alertIds = outstandingAlerts.getOrDefault(action.getType(), new ArrayList());
            String internalId = origin + "/" + alert.getAlertId();
            if (!alertIds.contains(internalId) ) {
                alertIds.add(internalId);
            }
            outstandingAlerts.put(action.getType(), alertIds);
            rebNode.updateEmergencyState(action.getType(), alertIds);
        }

        void processAlert(String origin, String alertId, AlertState state) {
            for (Map.Entry<RebPowerAction.Type, List<String>> e : outstandingAlerts.entrySet()) {
                List<String> alertIds = e.getValue();
                if (alertIds.remove(origin + "/" + alertId)) {
                    if (state == AlertState.ALARM) {
                        LOG.log(Level.SEVERE, "Something went really wrong in processing Reb Alerts.");
                    }
                    rebNode.updateEmergencyState(e.getKey(), alertIds);
                }
            }
        }
        
        void processDisconnection(String origin) {
            //Clear the alerts for the disconnected agent
            //Have to scan all outstanding alerts to check for their origin   
            for (Map.Entry<RebPowerAction.Type, List<String>> e : outstandingAlerts.entrySet()) {
                List<String> alertIds = e.getValue();
                Iterator<String> iter = alertIds.iterator();
                while ( iter.hasNext() ) {
                    String id = iter.next();
                    if (id.startsWith(origin + "/")) {
                        iter.remove();
                    }
                }
                rebNode.updateEmergencyState(e.getKey(), alertIds);
            }
        }
        
    }

    private class AlertPayloadListener implements AlertListener {

        //For each agent in the list of origins we have to first process
        //an Alert summary to make sure we process all existing alerts.
        //After the original initialization we can listen only to raised
        //alerts
        private final List<String> initialized = new CopyOnWriteArrayList<>();

        @Override
        public void onAlert(AlertEvent event) {
            if (originManager.getOrigins().contains(event.getSource())) {

                Alert alert = event.getAlert();
                AlertEvent.AlertEventType type = event.getType();
                String origin = event.getSource();

                if (null != type) {
                    synchronized(initialized) {
                        if (type != AlertEvent.AlertEventType.AGENT_DISCONNECTION && !initialized.contains(origin)) {
                            //Process initial summary
                            initialized.add(origin);
                            processAlertSummary(event.getSummary(), origin);
                        } else {
                            switch (type) {
                                case ALERT_RAISED:
                                    processAlert(origin, alert, event.getLevel());
                                    break;
                                case ALERT_CLEARED:
                                    processClearedAlerts(origin, event);
                                    break;
                                case AGENT_DISCONNECTION:
                                    processAgentDisconnection(origin);
                                    break;
                            }
                        }
                    }
                }
            }

        }

        private void processClearedAlerts(String origin, AlertEvent event) {
            List<String> clearedIds = event.getClearedIds();
            for (RebEmergencyResponseDelegate delegate : rebEmergencyResponseDelegates.values()) {
                for (String clearedId : clearedIds) {
                    delegate.processAlert(origin, clearedId, event.getStateForClearedAlert(clearedId));                    
                }
            }
            
        }
        
        private void processAlertsAfterDisconnection(String origin) {
            for (RebEmergencyResponseDelegate delegate : rebEmergencyResponseDelegates.values()) {
                delegate.processDisconnection(origin);
            }
            
        }

        private void processAlert(String origin, Alert alert, AlertState state) {
            //Check if the Alert contains a RebPowerAction
            RebPowerAction rebPowerAction = RebPowerAction.getRebPowerActionPayloadFromAlert(alert);
            if (rebPowerAction != null) {
                String rebPath = rebPowerAction.getRebPath();
                RebPowerSupplyNode rebNode = rebs.get(rebPath);
                if (rebNode == null) {
                    LOG.log(Level.SEVERE, "Could not find RebNode for path {0}", new Object[]{rebPath});
                } else {
                    RebEmergencyResponseDelegate delegate = rebEmergencyResponseDelegates.get(rebPath);
                    delegate.processEmergencyResponse(origin, alert, rebPowerAction);
                }
            } else {
                //Check if we only process alerts when they are cleared or also
                //when there is a state change.
                if ( ! processOnlyClearedAlerts ) {
                    //There is no action to carry out, so we need to see if this Alert
                    //is causing any Reb to have outstanding actions.
                    for (RebEmergencyResponseDelegate delegate : rebEmergencyResponseDelegates.values()) {
                        delegate.processAlert(origin, alert.getAlertId(), state);
                    }
                }
            }
        }

        void processAlertSummary(RaisedAlertSummary summary, String origin) {
            for (RaisedAlertHistory alertHistory : summary.getAllRaisedAlertHistories()) {
                if ( ! processOnlyClearedAlerts ) {
                    processAlert(origin, alertHistory.getLatestAlert(), alertHistory.getLatestAlertState());                    
                } else {
                    processAlert(origin, alertHistory.getHighestAlert(), alertHistory.getHighestAlertState());
                }
            }

        }
        
        void processAgentDisconnection(String origin) {
            initialized.remove(origin);
            processAlertsAfterDisconnection(origin);            
        }
    }

    private class OriginManager implements AgentPresenceListener {

        //This list contains the origins to listen to when processing Alerts.
        //It contains by default the name of the Agent in which this EmergencyReponseManager
        //is created and any external agent that specifies so in the Agent properties.
        private final List<String> origins = new CopyOnWriteArrayList<>();

        OriginManager(String agentName) {
            origins.add(agentName);
        }

        @Override
        public void connecting(AgentInfo... agents) {
            for (AgentInfo ai : agents) {
                if (ai.hasAgentProperty(RebPowerAction.class.getCanonicalName())) {
                    LOG.log(Level.INFO, "Adding agent {0} to list of origins for payload {1}", new Object[]{ai.getName(), RebPowerAction.class.getCanonicalName()});
                    origins.add(ai.getName());
                    RaisedAlertSummary s = alertService.getAllSummaries().get(ai.getName());
                    if ( s != null ) {
                        alertPayloadListener.processAlertSummary(s,ai.getName());
                    }
                }
            }
        }

        @Override
        public void disconnected(AgentInfo... agents) {
            for (AgentInfo ai : agents) {
                if (origins.remove(ai.getName())) {
                    LOG.log(Level.INFO, "Removing agent {0} from the list of origins for payload {1}", new Object[]{ai.getName(), RebPowerAction.class.getCanonicalName()});
                    alertPayloadListener.processAgentDisconnection(ai.getName());                
                }
            }
        }

        List<String> getOrigins() {
            return origins;
        }
    }

}
