package org.lsst.ccs.subsystem.power;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.actions.BulkPsPowerAction;
import org.lsst.ccs.subsystem.power.alerts.RebPowerAlertType;
import org.lsst.ccs.subsystem.power.data.PowerException;

/**
 * This class manages the emergency response actions based on Alert payloads.
 *
 * It uses the AgentPresenceManager to detect which Agents on the buses might be
 * sending Alert payloads to trigger emergency responses. This is done by adding
 * the payload fully qualified class name as part of the Agent properties.
 *
 *
 * @author The LSST CCS Team
 */
public class BulkPSEmergencyResponseManager implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(BulkPSEmergencyResponseManager.class.getName());

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    //The list of all the Rebs
    @LookupField(strategy = LookupField.Strategy.TREE)
    private QuadBox quadbox;

    private OriginManager originManager;
    private AlertPayloadListener alertPayloadListener = new AlertPayloadListener();

    private QuadboxEmergencyResponseDelegate quadboxEmergencyResponseDelegate;

    @Override
    public void build() {
        if ( quadbox == null ) {
            throw new RuntimeException("Could not find an instance of QuadBox in the tree.");
        }
    }

    @Override
    public void init() {
        alertService.startStatusAlertListening((info) -> {
            return info.isAgentWorkerOrService();
        });

        //Create the originManager
        originManager = new OriginManager(agent.getName());

        quadboxEmergencyResponseDelegate = new QuadboxEmergencyResponseDelegate(quadbox);
        
        //Add the oringManager to the AgentPresenceManager
        agent.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(originManager);
        
        //A ClearAlertHandler that always clears all raised Alerts
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
          
        };        
        
        //Register the raised alerts for the bulk PS switches
        for ( String switchName : quadbox.getSwitchNames() ) {
            alertService.registerAlert(RebPowerAlarm.getRebPowerAlert(RebPowerAlertType.EMERGENCY_ACTION_FAILED, switchName), alwaysClear);
        }
    }

    @Override
    public void start() {

        //Alert listener responsible to listen for Alert emergency payloads
        //published by the origins maintained by the OriginManager
        alertService.addListener(alertPayloadListener);

    }

    @Override
    public void shutdown() {
        //Remove the oringManager from the AgentPresenceManager
        agent.getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(originManager);
        
        alertService.removeListener(alertPayloadListener);
    }

    private class QuadboxEmergencyResponseDelegate {

        private final QuadBox quadbox;

        public QuadboxEmergencyResponseDelegate(QuadBox quadbox) {
            this.quadbox = quadbox;
        }
        
        synchronized void processEmergencyResponse(String origin, Alert alert, BulkPsPowerAction bulkPsSwitchAction) {
            if (bulkPsSwitchAction != null) {
                String switchName = bulkPsSwitchAction.getBulkPsSwitchName();
                if (!quadbox.getSwitchNames().contains(switchName)) {
                    LOG.log(Level.SEVERE, "Switch name {0} does not exist. The emergency action {1} could not be performed.", new Object[]{switchName, bulkPsSwitchAction.getType()});
                } else {
                    agent.getScheduler().execute(() -> {
                        //If there is an action to be carried out, do it right away.
                        switch (bulkPsSwitchAction.getType()) {
                            case TURN_BULK_PS_OFF:
                                LOG.log(Level.WARNING, "Turning OFF switch {0}", switchName);
                                try {
                                    quadbox.setSwitchOn(switchName, false);
                                } catch (PowerException pe) {
                                    LOG.log(Level.SEVERE, "Could not turn OFF switch " + switchName, pe);
                                    alertService.raiseAlert(RebPowerAlarm.getRebPowerAlert(RebPowerAlertType.EMERGENCY_ACTION_FAILED, switchName), AlertState.ALARM, "Could not turn OFF switch " + switchName);
                                }
                                break;
                            default:
                                LOG.log(Level.SEVERE, "Unknown Bulk PS Action {0} for switch {1}", new Object[]{bulkPsSwitchAction.getType(), switchName});
                        }
                    });
                }
            }

        }
    }


    private class AlertPayloadListener implements AlertListener {

        //For each agent in the list of origins we have to first process
        //an Alert summary to make sure we process all existing alerts.
        //After the original initialization we can listen only to raised
        //alerts
        private final List<String> initialized = new CopyOnWriteArrayList<>();

        @Override
        public void onAlert(AlertEvent event) {
            if (originManager.getOrigins().contains(event.getSource())) {

                Alert alert = event.getAlert();
                AlertEvent.AlertEventType type = event.getType();
                String origin = event.getSource();

                if (null != type) {
                    synchronized(initialized) {
                        if (type != AlertEvent.AlertEventType.AGENT_DISCONNECTION && !initialized.contains(origin)) {
                            //Process initial summary
                            initialized.add(origin);
                            processAlertSummary(event.getSummary(), origin);
                        } else {
                            switch (type) {
                                case ALERT_RAISED:
                                    processAlert(origin, alert, event.getLevel());
                                    break;
                            }
                        }
                    }
                }
            }

        }

        private void processAlert(String origin, Alert alert, AlertState state) {
            //Check if the Alert contains a BulkPsPowerAction
            BulkPsPowerAction rebPowerAction = BulkPsPowerAction.getBulkPsPowerActionPayloadFromAlert(alert);
            if (rebPowerAction != null) {
                quadboxEmergencyResponseDelegate.processEmergencyResponse(origin, alert, rebPowerAction);
            } 
        }

        void processAlertSummary(RaisedAlertSummary summary, String origin) {
            for (RaisedAlertHistory alertHistory : summary.getAllRaisedAlertHistories()) {
                processAlert(origin, alertHistory.getLatestAlert(), alertHistory.getLatestAlertState());                    
            }
        }
        
    }

    private class OriginManager implements AgentPresenceListener {

        //This list contains the origins to listen to when processing Alerts.
        //It contains by default the name of the Agent in which this EmergencyReponseManager
        //is created and any external agent that specifies so in the Agent properties.
        private final List<String> origins = new CopyOnWriteArrayList<>();

        OriginManager(String agentName) {
            origins.add(agentName);
        }

        @Override
        public void connecting(AgentInfo... agents) {
            for (AgentInfo ai : agents) {
                if (ai.hasAgentProperty(BulkPsPowerAction.class.getCanonicalName())) {
                    LOG.log(Level.INFO, "Adding agent {0} to list of origins for payload {1}", new Object[]{ai.getName(), BulkPsPowerAction.class.getCanonicalName()});
                    origins.add(ai.getName());
                    RaisedAlertSummary s = alertService.getAllSummaries().get(ai.getName());
                    if ( s != null ) {
                        alertPayloadListener.processAlertSummary(s,ai.getName());
                    }
                }
            }
        }

        List<String> getOrigins() {
            return origins;
        }
    }

}
