package org.lsst.ccs.subsystem.power;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.power.constants.MonitorControl;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;
import org.lsst.ccs.subsystem.power.constants.QuadBoxSwitches;
import org.lsst.ccs.subsystem.power.constants.QuadboxAlerts;
import org.lsst.ccs.subsystem.power.constants.SwitchState;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.data.QuadBoxState;
import org.lsst.ccs.subsystem.common.switches.SwitchesOnOffCustomLevels;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;

/**
 *  Implements the Quad Box control subsystem.
 *
 *  @author Owen Saxton
 */
public class QuadBox extends Subsystem implements HasLifecycle {

    static class DummyDevice implements SwitchControl {
        @Override
        public int getSwitchDevice() { return 0; }
        @Override
        public Boolean isSwitchOn(int sw) { return null; }
        @Override
        public void switchOn(int sw) {}
        @Override
        public void switchOff(int sw) {}
    }

    private static final Map<Integer, String> typeMap = new HashMap<>();
    static {
        typeMap.put(QuadBoxSwitches.DEVC_BFR, "BFR");
        typeMap.put(QuadBoxSwitches.DEVC_PDU_5V, "5V PDU");
        typeMap.put(QuadBoxSwitches.DEVC_PDU_24VC, "24V clean PDU");
        typeMap.put(QuadBoxSwitches.DEVC_PDU_24VD, "24V dirty PDU");
        typeMap.put(QuadBoxSwitches.DEVC_PDU_48V, "48V PDU");
        typeMap.put(QuadBoxSwitches.DEVC_REB_BULK, "REB bulk PS");
    }


    @ConfigurationParameter(isFinal = true, description = "Switches Custom Levels", units = "unitless")
    SwitchesOnOffCustomLevels switchesCustomLevels = new SwitchesOnOffCustomLevels();

    
    private static final Logger LOG = Logger.getLogger(QuadBox.class.getName());
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService aps;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService apts;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<SwitchControl> swDeviceList = new ArrayList<>();

    private final SwitchControl[] swDevices = new SwitchControl[QuadBoxSwitches.NUM_DEVICES];
    private final QuadBoxState qbState = new QuadBoxState();
    private MonitorTaskControl monitorControl;


    /**
     *  Constructor.
     */
    public QuadBox() {
        super("quadbox", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, MonitorControl.NODE_NAME);

        // Create and schedule an AgentPeriodicTask to update and publish the utility trunk system state
        AgentPeriodicTask pt = new AgentPeriodicTask("quadbox-state", () -> updateState()).withPeriod(Duration.ofMillis(1000));
        apts.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Registers the possible alerts.
     */
    @Override
    public void init()
    {
        for (String swName : QuadBoxSwitches.getSwitchNames()) {
            alertService.registerAlert(QuadboxAlerts.PDU_ALERT.newAlert(swName));                        
        }
    }
    

    /**
     *  Subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a Quad Box controller.
        aps.setAgentProperty(PowerAgentProperties.QUAD_BOX_AGENT, getClass().getCanonicalName());

        // Checks power devices and control channels
        boolean error = false;
        for (SwitchControl sd : swDeviceList) {
            swDevices[sd.getSwitchDevice()] = sd;
        }
        for (int j = 0; j < swDevices.length; j++) {
            if (swDevices[j] == null) {
                LOG.log(Level.SEVERE, "The {0} device has not been specified", typeMap.get(j));
                //swDevices[j] = new DummyDevice();
                error = true;
            }
        }
        if (error) {
            throw new RuntimeException("Fatal initialization error");
        }
    }


    /**
     *  Starts the subsystem.
     *
     */
    @Override
    public void postStart()
    {
        publishState();        // For any GUIs
        LOG.info("Quad Box subsystem started");
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  name  The switch name.
     *  @param  on  Whether to turn on or off
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch", autoAck=false)
    public void setSwitchOn(@Argument(description="The switch name") String switchName,
                            @Argument(description="Whether to turn on") boolean isOn) throws PowerException
    {

        Integer sw = QuadBoxSwitches.getSwitchId(switchName);
        if (sw == null) {
            throw new PowerException("Invalid switch name: " + switchName);
        }
        int devNum = sw >> 8;
        int swNum = (byte)(int)sw;  // Sign extend byte value
        if (devNum < 0 || devNum >= QuadBoxSwitches.NUM_DEVICES || swNum >= QuadBoxSwitches.NUM_SWITCHES[devNum]) {
            throw new PowerException("Programming error!! Invalid switch number: " + sw);
        }

        helper().precondition(switchesCustomLevels.isCommandRequestCompatibleWithCustomLevel(this, switchName, isOn),
			      "Insufficient lock level for turning the switch %s. Required custom level is: %d",
			      isOn?"On":"Off",
			      switchesCustomLevels.getCustomLevelForSwitch(switchName,isOn)).action(()-> {

				      //	     gotCommand = true;

	     if (isOn) {
		 swDevices[devNum].switchOn(swNum);
	     }
	     else {
		 swDevices[devNum].switchOff(swNum);
	     }


				  });


	
    }


    /**
     *  Gets the list of switch names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of switch names")
    public List<String> getSwitchNames()
    {
        return QuadBoxSwitches.getSwitchNames();
    }


    /**
     *  Gets the full quad box state.
     *
     *  @return  The full quad box state
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public QuadBoxState getFullState()
    {
        qbState.setTickMillis(monitorControl.getFastPeriod());
        return qbState;
    }


    /**
     *  Updates the quad box state.
     */
    private void updateState()
    {
        boolean changed = monitorControl.hasPeriodChanged();
        for (int devNum = 0; devNum < QuadBoxSwitches.NUM_DEVICES; devNum++) {
            for (int swNum = -1; swNum < QuadBoxSwitches.NUM_SWITCHES[devNum]; swNum++) {
                Boolean swOn = swDevices[devNum].isSwitchOn(swNum);
                Boolean alOn = swDevices[devNum].hasAlert(swNum);
                SwitchState ss = (swOn == null || alOn == null) ? SwitchState.OFFLINE :
                                  alOn ? SwitchState.ALERT : swOn ? SwitchState.ON : SwitchState.OFF;
                SwitchState ssCurr = qbState.getSwitchState(devNum, swNum);
                if (ss != ssCurr) {
                    qbState.setSwitchState(devNum, swNum, ss);
                    changed = true;
                    String swName = QuadBoxSwitches.getSwitchName(devNum, swNum);
                    if (ss == SwitchState.ALERT) {
                        alertService.raiseAlert(QuadboxAlerts.PDU_ALERT.newAlert(swName),
                                                AlertState.ALARM, "PDU alert condition set for " + swName + " channel");
                    }
                    else if (ssCurr == SwitchState.ALERT) {
                        alertService.raiseAlert(QuadboxAlerts.PDU_ALERT.newAlert(swName),
                                                AlertState.NOMINAL, "PDU alert condition cleared for " + swName + " channel");
                    }
                }
            }
        }
        if (changed) {
            publishState();
        }
    }


    /**
     *  Publishes the state of the quad box system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    @Command(type=CommandType.QUERY, level = 0, name = "publishFullState", description="Publish the full state")
    public void publishState()
    {
        KeyValueData kvd = new KeyValueData(QuadBoxState.KEY, getFullState());
        publishSubsystemDataOnStatusBus(kvd);
    }    

}
