package org.lsst.ccs.subsystem.power;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.actions.RebPowerAction;
import org.lsst.ccs.subsystem.power.alerts.RebPowerAlertType;
import org.lsst.ccs.subsystem.power.constants.RebPsEnum;
import org.lsst.ccs.subsystem.power.data.PowerException;
import org.lsst.ccs.subsystem.power.states.RebDPhiState;
import org.lsst.ccs.subsystem.power.states.RebHvBiasState;
import org.lsst.ccs.subsystem.power.states.RebPowerState;
import org.lsst.ccs.subsystem.power.states.RebHvControlState;

/**
 * A class representing an individual Reb.
 * It contains the information regarding the power supply powering this reb.
 * It provides commands to control the power to this reb.
 *
 * @author The LSST CCS Team
 */
public class RebPowerSupplyNode implements HasLifecycle {

    protected final PowerSupplyConfiguration powerSupplyConfiguration;
    protected final int channel;
    protected static final Logger LOG = Logger.getLogger(RebPowerSupplyNode.class.getName());

    private double post_hvset_delay = 0.05;
    
    @LookupName
    protected String rebName;

    @LookupPath
    protected String rebPath;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String, RebPsDevice> psDevices = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.TOP)
    protected Agent agent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService agentStateService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter = "Power")
    private Channel powerChannel;

    @ConfigurationParameter(name="hvBiasSetPoint", category="HVRegulation", range = "0..54.0", units = "Volt", description = "REB bias voltage")
    protected volatile double hvBiasSetPoint;

    @ConfigurationParameter(name="maxDac", category="HVRegulation", range = "0..3300", units = "unitless", description = "REB maximum allowed")
    protected volatile int maxDac = 0;   // intended to be ~dac@setpoint + 2/gain, temp dependent

    @ConfigurationParameter(name="firstDac", category="HVRegulation", range = "0..1600", units = "unitless", description = "REB first dac step")
    protected volatile int firstDac = 1200;

    @ConfigurationParameter(name="useVoltsPerStepCal", category="HVRegulation", units = "unitless", description = "Use VoltsPerStepCal: true|false")
    protected volatile boolean useVoltsPerStepCal = false;

    @ConfigurationParameter(name="voltsPerStepCal", category="HVRegulation", units = "unitless", description = "slope of volts vs. DAC counts at setpoint")
    protected volatile double voltsPerStepCal = 0.080;  // this is the median gain at the set point, range is 0.03..0.150

    @ConfigurationParameter(name="reff", category="HVRegulation", units = "unitless", description = "slope of V = reff * I * 1000")
    protected volatile double reff = 645.0;      // current behaves as I (mA) = V / reff;


    protected RebPsDevice powerSupplyDevice;

    private volatile boolean disableRebPowerOn = false;
    private volatile boolean disableRebHvBiasOn = false;

    //Lock used to prefent interference between the reb power On/Off sequences.
    private final Object powerOnOffLock = new Object();

    //Lock used to prefent interference between the hvBias On/Off sequences.
    private final Object hvBiasOnOffLock = new Object();

    private final Map<RebPowerAction.Type, List<String>> outstandingAlerts = new ConcurrentHashMap<>();
    
    public RebPowerSupplyNode(PowerSupplyConfiguration powerSupplyConfiguration, int channel) {
        this.channel = channel;
        this.powerSupplyConfiguration = powerSupplyConfiguration;
    }

    @Override
    public void build() {
        String psName = powerSupplyConfiguration.getName();
        for (Entry<String,RebPsDevice> e : psDevices.entrySet()) {
            if ( e.getKey().endsWith(psName)) {
                if ( powerSupplyDevice != null ) {
                    throw new RuntimeException("Found more than one power supply that matches name "+powerSupplyConfiguration.getName());                
                }
                powerSupplyDevice = e.getValue();
            }
        }
        if ( powerSupplyDevice == null ) {
            throw new RuntimeException("Could not find power supply device "+powerSupplyConfiguration.getName()+" corresponding to this reb "+rebName);
        }
    }

    @Override
    public void init() {
        //A ClearAlertHandler that always clears all raised Alerts
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
          
        };
        //Register Alerts
        alertService.registerAlert(RebPowerAlarm.getRebPowerAlert(RebPowerAlertType.EMERGENCY_ACTION_FAILED, rebPath), alwaysClear);

        // TODO resolve this state w/respect to restarts and perhaps use persistency
        agentStateService.updateAgentComponentState(this, RebHvControlState.ALLOWED);

    }

    // user command for setting hv bias dac
    @Command(type = Command.CommandType.ACTION, description = "sets the RebPS HV DAC", autoAck = false)
    public void setHvBiasDac(int hvBiasDac) throws PowerException {
        setHvBiasDac(hvBiasDac, false);
    }
    
    public void setHvBiasDac(int hvBiasDac, boolean fromRegulator) throws PowerException {
        // When called "fromRegulator" the command must be in the ALLOWED state
        // When called by anything else, the command must be in the BLOCKED state
        synchronized (agentStateService.getStateLock()) {
            if (agentStateService.isComponentInState(rebPath, RebPowerState.ON)) {

                if ((agentStateService.isComponentInState(rebPath, RebHvControlState.ALLOWED) && fromRegulator) ||
                            (agentStateService.isComponentInState(rebPath, RebHvControlState.BLOCKED) && !fromRegulator))  {

                    powerSupplyDevice.setBiasDac(channel, hvBiasDac);
                } else {
                    if (fromRegulator) {
                        throw new PowerException("RebHvControlState must be ALLOWED on channel "+rebPath);
                    } else {
                        throw new PowerException("RebHvControlState must be BLOCKED on channel "+rebPath);
                    }
                }
            }
        }
    }


    public double getHvBiasSetPoint() {
        return hvBiasSetPoint;
    }

    public int getMaxDac() {
        return maxDac;
    }

    public int getFirstDac() {
        return firstDac;
    }

    public double getReff() {
        return reff;
    }

    public double getVoltsPerStepCal() {
        return voltsPerStepCal;
    }

    public boolean getUseVoltsPerStepCal() {
        return useVoltsPerStepCal;
    }

    @Command(type = Command.CommandType.ACTION, description = "Turns on the power to this Reb", autoAck = false)
    public void powerRebOn() throws PowerException {
        synchronized (powerOnOffLock) {
            agent.helper()
                .precondition(!disableRebPowerOn, rebPath + 
                        ": powering on has been disabled due to outstanding alerts: " +
                        outstandingAlerts.get(RebPowerAction.Type.TURN_REB_OFF))

                .precondition(agentStateService.isComponentInState(rebPath, RebHvBiasState.OFF), 
                        "The Reb hvBias switch must be open.")

                .precondition((this instanceof CornerRaftRebPowerSupplyNode) ? 
                        agentStateService.isComponentInState(rebPath, RebDPhiState.OFF) : true, 
                        "The Reb dphi switch must be open.")

                .enterFaultOnException(true)
                .action(() -> {
                        powerSupplyDevice.sequencePower(channel, true);
                    });
        }
    }

    @Command(type = Command.CommandType.ACTION, description = "Turns off the power to this Reb")
    public void powerRebOff() throws PowerException {
        powerRebOff(false);
    }

    private void powerRebOff(boolean disable) throws PowerException {
        synchronized(powerOnOffLock) {
            disableRebPowerOn = disable;
            powerSupplyDevice.setBiasDac(channel, 0);
            powerSupplyDevice.sequencePower(channel, false);
            if (disable) {
                agentStateService.updateAgentComponentState(this, RebHvControlState.BLOCKED);
            }                       
        }
    }
    
    @Command(type = Command.CommandType.ACTION, description = "Turns on the HV bias switch, DAC must be 0", autoAck = false)
    public void hvBiasOn() throws PowerException {
        synchronized (hvBiasOnOffLock) {
            agent.helper()
                .precondition(agentStateService.isComponentInState(rebPath, RebPowerState.ON), 
                        "The Reb main power must be ON.")

                .precondition(!disableRebHvBiasOn,
                        rebPath + ": HvBias is disabled manually or due to outstanding alerts: " + 
                        outstandingAlerts.get(RebPowerAction.Type.TURN_REB_HVBIAS_OFF))

                .precondition((int) powerSupplyDevice.getDacs(true)[channel] == 0, "DAC must be 0")

                .enterFaultOnException(true).action(
                    () -> {
                        powerSupplyDevice.setPowerOn(channel, RebPsEnum.HVBIAS.getNumber(), true);
                    });
        }
    }

    
    @Command(type=Command.CommandType.ACTION, description="Opens the HV bias switch")
    public void hvBiasOff() throws PowerException {
        hvBiasOff(false);
    }

    protected void hvBiasOff(boolean disable) throws PowerException {
        synchronized(hvBiasOnOffLock) {
            disableRebHvBiasOn = disable;
            powerSupplyDevice.setPowerOn(channel, RebPsEnum.HVBIAS.getNumber(), false);
            if (disable) {
                agentStateService.updateAgentComponentState(this, RebHvControlState.BLOCKED);
                setHvBiasDac(0);
            }
        }
    }
    
    @Command(type=Command.CommandType.ACTION, description="sets whether HV bias regulation is allowed for this channel")
    public void allowHvControl(boolean allow)  {
        if (allow) {
            agentStateService.updateAgentComponentState(this, RebHvControlState.ALLOWED);
        } else {
            agentStateService.updateAgentComponentState(this, RebHvControlState.BLOCKED);
        }
    }

    
    @Command(type=Command.CommandType.QUERY, description="Is the REB HV bias regulation allowed?")
    public boolean isRebHvControlAllowed() {
        return(agentStateService.isComponentInState(rebPath, RebHvControlState.ALLOWED));
    }


    @Command(type = Command.CommandType.ACTION, description = "Read the HvBias DAC value")
    public double readHvBiasDac() throws PowerException {
        return powerSupplyDevice.getDacs(true)[channel];
    }

    /**
     *  Reads the maximum and minimum values of a channel.
     *
     *  @param  psName  The power supply enum
     *  @param  chan    The channel enum
     *  @return  A two-element array containing maximum (0) & minimum (0) values
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.QUERY, description="Read a channel's maximum & minimum values")
    public double[] getChanExtrema(@Argument(description="PS name")
                                   RebPsDevice.PsName psName,
                                   @Argument(description="Channel name")
                                   RebPsDevice.ChanName chan) throws PowerException {
        return powerSupplyDevice.readChanExtrema(channel, psName, chan);
    }


    /**
     *  Resets the maximum and minimum values of a channel.
     *
     *  @param  psName  The power supply enum
     *  @param  chan    The channel enum
     *  @throws  PowerException
     */
    @Command(type=Command.CommandType.ACTION, description="Reset a channel's maximum & minimum values")
    public void resetChanExtrema(@Argument(description="PS name")
                                 RebPsDevice.PsName psName,
                                 @Argument(description="Channel name")
                                 RebPsDevice.ChanName chan) throws PowerException {
        powerSupplyDevice.resetChanExtrema(channel, psName, chan);
    }

    RebPsDevice getRebPsDevice() {
        return powerSupplyDevice;
    }

    Channel getPowerChannel() {
        return powerChannel;
    }

    synchronized void processEmergencyResponse(String origin, Alert alert, RebPowerAction rebPowerAction) {
        if (rebPowerAction != null) {
            //If there is an action process it on a separate thread.
            if (rebPath.equals(rebPowerAction.getRebPath())) {
                agent.getScheduler().execute(() -> {
                    //If there is an action to be carried out, do it right away.
                    switch (rebPowerAction.getType()) {
                        case TURN_REB_HVBIAS_OFF:
                            LOG.log(Level.WARNING, "Turning HvBias OFF for reb {0}", rebPath);
                            try {
                                hvBiasOff(true);
                            } catch (PowerException pe) {
                                LOG.log(Level.SEVERE, "Could not turn HvBias OFF for reb " + rebPath, pe);
                                alertService.raiseAlert(
                                        RebPowerAlarm.getRebPowerAlert(RebPowerAlertType.EMERGENCY_ACTION_FAILED, rebPath),
                                        AlertState.ALARM, "Could not turn HvBias OFF for reb " + rebPath);
                            }
                            break;
                        case TURN_REB_OFF:
                            LOG.log(Level.WARNING, "Turning Power OFF for reb {0}", rebPath);
                            try {
                                powerRebOff(true);
                            } catch (PowerException pe) {
                                LOG.log(Level.SEVERE, "Could not turn Power OFF for reb " + rebPath, pe);
                                alertService.raiseAlert(
                                        RebPowerAlarm.getRebPowerAlert(RebPowerAlertType.EMERGENCY_ACTION_FAILED, rebPath),
                                        AlertState.ALARM, "Could not turn Power OFF for reb " + rebPath);
                            }
                            break;
                        default:
                            LOG.log(Level.SEVERE,
                                    "Unknown Reb Power Action {0} for Reb path {1}", new Object[]{rebPowerAction.getType(), rebPath});
                    }
                });
            }
        } 
    }
    
    void updateEmergencyState(RebPowerAction.Type actionType, List<String> outstandingAlerts) {
        this.outstandingAlerts.put(actionType,outstandingAlerts);
        if (outstandingAlerts.isEmpty()) {
            switch (actionType) {
                case TURN_REB_HVBIAS_OFF:
                    LOG.log(Level.INFO, "Enabling hvBiasOn command for {0}", rebPath);
                    disableRebHvBiasOn = false;
                    break;
                case TURN_REB_OFF:
                    LOG.log(Level.INFO, "Enabling powerRebOn command for {0}", rebPath);
                    disableRebPowerOn = false;
                    break;
            }
        }
    }
 

}
