package org.lsst.ccs.subsystem.power;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.auxelex.Bfr;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.power.constants.PowerConfig;
import org.lsst.ccs.subsystem.power.constants.QuadBoxSwitches;

/**
 *  Interface to the SLAC BFR.
 *
 *  @author Owen Saxton
 */
public class BfrDevice extends Device implements SwitchControl {

    /**
     *  Constants.
     */
    private static final Logger LOG = Logger.getLogger(BfrDevice.class.getName());
    private static final double
        MIN_SPECIAL_VALUE = 327.52,
        LINE_VOLTAGE = 220.0;
    private static final String
        NODE = "node";
    private static final int
        TYPE_CURRENT = 0,
        TYPE_POWER = 1;

    /**
     *  Data fields.
     */
    @ConfigurationParameter(name=NODE, category=PowerConfig.CATEGORY_QUADBOX, isFinal=true, units="unitless", description = "The node board id")
    private volatile Integer node;

    private final Bfr bfr = new Bfr();
    private double currents[] = null;
    private double totalPower = 0.0;
    private boolean initError = false;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (node == null) {
            ErrorUtils.reportConfigError(LOG, name, NODE, "is missing");
        }
        fullName = "BFR board (" + node + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    public void initialize()
    {
        try {
            bfr.open(node);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            bfr.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}:{1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Checks a monitoring channel's parameters for validity.
     *
     *  @param  ch  The channel to check
     *  @return  Two-element array containing the encoded type [0] and subtype [1] 
     *  @throws  Exception  If parameters are invalid
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        int[] types = {0, 0};
        String typeStr = ch.getTypeStr();
        switch (typeStr.toUpperCase()) {
        case "CURRENT":
            types[0] = TYPE_CURRENT;
            break;
        case "POWER":
            types[0] = TYPE_POWER;
            break;
        default:
            ErrorUtils.reportChannelError(LOG, ch.getPath(), "channel type", typeStr);
        }
        int hwChan = ch.getHwChan();
        if (hwChan < 0 || hwChan >= Bfr.NUM_SENSORS) {
            ErrorUtils.reportChannelError(LOG, ch.getPath(), "hw channel number", hwChan);
        }
        return types;
    }


    /**
     *  Gets the group for a channel.
     *
     *  @param  ch  The channel
     *  @return  The group name, or null if not in a group
     */
    @Override
    protected String getGroupForChannel(Channel ch) {
        return "currents";
    }


    /**
     *  Reads all monitored channels.
     * 
     *  @param  group  The group to read
     */
    @Override
    protected void readChannelGroup(String group)
    {
        currents = null;
        totalPower = 0.0;
        if (isOnline()) {
            try {
                currents = bfr.readCurrent(Bfr.CurrentType.AC);
                for (int j = 0; j < currents.length; j++) {
                    if (currents[j] >= MIN_SPECIAL_VALUE) {
                        currents[j] = Double.NaN;
                    }
                    else {
                        totalPower += currents[j];
                    }
                }
                totalPower *= LINE_VOLTAGE;
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0} currents: {1}", new Object[]{fullName, e});
                setOnline(false);
            }
        }
    }


    /**
     *  Reads a monitored channel.
     *
     *  @param  ch  The channel to read
     *  @return  The read value
     */
    @Override
    protected double readChannel(Channel ch)
    {
        if (ch.getType() == TYPE_CURRENT) {
            return currents == null ? Double.NaN : currents[ch.getHwChan()];
        }
        else {
            return totalPower;
        }
    }


    /**
     *  Gets the switch device type.
     * 
     *  @return  The switch device type enumeration
     */
    @Override
    public int getSwitchDevice()
    {
        return QuadBoxSwitches.DEVC_BFR;
    }


    /**
     *  Turns a relay on.
     *
     *  @param  relay  The relay number
     */
    @Override
    public void switchOn(int relay)
    {
        try {
            bfr.setRelayOn(relay);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error setting relay {0} on {1}: {2}", new Object[]{relay, fullName, e});
        }
        
    }


    /**
     *  Turns a relay off.
     *
     *  @param  relay  The relay number
     */
    @Override
    public void switchOff(int relay)
    {
        try {
            bfr.setRelayOff(relay);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error setting relay {0} on {1}: {2}", new Object[]{relay, fullName, e});
        }
        
    }


    /**
     *  Gets the state of a relay.
     *
     *  @param  relay  The relay number
     *  @return  Whether the relay is on or off
     */
    @Override
    public Boolean isSwitchOn(int relay)
    {
        if (!isOnline() || relay < 0) return null;
        try {
            return bfr.isRelayOn(relay);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error getting state for relay {0} on {1}: {2}", new Object[]{relay, fullName, e});
            return null;
        }
    }

}
